/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @flow

import type {
  SourcePacket,
  PausedPacket,
  ThreadFront,
  Target,
  DevToolsClient,
} from "./types";

import Actions from "../../actions";

import { createPause, prepareSourcePayload } from "./create";
import sourceQueue from "../../utils/source-queue";
import { recordEvent } from "../../utils/telemetry";
import { prefs } from "../../utils/prefs";

type Dependencies = {
  actions: typeof Actions,
  devToolsClient: DevToolsClient,
};

let actions: typeof Actions;
let isInterrupted: boolean;
let threadFrontListeners: WeakMap<ThreadFront, Array<Function>>;

function addThreadEventListeners(thread: ThreadFront): void {
  const removeListeners = [];
  Object.keys(clientEvents).forEach(eventName => {
    // EventEmitter.on returns a function that removes the event listener.
    removeListeners.push(
      thread.on(eventName, clientEvents[eventName].bind(null, thread))
    );
  });
  threadFrontListeners.set(thread, removeListeners);
}

function removeThreadEventListeners(thread: ThreadFront): void {
  const removeListeners = threadFrontListeners.get(thread) || [];
  for (const removeListener of removeListeners) {
    removeListener();
  }
}

function attachAllTargets(currentTarget: Target): boolean {
  return prefs.fission && currentTarget.isParentProcess;
}

function setupEvents(dependencies: Dependencies): void {
  actions = dependencies.actions;
  sourceQueue.initialize(actions);

  threadFrontListeners = new WeakMap();
}

async function paused(
  threadFront: ThreadFront,
  packet: PausedPacket
): Promise<*> {
  // If paused by an explicit interrupt, which are generated by the
  // slow script dialog and internal events such as setting
  // breakpoints, ignore the event.
  const { why } = packet;
  if (why.type === "interrupted" && !packet.why.onNext) {
    isInterrupted = true;
    return;
  }

  // Ignore attached events because they are not useful to the user.
  if (why.type == "alreadyPaused" || why.type == "attached") {
    return;
  }

  if (packet.frame) {
    // When reloading we might receive a pause event before the
    // top frame's source has arrived.
    await actions.ensureSourceActor(
      threadFront.actorID,
      packet.frame.where.actor
    );
  }

  const pause = createPause(threadFront.actor, packet);

  actions.paused(pause);
  recordEvent("pause", { reason: why.type });
}

function resumed(threadFront: ThreadFront): void {
  // NOTE: the client suppresses resumed events while interrupted
  // to prevent unintentional behavior.
  // see [client docs](../README.md#interrupted) for more information.
  if (isInterrupted) {
    isInterrupted = false;
    return;
  }

  actions.resumed(threadFront.actorID);
}

function newSource(threadFront: ThreadFront, { source }: SourcePacket): void {
  sourceQueue.queue({
    type: "generated",
    data: prepareSourcePayload(threadFront, source),
  });
}

const clientEvents = {
  paused,
  resumed,
  newSource,
};

export {
  setupEvents,
  clientEvents,
  addThreadEventListeners,
  removeThreadEventListeners,
  attachAllTargets,
};
