/**
 * UI infrastructure code and utility code for the gaia email app.
 **/

var mozL10n = navigator.mozL10n;

// Dependcy handling for Cards
// We match the first section of each card type to the key
// E.g., setup-progress, would load the 'setup' lazyCards.setup
var lazyCards = {
    compose: ['js/compose-cards.js', 'style/compose-cards.css'],
    settings: ['js/setup-cards.js', 'style/setup-cards.css'],
    setup: ['js/setup-cards.js', 'style/setup-cards.css']
};

function dieOnFatalError(msg) {
  console.error('FATAL:', msg);
  throw new Error(msg);
}

var fldNodes, msgNodes, cmpNodes, supNodes, tngNodes;
function processTemplNodes(prefix) {
  var holder = document.getElementById('templ-' + prefix),
      nodes = {},
      node = holder.firstElementChild,
      reInvariant = new RegExp('^' + prefix + '-');
  while (node) {
    var classes = node.classList, found = false;
    for (var i = 0; i < classes.length; i++) {
      if (reInvariant.test(classes[i])) {
        var name = classes[i].substring(prefix.length + 1);
        nodes[name] = node;
        found = true;
        break;
      }
    }
    if (!found) {
      console.warn('Bad template node for prefix "' + prefix +
                   '" for node with classes:', classes);
    }

    node = node.nextElementSibling;
  }

  return nodes;
}
function populateTemplateNodes() {
  fldNodes = processTemplNodes('fld');
  msgNodes = processTemplNodes('msg');
  cmpNodes = processTemplNodes('cmp');
  supNodes = processTemplNodes('sup');
  tngNodes = processTemplNodes('tng');
}

function addClass(domNode, name) {
  if (domNode) {
    domNode.classList.add(name);
  }
}

function removeClass(domNode, name) {
  if (domNode) {
    domNode.classList.remove(name);
  }
}

function batchAddClass(domNode, searchClass, classToAdd) {
  var nodes = domNode.getElementsByClassName(searchClass);
  for (var i = 0; i < nodes.length; i++) {
    nodes[i].classList.add(classToAdd);
  }
}

function batchRemoveClass(domNode, searchClass, classToRemove) {
  var nodes = domNode.getElementsByClassName(searchClass);
  for (var i = 0; i < nodes.length; i++) {
    nodes[i].classList.remove(classToRemove);
  }
}

var MATCHED_TEXT_CLASS = 'highlight';

function appendMatchItemTo(matchItem, node) {
  var text = matchItem.text;
  var idx = 0;
  for (var iRun = 0; iRun <= matchItem.matchRuns.length; iRun++) {
    var run;
    if (iRun === matchItem.matchRuns.length)
      run = { start: text.length, length: 0 };
    else
      run = matchItem.matchRuns[iRun];

    // generate the un-highlighted span
    if (run.start > idx) {
      var tnode = document.createTextNode(text.substring(idx, run.start));
      node.appendChild(tnode);
    }

    if (!run.length)
      continue;
    var hspan = document.createElement('span');
    hspan.classList.add(MATCHED_TEXT_CLASS);
    hspan.textContent = text.substr(run.start, run.length);
    node.appendChild(hspan);
    idx = run.start + run.length;
  }
}

/**
 * Add an event listener on a container that, when an event is encounted on
 * a descendant, walks up the tree to find the immediate child of the container
 * and tells us what the click was on.
 */
function bindContainerHandler(containerNode, eventName, func) {
  containerNode.addEventListener(eventName, function(event) {
    var node = event.target;
    // bail if they clicked on the container and not a child...
    if (node === containerNode)
      return;
    while (node && node.parentNode !== containerNode) {
      node = node.parentNode;
    }
    func(node, event);
  }, false);
}

/**
 * Bind both 'click' and 'contextmenu' (synthetically created by b2g), plus
 * handling click suppression that is currently required because we still
 * see the click event.  We also suppress contextmenu's default event so that
 * we don't trigger the browser's right-click menu when operating in firefox.
 */
function bindContainerClickAndHold(containerNode, clickFunc, holdFunc) {
  // Rather than tracking suppressClick ourselves in here, we maintain the
  // state globally in Cards.  The rationale is that popup menus will be
  // triggered on contextmenu, which transfers responsibility of the click
  // event to the popup handling logic.  There is also no chance for multiple
  // contextmenu events overlapping (that we would consider reasonable).
  bindContainerHandler(
    containerNode, 'click',
    function(node, event) {
      if (Cards._suppressClick) {
        Cards._suppressClick = false;
        return;
      }
      clickFunc(node, event);
    });
  bindContainerHandler(
    containerNode, 'contextmenu',
    function(node, event) {
      // Always preventDefault, as this terminates processing of the click as a
      // drag event.
      event.preventDefault();
      // suppress the subsequent click if this was actually a left click
      if (event.button === 0) {
        Cards._suppressClick = true;
      }

      return holdFunc(node, event);
    });
}

/**
 * Fairly simple card abstraction with support for simple horizontal animated
 * transitions.  We are cribbing from deuxdrop's mobile UI's cards.js
 * implementation created jrburke.
 */
var Cards = {
  /* @dictof[
   *   @key[name String]
   *   @value[@dict[
   *     @key[name String]{
   *       The name of the card, which should also be the name of the css class
   *       used for the card when 'card-' is prepended.
   *     }
   *     @key[modes @dictof[
   *       @key[modeName String]
   *       @value[modeDef @dict[
   *         @key[tray Boolean]{
   *           Should this card be displayed as a tray that leaves the edge of
   *           the adjacent card visible?  (The width of the edge being a
   *           value consistent across all cards.)
   *         }
   *       ]
   *     ]]
   *     @key[constructor Function]{
   *       The constructor to use to create an instance of the card.
   *     }
   *   ]]
   * ]
   */
  _cardDefs: {},

  /* @listof[@typedef[CardInstance @dict[
   *   @key[domNode]{
   *   }
   *   @key[cardDef]
   *   @key[modeDef]
   *   @key[left Number]{
   *     Left offset of the card in #cards.
   *   }
   *   @key[cardImpl]{
   *     The result of calling the card's constructor.
   *   }
   * ]]]{
   *   Existing cards, left-to-right, new cards getting pushed onto the right.
   * }
   */
  _cardStack: [],
  activeCardIndex: -1,
  /*
   * @oneof[null @listof[cardName modeName]]{
   *   If a lazy load is causing us to have to wait before we push a card, this
   *   is the type of card we are planning to push.  This is used by hasCard
   *   to avoid returning misleading answers while an async push is happening.
   * }
   */
  _pendingPush: null,

  /**
   * Cards can stack on top of each other, make sure the stacked set is
   * visible over the lower sets.
   */
  _zIndex: 0,

  /**
   * The DOM node that contains the _containerNode ("#cardContainer") and which
   * we inject popup and masking layers into.  The choice of doing the popup
   * stuff at this layer is arbitrary.
   */
  _rootNode: null,
  /**
   * The "#cardContainer" node which serves as the scroll container for the
   * contained _cardsNode ("#cards").  It is as wide as the viewport.
   */
  _containerNode: null,
  /**
   * The "#cards" node that holds the cards; it is as wide as all of the cards
   * it contains and has its left offset changed in order to change what card
   * is visible.
   */
  _cardsNode: null,
  /**
   * DOM template nodes for the cards.
   */
  _templateNodes: null,

  /**
   * The DOM nodes that should be removed from their parent when our current
   * transition ends.
   */
  _animatingDeadDomNodes: [],

  /**
   * Tracks the number of transition events per card animation. Since each
   * animation ends up with two transitionend events since two cards are
   * moving, need to wait for the last one to be finished before doing
   * cleanup, like DOM removal.
   */
  _transitionCount: 0,

  /**
   * Annoying logic related to contextmenu event handling; search for the uses
   * for more info.
   */
  _suppressClick: false,
  /**
   * Is a tray card visible, suggesting that we need to intercept clicks in the
   * tray region so that we can transition back to the thing visible because of
   * the tray and avoid the click triggering that card's logic.
   */
  _trayActive: false,
  /**
   * Is a popup visible, suggesting that any click that is not on the popup
   * should be taken as a desire to close the popup?  This is not a boolean,
   * but rather info on the active popup.
   */
  _popupActive: null,
  /**
   * Are we eating all click events we see until we transition to the next
   * card (possibly due to a call to pushCard that has not yet occurred?).
   * Set by calling `eatEventsUntilNextCard`.
   */
  _eatingEventsUntilNextCard: false,

  /**
   * Initialize and bind ourselves to the DOM which should now be fully loaded.
   */
  _init: function() {
    this._rootNode = document.body;
    this._containerNode = document.getElementById('cardContainer');
    this._cardsNode = document.getElementById('cards');
    this._templateNodes = processTemplNodes('card');

    this._containerNode.addEventListener('click',
                                         this._onMaybeIntercept.bind(this),
                                         true);
    this._containerNode.addEventListener('contextmenu',
                                         this._onMaybeIntercept.bind(this),
                                         true);

    // XXX be more platform detecty. or just add more events. unless the
    // prefixes are already gone with webkit and opera?
    this._cardsNode.addEventListener('transitionend',
                                     this._onTransitionEnd.bind(this),
                                     false);
  },

  /**
   * If the tray is active and a click happens in the tray area, transition
   * back to the visible thing (which must be to our right currently.)
   */
  _onMaybeIntercept: function(event) {
    // Contextmenu-derived click suppression wants to gobble an explicitly
    // expected event, and so takes priority over other types of suppression.
    if (event.type === 'click' && this._suppressClick) {
      this._suppressClick = false;
      event.stopPropagation();
      return;
    }
    if (this._eatingEventsUntilNextCard) {
      event.stopPropagation();
      return;
    }
    if (this._popupActive) {
      event.stopPropagation();
      this._popupActive.close();
      return;
    }

    // Find the card containing the event target.
    var cardNode = event.target;
    for (cardNode = event.target; cardNode; cardNode = cardNode.parentNode) {
      if (cardNode.classList.contains('card'))
        break;
    }

    // If tray is active and the click is in the card that is after
    // current card (in the gutter), then just transition back to
    // that card.
    if (this._trayActive && cardNode && cardNode.classList.contains('after')) {
      event.stopPropagation();

      // Look for a card with a data-tray-target attribute
      var targetIndex = -1;
      this._cardStack.some(function(card, i) {
        if (card.domNode.hasAttribute('data-tray-target')) {
          targetIndex = i;
          return true;
        }
      });

      // Choose a default of one card ahead
      if (targetIndex === -1)
        targetIndex = this.activeCardIndex + 1;

      var indexDiff = targetIndex - (this.activeCardIndex + 1);
      if (indexDiff > 0) {
        this._afterTransitionAction = (function() {
          this.removeCardAndSuccessors(this._cardStack[0].domNode,
                                       'none', indexDiff);
          this.moveToCard(targetIndex, 'animate', 'forward');
        }.bind(this));
      }

      this.moveToCard(this.activeCardIndex + 1, 'animate', 'forward');
    }
  },

  defineCard: function(cardDef) {
    if (!cardDef.name)
      throw new Error('The card type needs a name');
    if (this._cardDefs.hasOwnProperty(cardDef.name))
      throw new Error('Duplicate card name: ' + cardDef.name);
    this._cardDefs[cardDef.name] = cardDef;

    // normalize the modes
    for (var modeName in cardDef.modes) {
      var mode = cardDef.modes[modeName];
      if (!mode.hasOwnProperty('tray'))
        mode.tray = false;
      mode.name = modeName;
    }
  },

  defineCardWithDefaultMode: function(name, defaultMode, constructor) {
    var cardDef = {
      name: name,
      modes: {},
      constructor: constructor
    };
    cardDef.modes['default'] = defaultMode;
    this.defineCard(cardDef);
  },

  /**
   * Push a card onto the card-stack.
   */
  /* @args[
   *   @param[type]
   *   @param[mode String]{
   *   }
   *   @param[showMethod @oneof[
   *     @case['animate']{
   *       Perform an animated scrolling transition.
   *     }
   *     @case['immediate']{
   *       Immediately warp to the card without animation.
   *     }
   *     @case['none']{
   *       Don't touch the view at all.
   *     }
   *   ]]
   *   @param[args Object]{
   *     An arguments object to provide to the card's constructor when
   *     instantiating.
   *   }
   *   @param[placement #:optional @oneof[
   *     @case[undefined]{
   *       The card gets pushed onto the end of the stack.
   *     }
   *     @case['left']{
   *       The card gets inserted to the left of the current card.
   *     }
   *     @case['right']{
   *       The card gets inserted to the right of the current card.
   *     }
   *   }
   * ]
   */
  pushCard: function(type, mode, showMethod, args, placement) {
    var cardDef = this._cardDefs[type];
    var typePrefix = type.split('-')[0];

    if (!cardDef && lazyCards[typePrefix]) {
      this._pendingPush = [type, mode];
      var saveArgs = Array.slice(arguments);
      var callback = function() {
        this.pushCard.apply(this, saveArgs);
      };

      this.eatEventsUntilNextCard();
      App.loader.load(lazyCards[typePrefix], callback.bind(this));
      return;
    } else if (!cardDef) {
      throw new Error('No such card def type: ' + type);
    }
    this._pendingPush = null;

    var modeDef = cardDef.modes[mode];
    if (!modeDef)
      throw new Error('No such card mode: ' + mode);

    var domNode = this._templateNodes[type].cloneNode(true);

    var cardImpl = new cardDef.constructor(domNode, mode, args);
    var cardInst = {
      domNode: domNode,
      cardDef: cardDef,
      modeDef: modeDef,
      cardImpl: cardImpl
    };
    var cardIndex, insertBuddy;
    if (!placement) {
      cardIndex = this._cardStack.length;
      insertBuddy = null;
      domNode.classList.add(cardIndex === 0 ? 'before' : 'after');
    }
    else if (placement === 'left') {
      cardIndex = this.activeCardIndex++;
      insertBuddy = this._cardsNode.children[cardIndex];
      domNode.classList.add('before');
    }
    else if (placement === 'right') {
      cardIndex = this.activeCardIndex + 1;
      if (cardIndex >= this._cardStack.length)
        insertBuddy = null;
      else
        insertBuddy = this._cardsNode.children[cardIndex];
      domNode.classList.add('after');
    }
    this._cardStack.splice(cardIndex, 0, cardInst);
    this._cardsNode.insertBefore(domNode, insertBuddy);

    // If the card has any <button type="reset"> buttons,
    // make them clear the field they're next to and not the entire form.
    // See input_areas.js and shared/style/input_areas.css.
    hookupInputAreaResetButtons(domNode);

    if ('postInsert' in cardImpl)
      cardImpl.postInsert();

    if (showMethod !== 'none') {
      // make sure the reflow sees the new node so that the animation
      // later is smooth.
      domNode.clientWidth;

      this._showCard(cardIndex, showMethod, 'forward');
    }
  },

  _findCardUsingTypeAndMode: function(type, mode) {
    for (var i = 0; i < this._cardStack.length; i++) {
      var cardInst = this._cardStack[i];
      if (cardInst.cardDef.name === type &&
          cardInst.modeDef.name === mode) {
        return i;
      }
    }
  },

  _findCardUsingImpl: function(impl) {
    for (var i = 0; i < this._cardStack.length; i++) {
      var cardInst = this._cardStack[i];
      if (cardInst.cardImpl === impl)
        return i;
    }
  },

  _findCard: function(query, skipFail) {
    var result;
    if (Array.isArray(query))
      result = this._findCardUsingTypeAndMode(query[0], query[1], skipFail);
    else if (typeof(query) === 'number') // index number
      result = query;
    else
      result = this._findCardUsingImpl(query);

    if (result > -1)
      return result;
    else if (!skipFail)
      throw new Error('Unable to find card with query:', query);
    else
      // Returning undefined explicitly so that index comparisons, like
      // the one in hasCard, are correct.
      return undefined;
  },

  hasCard: function(query) {
    if (this._pendingPush && Array.isArray(query) && query.length === 2 &&
        this._pendingPush[0] === query[0] &&
        this._pendingPush[1] === query[1])
      return true;

    return this._findCard(query, true) > -1;
  },

  findCardObject: function(query) {
    return this._cardStack[this._findCard(query)];
  },

  folderSelector: function(callback) {
    var self = this;

    App.loader.load(
      ['style/value_selector.css', 'js/value_selector.js'],
      function() {
        // XXX: Unified folders will require us to make sure we get
        //      the folder list for the account the message originates from.
        if (!self.folderPrompt) {
          var selectorTitle = mozL10n.get('messages-folder-select');
          self.folderPrompt = new ValueSelector(selectorTitle);
        }

        var folderCardObj =
          Cards.findCardObject(['folder-picker', 'navigation']);
        var folderImpl = folderCardObj.cardImpl;
        var folders = folderImpl.foldersSlice.items;
        for (var i = 0; i < folders.length; i++) {
          var folder = folders[i];
          self.folderPrompt.addToList(folder.name, folder.depth,
            function(folder) {
              return function() {
                self.folderPrompt.hide();
                callback(folder);
              }
            }(folder));

        }
        self.folderPrompt.show();
      });
  },

  moveToCard: function(query, showMethod) {
    this._showCard(this._findCard(query), showMethod || 'animate');
  },

  tellCard: function(query, what) {
    var cardIndex = this._findCard(query),
        cardInst = this._cardStack[cardIndex];
    if (!('told' in cardInst.cardImpl))
      console.warn("Tried to tell a card that's not listening!", query, what);
    else
      cardInst.cardImpl.told(what);
  },

  /**
   * Create a mask that shows only the given node by creating 2 or 4 div's,
   * returning the container that holds those divs.  It's not clear if a single
   * div with some type of fancy clipping would be better.
   */
  _createMaskForNode: function(domNode, bounds) {
    var anchorIn = this._rootNode, cleanupDivs = [];
    var uiWidth = this._containerNode.offsetWidth,
        uiHeight = this._containerNode.offsetHeight;

    // inclusive pixel coverage
    function addMask(left, top, right, bottom) {
      var node = document.createElement('div');
      node.classList.add('popup-mask');
      node.style.left = left + 'px';
      node.style.top = top + 'px';
      node.style.width = (right - left + 1) + 'px';
      node.style.height = (bottom - top + 1) + 'px';
      cleanupDivs.push(node);
      anchorIn.appendChild(node);
    }
    if (bounds.left > 1)
      addMask(0, bounds.top, bounds.left - 1, bounds.bottom);
    if (bounds.top > 0)
      addMask(0, 0, uiWidth - 1, bounds.top - 1);
    if (bounds.right < uiWidth - 1)
      addMask(bounds.right + 1, bounds.top, uiWidth - 1, bounds.bottom);
    if (bounds.bottom < uiHeight - 1)
      addMask(0, bounds.bottom + 1, uiWidth - 1, uiHeight - 1);
    return function() {
      for (var i = 0; i < cleanupDivs.length; i++) {
        anchorIn.removeChild(cleanupDivs[i]);
      }
    };
  },

  /**
   * Remove the card identified by its DOM node and all the cards to its right.
   * Pass null to remove all of the cards!
   */
  /* @args[
   *   @param[cardDomNode]{
   *     The DOM node that is the first card to remove; all of the cards to its
   *     right will also be removed.  If null is passed it is understood you
   *     want to remove all cards.
   *   }
   *   @param[showMethod @oneof[
   *     @case['animate']{
   *       Perform an animated scrolling transition.
   *     }
   *     @case['immediate']{
   *       Immediately warp to the card without animation.
   *     }
   *     @case['none']{
   *       Remove the nodes immediately, don't do anything about the view
   *       position.  You only want to do this if you are going to push one
   *       or more cards and the last card will use a transition of 'immediate'.
   *     }
   *   ]]
   *   @param[numCards #:optional Number]{
   *     The number of cards to remove.  If omitted, all the cards to the right
   *     of this card are removed as well.
   *   }
   *   @param[nextCardSpec #:optional]{
   *     If a showMethod is not 'none', the card to show after removal.
   *   }
   * ]
   */
  removeCardAndSuccessors: function(cardDomNode, showMethod, numCards,
                                    nextCardSpec) {
    if (!this._cardStack.length)
      return;

    var firstIndex, iCard, cardInst;
    if (cardDomNode === undefined) {
      throw new Error('undefined is not a valid card spec!');
    }
    else if (cardDomNode === null) {
      firstIndex = 0;
      // reset the z-index to 0 since we may have cards in the stack that
      // adjusted the z-index (and we are definitively clearing all cards).
      this._zIndex = 0;
    }
    else {
      for (iCard = this._cardStack.length - 1; iCard >= 0; iCard--) {
        cardInst = this._cardStack[iCard];
        if (cardInst.domNode === cardDomNode) {
          firstIndex = iCard;
          break;
        }
      }
      if (firstIndex === undefined)
        throw new Error('No card represented by that DOM node');
    }
    if (!numCards)
      numCards = this._cardStack.length - firstIndex;

    if (showMethod !== 'none') {
      var nextCardIndex = null;
      if (nextCardSpec)
        nextCardIndex = this._findCard(nextCardSpec);
      else if (this._cardStack.length)
        nextCardIndex = Math.min(firstIndex - 1, this._cardStack.length - 1);

      this._showCard(nextCardIndex, showMethod, 'back');
    }

    // Update activeCardIndex if nodes were removed that would affect its
    // value.
    if (firstIndex <= this.activeCardIndex) {
      this.activeCardIndex -= numCards;
      if (this.activeCardIndex < -1) {
        this.activeCardIndex = -1;
      }
    }

    var deadCardInsts = this._cardStack.splice(
                          firstIndex, numCards);
    for (iCard = 0; iCard < deadCardInsts.length; iCard++) {
      cardInst = deadCardInsts[iCard];
      try {
        cardInst.cardImpl.die();
      }
      catch (ex) {
        console.warn('Problem cleaning up card:', ex, '\n', ex.stack);
      }
      switch (showMethod) {
        case 'animate':
        case 'immediate': // XXX handle properly
          this._animatingDeadDomNodes.push(cardInst.domNode);
          break;
        case 'none':
          cardInst.domNode.parentNode.removeChild(cardInst.domNode);
          break;
      }
    }
  },

  /**
   * Shortcut for removing all the cards
   */
  removeAllCards: function() {
    return this.removeCardAndSuccessors(null, 'none');
  },

  _showCard: function(cardIndex, showMethod, navDirection) {
    // Do not do anything if this is a show card for the current card.
    if (cardIndex === this.activeCardIndex) {
      return;
    }

    if (cardIndex > this._cardStack.length - 1) {
      // Some cards were removed, adjust.
      cardIndex = this._cardStack.length - 1;
    }
    if (this.activeCardIndex > this._cardStack.length - 1) {
      this.activeCardIndex = -1;
    }

    if (this.activeCardIndex === -1) {
      this.activeCardIndex = cardIndex === 0 ? cardIndex : cardIndex - 1;
    }

    var cardInst = (cardIndex !== null) ? this._cardStack[cardIndex] : null;
    var beginNode = this._cardStack[this.activeCardIndex].domNode;
    var endNode = this._cardStack[cardIndex].domNode;
    var isForward = navDirection === 'forward';

    if (this._cardStack.length === 1) {
      // Reset zIndex so that it does not grow ever higher when all but
      // one card are removed
      this._zIndex = 0;
    }

    // If going forward and it is an overlay node, then do not animate the
    // beginning node, it will just sit under the overlay.
    if (isForward && endNode.classList.contains('anim-overlay')) {
      beginNode = null;

      // anim-overlays are the transitions to new layers in the stack. If
      // starting a new one, it is forward movement and needs a new zIndex.
      // Otherwise, going back to
      this._zIndex += 10;
    }

    // If going back and the beginning node was an overlay, do not animate
    // the end node, since it should just be hidden under the overlay.
    if (beginNode && beginNode.classList.contains('anim-overlay')) {
      if (isForward) {
        // If a forward animation and overlay had a vertical transition,
        // disable it, use normal horizontal transition.
        if (showMethod !== 'immediate' &&
            beginNode.classList.contains('anim-vertical')) {
          removeClass(beginNode, 'anim-vertical');
          addClass(beginNode, 'disabled-anim-vertical');
        }
      } else {
        endNode = null;
        this._zIndex -= 10;
      }
    }

    // If the zindex is not zero, then in an overlay stack, adjust zindex
    // accordingly.
    if (endNode && isForward && this._zIndex) {
      endNode.style.zIndex = this._zIndex;
    }

    var cardsNode = this._cardsNode;

    if (showMethod === 'immediate') {
      addClass(beginNode, 'no-anim');
      addClass(endNode, 'no-anim');

      // make sure the reflow sees the transition is turned off.
      cardsNode.clientWidth;
      // explicitly clear since there will be no animation
      this._eatingEventsUntilNextCard = false;
    }
    else {
      this._transitionCount = (beginNode && endNode) ? 2 : 1;
      this._eatingEventsUntilNextCard = true;
    }

    if (this.activeCardIndex === cardIndex) {
      // same node, no transition, just bootstrapping UI.
      removeClass(beginNode, 'before');
      removeClass(beginNode, 'after');
      addClass(beginNode, 'center');
    } else if (this.activeCardIndex > cardIndex) {
      // back
      removeClass(beginNode, 'center');
      addClass(beginNode, 'after');

      removeClass(endNode, 'before');
      addClass(endNode, 'center');
    } else {
      // forward
      removeClass(beginNode, 'center');
      addClass(beginNode, 'before');

      removeClass(endNode, 'after');
      addClass(endNode, 'center');
    }

    if (showMethod === 'immediate') {
      // make sure the instantaneous transition is seen before we turn
      // transitions back on.
      cardsNode.clientWidth;

      removeClass(beginNode, 'no-anim');
      removeClass(endNode, 'no-anim');
    }

    // Hide toaster while active card index changed:
    Toaster.hide();

    this.activeCardIndex = cardIndex;
    if (cardInst)
      this._trayActive = cardInst.modeDef.tray;
  },

  _onTransitionEnd: function(event) {
    // Multiple cards can animate, so there can be multiple transitionend
    // events. Only do the end work when all have finished animating.
    if (this._transitionCount > 0)
      this._transitionCount -= 1;

    if (this._transitionCount === 0) {
      if (this._eatingEventsUntilNextCard)
        this._eatingEventsUntilNextCard = false;
      if (this._animatingDeadDomNodes.length) {
        // Use a setTimeout to give the animation some space to settle.
        setTimeout(function() {
          this._animatingDeadDomNodes.forEach(function(domNode) {
            if (domNode.parentNode)
              domNode.parentNode.removeChild(domNode);
          });
          this._animatingDeadDomNodes = [];
        }.bind(this), 100);
      }

      // If an vertical overlay transition was was disabled, if
      // current node index is an overlay, enable it again.
      var endNode = this._cardStack[this.activeCardIndex].domNode;
      if (endNode.classList.contains('disabled-anim-vertical')) {
        removeClass(endNode, 'disabled-anim-vertical');
        addClass(endNode, 'anim-vertical');
      }

      // Popup toaster that pended for previous card view.
      var pendingToaster = Toaster.pendingStack.slice(-1)[0];
      if (pendingToaster) {
        pendingToaster();
        Toaster.pendingStack.pop();
      }

      // If any action to to at the end of transition trigger now.
      if (this._afterTransitionAction) {
        var afterTransitionAction = this._afterTransitionAction;
        this._afterTransitionAction = null;
        afterTransitionAction();
      }
    }
  },

  /**
   * Helper that causes (some) events targeted at our cards to be eaten until
   * we get to the next card.  The idea is to avoid bugs caused by the user
   * still being able to click things while our cards are transitioning or
   * while we are performing a (reliable) async wait before we actually initiate
   * a pushCard in response to user stimulus.
   *
   * This is automatically triggered when performing an animated transition;
   * other code should only call this in the async wait case mentioned above.
   *
   * For example, we don't want the user to have 2 message readers happening
   * at the same time because they managed to click on a second message before
   * the first reader got displayed.
   */
  eatEventsUntilNextCard: function() {
    this._eatingEventsUntilNextCard = true;
  },

  /**
   * Stop eating events, presumably because eatEventsUntilNextCard was used
   * as a hack for a known-fast async operation to avoid bugs (where we knew
   * full well that we weren't going to show a card).
   */
  stopEatingEvents: function() {
    this._eatingEventsUntilNextCard = false;
  },

  /**
   * If there are any cards on the deck right now, log an error and clear them
   * all out.  Our caller is strongly asserting that there should be no cards
   * and the presence of any indicates a bug.
   */
  assertNoCards: function() {
    if (this._cardStack.length)
      throw new Error('There are ' + this._cardStack.length + ' cards but' +
                      ' there should be ZERO');
  }
};

/**
 * Central tracker of poptart messages; specifically, ongoing message sends,
 * failed sends, and recently performed undoable mutations.
 */
var Toaster = {
  get body() {
    delete this.body;
    return this.body =
           document.querySelector('section[role="status"]');
  },
  get text() {
    delete this.text;
    return this.text =
           document.querySelector('section[role="status"] p');
  },
  get undoBtn() {
    delete this.undoBtn;
    return this.undoBtn =
           document.querySelector('.toaster-banner-undo');
  },
  get retryBtn() {
    delete this.retryBtn;
    return this.retryBtn =
           document.querySelector('.toaster-banner-retry');
  },

  undoableOp: null,
  retryCallback: null,

  /**
   * Toaster timeout setting.
   */
  _timeout: 5000,
  /**
   * Toaster fadeout animation event handling.
   */
  _animationHandler: function() {
    this.body.addEventListener('transitionend', this, false);
    this.body.classList.add('fadeout');
  },
  /**
   * The list of cards that want to hear about what's up with the toaster.  For
   * now this will just be the message-list, but it might also be the
   * message-search card as well.  If it ends up being more, then we probably
   * want to rejigger things so we can just overlay stuff on most cards...
   */
  _listeners: [],

  pendingStack: [],

  /**
   * Tell toaster listeners about a mutation we just made.
   *
   * @param {Object} undoableOp undoable operation.
   * @param {Boolean} pending
   *   If true, indicates that we should wait to display this banner until we
   *   transition to the next card.  This is appropriate for things like
   *   deleting the message that is displayed on the current card (and which
   *   will be imminently closed).
   */
  logMutation: function(undoableOp, pending) {
    if (pending) {
      this.pendingStack.push(this.show.bind(this, 'undo', undoableOp));
    } else {
      this.show('undo', undoableOp);
    }
  },

  /**
   * Something failed that it makes sense to let the user explicitly trigger
   * a retry of!  For example, failure to synchronize.
   */
  logRetryable: function(retryStringId, retryCallback) {
    this.show('retry', retryStringId, retryCallback);
  },

  handleEvent: function(evt) {
    switch (evt.type) {
      case 'click' :
        var classList = evt.target.classList;
        if (classList.contains('toaster-banner-undo')) {
          this.undoableOp.undo();
          this.hide();
        } else if (classList.contains('toaster-banner-retry')) {
          if (this.retryCallback)
            this.retryCallback();
          this.hide();
        } else if (classList.contains('toaster-cancel-btn')) {
          this.hide();
        }
        break;
      case 'transitionend' :
        this.hide();
        break;
    }
  },

  show: function(type, operation, callback) {
    // Close previous toaster before showing the new one.
    if (!this.body.classList.contains('collapsed')) {
      this.hide();
    }

    var text, textId, showUndo = false;
    var undoBtn = this.body.querySelector('.toaster-banner-undo');
    if (type === 'undo') {
      this.undoableOp = operation;
      // There is no need to show toaster if affected message count < 1
      if (!this.undoableOp || this.undoableOp.affectedCount < 1) {
        return;
      }
      textId = 'toaster-message-' + this.undoableOp.operation;
      text = mozL10n.get(textId, { n: this.undoableOp.affectedCount });
      // https://bugzilla.mozilla.org/show_bug.cgi?id=804916
      // Remove undo email move/delete UI for V1.
      showUndo = (this.undoableOp.operation !== 'move' &&
                  this.undoableOp.operation !== 'delete');
    } else if (type === 'retry') {
      textId = 'toaster-retryable-' + operation;
      text = mozL10n.get(textId);
      this.retryCallback = callback;
    // XXX I assume this is for debug purposes?
    } else if (type === 'text') {
      text = operation;
    }

    if (type === 'undo' && showUndo)
      this.undoBtn.classList.remove('collapsed');
    else
      this.undoBtn.classList.add('collapsed');
    if (type === 'retry')
      this.retryBtn.classList.remove('collapsed');
    else
      this.retryBtn.classList.add('collapsed');

    this.body.title = type;
    this.text.textContent = text;
    this.body.addEventListener('click', this, false);
    this.body.classList.remove('collapsed');
    this.fadeTimeout = window.setTimeout(this._animationHandler.bind(this),
                                         this._timeout);
  },

  hide: function() {
    this.body.classList.add('collapsed');
    this.body.classList.remove('fadeout');
    window.clearTimeout(this.fadeTimeout);
    this.fadeTimeout = null;
    this.body.removeEventListener('click', this);
    this.body.removeEventListener('transitionend', this);

    // Clear operations:
    this.undoableOp = null;
    this.retryCallback = null;
  }
};

/**
 * Confirm dialog helper function. Display the dialog by providing dialog body
 * element and button id/handler function.
 *
 */
var ConfirmDialog = {
  dialog: null,
  show: function(dialog, confirm, cancel) {
    this.dialog = dialog;
    var formSubmit = function(evt) {
      this.hide();
      switch (evt.explicitOriginalTarget.id) {
        case confirm.id:
          confirm.handler();
          break;
        case cancel.id:
          if (cancel.handler)
            cancel.handler();
          break;
      }
      return false;
    };
    dialog.addEventListener('submit', formSubmit.bind(this));
    document.body.appendChild(dialog);
  },
  hide: function() {
    document.body.removeChild(this.dialog);
  }
};
////////////////////////////////////////////////////////////////////////////////
// Attachment Formatting Helpers

/**
 * Display a human-readable file size.  Currently we always display things in
 * kilobytes because we are targeting a mobile device and we want bigger sizes
 * (like megabytes) to be obviously large numbers.
 */
function prettyFileSize(sizeInBytes) {
  var kilos = Math.ceil(sizeInBytes / 1024);
  return mozL10n.get('attachment-size-kib', { kilobytes: kilos });
}

/**
 * Display a human-readable relative timestamp.
 */
function prettyDate(time, useCompactFormat) {
  var f = new mozL10n.DateTimeFormat();
  return f.fromNow(time, useCompactFormat);
}

(function() {
  var formatter = new mozL10n.DateTimeFormat();
  var updatePrettyDate = function updatePrettyDate() {
    var labels = document.querySelectorAll('[data-time]');
    var i = labels.length;
    while (i--) {
      labels[i].textContent = formatter.fromNow(
        labels[i].dataset.time,
        // the presence of the attribute is our indicator; not its value
        'compactFormat' in labels[i].dataset);
    }
  };
  var timer = setInterval(updatePrettyDate, 60 * 1000);

  window.addEventListener('message', function visibleAppUpdatePrettyDate(evt) {
    var data = evt.data;
    if (!data || (typeof(data) !== 'object') ||
        !('message' in data) || data.message !== 'visibilitychange')
      return;
    clearTimeout(timer);
    if (!data.hidden) {
      updatePrettyDate();
      timer = setInterval(updatePrettyDate, 60 * 1000);
    }
  });
})();

////////////////////////////////////////////////////////////////////////////////

/**
 * Class to handle form input navigation.
 *
 * If 'Enter' is hit, next input element will be focused,
 * and if the input element is the last one, trigger 'onLast' callback.
 *
 * options:
 *   {
 *     formElem: element,             // The form element
 *     checkFormValidity: function    // Function to check form validity
 *     onLast: function               // Callback when 'Enter' in the last input
 *   }
 */
function FormNavigation(options) {
  function extend(destination, source) {
    for (var property in source)
      destination[property] = source[property];
    return destination;
  }

  if (!options.formElem) {
    throw new Error('The form element should be defined.');
  }

  var self = this;
  this.options = extend({
    formElem: null,
    checkFormValidity: function checkFormValidity() {
      return self.options.formElem.checkValidity();
    },
    onLast: function() {}
  }, options);

  this.options.formElem.addEventListener('keypress',
    this.onKeyPress.bind(this));
}

FormNavigation.prototype = {
  onKeyPress: function formNav_onKeyPress(event) {
    if (event.keyCode === 13) {
      // If the user hit enter, focus the next form element, or, if the current
      // element is the last one and the form is valid, submit the form.
      var nextInput = this.focusNextInput(event);
      if (!nextInput && this.options.checkFormValidity()) {
        this.options.onLast();
      }
    }
  },

  focusNextInput: function formNav_focusNextInput(event) {
    var currentInput = event.target;
    var inputElems = this.options.formElem.getElementsByTagName('input');
    var currentInputFound = false;

    for (var i = 0; i < inputElems.length; i++) {
      var input = inputElems[i];
      if (currentInput === input) {
        currentInputFound = true;
        continue;
      } else if (!currentInputFound) {
        continue;
      }

      if (input.type === 'hidden' || input.type === 'button') {
        continue;
      }

      input.focus();
      if (document.activeElement !== input) {
        // We couldn't focus the element we wanted.  Try with the next one.
        continue;
      }
      return input;
    }

    // If we couldn't find anything to focus, just blur the initial element.
    currentInput.blur();
    return null;
  }
};
