/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @flow

/**
 * This file is for use by unit tests for isolated debugger components that do
 * not need to interact with the redux store. When these tests need to construct
 * debugger objects, these interfaces should be used instead of plain object
 * literals.
 */

import type {
  ActorId,
  Breakpoint,
  DisplaySource,
  Expression,
  Frame,
  FrameId,
  Scope,
  Source,
  SourceId,
  SourceWithContentAndType,
  SourceWithContent,
  TextSourceContent,
  URL,
  WasmSourceContent,
  Why,
  Thread,
} from "../types";
import * as asyncValue from "./async-value";

import { initialState } from "../reducers/index";

import type { SourceBase } from "../reducers/sources";
import type { State } from "../reducers/types";
import type { FulfilledValue } from "./async-value";

function makeMockSource(url: URL = "url", id: SourceId = "source"): SourceBase {
  return {
    id,
    url,
    isBlackBoxed: false,
    isPrettyPrinted: false,
    relativeUrl: url,
    isWasm: false,
    extensionName: null,
    isExtension: false,
    isOriginal: id.includes("originalSource"),
  };
}

function makeMockDisplaySource(
  url: URL = "url",
  id: SourceId = "source"
): DisplaySource {
  return {
    ...makeMockSource(url, id),
    displayURL: url,
  };
}

function makeMockSourceWithContent(
  url?: string,
  id?: SourceId,
  contentType?: string = "text/javascript",
  text?: string = ""
): SourceWithContent {
  const source = makeMockSource(url, id);

  return {
    ...source,
    content: text
      ? asyncValue.fulfilled({
          type: "text",
          value: text,
          contentType,
        })
      : null,
  };
}

function makeMockSourceAndContent(
  url?: string,
  id?: SourceId,
  contentType?: string = "text/javascript",
  text: string = ""
): { ...SourceBase, content: TextSourceContent } {
  const source = makeMockSource(url, id);

  return {
    ...source,
    content: {
      type: "text",
      value: text,
      contentType,
    },
  };
}

function makeFullfilledMockSourceContent(
  text: string = "",
  contentType?: string = "text/javascript"
): FulfilledValue<TextSourceContent> {
  return asyncValue.fulfilled({
    type: "text",
    value: text,
    contentType,
  });
}

function makeMockWasmSource(): SourceBase {
  return {
    id: "wasm-source-id",
    url: "url",
    isBlackBoxed: false,
    isPrettyPrinted: false,
    relativeUrl: "url",
    isWasm: true,
    extensionName: null,
    isExtension: false,
    isOriginal: false,
  };
}

function makeMockWasmSourceWithContent(text: {|
  binary: Object,
|}): SourceWithContentAndType<WasmSourceContent> {
  const source = makeMockWasmSource();

  return {
    ...source,
    content: asyncValue.fulfilled({
      type: "wasm",
      value: text,
    }),
  };
}

function makeMockScope(
  actor: ActorId = "scope-actor",
  type: string = "block",
  parent: ?Scope = null
): Scope {
  return {
    actor,
    parent,
    bindings: {
      arguments: [],
      variables: {},
    },
    object: null,
    function: null,
    type,
    scopeKind: "",
  };
}

function mockScopeAddVariable(scope: Scope, name: string) {
  if (!scope.bindings) {
    throw new Error("no scope bindings");
  }
  scope.bindings.variables[name] = { value: null };
}

function makeMockBreakpoint(
  source: Source = makeMockSource(),
  line: number = 1,
  column: ?number
): Breakpoint {
  const location = column
    ? { sourceId: source.id, line, column }
    : { sourceId: source.id, line };
  return {
    id: "breakpoint",
    location,
    astLocation: null,
    generatedLocation: location,
    disabled: false,
    text: "text",
    originalText: "text",
    options: {},
  };
}

function makeMockFrame(
  id: FrameId = "frame",
  source: Source = makeMockSource("url"),
  scope: Scope = makeMockScope(),
  line: number = 4,
  displayName: string = `display-${id}`,
  index: number = 0
): Frame {
  const location = { sourceId: source.id, line };
  return {
    id,
    thread: "FakeThread",
    displayName,
    location,
    generatedLocation: location,
    source,
    scope,
    this: {},
    index,
    asyncCause: null,
    state: "on-stack",
    type: "call",
  };
}

function makeMockFrameWithURL(url: URL): Frame {
  return makeMockFrame(undefined, makeMockSource(url));
}

function makeWhyNormal(frameReturnValue: any = undefined): Why {
  if (frameReturnValue) {
    return { type: "why-normal", frameFinished: { return: frameReturnValue } };
  }
  return { type: "why-normal" };
}

function makeWhyThrow(frameThrowValue: any): Why {
  return { type: "why-throw", frameFinished: { throw: frameThrowValue } };
}

function makeMockExpression(value: Object): Expression {
  return {
    input: "input",
    value,
    from: "from",
    updating: false,
  };
}

// Mock contexts for use in tests that do not create a redux store.
const mockcx = { navigateCounter: 0 };
const mockthreadcx = {
  navigateCounter: 0,
  thread: "FakeThread",
  pauseCounter: 0,
  isPaused: false,
};

function makeMockThread(fields: $Shape<Thread>) {
  return {
    actor: "test",
    url: "example.com",
    type: "worker",
    name: "test",
    ...fields,
  };
}

function makeMockState(state: $Shape<State>) {
  return {
    ...initialState(),
    ...state,
  };
}

export {
  makeMockDisplaySource,
  makeMockSource,
  makeMockSourceWithContent,
  makeMockSourceAndContent,
  makeMockWasmSource,
  makeMockWasmSourceWithContent,
  makeMockScope,
  mockScopeAddVariable,
  makeMockBreakpoint,
  makeMockFrame,
  makeMockFrameWithURL,
  makeWhyNormal,
  makeWhyThrow,
  makeMockExpression,
  mockcx,
  mockthreadcx,
  makeMockState,
  makeMockThread,
  makeFullfilledMockSourceContent,
};
