# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

from mach.decorators import Command, CommandArgument

LICENSE_HEADER = """# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
"""

GENERATED_HEADER = """
### This file was AUTOMATICALLY GENERATED by `./mach update-glean-tags`
### DO NOT edit it by hand.
"""


@Command(
    "data-review",
    category="misc",
    description="Generate a skeleton data review request form for a given bug's data",
)
@CommandArgument(
    "bug", default=None, nargs="?", type=str, help="bug number or search pattern"
)
def data_review(command_context, bug=None):
    # Get the metrics_index's list of metrics indices
    # by loading the index as a module.
    from os import path
    import sys

    sys.path.append(path.join(path.dirname(__file__), path.pardir))
    from metrics_index import metrics_yamls

    from glean_parser import data_review
    from pathlib import Path

    return data_review.generate(
        bug, [Path(command_context.topsrcdir) / x for x in metrics_yamls]
    )


@Command(
    "perf-data-review",
    category="misc",
    description="Generate a skeleton performance data review request form for a given bug's data",
)
@CommandArgument(
    "bug", default=None, nargs="?", type=str, help="bug number or search pattern"
)
def perf_data_review(command_context, bug=None):
    # Get the metrics_index's list of metrics indices
    # by loading the index as a module.
    from os import path
    import sys

    sys.path.append(path.join(path.dirname(__file__), path.pardir))
    from metrics_index import metrics_yamls

    sys.path.append(path.dirname(__file__))
    import perf_data_review
    from pathlib import Path

    return perf_data_review.generate(
        bug, [Path(command_context.topsrcdir) / x for x in metrics_yamls]
    )


@Command(
    "update-glean-tags",
    category="misc",
    description=(
        "Creates a list of valid glean tags based on in-tree bugzilla component definitions"
    ),
)
def update_glean_tags(command_context):
    from pathlib import Path
    import yaml
    from mozbuild.frontend.reader import BuildReader, EmptyConfig

    config = EmptyConfig(str((Path(__file__).parent / "../../../../").resolve()))
    reader = BuildReader(config)
    bug_components = set()
    for p in reader.read_topsrcdir():
        if p.get("BUG_COMPONENT"):
            bug_components.add(p["BUG_COMPONENT"])

    tags_filename = (Path(__file__).parent / "../tags.yaml").resolve()

    tags = {"$schema": "moz://mozilla.org/schemas/glean/tags/1-0-0"}
    for bug_component in bug_components:
        product = bug_component.product.strip()
        component = bug_component.component.strip()
        tags["{} :: {}".format(product, component)] = {
            "description": "The Bugzilla component which applies to this object."
        }

    open(tags_filename, "w").write(
        "{}\n{}\n\n".format(LICENSE_HEADER, GENERATED_HEADER)
        + yaml.dump(tags, width=78, explicit_start=True)
    )
