/**
 * CommonJS JavaScript code that provides the puppeteer utilities. See the
 * [README](https://github.com/puppeteer/puppeteer/blob/main/src/injected/README.md)
 * for injection for more information.
 *
 * @internal
 */
export const source = "\"use strict\";\nvar __defProp = Object.defineProperty;\nvar __getOwnPropDesc = Object.getOwnPropertyDescriptor;\nvar __getOwnPropNames = Object.getOwnPropertyNames;\nvar __hasOwnProp = Object.prototype.hasOwnProperty;\nvar __export = (target, all) => {\n  for (var name in all)\n    __defProp(target, name, { get: all[name], enumerable: true });\n};\nvar __copyProps = (to, from, except, desc) => {\n  if (from && typeof from === \"object\" || typeof from === \"function\") {\n    for (let key of __getOwnPropNames(from))\n      if (!__hasOwnProp.call(to, key) && key !== except)\n        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });\n  }\n  return to;\n};\nvar __toCommonJS = (mod) => __copyProps(__defProp({}, \"__esModule\", { value: true }), mod);\n\n// src/injected/injected.ts\nvar injected_exports = {};\n__export(injected_exports, {\n  default: () => injected_default\n});\nmodule.exports = __toCommonJS(injected_exports);\n\n// src/common/Errors.ts\nvar CustomError = class extends Error {\n  constructor(message) {\n    super(message);\n    this.name = this.constructor.name;\n    Error.captureStackTrace(this, this.constructor);\n  }\n};\nvar TimeoutError = class extends CustomError {\n};\nvar ProtocolError = class extends CustomError {\n  #code;\n  #originalMessage = \"\";\n  set code(code) {\n    this.#code = code;\n  }\n  get code() {\n    return this.#code;\n  }\n  set originalMessage(originalMessage) {\n    this.#originalMessage = originalMessage;\n  }\n  get originalMessage() {\n    return this.#originalMessage;\n  }\n};\nvar errors = Object.freeze({\n  TimeoutError,\n  ProtocolError\n});\n\n// src/util/DeferredPromise.ts\nfunction createDeferredPromise(opts) {\n  let isResolved = false;\n  let isRejected = false;\n  let resolver;\n  let rejector;\n  const taskPromise = new Promise((resolve, reject) => {\n    resolver = resolve;\n    rejector = reject;\n  });\n  const timeoutId = opts && opts.timeout > 0 ? setTimeout(() => {\n    isRejected = true;\n    rejector(new TimeoutError(opts.message));\n  }, opts.timeout) : void 0;\n  return Object.assign(taskPromise, {\n    resolved: () => {\n      return isResolved;\n    },\n    finished: () => {\n      return isResolved || isRejected;\n    },\n    resolve: (value) => {\n      if (timeoutId) {\n        clearTimeout(timeoutId);\n      }\n      isResolved = true;\n      resolver(value);\n    },\n    reject: (err) => {\n      clearTimeout(timeoutId);\n      isRejected = true;\n      rejector(err);\n    }\n  });\n}\n\n// src/util/Function.ts\nvar createdFunctions = /* @__PURE__ */ new Map();\nvar createFunction = (functionValue) => {\n  let fn = createdFunctions.get(functionValue);\n  if (fn) {\n    return fn;\n  }\n  fn = new Function(`return ${functionValue}`)();\n  createdFunctions.set(functionValue, fn);\n  return fn;\n};\n\n// src/injected/ARIAQuerySelector.ts\nvar ARIAQuerySelector_exports = {};\n__export(ARIAQuerySelector_exports, {\n  ariaQuerySelector: () => ariaQuerySelector,\n  ariaQuerySelectorAll: () => ariaQuerySelectorAll\n});\nvar ariaQuerySelector = (root, selector) => {\n  return window.__ariaQuerySelector(root, selector);\n};\nvar ariaQuerySelectorAll = async function* (root, selector) {\n  yield* await window.__ariaQuerySelectorAll(root, selector);\n};\n\n// src/injected/CustomQuerySelector.ts\nvar CustomQuerySelector_exports = {};\n__export(CustomQuerySelector_exports, {\n  customQuerySelectors: () => customQuerySelectors\n});\nvar CustomQuerySelectorRegistry = class {\n  #selectors = /* @__PURE__ */ new Map();\n  register(name, handler) {\n    if (!handler.queryOne && handler.queryAll) {\n      const querySelectorAll = handler.queryAll;\n      handler.queryOne = (node, selector) => {\n        for (const result of querySelectorAll(node, selector)) {\n          return result;\n        }\n        return null;\n      };\n    } else if (handler.queryOne && !handler.queryAll) {\n      const querySelector = handler.queryOne;\n      handler.queryAll = (node, selector) => {\n        const result = querySelector(node, selector);\n        return result ? [result] : [];\n      };\n    } else if (!handler.queryOne || !handler.queryAll) {\n      throw new Error(\"At least one query method must be defined.\");\n    }\n    this.#selectors.set(name, {\n      querySelector: handler.queryOne,\n      querySelectorAll: handler.queryAll\n    });\n  }\n  unregister(name) {\n    this.#selectors.delete(name);\n  }\n  get(name) {\n    return this.#selectors.get(name);\n  }\n  clear() {\n    this.#selectors.clear();\n  }\n};\nvar customQuerySelectors = new CustomQuerySelectorRegistry();\n\n// src/injected/PierceQuerySelector.ts\nvar PierceQuerySelector_exports = {};\n__export(PierceQuerySelector_exports, {\n  pierceQuerySelector: () => pierceQuerySelector,\n  pierceQuerySelectorAll: () => pierceQuerySelectorAll\n});\nvar pierceQuerySelector = (root, selector) => {\n  let found = null;\n  const search = (root2) => {\n    const iter = document.createTreeWalker(root2, NodeFilter.SHOW_ELEMENT);\n    do {\n      const currentNode = iter.currentNode;\n      if (currentNode.shadowRoot) {\n        search(currentNode.shadowRoot);\n      }\n      if (currentNode instanceof ShadowRoot) {\n        continue;\n      }\n      if (currentNode !== root2 && !found && currentNode.matches(selector)) {\n        found = currentNode;\n      }\n    } while (!found && iter.nextNode());\n  };\n  if (root instanceof Document) {\n    root = root.documentElement;\n  }\n  search(root);\n  return found;\n};\nvar pierceQuerySelectorAll = (element, selector) => {\n  const result = [];\n  const collect = (root) => {\n    const iter = document.createTreeWalker(root, NodeFilter.SHOW_ELEMENT);\n    do {\n      const currentNode = iter.currentNode;\n      if (currentNode.shadowRoot) {\n        collect(currentNode.shadowRoot);\n      }\n      if (currentNode instanceof ShadowRoot) {\n        continue;\n      }\n      if (currentNode !== root && currentNode.matches(selector)) {\n        result.push(currentNode);\n      }\n    } while (iter.nextNode());\n  };\n  if (element instanceof Document) {\n    element = element.documentElement;\n  }\n  collect(element);\n  return result;\n};\n\n// src/util/assert.ts\nvar assert = (value, message) => {\n  if (!value) {\n    throw new Error(message);\n  }\n};\n\n// src/injected/Poller.ts\nvar MutationPoller = class {\n  #fn;\n  #root;\n  #observer;\n  #promise;\n  constructor(fn, root) {\n    this.#fn = fn;\n    this.#root = root;\n  }\n  async start() {\n    const promise = this.#promise = createDeferredPromise();\n    const result = await this.#fn();\n    if (result) {\n      promise.resolve(result);\n      return;\n    }\n    this.#observer = new MutationObserver(async () => {\n      const result2 = await this.#fn();\n      if (!result2) {\n        return;\n      }\n      promise.resolve(result2);\n      await this.stop();\n    });\n    this.#observer.observe(this.#root, {\n      childList: true,\n      subtree: true,\n      attributes: true\n    });\n  }\n  async stop() {\n    assert(this.#promise, \"Polling never started.\");\n    if (!this.#promise.finished()) {\n      this.#promise.reject(new Error(\"Polling stopped\"));\n    }\n    if (this.#observer) {\n      this.#observer.disconnect();\n      this.#observer = void 0;\n    }\n  }\n  result() {\n    assert(this.#promise, \"Polling never started.\");\n    return this.#promise;\n  }\n};\nvar RAFPoller = class {\n  #fn;\n  #promise;\n  constructor(fn) {\n    this.#fn = fn;\n  }\n  async start() {\n    const promise = this.#promise = createDeferredPromise();\n    const result = await this.#fn();\n    if (result) {\n      promise.resolve(result);\n      return;\n    }\n    const poll = async () => {\n      if (promise.finished()) {\n        return;\n      }\n      const result2 = await this.#fn();\n      if (!result2) {\n        window.requestAnimationFrame(poll);\n        return;\n      }\n      promise.resolve(result2);\n      await this.stop();\n    };\n    window.requestAnimationFrame(poll);\n  }\n  async stop() {\n    assert(this.#promise, \"Polling never started.\");\n    if (!this.#promise.finished()) {\n      this.#promise.reject(new Error(\"Polling stopped\"));\n    }\n  }\n  result() {\n    assert(this.#promise, \"Polling never started.\");\n    return this.#promise;\n  }\n};\nvar IntervalPoller = class {\n  #fn;\n  #ms;\n  #interval;\n  #promise;\n  constructor(fn, ms) {\n    this.#fn = fn;\n    this.#ms = ms;\n  }\n  async start() {\n    const promise = this.#promise = createDeferredPromise();\n    const result = await this.#fn();\n    if (result) {\n      promise.resolve(result);\n      return;\n    }\n    this.#interval = setInterval(async () => {\n      const result2 = await this.#fn();\n      if (!result2) {\n        return;\n      }\n      promise.resolve(result2);\n      await this.stop();\n    }, this.#ms);\n  }\n  async stop() {\n    assert(this.#promise, \"Polling never started.\");\n    if (!this.#promise.finished()) {\n      this.#promise.reject(new Error(\"Polling stopped\"));\n    }\n    if (this.#interval) {\n      clearInterval(this.#interval);\n      this.#interval = void 0;\n    }\n  }\n  result() {\n    assert(this.#promise, \"Polling never started.\");\n    return this.#promise;\n  }\n};\n\n// src/injected/PQuerySelector.ts\nvar PQuerySelector_exports = {};\n__export(PQuerySelector_exports, {\n  pQuerySelector: () => pQuerySelector,\n  pQuerySelectorAll: () => pQuerySelectorAll\n});\n\n// src/util/AsyncIterableUtil.ts\nvar AsyncIterableUtil = class {\n  static async *map(iterable, map) {\n    for await (const value of iterable) {\n      yield await map(value);\n    }\n  }\n  static async *flatMap(iterable, map) {\n    for await (const value of iterable) {\n      yield* map(value);\n    }\n  }\n  static async collect(iterable) {\n    const result = [];\n    for await (const value of iterable) {\n      result.push(value);\n    }\n    return result;\n  }\n  static async first(iterable) {\n    for await (const value of iterable) {\n      return value;\n    }\n    return;\n  }\n};\n\n// src/util/ErrorLike.ts\nfunction isErrorLike(obj) {\n  return typeof obj === \"object\" && obj !== null && \"name\" in obj && \"message\" in obj;\n}\n\n// src/injected/PSelectorParser.ts\nvar PUPPETEER_PSEUDO_ELEMENT = /^::-p-([-a-zA-Z_]+)\\(/;\nvar PSelectorParser = class {\n  #input;\n  #escaped = false;\n  #quoted = false;\n  #selectors = [[[]]];\n  constructor(input) {\n    this.#input = input;\n  }\n  get selectors() {\n    return this.#selectors;\n  }\n  parse() {\n    for (let i = 0; i < this.#input.length; ++i) {\n      if (this.#escaped) {\n        this.#escaped = false;\n        continue;\n      }\n      switch (this.#input[i]) {\n        case \"\\\\\": {\n          this.#escaped = true;\n          break;\n        }\n        case '\"': {\n          this.#quoted = !this.#quoted;\n          break;\n        }\n        default: {\n          if (this.#quoted) {\n            break;\n          }\n          const remainder = this.#input.slice(i);\n          if (remainder.startsWith(\">>>>\")) {\n            this.#push(this.#input.slice(0, i));\n            this.#input = remainder.slice(\">>>>\".length);\n            this.#parseDeepChild();\n          } else if (remainder.startsWith(\">>>\")) {\n            this.#push(this.#input.slice(0, i));\n            this.#input = remainder.slice(\">>>\".length);\n            this.#parseDeepDescendent();\n          } else {\n            const result = PUPPETEER_PSEUDO_ELEMENT.exec(remainder);\n            if (!result) {\n              continue;\n            }\n            const [match, name] = result;\n            this.#push(this.#input.slice(0, i));\n            this.#input = remainder.slice(match.length);\n            this.#push({\n              name,\n              value: this.#scanParameter()\n            });\n          }\n        }\n      }\n    }\n    this.#push(this.#input);\n  }\n  #push(selector) {\n    if (typeof selector === \"string\") {\n      selector = selector.trimEnd();\n      if (selector.length === 0) {\n        return;\n      }\n    }\n    const roots = this.#selectors[this.#selectors.length - 1];\n    roots[roots.length - 1].push(selector);\n  }\n  #parseDeepChild() {\n    this.#selectors[this.#selectors.length - 1].push([]);\n  }\n  #parseDeepDescendent() {\n    this.#selectors.push([[]]);\n  }\n  #scanParameter() {\n    const char = this.#input[0];\n    switch (char) {\n      case \"'\":\n      case '\"':\n        this.#input = this.#input.slice(1);\n        const parameter = this.#scanEscapedValueTill(char);\n        if (!this.#input.startsWith(\")\")) {\n          throw new Error(\"Expected ')'\");\n        }\n        this.#input = this.#input.slice(1);\n        return parameter;\n      default:\n        return this.#scanEscapedValueTill(\")\");\n    }\n  }\n  #scanEscapedValueTill(end) {\n    let string = \"\";\n    for (let i = 0; i < this.#input.length; ++i) {\n      if (this.#escaped) {\n        this.#escaped = false;\n        string += this.#input[i];\n        continue;\n      }\n      switch (this.#input[i]) {\n        case \"\\\\\": {\n          this.#escaped = true;\n          break;\n        }\n        case end: {\n          this.#input = this.#input.slice(i + 1);\n          return string;\n        }\n        default: {\n          string += this.#input[i];\n        }\n      }\n    }\n    throw new Error(`Expected \\`${end}\\``);\n  }\n};\nfunction parsePSelectors(selector) {\n  const parser = new PSelectorParser(selector);\n  parser.parse();\n  return parser.selectors;\n}\n\n// src/injected/TextQuerySelector.ts\nvar TextQuerySelector_exports = {};\n__export(TextQuerySelector_exports, {\n  textQuerySelectorAll: () => textQuerySelectorAll\n});\n\n// src/injected/TextContent.ts\nvar TRIVIAL_VALUE_INPUT_TYPES = /* @__PURE__ */ new Set([\"checkbox\", \"image\", \"radio\"]);\nvar isNonTrivialValueNode = (node) => {\n  if (node instanceof HTMLSelectElement) {\n    return true;\n  }\n  if (node instanceof HTMLTextAreaElement) {\n    return true;\n  }\n  if (node instanceof HTMLInputElement && !TRIVIAL_VALUE_INPUT_TYPES.has(node.type)) {\n    return true;\n  }\n  return false;\n};\nvar UNSUITABLE_NODE_NAMES = /* @__PURE__ */ new Set([\"SCRIPT\", \"STYLE\"]);\nvar isSuitableNodeForTextMatching = (node) => {\n  return !UNSUITABLE_NODE_NAMES.has(node.nodeName) && !document.head?.contains(node);\n};\nvar textContentCache = /* @__PURE__ */ new WeakMap();\nvar eraseFromCache = (node) => {\n  while (node) {\n    textContentCache.delete(node);\n    if (node instanceof ShadowRoot) {\n      node = node.host;\n    } else {\n      node = node.parentNode;\n    }\n  }\n};\nvar observedNodes = /* @__PURE__ */ new WeakSet();\nvar textChangeObserver = new MutationObserver((mutations) => {\n  for (const mutation of mutations) {\n    eraseFromCache(mutation.target);\n  }\n});\nvar createTextContent = (root) => {\n  let value = textContentCache.get(root);\n  if (value) {\n    return value;\n  }\n  value = { full: \"\", immediate: [] };\n  if (!isSuitableNodeForTextMatching(root)) {\n    return value;\n  }\n  let currentImmediate = \"\";\n  if (isNonTrivialValueNode(root)) {\n    value.full = root.value;\n    value.immediate.push(root.value);\n    root.addEventListener(\n      \"input\",\n      (event) => {\n        eraseFromCache(event.target);\n      },\n      { once: true, capture: true }\n    );\n  } else {\n    for (let child = root.firstChild; child; child = child.nextSibling) {\n      if (child.nodeType === Node.TEXT_NODE) {\n        value.full += child.nodeValue ?? \"\";\n        currentImmediate += child.nodeValue ?? \"\";\n        continue;\n      }\n      if (currentImmediate) {\n        value.immediate.push(currentImmediate);\n      }\n      currentImmediate = \"\";\n      if (child.nodeType === Node.ELEMENT_NODE) {\n        value.full += createTextContent(child).full;\n      }\n    }\n    if (currentImmediate) {\n      value.immediate.push(currentImmediate);\n    }\n    if (root instanceof Element && root.shadowRoot) {\n      value.full += createTextContent(root.shadowRoot).full;\n    }\n    if (!observedNodes.has(root)) {\n      textChangeObserver.observe(root, {\n        childList: true,\n        characterData: true\n      });\n      observedNodes.add(root);\n    }\n  }\n  textContentCache.set(root, value);\n  return value;\n};\n\n// src/injected/TextQuerySelector.ts\nvar textQuerySelectorAll = function* (root, selector) {\n  let yielded = false;\n  for (const node of root.childNodes) {\n    if (node instanceof Element && isSuitableNodeForTextMatching(node)) {\n      let matches;\n      if (!node.shadowRoot) {\n        matches = textQuerySelectorAll(node, selector);\n      } else {\n        matches = textQuerySelectorAll(node.shadowRoot, selector);\n      }\n      for (const match of matches) {\n        yield match;\n        yielded = true;\n      }\n    }\n  }\n  if (yielded) {\n    return;\n  }\n  if (root instanceof Element && isSuitableNodeForTextMatching(root)) {\n    const textContent = createTextContent(root);\n    if (textContent.full.includes(selector)) {\n      yield root;\n    }\n  }\n};\n\n// src/injected/util.ts\nvar util_exports = {};\n__export(util_exports, {\n  checkVisibility: () => checkVisibility,\n  deepChildren: () => deepChildren,\n  deepDescendents: () => deepDescendents\n});\nvar HIDDEN_VISIBILITY_VALUES = [\"hidden\", \"collapse\"];\nvar checkVisibility = (node, visible) => {\n  if (!node) {\n    return visible === false;\n  }\n  if (visible === void 0) {\n    return node;\n  }\n  const element = node.nodeType === Node.TEXT_NODE ? node.parentElement : node;\n  const style = window.getComputedStyle(element);\n  const isVisible = style && !HIDDEN_VISIBILITY_VALUES.includes(style.visibility) && !isBoundingBoxEmpty(element);\n  return visible === isVisible ? node : false;\n};\nfunction isBoundingBoxEmpty(element) {\n  const rect = element.getBoundingClientRect();\n  return rect.width === 0 || rect.height === 0;\n}\nfunction* deepChildren(root) {\n  const walker = document.createTreeWalker(root, NodeFilter.SHOW_ELEMENT);\n  let node = walker.nextNode();\n  for (; node; node = walker.nextNode()) {\n    yield node.shadowRoot ?? node;\n  }\n}\nfunction* deepDescendents(root) {\n  const walkers = [document.createTreeWalker(root, NodeFilter.SHOW_ELEMENT)];\n  let walker;\n  while (walker = walkers.shift()) {\n    for (let node = walker.nextNode(); node; node = walker.nextNode()) {\n      if (!node.shadowRoot) {\n        yield node;\n        continue;\n      }\n      walkers.push(\n        document.createTreeWalker(node.shadowRoot, NodeFilter.SHOW_ELEMENT)\n      );\n      yield node.shadowRoot;\n    }\n  }\n}\n\n// src/injected/XPathQuerySelector.ts\nvar XPathQuerySelector_exports = {};\n__export(XPathQuerySelector_exports, {\n  xpathQuerySelectorAll: () => xpathQuerySelectorAll\n});\nvar xpathQuerySelectorAll = function* (root, selector) {\n  const doc = root.ownerDocument || document;\n  const iterator = doc.evaluate(\n    selector,\n    root,\n    null,\n    XPathResult.ORDERED_NODE_ITERATOR_TYPE\n  );\n  let item;\n  while (item = iterator.iterateNext()) {\n    yield item;\n  }\n};\n\n// src/injected/PQuerySelector.ts\nvar SelectorError = class extends Error {\n  constructor(selector, message) {\n    super(`${selector} is not a valid selector: ${message}`);\n  }\n};\nvar PQueryEngine = class {\n  #input;\n  #deepShadowSelectors;\n  #shadowSelectors;\n  #selectors;\n  #selector;\n  elements;\n  constructor(element, selector) {\n    this.#input = selector.trim();\n    if (this.#input.length === 0) {\n      throw new SelectorError(this.#input, \"The provided selector is empty.\");\n    }\n    try {\n      this.#deepShadowSelectors = parsePSelectors(this.#input);\n    } catch (error) {\n      if (!isErrorLike(error)) {\n        throw new SelectorError(this.#input, String(error));\n      }\n      throw new SelectorError(this.#input, error.message);\n    }\n    if (this.#deepShadowSelectors.some((shadowSelectors) => {\n      return shadowSelectors.some((selectors) => {\n        return selectors.length === 0;\n      });\n    })) {\n      throw new SelectorError(\n        this.#input,\n        \"Multiple deep combinators found in sequence.\"\n      );\n    }\n    this.#shadowSelectors = this.#deepShadowSelectors.shift();\n    this.#selectors = this.#shadowSelectors.shift();\n    this.#selector = this.#selectors.shift();\n    this.elements = [element];\n  }\n  async run() {\n    if (typeof this.#selector === \"string\") {\n      switch (this.#selector.trimStart()) {\n        case \":scope\":\n          this.#next();\n          break;\n        default:\n          this.#selector = ` ${this.#selector}`;\n          break;\n      }\n    }\n    for (; this.#selector !== void 0; this.#next()) {\n      const selector = this.#selector;\n      const input = this.#input;\n      this.elements = AsyncIterableUtil.flatMap(\n        this.elements,\n        async function* (element) {\n          if (typeof selector === \"string\") {\n            if (!element.parentElement) {\n              yield* element.querySelectorAll(selector);\n              return;\n            }\n            let index = 0;\n            for (const child of element.parentElement.children) {\n              ++index;\n              if (child === element) {\n                break;\n              }\n            }\n            yield* element.parentElement.querySelectorAll(\n              `:scope > :nth-child(${index})${selector}`\n            );\n            return;\n          }\n          switch (selector.name) {\n            case \"text\":\n              yield* textQuerySelectorAll(element, selector.value);\n              break;\n            case \"xpath\":\n              yield* xpathQuerySelectorAll(element, selector.value);\n              break;\n            case \"aria\":\n              yield* ariaQuerySelectorAll(element, selector.value);\n              break;\n            default:\n              const querySelector = customQuerySelectors.get(selector.name);\n              if (!querySelector) {\n                throw new SelectorError(\n                  input,\n                  `Unknown selector type: ${selector.name}`\n                );\n              }\n              yield* querySelector.querySelectorAll(element, selector.value);\n          }\n        }\n      );\n    }\n  }\n  #next() {\n    if (this.#selectors.length === 0) {\n      if (this.#shadowSelectors.length === 0) {\n        if (this.#deepShadowSelectors.length === 0) {\n          this.#selector = void 0;\n          return;\n        }\n        this.elements = AsyncIterableUtil.flatMap(\n          this.elements,\n          function* (element) {\n            yield* deepDescendents(element);\n          }\n        );\n        this.#shadowSelectors = this.#deepShadowSelectors.shift();\n      }\n      this.elements = AsyncIterableUtil.flatMap(\n        this.elements,\n        function* (element) {\n          yield* deepChildren(element);\n        }\n      );\n      this.#selectors = this.#shadowSelectors.shift();\n    }\n    this.#selector = this.#selectors.shift();\n  }\n};\nvar pQuerySelectorAll = async function* (root, selector) {\n  const query = new PQueryEngine(root, selector);\n  query.run();\n  yield* query.elements;\n};\nvar pQuerySelector = async function(root, selector) {\n  for await (const element of pQuerySelectorAll(root, selector)) {\n    return element;\n  }\n  return null;\n};\n\n// src/injected/injected.ts\nvar PuppeteerUtil = Object.freeze({\n  ...ARIAQuerySelector_exports,\n  ...CustomQuerySelector_exports,\n  ...PierceQuerySelector_exports,\n  ...PQuerySelector_exports,\n  ...TextQuerySelector_exports,\n  ...util_exports,\n  ...XPathQuerySelector_exports,\n  createDeferredPromise,\n  createFunction,\n  createTextContent,\n  IntervalPoller,\n  isSuitableNodeForTextMatching,\n  MutationPoller,\n  RAFPoller\n});\nvar injected_default = PuppeteerUtil;\n";
