// Copyright (c) 2015 Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// ntstatus_reason_win.h: Windows NTSTATUS code to string.
//
// Provides a means to convert NTSTATUS codes to strings.
//
// Author: Ben Wagner

#include <algorithm>
#include <string>

#include "common/stdio_wrapper.h"
#include "google_breakpad/common/breakpad_types.h"
#include "google_breakpad/common/minidump_exception_win32.h"
#include "processor/symbolic_constants_win.h"

namespace google_breakpad {

// The content of this array was created from ntstatus.h in the 10 SDK
// (version 10.0.19041.0) with
//
// egrep '#define [A-Z_0-9]+\s+\(\(NTSTATUS\)0x[048C][0-9A-F]+L\)' ntstatus.h
// | tr -d '\r'
// | sed -r 's@#define ([A-Z_0-9]+)\s+\(\(NTSTATUS\)(0x[048C][0-9A-F]+)L\).*@\2 \1@'
// | sort
// | sed -r 's@(0x[048C][0-9A-F]+) ([A-Z_0-9]+)@  MD_NTSTATUS_WIN_\2,@'
//
// With easy copy to clipboard with
// | xclip -selection c  # on linux
// | clip  # on windows
// | pbcopy  # on mac
static const uint32_t kNTStatusValues[] = {
  MD_NTSTATUS_WIN_STATUS_SUCCESS,
  MD_NTSTATUS_WIN_STATUS_WAIT_0,
  MD_NTSTATUS_WIN_STATUS_WAIT_1,
  MD_NTSTATUS_WIN_STATUS_WAIT_2,
  MD_NTSTATUS_WIN_STATUS_WAIT_3,
  MD_NTSTATUS_WIN_STATUS_WAIT_63,
  MD_NTSTATUS_WIN_STATUS_ABANDONED,
  MD_NTSTATUS_WIN_STATUS_ABANDONED_WAIT_0,
  MD_NTSTATUS_WIN_STATUS_ABANDONED_WAIT_63,
  MD_NTSTATUS_WIN_STATUS_USER_APC,
  MD_NTSTATUS_WIN_STATUS_ALREADY_COMPLETE,
  MD_NTSTATUS_WIN_STATUS_KERNEL_APC,
  MD_NTSTATUS_WIN_STATUS_ALERTED,
  MD_NTSTATUS_WIN_STATUS_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_PENDING,
  MD_NTSTATUS_WIN_STATUS_REPARSE,
  MD_NTSTATUS_WIN_STATUS_MORE_ENTRIES,
  MD_NTSTATUS_WIN_STATUS_NOT_ALL_ASSIGNED,
  MD_NTSTATUS_WIN_STATUS_SOME_NOT_MAPPED,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_BREAK_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_VOLUME_MOUNTED,
  MD_NTSTATUS_WIN_STATUS_RXACT_COMMITTED,
  MD_NTSTATUS_WIN_STATUS_NOTIFY_CLEANUP,
  MD_NTSTATUS_WIN_STATUS_NOTIFY_ENUM_DIR,
  MD_NTSTATUS_WIN_STATUS_NO_QUOTAS_FOR_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_TRANSITION,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_DEMAND_ZERO,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_COPY_ON_WRITE,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_GUARD_PAGE,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_PAGING_FILE,
  MD_NTSTATUS_WIN_STATUS_CACHE_PAGE_LOCKED,
  MD_NTSTATUS_WIN_STATUS_CRASH_DUMP,
  MD_NTSTATUS_WIN_STATUS_BUFFER_ALL_ZEROS,
  MD_NTSTATUS_WIN_STATUS_REPARSE_OBJECT,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_REQUIREMENTS_CHANGED,
  MD_NTSTATUS_WIN_STATUS_TRANSLATION_COMPLETE,
  MD_NTSTATUS_WIN_STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY,
  MD_NTSTATUS_WIN_STATUS_NOTHING_TO_TERMINATE,
  MD_NTSTATUS_WIN_STATUS_PROCESS_NOT_IN_JOB,
  MD_NTSTATUS_WIN_STATUS_PROCESS_IN_JOB,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_HIBERNATE_READY,
  MD_NTSTATUS_WIN_STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY,
  MD_NTSTATUS_WIN_STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_INTERRUPT_STILL_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_PROCESS_CLONED,
  MD_NTSTATUS_WIN_STATUS_FILE_LOCKED_WITH_ONLY_READERS,
  MD_NTSTATUS_WIN_STATUS_FILE_LOCKED_WITH_WRITERS,
  MD_NTSTATUS_WIN_STATUS_VALID_IMAGE_HASH,
  MD_NTSTATUS_WIN_STATUS_VALID_CATALOG_HASH,
  MD_NTSTATUS_WIN_STATUS_VALID_STRONG_CODE_HASH,
  MD_NTSTATUS_WIN_STATUS_GHOSTED,
  MD_NTSTATUS_WIN_STATUS_DATA_OVERWRITTEN,
  MD_NTSTATUS_WIN_STATUS_RESOURCEMANAGER_READ_ONLY,
  MD_NTSTATUS_WIN_STATUS_RING_PREVIOUSLY_EMPTY,
  MD_NTSTATUS_WIN_STATUS_RING_PREVIOUSLY_FULL,
  MD_NTSTATUS_WIN_STATUS_RING_PREVIOUSLY_ABOVE_QUOTA,
  MD_NTSTATUS_WIN_STATUS_RING_NEWLY_EMPTY,
  MD_NTSTATUS_WIN_STATUS_RING_SIGNAL_OPPOSITE_ENDPOINT,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_SWITCHED_TO_NEW_HANDLE,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_HANDLE_CLOSED,
  MD_NTSTATUS_WIN_STATUS_WAIT_FOR_OPLOCK,
  MD_NTSTATUS_WIN_STATUS_REPARSE_GLOBAL,
  MD_NTSTATUS_WIN_DBG_EXCEPTION_HANDLED,
  MD_NTSTATUS_WIN_DBG_CONTINUE,
  MD_NTSTATUS_WIN_STATUS_FLT_IO_COMPLETE,
  MD_NTSTATUS_WIN_STATUS_RTPM_CONTEXT_CONTINUE,
  MD_NTSTATUS_WIN_STATUS_RTPM_CONTEXT_COMPLETE,
  MD_NTSTATUS_WIN_STATUS_HV_PENDING_PAGE_REQUESTS,
  MD_NTSTATUS_WIN_STATUS_SPACES_REPAIRED,
  MD_NTSTATUS_WIN_STATUS_SPACES_PAUSE,
  MD_NTSTATUS_WIN_STATUS_SPACES_COMPLETE,
  MD_NTSTATUS_WIN_STATUS_SPACES_REDIRECT,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_EXISTS,
  MD_NTSTATUS_WIN_STATUS_THREAD_WAS_SUSPENDED,
  MD_NTSTATUS_WIN_STATUS_WORKING_SET_LIMIT_RANGE,
  MD_NTSTATUS_WIN_STATUS_IMAGE_NOT_AT_BASE,
  MD_NTSTATUS_WIN_STATUS_RXACT_STATE_CREATED,
  MD_NTSTATUS_WIN_STATUS_SEGMENT_NOTIFICATION,
  MD_NTSTATUS_WIN_STATUS_LOCAL_USER_SESSION_KEY,
  MD_NTSTATUS_WIN_STATUS_BAD_CURRENT_DIRECTORY,
  MD_NTSTATUS_WIN_STATUS_SERIAL_MORE_WRITES,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_RECOVERED,
  MD_NTSTATUS_WIN_STATUS_FT_READ_RECOVERY_FROM_BACKUP,
  MD_NTSTATUS_WIN_STATUS_FT_WRITE_RECOVERY,
  MD_NTSTATUS_WIN_STATUS_SERIAL_COUNTER_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_NULL_LM_PASSWORD,
  MD_NTSTATUS_WIN_STATUS_IMAGE_MACHINE_TYPE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_RECEIVE_PARTIAL,
  MD_NTSTATUS_WIN_STATUS_RECEIVE_EXPEDITED,
  MD_NTSTATUS_WIN_STATUS_RECEIVE_PARTIAL_EXPEDITED,
  MD_NTSTATUS_WIN_STATUS_EVENT_DONE,
  MD_NTSTATUS_WIN_STATUS_EVENT_PENDING,
  MD_NTSTATUS_WIN_STATUS_CHECKING_FILE_SYSTEM,
  MD_NTSTATUS_WIN_STATUS_FATAL_APP_EXIT,
  MD_NTSTATUS_WIN_STATUS_PREDEFINED_HANDLE,
  MD_NTSTATUS_WIN_STATUS_WAS_UNLOCKED,
  MD_NTSTATUS_WIN_STATUS_SERVICE_NOTIFICATION,
  MD_NTSTATUS_WIN_STATUS_WAS_LOCKED,
  MD_NTSTATUS_WIN_STATUS_LOG_HARD_ERROR,
  MD_NTSTATUS_WIN_STATUS_ALREADY_WIN32,
  MD_NTSTATUS_WIN_STATUS_WX86_UNSIMULATE,
  MD_NTSTATUS_WIN_STATUS_WX86_CONTINUE,
  MD_NTSTATUS_WIN_STATUS_WX86_SINGLE_STEP,
  MD_NTSTATUS_WIN_STATUS_WX86_BREAKPOINT,
  MD_NTSTATUS_WIN_STATUS_WX86_EXCEPTION_CONTINUE,
  MD_NTSTATUS_WIN_STATUS_WX86_EXCEPTION_LASTCHANCE,
  MD_NTSTATUS_WIN_STATUS_WX86_EXCEPTION_CHAIN,
  MD_NTSTATUS_WIN_STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE,
  MD_NTSTATUS_WIN_STATUS_NO_YIELD_PERFORMED,
  MD_NTSTATUS_WIN_STATUS_TIMER_RESUME_IGNORED,
  MD_NTSTATUS_WIN_STATUS_ARBITRATION_UNHANDLED,
  MD_NTSTATUS_WIN_STATUS_CARDBUS_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_WX86_CREATEWX86TIB,
  MD_NTSTATUS_WIN_STATUS_MP_PROCESSOR_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_HIBERNATED,
  MD_NTSTATUS_WIN_STATUS_RESUME_HIBERNATION,
  MD_NTSTATUS_WIN_STATUS_FIRMWARE_UPDATED,
  MD_NTSTATUS_WIN_STATUS_DRIVERS_LEAKING_LOCKED_PAGES,
  MD_NTSTATUS_WIN_STATUS_MESSAGE_RETRIEVED,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_POWERSTATE_TRANSITION,
  MD_NTSTATUS_WIN_STATUS_ALPC_CHECK_COMPLETION_LIST,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION,
  MD_NTSTATUS_WIN_STATUS_ACCESS_AUDIT_BY_POLICY,
  MD_NTSTATUS_WIN_STATUS_ABANDON_HIBERFILE,
  MD_NTSTATUS_WIN_STATUS_BIZRULES_NOT_ENABLED,
  MD_NTSTATUS_WIN_STATUS_FT_READ_FROM_COPY,
  MD_NTSTATUS_WIN_STATUS_IMAGE_AT_DIFFERENT_BASE,
  MD_NTSTATUS_WIN_STATUS_PATCH_DEFERRED,
  MD_NTSTATUS_WIN_STATUS_WAKE_SYSTEM,
  MD_NTSTATUS_WIN_STATUS_DS_SHUTTING_DOWN,
  MD_NTSTATUS_WIN_STATUS_DISK_REPAIR_REDIRECTED,
  MD_NTSTATUS_WIN_STATUS_SERVICES_FAILED_AUTOSTART,
  MD_NTSTATUS_WIN_DBG_REPLY_LATER,
  MD_NTSTATUS_WIN_DBG_UNABLE_TO_PROVIDE_HANDLE,
  MD_NTSTATUS_WIN_DBG_TERMINATE_THREAD,
  MD_NTSTATUS_WIN_DBG_TERMINATE_PROCESS,
  MD_NTSTATUS_WIN_DBG_CONTROL_C,
  MD_NTSTATUS_WIN_DBG_PRINTEXCEPTION_C,
  MD_NTSTATUS_WIN_DBG_RIPEXCEPTION,
  MD_NTSTATUS_WIN_DBG_CONTROL_BREAK,
  MD_NTSTATUS_WIN_DBG_COMMAND_EXCEPTION,
  MD_NTSTATUS_WIN_DBG_PRINTEXCEPTION_WIDE_C,
  MD_NTSTATUS_WIN_RPC_NT_UUID_LOCAL_ONLY,
  MD_NTSTATUS_WIN_RPC_NT_SEND_INCOMPLETE,
  MD_NTSTATUS_WIN_STATUS_CTX_CDM_CONNECT,
  MD_NTSTATUS_WIN_STATUS_CTX_CDM_DISCONNECT,
  MD_NTSTATUS_WIN_STATUS_SXS_RELEASE_ACTIVATION_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_HEURISTIC_DAMAGE_POSSIBLE,
  MD_NTSTATUS_WIN_STATUS_RECOVERY_NOT_NEEDED,
  MD_NTSTATUS_WIN_STATUS_RM_ALREADY_STARTED,
  MD_NTSTATUS_WIN_STATUS_LOG_NO_RESTART,
  MD_NTSTATUS_WIN_STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PARTIAL_DATA_POPULATED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SKIP_ALLOCATION_PREPARATION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_NOT_PINNED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_PREFERRED_MODE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DATASET_IS_EMPTY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_LEADLINK_START_DEFERRED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_START_DEFERRED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DEPENDABLE_CHILD_STATUS,
  MD_NTSTATUS_WIN_STATUS_NDIS_INDICATION_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_PCP_UNSUPPORTED_PSS_SALT,
  MD_NTSTATUS_WIN_STATUS_GUARD_PAGE_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_DATATYPE_MISALIGNMENT,
  MD_NTSTATUS_WIN_STATUS_BREAKPOINT,
  MD_NTSTATUS_WIN_STATUS_SINGLE_STEP,
  MD_NTSTATUS_WIN_STATUS_BUFFER_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_FILES,
  MD_NTSTATUS_WIN_STATUS_WAKE_SYSTEM_DEBUGGER,
  MD_NTSTATUS_WIN_STATUS_HANDLES_CLOSED,
  MD_NTSTATUS_WIN_STATUS_NO_INHERITANCE,
  MD_NTSTATUS_WIN_STATUS_GUID_SUBSTITUTION_MADE,
  MD_NTSTATUS_WIN_STATUS_PARTIAL_COPY,
  MD_NTSTATUS_WIN_STATUS_DEVICE_PAPER_EMPTY,
  MD_NTSTATUS_WIN_STATUS_DEVICE_POWERED_OFF,
  MD_NTSTATUS_WIN_STATUS_DEVICE_OFF_LINE,
  MD_NTSTATUS_WIN_STATUS_DEVICE_BUSY,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_EAS,
  MD_NTSTATUS_WIN_STATUS_INVALID_EA_NAME,
  MD_NTSTATUS_WIN_STATUS_EA_LIST_INCONSISTENT,
  MD_NTSTATUS_WIN_STATUS_INVALID_EA_FLAG,
  MD_NTSTATUS_WIN_STATUS_VERIFY_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_EXTRANEOUS_INFORMATION,
  MD_NTSTATUS_WIN_STATUS_RXACT_COMMIT_NECESSARY,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_ENTRIES,
  MD_NTSTATUS_WIN_STATUS_FILEMARK_DETECTED,
  MD_NTSTATUS_WIN_STATUS_MEDIA_CHANGED,
  MD_NTSTATUS_WIN_STATUS_BUS_RESET,
  MD_NTSTATUS_WIN_STATUS_END_OF_MEDIA,
  MD_NTSTATUS_WIN_STATUS_BEGINNING_OF_MEDIA,
  MD_NTSTATUS_WIN_STATUS_MEDIA_CHECK,
  MD_NTSTATUS_WIN_STATUS_SETMARK_DETECTED,
  MD_NTSTATUS_WIN_STATUS_NO_DATA_DETECTED,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_HAS_OPEN_HANDLES,
  MD_NTSTATUS_WIN_STATUS_SERVER_HAS_OPEN_HANDLES,
  MD_NTSTATUS_WIN_STATUS_ALREADY_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_LONGJUMP,
  MD_NTSTATUS_WIN_STATUS_CLEANER_CARTRIDGE_INSTALLED,
  MD_NTSTATUS_WIN_STATUS_PLUGPLAY_QUERY_VETOED,
  MD_NTSTATUS_WIN_STATUS_UNWIND_CONSOLIDATE,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_HIVE_RECOVERED,
  MD_NTSTATUS_WIN_STATUS_DLL_MIGHT_BE_INSECURE,
  MD_NTSTATUS_WIN_STATUS_DLL_MIGHT_BE_INCOMPATIBLE,
  MD_NTSTATUS_WIN_STATUS_STOPPED_ON_SYMLINK,
  MD_NTSTATUS_WIN_STATUS_CANNOT_GRANT_REQUESTED_OPLOCK,
  MD_NTSTATUS_WIN_STATUS_NO_ACE_CONDITION,
  MD_NTSTATUS_WIN_STATUS_DEVICE_SUPPORT_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_DEVICE_POWER_CYCLE_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_NO_WORK_DONE,
  MD_NTSTATUS_WIN_STATUS_RETURN_ADDRESS_HIJACK_ATTEMPT,
  MD_NTSTATUS_WIN_STATUS_DEVICE_REQUIRES_CLEANING,
  MD_NTSTATUS_WIN_STATUS_DEVICE_DOOR_OPEN,
  MD_NTSTATUS_WIN_STATUS_DATA_LOST_REPAIR,
  MD_NTSTATUS_WIN_STATUS_GPIO_INTERRUPT_ALREADY_UNMASKED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS,
  MD_NTSTATUS_WIN_DBG_EXCEPTION_NOT_HANDLED,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_ALREADY_UP,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_ALREADY_DOWN,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_ALREADY_ONLINE,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_ALREADY_MEMBER,
  MD_NTSTATUS_WIN_STATUS_COULD_NOT_RESIZE_LOG,
  MD_NTSTATUS_WIN_STATUS_NO_TXF_METADATA,
  MD_NTSTATUS_WIN_STATUS_CANT_RECOVER_WITH_HANDLE_OPEN,
  MD_NTSTATUS_WIN_STATUS_TXF_METADATA_ALREADY_PRESENT,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET,
  MD_NTSTATUS_WIN_STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED,
  MD_NTSTATUS_WIN_STATUS_FLT_BUFFER_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_FVE_PARTIAL_METADATA,
  MD_NTSTATUS_WIN_STATUS_FVE_TRANSIENT_STATE,
  MD_NTSTATUS_WIN_STATUS_VID_REMOTE_NODE_PARENT_GPA_PAGES_USED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_INCOMPLETE_REGENERATION,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_INCOMPLETE_DISK_MIGRATION,
  MD_NTSTATUS_WIN_STATUS_BCD_NOT_ALL_ENTRIES_IMPORTED,
  MD_NTSTATUS_WIN_STATUS_BCD_NOT_ALL_ENTRIES_SYNCHRONIZED,
  MD_NTSTATUS_WIN_STATUS_QUERY_STORAGE_ERROR,
  MD_NTSTATUS_WIN_STATUS_GDI_HANDLE_LEAK,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_NOT_ENABLED,
  MD_NTSTATUS_WIN_STATUS_UNSUCCESSFUL,
  MD_NTSTATUS_WIN_STATUS_NOT_IMPLEMENTED,
  MD_NTSTATUS_WIN_STATUS_INVALID_INFO_CLASS,
  MD_NTSTATUS_WIN_STATUS_INFO_LENGTH_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_ACCESS_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_IN_PAGE_ERROR,
  MD_NTSTATUS_WIN_STATUS_PAGEFILE_QUOTA,
  MD_NTSTATUS_WIN_STATUS_INVALID_HANDLE,
  MD_NTSTATUS_WIN_STATUS_BAD_INITIAL_STACK,
  MD_NTSTATUS_WIN_STATUS_BAD_INITIAL_PC,
  MD_NTSTATUS_WIN_STATUS_INVALID_CID,
  MD_NTSTATUS_WIN_STATUS_TIMER_NOT_CANCELED,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_DEVICE,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_FILE,
  MD_NTSTATUS_WIN_STATUS_INVALID_DEVICE_REQUEST,
  MD_NTSTATUS_WIN_STATUS_END_OF_FILE,
  MD_NTSTATUS_WIN_STATUS_WRONG_VOLUME,
  MD_NTSTATUS_WIN_STATUS_NO_MEDIA_IN_DEVICE,
  MD_NTSTATUS_WIN_STATUS_UNRECOGNIZED_MEDIA,
  MD_NTSTATUS_WIN_STATUS_NONEXISTENT_SECTOR,
  MD_NTSTATUS_WIN_STATUS_MORE_PROCESSING_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_NO_MEMORY,
  MD_NTSTATUS_WIN_STATUS_CONFLICTING_ADDRESSES,
  MD_NTSTATUS_WIN_STATUS_NOT_MAPPED_VIEW,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_FREE_VM,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_DELETE_SECTION,
  MD_NTSTATUS_WIN_STATUS_INVALID_SYSTEM_SERVICE,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_INSTRUCTION,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOCK_SEQUENCE,
  MD_NTSTATUS_WIN_STATUS_INVALID_VIEW_SIZE,
  MD_NTSTATUS_WIN_STATUS_INVALID_FILE_FOR_SECTION,
  MD_NTSTATUS_WIN_STATUS_ALREADY_COMMITTED,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_BUFFER_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_OBJECT_TYPE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_NONCONTINUABLE_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_INVALID_DISPOSITION,
  MD_NTSTATUS_WIN_STATUS_UNWIND,
  MD_NTSTATUS_WIN_STATUS_BAD_STACK,
  MD_NTSTATUS_WIN_STATUS_INVALID_UNWIND_TARGET,
  MD_NTSTATUS_WIN_STATUS_NOT_LOCKED,
  MD_NTSTATUS_WIN_STATUS_PARITY_ERROR,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_DECOMMIT_VM,
  MD_NTSTATUS_WIN_STATUS_NOT_COMMITTED,
  MD_NTSTATUS_WIN_STATUS_INVALID_PORT_ATTRIBUTES,
  MD_NTSTATUS_WIN_STATUS_PORT_MESSAGE_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_MIX,
  MD_NTSTATUS_WIN_STATUS_INVALID_QUOTA_LOWER,
  MD_NTSTATUS_WIN_STATUS_DISK_CORRUPT_ERROR,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_INVALID,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_COLLISION,
  MD_NTSTATUS_WIN_STATUS_PORT_DO_NOT_DISTURB,
  MD_NTSTATUS_WIN_STATUS_PORT_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_DEVICE_ALREADY_ATTACHED,
  MD_NTSTATUS_WIN_STATUS_OBJECT_PATH_INVALID,
  MD_NTSTATUS_WIN_STATUS_OBJECT_PATH_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_OBJECT_PATH_SYNTAX_BAD,
  MD_NTSTATUS_WIN_STATUS_DATA_OVERRUN,
  MD_NTSTATUS_WIN_STATUS_DATA_LATE_ERROR,
  MD_NTSTATUS_WIN_STATUS_DATA_ERROR,
  MD_NTSTATUS_WIN_STATUS_CRC_ERROR,
  MD_NTSTATUS_WIN_STATUS_SECTION_TOO_BIG,
  MD_NTSTATUS_WIN_STATUS_PORT_CONNECTION_REFUSED,
  MD_NTSTATUS_WIN_STATUS_INVALID_PORT_HANDLE,
  MD_NTSTATUS_WIN_STATUS_SHARING_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_INVALID_PAGE_PROTECTION,
  MD_NTSTATUS_WIN_STATUS_MUTANT_NOT_OWNED,
  MD_NTSTATUS_WIN_STATUS_SEMAPHORE_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_PORT_ALREADY_SET,
  MD_NTSTATUS_WIN_STATUS_SECTION_NOT_IMAGE,
  MD_NTSTATUS_WIN_STATUS_SUSPEND_COUNT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_THREAD_IS_TERMINATING,
  MD_NTSTATUS_WIN_STATUS_BAD_WORKING_SET_LIMIT,
  MD_NTSTATUS_WIN_STATUS_INCOMPATIBLE_FILE_MAP,
  MD_NTSTATUS_WIN_STATUS_SECTION_PROTECTION,
  MD_NTSTATUS_WIN_STATUS_EAS_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_EA_TOO_LARGE,
  MD_NTSTATUS_WIN_STATUS_NONEXISTENT_EA_ENTRY,
  MD_NTSTATUS_WIN_STATUS_NO_EAS_ON_FILE,
  MD_NTSTATUS_WIN_STATUS_EA_CORRUPT_ERROR,
  MD_NTSTATUS_WIN_STATUS_FILE_LOCK_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_LOCK_NOT_GRANTED,
  MD_NTSTATUS_WIN_STATUS_DELETE_PENDING,
  MD_NTSTATUS_WIN_STATUS_CTL_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_UNKNOWN_REVISION,
  MD_NTSTATUS_WIN_STATUS_REVISION_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_INVALID_OWNER,
  MD_NTSTATUS_WIN_STATUS_INVALID_PRIMARY_GROUP,
  MD_NTSTATUS_WIN_STATUS_NO_IMPERSONATION_TOKEN,
  MD_NTSTATUS_WIN_STATUS_CANT_DISABLE_MANDATORY,
  MD_NTSTATUS_WIN_STATUS_NO_LOGON_SERVERS,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_LOGON_SESSION,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_PRIVILEGE,
  MD_NTSTATUS_WIN_STATUS_PRIVILEGE_NOT_HELD,
  MD_NTSTATUS_WIN_STATUS_INVALID_ACCOUNT_NAME,
  MD_NTSTATUS_WIN_STATUS_USER_EXISTS,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_USER,
  MD_NTSTATUS_WIN_STATUS_GROUP_EXISTS,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_GROUP,
  MD_NTSTATUS_WIN_STATUS_MEMBER_IN_GROUP,
  MD_NTSTATUS_WIN_STATUS_MEMBER_NOT_IN_GROUP,
  MD_NTSTATUS_WIN_STATUS_LAST_ADMIN,
  MD_NTSTATUS_WIN_STATUS_WRONG_PASSWORD,
  MD_NTSTATUS_WIN_STATUS_ILL_FORMED_PASSWORD,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_RESTRICTION,
  MD_NTSTATUS_WIN_STATUS_LOGON_FAILURE,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_RESTRICTION,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOGON_HOURS,
  MD_NTSTATUS_WIN_STATUS_INVALID_WORKSTATION,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_DISABLED,
  MD_NTSTATUS_WIN_STATUS_NONE_MAPPED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_LUIDS_REQUESTED,
  MD_NTSTATUS_WIN_STATUS_LUIDS_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_INVALID_SUB_AUTHORITY,
  MD_NTSTATUS_WIN_STATUS_INVALID_ACL,
  MD_NTSTATUS_WIN_STATUS_INVALID_SID,
  MD_NTSTATUS_WIN_STATUS_INVALID_SECURITY_DESCR,
  MD_NTSTATUS_WIN_STATUS_PROCEDURE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_FORMAT,
  MD_NTSTATUS_WIN_STATUS_NO_TOKEN,
  MD_NTSTATUS_WIN_STATUS_BAD_INHERITANCE_ACL,
  MD_NTSTATUS_WIN_STATUS_RANGE_NOT_LOCKED,
  MD_NTSTATUS_WIN_STATUS_DISK_FULL,
  MD_NTSTATUS_WIN_STATUS_SERVER_DISABLED,
  MD_NTSTATUS_WIN_STATUS_SERVER_NOT_DISABLED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_GUIDS_REQUESTED,
  MD_NTSTATUS_WIN_STATUS_GUIDS_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_INVALID_ID_AUTHORITY,
  MD_NTSTATUS_WIN_STATUS_AGENTS_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_INVALID_VOLUME_LABEL,
  MD_NTSTATUS_WIN_STATUS_SECTION_NOT_EXTENDED,
  MD_NTSTATUS_WIN_STATUS_NOT_MAPPED_DATA,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_DATA_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_TYPE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_NAME_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_ARRAY_BOUNDS_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_FLOAT_DENORMAL_OPERAND,
  MD_NTSTATUS_WIN_STATUS_FLOAT_DIVIDE_BY_ZERO,
  MD_NTSTATUS_WIN_STATUS_FLOAT_INEXACT_RESULT,
  MD_NTSTATUS_WIN_STATUS_FLOAT_INVALID_OPERATION,
  MD_NTSTATUS_WIN_STATUS_FLOAT_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_FLOAT_STACK_CHECK,
  MD_NTSTATUS_WIN_STATUS_FLOAT_UNDERFLOW,
  MD_NTSTATUS_WIN_STATUS_INTEGER_DIVIDE_BY_ZERO,
  MD_NTSTATUS_WIN_STATUS_INTEGER_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_PRIVILEGED_INSTRUCTION,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_PAGING_FILES,
  MD_NTSTATUS_WIN_STATUS_FILE_INVALID,
  MD_NTSTATUS_WIN_STATUS_ALLOTTED_SPACE_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_DFS_EXIT_PATH_FOUND,
  MD_NTSTATUS_WIN_STATUS_DEVICE_DATA_ERROR,
  MD_NTSTATUS_WIN_STATUS_DEVICE_NOT_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_DEVICE_POWER_FAILURE,
  MD_NTSTATUS_WIN_STATUS_FREE_VM_NOT_AT_BASE,
  MD_NTSTATUS_WIN_STATUS_MEMORY_NOT_ALLOCATED,
  MD_NTSTATUS_WIN_STATUS_WORKING_SET_QUOTA,
  MD_NTSTATUS_WIN_STATUS_MEDIA_WRITE_PROTECTED,
  MD_NTSTATUS_WIN_STATUS_DEVICE_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_INVALID_GROUP_ATTRIBUTES,
  MD_NTSTATUS_WIN_STATUS_BAD_IMPERSONATION_LEVEL,
  MD_NTSTATUS_WIN_STATUS_CANT_OPEN_ANONYMOUS,
  MD_NTSTATUS_WIN_STATUS_BAD_VALIDATION_CLASS,
  MD_NTSTATUS_WIN_STATUS_BAD_TOKEN_TYPE,
  MD_NTSTATUS_WIN_STATUS_BAD_MASTER_BOOT_RECORD,
  MD_NTSTATUS_WIN_STATUS_INSTRUCTION_MISALIGNMENT,
  MD_NTSTATUS_WIN_STATUS_INSTANCE_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_PIPE_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_INVALID_PIPE_STATE,
  MD_NTSTATUS_WIN_STATUS_PIPE_BUSY,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_FUNCTION,
  MD_NTSTATUS_WIN_STATUS_PIPE_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_PIPE_CLOSING,
  MD_NTSTATUS_WIN_STATUS_PIPE_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_PIPE_LISTENING,
  MD_NTSTATUS_WIN_STATUS_INVALID_READ_MODE,
  MD_NTSTATUS_WIN_STATUS_IO_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_FILE_FORCED_CLOSED,
  MD_NTSTATUS_WIN_STATUS_PROFILING_NOT_STARTED,
  MD_NTSTATUS_WIN_STATUS_PROFILING_NOT_STOPPED,
  MD_NTSTATUS_WIN_STATUS_COULD_NOT_INTERPRET,
  MD_NTSTATUS_WIN_STATUS_FILE_IS_A_DIRECTORY,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_REMOTE_NOT_LISTENING,
  MD_NTSTATUS_WIN_STATUS_DUPLICATE_NAME,
  MD_NTSTATUS_WIN_STATUS_BAD_NETWORK_PATH,
  MD_NTSTATUS_WIN_STATUS_NETWORK_BUSY,
  MD_NTSTATUS_WIN_STATUS_DEVICE_DOES_NOT_EXIST,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_COMMANDS,
  MD_NTSTATUS_WIN_STATUS_ADAPTER_HARDWARE_ERROR,
  MD_NTSTATUS_WIN_STATUS_INVALID_NETWORK_RESPONSE,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_NETWORK_ERROR,
  MD_NTSTATUS_WIN_STATUS_BAD_REMOTE_ADAPTER,
  MD_NTSTATUS_WIN_STATUS_PRINT_QUEUE_FULL,
  MD_NTSTATUS_WIN_STATUS_NO_SPOOL_SPACE,
  MD_NTSTATUS_WIN_STATUS_PRINT_CANCELLED,
  MD_NTSTATUS_WIN_STATUS_NETWORK_NAME_DELETED,
  MD_NTSTATUS_WIN_STATUS_NETWORK_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_BAD_DEVICE_TYPE,
  MD_NTSTATUS_WIN_STATUS_BAD_NETWORK_NAME,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_NAMES,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SESSIONS,
  MD_NTSTATUS_WIN_STATUS_SHARING_PAUSED,
  MD_NTSTATUS_WIN_STATUS_REQUEST_NOT_ACCEPTED,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_PAUSED,
  MD_NTSTATUS_WIN_STATUS_NET_WRITE_FAULT,
  MD_NTSTATUS_WIN_STATUS_PROFILING_AT_LIMIT,
  MD_NTSTATUS_WIN_STATUS_NOT_SAME_DEVICE,
  MD_NTSTATUS_WIN_STATUS_FILE_RENAMED,
  MD_NTSTATUS_WIN_STATUS_VIRTUAL_CIRCUIT_CLOSED,
  MD_NTSTATUS_WIN_STATUS_NO_SECURITY_ON_OBJECT,
  MD_NTSTATUS_WIN_STATUS_CANT_WAIT,
  MD_NTSTATUS_WIN_STATUS_PIPE_EMPTY,
  MD_NTSTATUS_WIN_STATUS_CANT_ACCESS_DOMAIN_INFO,
  MD_NTSTATUS_WIN_STATUS_CANT_TERMINATE_SELF,
  MD_NTSTATUS_WIN_STATUS_INVALID_SERVER_STATE,
  MD_NTSTATUS_WIN_STATUS_INVALID_DOMAIN_STATE,
  MD_NTSTATUS_WIN_STATUS_INVALID_DOMAIN_ROLE,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_DOMAIN,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_EXISTS,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_NOT_GRANTED,
  MD_NTSTATUS_WIN_STATUS_INVALID_OPLOCK_PROTOCOL,
  MD_NTSTATUS_WIN_STATUS_INTERNAL_DB_CORRUPTION,
  MD_NTSTATUS_WIN_STATUS_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_GENERIC_NOT_MAPPED,
  MD_NTSTATUS_WIN_STATUS_BAD_DESCRIPTOR_FORMAT,
  MD_NTSTATUS_WIN_STATUS_INVALID_USER_BUFFER,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_IO_ERROR,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_MM_CREATE_ERR,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_MM_MAP_ERROR,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_MM_EXTEND_ERR,
  MD_NTSTATUS_WIN_STATUS_NOT_LOGON_PROCESS,
  MD_NTSTATUS_WIN_STATUS_LOGON_SESSION_EXISTS,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_1,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_2,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_3,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_4,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_5,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_6,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_7,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_8,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_9,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_10,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_11,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_12,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_NOT_STARTED,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_STARTED,
  MD_NTSTATUS_WIN_STATUS_STACK_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_PACKAGE,
  MD_NTSTATUS_WIN_STATUS_BAD_FUNCTION_TABLE,
  MD_NTSTATUS_WIN_STATUS_VARIABLE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_NOT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_FILE_CORRUPT_ERROR,
  MD_NTSTATUS_WIN_STATUS_NOT_A_DIRECTORY,
  MD_NTSTATUS_WIN_STATUS_BAD_LOGON_SESSION_STATE,
  MD_NTSTATUS_WIN_STATUS_LOGON_SESSION_COLLISION,
  MD_NTSTATUS_WIN_STATUS_NAME_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_FILES_OPEN,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_IN_USE,
  MD_NTSTATUS_WIN_STATUS_MESSAGE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PROCESS_IS_TERMINATING,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOGON_TYPE,
  MD_NTSTATUS_WIN_STATUS_NO_GUID_TRANSLATION,
  MD_NTSTATUS_WIN_STATUS_CANNOT_IMPERSONATE,
  MD_NTSTATUS_WIN_STATUS_IMAGE_ALREADY_LOADED,
  MD_NTSTATUS_WIN_STATUS_ABIOS_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_ABIOS_LID_NOT_EXIST,
  MD_NTSTATUS_WIN_STATUS_ABIOS_LID_ALREADY_OWNED,
  MD_NTSTATUS_WIN_STATUS_ABIOS_NOT_LID_OWNER,
  MD_NTSTATUS_WIN_STATUS_ABIOS_INVALID_COMMAND,
  MD_NTSTATUS_WIN_STATUS_ABIOS_INVALID_LID,
  MD_NTSTATUS_WIN_STATUS_ABIOS_SELECTOR_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_ABIOS_INVALID_SELECTOR,
  MD_NTSTATUS_WIN_STATUS_NO_LDT,
  MD_NTSTATUS_WIN_STATUS_INVALID_LDT_SIZE,
  MD_NTSTATUS_WIN_STATUS_INVALID_LDT_OFFSET,
  MD_NTSTATUS_WIN_STATUS_INVALID_LDT_DESCRIPTOR,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_NE_FORMAT,
  MD_NTSTATUS_WIN_STATUS_RXACT_INVALID_STATE,
  MD_NTSTATUS_WIN_STATUS_RXACT_COMMIT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_MAPPED_FILE_SIZE_ZERO,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_OPENED_FILES,
  MD_NTSTATUS_WIN_STATUS_CANCELLED,
  MD_NTSTATUS_WIN_STATUS_CANNOT_DELETE,
  MD_NTSTATUS_WIN_STATUS_INVALID_COMPUTER_NAME,
  MD_NTSTATUS_WIN_STATUS_FILE_DELETED,
  MD_NTSTATUS_WIN_STATUS_SPECIAL_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_SPECIAL_GROUP,
  MD_NTSTATUS_WIN_STATUS_SPECIAL_USER,
  MD_NTSTATUS_WIN_STATUS_MEMBERS_PRIMARY_GROUP,
  MD_NTSTATUS_WIN_STATUS_FILE_CLOSED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_THREADS,
  MD_NTSTATUS_WIN_STATUS_THREAD_NOT_IN_PROCESS,
  MD_NTSTATUS_WIN_STATUS_TOKEN_ALREADY_IN_USE,
  MD_NTSTATUS_WIN_STATUS_PAGEFILE_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_COMMITMENT_LIMIT,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_LE_FORMAT,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_NOT_MZ,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_PROTECT,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_WIN_16,
  MD_NTSTATUS_WIN_STATUS_LOGON_SERVER_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_TIME_DIFFERENCE_AT_DC,
  MD_NTSTATUS_WIN_STATUS_SYNCHRONIZATION_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_DLL_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_OPEN_FAILED,
  MD_NTSTATUS_WIN_STATUS_IO_PRIVILEGE_FAILED,
  MD_NTSTATUS_WIN_STATUS_ORDINAL_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_ENTRYPOINT_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CONTROL_C_EXIT,
  MD_NTSTATUS_WIN_STATUS_LOCAL_DISCONNECT,
  MD_NTSTATUS_WIN_STATUS_REMOTE_DISCONNECT,
  MD_NTSTATUS_WIN_STATUS_REMOTE_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_LINK_FAILED,
  MD_NTSTATUS_WIN_STATUS_LINK_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_INVALID_CONNECTION,
  MD_NTSTATUS_WIN_STATUS_INVALID_ADDRESS,
  MD_NTSTATUS_WIN_STATUS_DLL_INIT_FAILED,
  MD_NTSTATUS_WIN_STATUS_MISSING_SYSTEMFILE,
  MD_NTSTATUS_WIN_STATUS_UNHANDLED_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_APP_INIT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_PAGEFILE_CREATE_FAILED,
  MD_NTSTATUS_WIN_STATUS_NO_PAGEFILE,
  MD_NTSTATUS_WIN_STATUS_INVALID_LEVEL,
  MD_NTSTATUS_WIN_STATUS_WRONG_PASSWORD_CORE,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_FLOAT_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_PIPE_BROKEN,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_IO_FAILED,
  MD_NTSTATUS_WIN_STATUS_NO_EVENT_PAIR,
  MD_NTSTATUS_WIN_STATUS_UNRECOGNIZED_VOLUME,
  MD_NTSTATUS_WIN_STATUS_SERIAL_NO_DEVICE_INITED,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_ALIAS,
  MD_NTSTATUS_WIN_STATUS_MEMBER_NOT_IN_ALIAS,
  MD_NTSTATUS_WIN_STATUS_MEMBER_IN_ALIAS,
  MD_NTSTATUS_WIN_STATUS_ALIAS_EXISTS,
  MD_NTSTATUS_WIN_STATUS_LOGON_NOT_GRANTED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SECRETS,
  MD_NTSTATUS_WIN_STATUS_SECRET_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_INTERNAL_DB_ERROR,
  MD_NTSTATUS_WIN_STATUS_FULLSCREEN_MODE,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_CONTEXT_IDS,
  MD_NTSTATUS_WIN_STATUS_LOGON_TYPE_NOT_GRANTED,
  MD_NTSTATUS_WIN_STATUS_NOT_REGISTRY_FILE,
  MD_NTSTATUS_WIN_STATUS_NT_CROSS_ENCRYPTION_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_CTRLR_CONFIG_ERROR,
  MD_NTSTATUS_WIN_STATUS_FT_MISSING_MEMBER,
  MD_NTSTATUS_WIN_STATUS_ILL_FORMED_SERVICE_ENTRY,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_CHARACTER,
  MD_NTSTATUS_WIN_STATUS_UNMAPPABLE_CHARACTER,
  MD_NTSTATUS_WIN_STATUS_UNDEFINED_CHARACTER,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_VOLUME,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_ID_MARK_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_WRONG_CYLINDER,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_UNKNOWN_ERROR,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_BAD_REGISTERS,
  MD_NTSTATUS_WIN_STATUS_DISK_RECALIBRATE_FAILED,
  MD_NTSTATUS_WIN_STATUS_DISK_OPERATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_DISK_RESET_FAILED,
  MD_NTSTATUS_WIN_STATUS_SHARED_IRQ_BUSY,
  MD_NTSTATUS_WIN_STATUS_FT_ORPHANING,
  MD_NTSTATUS_WIN_STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT,
  MD_NTSTATUS_WIN_STATUS_PARTITION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_INVALID_BLOCK_LENGTH,
  MD_NTSTATUS_WIN_STATUS_DEVICE_NOT_PARTITIONED,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_LOCK_MEDIA,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_UNLOAD_MEDIA,
  MD_NTSTATUS_WIN_STATUS_EOM_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_NO_MEDIA,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_MEMBER,
  MD_NTSTATUS_WIN_STATUS_INVALID_MEMBER,
  MD_NTSTATUS_WIN_STATUS_KEY_DELETED,
  MD_NTSTATUS_WIN_STATUS_NO_LOG_SPACE,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SIDS,
  MD_NTSTATUS_WIN_STATUS_LM_CROSS_ENCRYPTION_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_KEY_HAS_CHILDREN,
  MD_NTSTATUS_WIN_STATUS_CHILD_MUST_BE_VOLATILE,
  MD_NTSTATUS_WIN_STATUS_DEVICE_CONFIGURATION_ERROR,
  MD_NTSTATUS_WIN_STATUS_DRIVER_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_INVALID_DEVICE_STATE,
  MD_NTSTATUS_WIN_STATUS_IO_DEVICE_ERROR,
  MD_NTSTATUS_WIN_STATUS_DEVICE_PROTOCOL_ERROR,
  MD_NTSTATUS_WIN_STATUS_BACKUP_CONTROLLER,
  MD_NTSTATUS_WIN_STATUS_LOG_FILE_FULL,
  MD_NTSTATUS_WIN_STATUS_TOO_LATE,
  MD_NTSTATUS_WIN_STATUS_NO_TRUST_LSA_SECRET,
  MD_NTSTATUS_WIN_STATUS_NO_TRUST_SAM_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_TRUSTED_DOMAIN_FAILURE,
  MD_NTSTATUS_WIN_STATUS_TRUSTED_RELATIONSHIP_FAILURE,
  MD_NTSTATUS_WIN_STATUS_EVENTLOG_FILE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_EVENTLOG_CANT_START,
  MD_NTSTATUS_WIN_STATUS_TRUST_FAILURE,
  MD_NTSTATUS_WIN_STATUS_MUTANT_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_NETLOGON_NOT_STARTED,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_POSSIBLE_DEADLOCK,
  MD_NTSTATUS_WIN_STATUS_NETWORK_CREDENTIAL_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_REMOTE_SESSION_LIMIT,
  MD_NTSTATUS_WIN_STATUS_EVENTLOG_FILE_CHANGED,
  MD_NTSTATUS_WIN_STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_NOLOGON_SERVER_TRUST_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_TRUST_INCONSISTENT,
  MD_NTSTATUS_WIN_STATUS_FS_DRIVER_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_IMAGE_ALREADY_LOADED_AS_DLL,
  MD_NTSTATUS_WIN_STATUS_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING,
  MD_NTSTATUS_WIN_STATUS_SHORT_NAMES_NOT_ENABLED_ON_VOLUME,
  MD_NTSTATUS_WIN_STATUS_SECURITY_STREAM_IS_INCONSISTENT,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOCK_RANGE,
  MD_NTSTATUS_WIN_STATUS_INVALID_ACE_CONDITION,
  MD_NTSTATUS_WIN_STATUS_IMAGE_SUBSYSTEM_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_NOTIFICATION_GUID_ALREADY_DEFINED,
  MD_NTSTATUS_WIN_STATUS_INVALID_EXCEPTION_HANDLER,
  MD_NTSTATUS_WIN_STATUS_DUPLICATE_PRIVILEGES,
  MD_NTSTATUS_WIN_STATUS_NOT_ALLOWED_ON_SYSTEM_FILE,
  MD_NTSTATUS_WIN_STATUS_REPAIR_NEEDED,
  MD_NTSTATUS_WIN_STATUS_QUOTA_NOT_ENABLED,
  MD_NTSTATUS_WIN_STATUS_NO_APPLICATION_PACKAGE,
  MD_NTSTATUS_WIN_STATUS_FILE_METADATA_OPTIMIZATION_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_NOT_SAME_OBJECT,
  MD_NTSTATUS_WIN_STATUS_FATAL_MEMORY_EXHAUSTION,
  MD_NTSTATUS_WIN_STATUS_ERROR_PROCESS_NOT_IN_JOB,
  MD_NTSTATUS_WIN_STATUS_CPU_SET_INVALID,
  MD_NTSTATUS_WIN_STATUS_IO_DEVICE_INVALID_DATA,
  MD_NTSTATUS_WIN_STATUS_IO_UNALIGNED_WRITE,
  MD_NTSTATUS_WIN_STATUS_CONTROL_STACK_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_NETWORK_OPEN_RESTRICTION,
  MD_NTSTATUS_WIN_STATUS_NO_USER_SESSION_KEY,
  MD_NTSTATUS_WIN_STATUS_USER_SESSION_DELETED,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_LANG_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_INSUFF_SERVER_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_INVALID_BUFFER_SIZE,
  MD_NTSTATUS_WIN_STATUS_INVALID_ADDRESS_COMPONENT,
  MD_NTSTATUS_WIN_STATUS_INVALID_ADDRESS_WILDCARD,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_ADDRESSES,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_CLOSED,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_RESET,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_NODES,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_ABORTED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_TIMED_OUT,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NO_RELEASE,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NO_MATCH,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_RESPONDED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INVALID_ID,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INVALID_TYPE,
  MD_NTSTATUS_WIN_STATUS_NOT_SERVER_SESSION,
  MD_NTSTATUS_WIN_STATUS_NOT_CLIENT_SESSION,
  MD_NTSTATUS_WIN_STATUS_CANNOT_LOAD_REGISTRY_FILE,
  MD_NTSTATUS_WIN_STATUS_DEBUG_ATTACH_FAILED,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_PROCESS_TERMINATED,
  MD_NTSTATUS_WIN_STATUS_DATA_NOT_ACCEPTED,
  MD_NTSTATUS_WIN_STATUS_NO_BROWSER_SERVERS_FOUND,
  MD_NTSTATUS_WIN_STATUS_VDM_HARD_ERROR,
  MD_NTSTATUS_WIN_STATUS_DRIVER_CANCEL_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_REPLY_MESSAGE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_MAPPED_ALIGNMENT,
  MD_NTSTATUS_WIN_STATUS_IMAGE_CHECKSUM_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA,
  MD_NTSTATUS_WIN_STATUS_CLIENT_SERVER_PARAMETERS_INVALID,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_MUST_CHANGE,
  MD_NTSTATUS_WIN_STATUS_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NOT_TINY_STREAM,
  MD_NTSTATUS_WIN_STATUS_RECOVERY_FAILURE,
  MD_NTSTATUS_WIN_STATUS_STACK_OVERFLOW_READ,
  MD_NTSTATUS_WIN_STATUS_FAIL_CHECK,
  MD_NTSTATUS_WIN_STATUS_DUPLICATE_OBJECTID,
  MD_NTSTATUS_WIN_STATUS_OBJECTID_EXISTS,
  MD_NTSTATUS_WIN_STATUS_CONVERT_TO_LARGE,
  MD_NTSTATUS_WIN_STATUS_RETRY,
  MD_NTSTATUS_WIN_STATUS_FOUND_OUT_OF_SCOPE,
  MD_NTSTATUS_WIN_STATUS_ALLOCATE_BUCKET,
  MD_NTSTATUS_WIN_STATUS_PROPSET_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_MARSHALL_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_INVALID_VARIANT,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_CONTROLLER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_LOCKED_OUT,
  MD_NTSTATUS_WIN_STATUS_HANDLE_NOT_CLOSABLE,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_REFUSED,
  MD_NTSTATUS_WIN_STATUS_GRACEFUL_DISCONNECT,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_ALREADY_ASSOCIATED,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_NOT_ASSOCIATED,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_INVALID,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_NETWORK_UNREACHABLE,
  MD_NTSTATUS_WIN_STATUS_HOST_UNREACHABLE,
  MD_NTSTATUS_WIN_STATUS_PROTOCOL_UNREACHABLE,
  MD_NTSTATUS_WIN_STATUS_PORT_UNREACHABLE,
  MD_NTSTATUS_WIN_STATUS_REQUEST_ABORTED,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_ABORTED,
  MD_NTSTATUS_WIN_STATUS_BAD_COMPRESSION_BUFFER,
  MD_NTSTATUS_WIN_STATUS_USER_MAPPED_FILE,
  MD_NTSTATUS_WIN_STATUS_AUDIT_FAILED,
  MD_NTSTATUS_WIN_STATUS_TIMER_RESOLUTION_NOT_SET,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_COUNT_LIMIT,
  MD_NTSTATUS_WIN_STATUS_LOGIN_TIME_RESTRICTION,
  MD_NTSTATUS_WIN_STATUS_LOGIN_WKSTA_RESTRICTION,
  MD_NTSTATUS_WIN_STATUS_IMAGE_MP_UP_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_LOGON_INFO,
  MD_NTSTATUS_WIN_STATUS_BAD_DLL_ENTRYPOINT,
  MD_NTSTATUS_WIN_STATUS_BAD_SERVICE_ENTRYPOINT,
  MD_NTSTATUS_WIN_STATUS_LPC_REPLY_LOST,
  MD_NTSTATUS_WIN_STATUS_IP_ADDRESS_CONFLICT1,
  MD_NTSTATUS_WIN_STATUS_IP_ADDRESS_CONFLICT2,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_QUOTA_LIMIT,
  MD_NTSTATUS_WIN_STATUS_PATH_NOT_COVERED,
  MD_NTSTATUS_WIN_STATUS_NO_CALLBACK_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_LICENSE_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_PWD_TOO_SHORT,
  MD_NTSTATUS_WIN_STATUS_PWD_TOO_RECENT,
  MD_NTSTATUS_WIN_STATUS_PWD_HISTORY_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_PLUGPLAY_NO_DEVICE,
  MD_NTSTATUS_WIN_STATUS_UNSUPPORTED_COMPRESSION,
  MD_NTSTATUS_WIN_STATUS_INVALID_HW_PROFILE,
  MD_NTSTATUS_WIN_STATUS_INVALID_PLUGPLAY_DEVICE_PATH,
  MD_NTSTATUS_WIN_STATUS_DRIVER_ORDINAL_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_DRIVER_ENTRYPOINT_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_NOT_OWNED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_LINKS,
  MD_NTSTATUS_WIN_STATUS_QUOTA_LIST_INCONSISTENT,
  MD_NTSTATUS_WIN_STATUS_FILE_IS_OFFLINE,
  MD_NTSTATUS_WIN_STATUS_EVALUATION_EXPIRATION,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_DLL_RELOCATION,
  MD_NTSTATUS_WIN_STATUS_LICENSE_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_DLL_INIT_FAILED_LOGOFF,
  MD_NTSTATUS_WIN_STATUS_DRIVER_UNABLE_TO_LOAD,
  MD_NTSTATUS_WIN_STATUS_DFS_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_VOLUME_DISMOUNTED,
  MD_NTSTATUS_WIN_STATUS_WX86_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_WX86_FLOAT_STACK_CHECK,
  MD_NTSTATUS_WIN_STATUS_VALIDATE_CONTINUE,
  MD_NTSTATUS_WIN_STATUS_NO_MATCH,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_MATCHES,
  MD_NTSTATUS_WIN_STATUS_NOT_A_REPARSE_POINT,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_TAG_INVALID,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_TAG_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_DATA_INVALID,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_TAG_NOT_HANDLED,
  MD_NTSTATUS_WIN_STATUS_PWD_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_STOWED_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_CONTEXT_STOWED_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_REPARSE_POINT_NOT_RESOLVED,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_IS_A_REPARSE_POINT,
  MD_NTSTATUS_WIN_STATUS_RANGE_LIST_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_SOURCE_ELEMENT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_DESTINATION_ELEMENT_FULL,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_ELEMENT_ADDRESS,
  MD_NTSTATUS_WIN_STATUS_MAGAZINE_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_REINITIALIZATION_NEEDED,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTION_FAILED,
  MD_NTSTATUS_WIN_STATUS_DECRYPTION_FAILED,
  MD_NTSTATUS_WIN_STATUS_RANGE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NO_RECOVERY_POLICY,
  MD_NTSTATUS_WIN_STATUS_NO_EFS,
  MD_NTSTATUS_WIN_STATUS_WRONG_EFS,
  MD_NTSTATUS_WIN_STATUS_NO_USER_KEYS,
  MD_NTSTATUS_WIN_STATUS_FILE_NOT_ENCRYPTED,
  MD_NTSTATUS_WIN_STATUS_NOT_EXPORT_FORMAT,
  MD_NTSTATUS_WIN_STATUS_FILE_ENCRYPTED,
  MD_NTSTATUS_WIN_STATUS_WMI_GUID_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_WMI_INSTANCE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_WMI_ITEMID_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_WMI_TRY_AGAIN,
  MD_NTSTATUS_WIN_STATUS_SHARED_POLICY,
  MD_NTSTATUS_WIN_STATUS_POLICY_OBJECT_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_POLICY_ONLY_IN_DS,
  MD_NTSTATUS_WIN_STATUS_VOLUME_NOT_UPGRADED,
  MD_NTSTATUS_WIN_STATUS_REMOTE_STORAGE_NOT_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_REMOTE_STORAGE_MEDIA_ERROR,
  MD_NTSTATUS_WIN_STATUS_NO_TRACKING_SERVICE,
  MD_NTSTATUS_WIN_STATUS_SERVER_SID_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_DS_NO_ATTRIBUTE_OR_VALUE,
  MD_NTSTATUS_WIN_STATUS_DS_INVALID_ATTRIBUTE_SYNTAX,
  MD_NTSTATUS_WIN_STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED,
  MD_NTSTATUS_WIN_STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS,
  MD_NTSTATUS_WIN_STATUS_DS_BUSY,
  MD_NTSTATUS_WIN_STATUS_DS_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_DS_NO_RIDS_ALLOCATED,
  MD_NTSTATUS_WIN_STATUS_DS_NO_MORE_RIDS,
  MD_NTSTATUS_WIN_STATUS_DS_INCORRECT_ROLE_OWNER,
  MD_NTSTATUS_WIN_STATUS_DS_RIDMGR_INIT_ERROR,
  MD_NTSTATUS_WIN_STATUS_DS_OBJ_CLASS_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_ON_NON_LEAF,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_ON_RDN,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_MOD_OBJ_CLASS,
  MD_NTSTATUS_WIN_STATUS_DS_CROSS_DOM_MOVE_FAILED,
  MD_NTSTATUS_WIN_STATUS_DS_GC_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_SERVICE_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_REPARSE_ATTRIBUTE_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_CANT_ENABLE_DENY_ONLY,
  MD_NTSTATUS_WIN_STATUS_FLOAT_MULTIPLE_FAULTS,
  MD_NTSTATUS_WIN_STATUS_FLOAT_MULTIPLE_TRAPS,
  MD_NTSTATUS_WIN_STATUS_DEVICE_REMOVED,
  MD_NTSTATUS_WIN_STATUS_JOURNAL_DELETE_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_JOURNAL_NOT_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_NOINTERFACE,
  MD_NTSTATUS_WIN_STATUS_DS_RIDMGR_DISABLED,
  MD_NTSTATUS_WIN_STATUS_DS_ADMIN_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_DRIVER_FAILED_SLEEP,
  MD_NTSTATUS_WIN_STATUS_MUTUAL_AUTHENTICATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_SYSTEM_FILE,
  MD_NTSTATUS_WIN_STATUS_DATATYPE_MISALIGNMENT_ERROR,
  MD_NTSTATUS_WIN_STATUS_WMI_READ_ONLY,
  MD_NTSTATUS_WIN_STATUS_WMI_SET_FAILURE,
  MD_NTSTATUS_WIN_STATUS_COMMITMENT_MINIMUM,
  MD_NTSTATUS_WIN_STATUS_REG_NAT_CONSUMPTION,
  MD_NTSTATUS_WIN_STATUS_TRANSPORT_FULL,
  MD_NTSTATUS_WIN_STATUS_DS_SAM_INIT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_ONLY_IF_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_DS_SENSITIVE_GROUP_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_PNP_RESTART_ENUMERATION,
  MD_NTSTATUS_WIN_STATUS_JOURNAL_ENTRY_DELETED,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_MOD_PRIMARYGROUPID,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_IMAGE_BAD_SIGNATURE,
  MD_NTSTATUS_WIN_STATUS_PNP_REBOOT_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_POWER_STATE_INVALID,
  MD_NTSTATUS_WIN_STATUS_DS_INVALID_GROUP_TYPE,
  MD_NTSTATUS_WIN_STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN,
  MD_NTSTATUS_WIN_STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN,
  MD_NTSTATUS_WIN_STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER,
  MD_NTSTATUS_WIN_STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER,
  MD_NTSTATUS_WIN_STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER,
  MD_NTSTATUS_WIN_STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER,
  MD_NTSTATUS_WIN_STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER,
  MD_NTSTATUS_WIN_STATUS_DS_HAVE_PRIMARY_MEMBERS,
  MD_NTSTATUS_WIN_STATUS_WMI_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_POWER,
  MD_NTSTATUS_WIN_STATUS_SAM_NEED_BOOTKEY_PASSWORD,
  MD_NTSTATUS_WIN_STATUS_SAM_NEED_BOOTKEY_FLOPPY,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_START,
  MD_NTSTATUS_WIN_STATUS_DS_INIT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_SAM_INIT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_DS_GC_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY,
  MD_NTSTATUS_WIN_STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS,
  MD_NTSTATUS_WIN_STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_MULTIPLE_FAULT_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_CURRENT_DOMAIN_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_CANNOT_MAKE,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_SHUTDOWN,
  MD_NTSTATUS_WIN_STATUS_DS_INIT_FAILURE_CONSOLE,
  MD_NTSTATUS_WIN_STATUS_DS_SAM_INIT_FAILURE_CONSOLE,
  MD_NTSTATUS_WIN_STATUS_UNFINISHED_CONTEXT_DELETED,
  MD_NTSTATUS_WIN_STATUS_NO_TGT_REPLY,
  MD_NTSTATUS_WIN_STATUS_OBJECTID_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NO_IP_ADDRESSES,
  MD_NTSTATUS_WIN_STATUS_WRONG_CREDENTIAL_HANDLE,
  MD_NTSTATUS_WIN_STATUS_CRYPTO_SYSTEM_INVALID,
  MD_NTSTATUS_WIN_STATUS_MAX_REFERRALS_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_MUST_BE_KDC,
  MD_NTSTATUS_WIN_STATUS_STRONG_CRYPTO_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_PRINCIPALS,
  MD_NTSTATUS_WIN_STATUS_NO_PA_DATA,
  MD_NTSTATUS_WIN_STATUS_PKINIT_NAME_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_LOGON_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_KDC_INVALID_REQUEST,
  MD_NTSTATUS_WIN_STATUS_KDC_UNABLE_TO_REFER,
  MD_NTSTATUS_WIN_STATUS_KDC_UNKNOWN_ETYPE,
  MD_NTSTATUS_WIN_STATUS_SHUTDOWN_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_SERVER_SHUTDOWN_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_ON_SBS,
  MD_NTSTATUS_WIN_STATUS_WMI_GUID_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_WMI_ALREADY_DISABLED,
  MD_NTSTATUS_WIN_STATUS_WMI_ALREADY_ENABLED,
  MD_NTSTATUS_WIN_STATUS_MFT_TOO_FRAGMENTED,
  MD_NTSTATUS_WIN_STATUS_COPY_PROTECTION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_CSS_AUTHENTICATION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_CSS_KEY_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_CSS_KEY_NOT_ESTABLISHED,
  MD_NTSTATUS_WIN_STATUS_CSS_SCRAMBLED_SECTOR,
  MD_NTSTATUS_WIN_STATUS_CSS_REGION_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_CSS_RESETS_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_CHANGE_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_LOST_MODE_LOGON_RESTRICTION,
  MD_NTSTATUS_WIN_STATUS_PKINIT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_SUBSYSTEM_FAILURE,
  MD_NTSTATUS_WIN_STATUS_NO_KERB_KEY,
  MD_NTSTATUS_WIN_STATUS_HOST_DOWN,
  MD_NTSTATUS_WIN_STATUS_UNSUPPORTED_PREAUTH,
  MD_NTSTATUS_WIN_STATUS_EFS_ALG_BLOB_TOO_BIG,
  MD_NTSTATUS_WIN_STATUS_PORT_NOT_SET,
  MD_NTSTATUS_WIN_STATUS_DEBUGGER_INACTIVE,
  MD_NTSTATUS_WIN_STATUS_DS_VERSION_CHECK_FAILURE,
  MD_NTSTATUS_WIN_STATUS_AUDITING_DISABLED,
  MD_NTSTATUS_WIN_STATUS_PRENT4_MACHINE_ACCOUNT,
  MD_NTSTATUS_WIN_STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_WIN_32,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_WIN_64,
  MD_NTSTATUS_WIN_STATUS_BAD_BINDINGS,
  MD_NTSTATUS_WIN_STATUS_NETWORK_SESSION_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_APPHELP_BLOCK,
  MD_NTSTATUS_WIN_STATUS_ALL_SIDS_FILTERED,
  MD_NTSTATUS_WIN_STATUS_NOT_SAFE_MODE_DRIVER,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_PATH,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_OTHER,
  MD_NTSTATUS_WIN_STATUS_FAILED_DRIVER_ENTRY,
  MD_NTSTATUS_WIN_STATUS_DEVICE_ENUMERATION_ERROR,
  MD_NTSTATUS_WIN_STATUS_MOUNT_POINT_NOT_RESOLVED,
  MD_NTSTATUS_WIN_STATUS_INVALID_DEVICE_OBJECT_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_MCA_OCCURED,
  MD_NTSTATUS_WIN_STATUS_DRIVER_BLOCKED_CRITICAL,
  MD_NTSTATUS_WIN_STATUS_DRIVER_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_DRIVER_DATABASE_ERROR,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_HIVE_TOO_LARGE,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMPORT_OF_NON_DLL,
  MD_NTSTATUS_WIN_STATUS_NO_SECRETS,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY,
  MD_NTSTATUS_WIN_STATUS_FAILED_STACK_SWITCH,
  MD_NTSTATUS_WIN_STATUS_HEAP_CORRUPTION,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_WRONG_PIN,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CARD_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_CARD,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_KEY_CONTAINER,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_CERTIFICATE,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_KEYSET,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_IO_ERROR,
  MD_NTSTATUS_WIN_STATUS_DOWNGRADE_DETECTED,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CERT_REVOKED,
  MD_NTSTATUS_WIN_STATUS_ISSUING_CA_UNTRUSTED,
  MD_NTSTATUS_WIN_STATUS_REVOCATION_OFFLINE_C,
  MD_NTSTATUS_WIN_STATUS_PKINIT_CLIENT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CERT_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_DRIVER_FAILED_PRIOR_UNLOAD,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_SILENT_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_PER_USER_TRUST_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_DS_NAME_NOT_UNIQUE,
  MD_NTSTATUS_WIN_STATUS_DS_DUPLICATE_ID_FOUND,
  MD_NTSTATUS_WIN_STATUS_DS_GROUP_CONVERSION_ERROR,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_PREPARE_HIBERNATE,
  MD_NTSTATUS_WIN_STATUS_USER2USER_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_STACK_BUFFER_OVERRUN,
  MD_NTSTATUS_WIN_STATUS_NO_S4U_PROT_SUPPORT,
  MD_NTSTATUS_WIN_STATUS_CROSSREALM_DELEGATION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_REVOCATION_OFFLINE_KDC,
  MD_NTSTATUS_WIN_STATUS_ISSUING_CA_UNTRUSTED_KDC,
  MD_NTSTATUS_WIN_STATUS_KDC_CERT_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_KDC_CERT_REVOKED,
  MD_NTSTATUS_WIN_STATUS_PARAMETER_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_HIBERNATION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_DELAY_LOAD_FAILED,
  MD_NTSTATUS_WIN_STATUS_AUTHENTICATION_FIREWALL_FAILED,
  MD_NTSTATUS_WIN_STATUS_VDM_DISALLOWED,
  MD_NTSTATUS_WIN_STATUS_HUNG_DISPLAY_DRIVER_THREAD,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE,
  MD_NTSTATUS_WIN_STATUS_INVALID_CRUNTIME_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_NTLM_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_DS_SRC_SID_EXISTS_IN_FOREST,
  MD_NTSTATUS_WIN_STATUS_DS_DOMAIN_NAME_EXISTS_IN_FOREST,
  MD_NTSTATUS_WIN_STATUS_DS_FLAT_NAME_EXISTS_IN_FOREST,
  MD_NTSTATUS_WIN_STATUS_INVALID_USER_PRINCIPAL_NAME,
  MD_NTSTATUS_WIN_STATUS_FATAL_USER_CALLBACK_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_ASSERTION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_VERIFIER_STOP,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_POP_STACK,
  MD_NTSTATUS_WIN_STATUS_INCOMPATIBLE_DRIVER_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_HIVE_UNLOADED,
  MD_NTSTATUS_WIN_STATUS_COMPRESSION_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_LIMITATION,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_HASH,
  MD_NTSTATUS_WIN_STATUS_NOT_CAPABLE,
  MD_NTSTATUS_WIN_STATUS_REQUEST_OUT_OF_SEQUENCE,
  MD_NTSTATUS_WIN_STATUS_IMPLEMENTATION_LIMIT,
  MD_NTSTATUS_WIN_STATUS_ELEVATION_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_NO_SECURITY_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_PKU2U_CERT_FAILURE,
  MD_NTSTATUS_WIN_STATUS_BEYOND_VDL,
  MD_NTSTATUS_WIN_STATUS_ENCOUNTERED_WRITE_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_PTE_CHANGED,
  MD_NTSTATUS_WIN_STATUS_PURGE_FAILED,
  MD_NTSTATUS_WIN_STATUS_CRED_REQUIRES_CONFIRMATION,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_FILE_NOT_CSE,
  MD_NTSTATUS_WIN_STATUS_INVALID_LABEL,
  MD_NTSTATUS_WIN_STATUS_DRIVER_PROCESS_TERMINATED,
  MD_NTSTATUS_WIN_STATUS_AMBIGUOUS_SYSTEM_DEVICE,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_DEVICE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_RESTART_BOOT_APPLICATION,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_NVRAM_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_INVALID_SESSION,
  MD_NTSTATUS_WIN_STATUS_THREAD_ALREADY_IN_SESSION,
  MD_NTSTATUS_WIN_STATUS_THREAD_NOT_IN_SESSION,
  MD_NTSTATUS_WIN_STATUS_INVALID_WEIGHT,
  MD_NTSTATUS_WIN_STATUS_REQUEST_PAUSED,
  MD_NTSTATUS_WIN_STATUS_NO_RANGES_PROCESSED,
  MD_NTSTATUS_WIN_STATUS_DISK_RESOURCES_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_NEEDS_REMEDIATION,
  MD_NTSTATUS_WIN_STATUS_DEVICE_FEATURE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_DEVICE_UNREACHABLE,
  MD_NTSTATUS_WIN_STATUS_INVALID_TOKEN,
  MD_NTSTATUS_WIN_STATUS_SERVER_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_FILE_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_DEVICE_INSUFFICIENT_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_PACKAGE_UPDATING,
  MD_NTSTATUS_WIN_STATUS_NOT_READ_FROM_COPY,
  MD_NTSTATUS_WIN_STATUS_FT_WRITE_FAILURE,
  MD_NTSTATUS_WIN_STATUS_FT_DI_SCAN_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NOT_EXTERNALLY_BACKED,
  MD_NTSTATUS_WIN_STATUS_EXTERNAL_BACKING_PROVIDER_UNKNOWN,
  MD_NTSTATUS_WIN_STATUS_COMPRESSION_NOT_BENEFICIAL,
  MD_NTSTATUS_WIN_STATUS_DATA_CHECKSUM_ERROR,
  MD_NTSTATUS_WIN_STATUS_INTERMIXED_KERNEL_EA_OPERATION,
  MD_NTSTATUS_WIN_STATUS_TRIM_READ_ZERO_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SEGMENT_DESCRIPTORS,
  MD_NTSTATUS_WIN_STATUS_INVALID_OFFSET_ALIGNMENT,
  MD_NTSTATUS_WIN_STATUS_INVALID_FIELD_IN_PARAMETER_LIST,
  MD_NTSTATUS_WIN_STATUS_OPERATION_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_INVALID_INITIATOR_TARGET_PATH,
  MD_NTSTATUS_WIN_STATUS_SCRUB_DATA_DISABLED,
  MD_NTSTATUS_WIN_STATUS_NOT_REDUNDANT_STORAGE,
  MD_NTSTATUS_WIN_STATUS_RESIDENT_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_COMPRESSED_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_IO_OPERATION_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_NEEDS_REMEDIATION,
  MD_NTSTATUS_WIN_STATUS_APPX_INTEGRITY_FAILURE_CLR_NGEN,
  MD_NTSTATUS_WIN_STATUS_SHARE_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_APISET_NOT_HOSTED,
  MD_NTSTATUS_WIN_STATUS_APISET_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_DEVICE_HARDWARE_ERROR,
  MD_NTSTATUS_WIN_STATUS_FIRMWARE_SLOT_INVALID,
  MD_NTSTATUS_WIN_STATUS_FIRMWARE_IMAGE_INVALID,
  MD_NTSTATUS_WIN_STATUS_STORAGE_TOPOLOGY_ID_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_WIM_NOT_BOOTABLE,
  MD_NTSTATUS_WIN_STATUS_BLOCKED_BY_PARENTAL_CONTROLS,
  MD_NTSTATUS_WIN_STATUS_NEEDS_REGISTRATION,
  MD_NTSTATUS_WIN_STATUS_QUOTA_ACTIVITY,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_INVOKE_INLINE,
  MD_NTSTATUS_WIN_STATUS_BLOCK_TOO_MANY_REFERENCES,
  MD_NTSTATUS_WIN_STATUS_MARKED_TO_DISALLOW_WRITES,
  MD_NTSTATUS_WIN_STATUS_NETWORK_ACCESS_DENIED_EDP,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_FAILURE,
  MD_NTSTATUS_WIN_STATUS_PNP_NO_COMPAT_DRIVERS,
  MD_NTSTATUS_WIN_STATUS_PNP_DRIVER_PACKAGE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PNP_DRIVER_CONFIGURATION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PNP_DRIVER_CONFIGURATION_INCOMPLETE,
  MD_NTSTATUS_WIN_STATUS_PNP_FUNCTION_DRIVER_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_PNP_DEVICE_CONFIGURATION_PENDING,
  MD_NTSTATUS_WIN_STATUS_DEVICE_HINT_NAME_BUFFER_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_PACKAGE_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_DEVICE_IN_MAINTENANCE,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_ON_DAX,
  MD_NTSTATUS_WIN_STATUS_FREE_SPACE_TOO_FRAGMENTED,
  MD_NTSTATUS_WIN_STATUS_DAX_MAPPING_EXISTS,
  MD_NTSTATUS_WIN_STATUS_CHILD_PROCESS_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_STORAGE_LOST_DATA_PERSISTENCE,
  MD_NTSTATUS_WIN_STATUS_VRF_CFG_AND_IO_ENABLED,
  MD_NTSTATUS_WIN_STATUS_PARTITION_TERMINATING,
  MD_NTSTATUS_WIN_STATUS_EXTERNAL_SYSKEY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_FILE_PROTECTED_UNDER_DPL,
  MD_NTSTATUS_WIN_STATUS_VOLUME_NOT_CLUSTER_ALIGNED,
  MD_NTSTATUS_WIN_STATUS_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND,
  MD_NTSTATUS_WIN_STATUS_APPX_FILE_NOT_ENCRYPTED,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_FT_READ_FAILURE,
  MD_NTSTATUS_WIN_STATUS_PATCH_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_ID_INVALID,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_DOES_NOT_EXIST,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_NOT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_NOT_A_DAX_VOLUME,
  MD_NTSTATUS_WIN_STATUS_NOT_DAX_MAPPABLE,
  MD_NTSTATUS_WIN_STATUS_CASE_DIFFERING_NAMES_IN_DIR,
  MD_NTSTATUS_WIN_STATUS_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_WITH_BTT,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTION_DISABLED,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTING_METADATA_DISALLOWED,
  MD_NTSTATUS_WIN_STATUS_CANT_CLEAR_ENCRYPTION_FLAG,
  MD_NTSTATUS_WIN_STATUS_UNSATISFIED_DEPENDENCIES,
  MD_NTSTATUS_WIN_STATUS_CASE_SENSITIVE_PATH,
  MD_NTSTATUS_WIN_STATUS_HAS_SYSTEM_CRITICAL_FILES,
  MD_NTSTATUS_WIN_STATUS_INVALID_TASK_NAME,
  MD_NTSTATUS_WIN_STATUS_INVALID_TASK_INDEX,
  MD_NTSTATUS_WIN_STATUS_THREAD_ALREADY_IN_TASK,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_BYPASS,
  MD_NTSTATUS_WIN_STATUS_UNDEFINED_SCOPE,
  MD_NTSTATUS_WIN_STATUS_INVALID_CAP,
  MD_NTSTATUS_WIN_STATUS_NOT_GUI_PROCESS,
  MD_NTSTATUS_WIN_STATUS_DEVICE_HUNG,
  MD_NTSTATUS_WIN_STATUS_CONTAINER_ASSIGNED,
  MD_NTSTATUS_WIN_STATUS_JOB_NO_CONTAINER,
  MD_NTSTATUS_WIN_STATUS_DEVICE_UNRESPONSIVE,
  MD_NTSTATUS_WIN_STATUS_REPARSE_POINT_ENCOUNTERED,
  MD_NTSTATUS_WIN_STATUS_ATTRIBUTE_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_NOT_A_TIERED_VOLUME,
  MD_NTSTATUS_WIN_STATUS_ALREADY_HAS_STREAM_ID,
  MD_NTSTATUS_WIN_STATUS_JOB_NOT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_ALREADY_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_NOT_TERMINATED,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_IS_TERMINATING,
  MD_NTSTATUS_WIN_STATUS_SMB1_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_SMR_GARBAGE_COLLECTION_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_INTERRUPTED,
  MD_NTSTATUS_WIN_STATUS_THREAD_NOT_RUNNING,
  MD_NTSTATUS_WIN_STATUS_FAIL_FAST_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_IMAGE_CERT_REVOKED,
  MD_NTSTATUS_WIN_STATUS_DYNAMIC_CODE_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_IMAGE_CERT_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_STRICT_CFG_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_SET_CONTEXT_DENIED,
  MD_NTSTATUS_WIN_STATUS_CROSS_PARTITION_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_PORT_CLOSED,
  MD_NTSTATUS_WIN_STATUS_MESSAGE_LOST,
  MD_NTSTATUS_WIN_STATUS_INVALID_MESSAGE,
  MD_NTSTATUS_WIN_STATUS_REQUEST_CANCELED,
  MD_NTSTATUS_WIN_STATUS_RECURSIVE_DISPATCH,
  MD_NTSTATUS_WIN_STATUS_LPC_RECEIVE_BUFFER_EXPECTED,
  MD_NTSTATUS_WIN_STATUS_LPC_INVALID_CONNECTION_USAGE,
  MD_NTSTATUS_WIN_STATUS_LPC_REQUESTS_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_IN_USE,
  MD_NTSTATUS_WIN_STATUS_HARDWARE_MEMORY_ERROR,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_HANDLE_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_RELEASED_DURING_OPERATION,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING,
  MD_NTSTATUS_WIN_STATUS_APC_RETURNED_WHILE_IMPERSONATING,
  MD_NTSTATUS_WIN_STATUS_PROCESS_IS_PROTECTED,
  MD_NTSTATUS_WIN_STATUS_MCA_EXCEPTION,
  MD_NTSTATUS_WIN_STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE,
  MD_NTSTATUS_WIN_STATUS_SYMLINK_CLASS_DISABLED,
  MD_NTSTATUS_WIN_STATUS_INVALID_IDN_NORMALIZATION,
  MD_NTSTATUS_WIN_STATUS_NO_UNICODE_TRANSLATION,
  MD_NTSTATUS_WIN_STATUS_ALREADY_REGISTERED,
  MD_NTSTATUS_WIN_STATUS_CONTEXT_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_PORT_ALREADY_HAS_COMPLETION_LIST,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_THREAD_PRIORITY,
  MD_NTSTATUS_WIN_STATUS_INVALID_THREAD,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_LDR_LOCK,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_LANG,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_PRI_BACK,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_THREAD_AFFINITY,
  MD_NTSTATUS_WIN_STATUS_LPC_HANDLE_COUNT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_EXECUTABLE_MEMORY_WRITE,
  MD_NTSTATUS_WIN_STATUS_KERNEL_EXECUTABLE_MEMORY_WRITE,
  MD_NTSTATUS_WIN_STATUS_ATTACHED_EXECUTABLE_MEMORY_WRITE,
  MD_NTSTATUS_WIN_STATUS_TRIGGERED_EXECUTABLE_MEMORY_WRITE,
  MD_NTSTATUS_WIN_STATUS_DISK_REPAIR_DISABLED,
  MD_NTSTATUS_WIN_STATUS_DS_DOMAIN_RENAME_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_DISK_QUOTA_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_CONTENT_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_BAD_CLUSTERS,
  MD_NTSTATUS_WIN_STATUS_VOLUME_DIRTY,
  MD_NTSTATUS_WIN_STATUS_DISK_REPAIR_UNSUCCESSFUL,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_OVERFULL,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_CORRUPTED,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_DELETED_FULL,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_CLEARED,
  MD_NTSTATUS_WIN_STATUS_ORPHAN_NAME_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_PROACTIVE_SCAN_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTED_IO_NOT_POSSIBLE,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_UPLEVEL_RECORDS,
  MD_NTSTATUS_WIN_STATUS_FILE_CHECKED_OUT,
  MD_NTSTATUS_WIN_STATUS_CHECKOUT_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_BAD_FILE_TYPE,
  MD_NTSTATUS_WIN_STATUS_FILE_TOO_LARGE,
  MD_NTSTATUS_WIN_STATUS_FORMS_AUTH_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_VIRUS_INFECTED,
  MD_NTSTATUS_WIN_STATUS_VIRUS_DELETED,
  MD_NTSTATUS_WIN_STATUS_BAD_MCFG_TABLE,
  MD_NTSTATUS_WIN_STATUS_CANNOT_BREAK_OPLOCK,
  MD_NTSTATUS_WIN_STATUS_BAD_KEY,
  MD_NTSTATUS_WIN_STATUS_BAD_DATA,
  MD_NTSTATUS_WIN_STATUS_NO_KEY,
  MD_NTSTATUS_WIN_STATUS_FILE_HANDLE_REVOKED,
  MD_NTSTATUS_WIN_STATUS_WOW_ASSERTION,
  MD_NTSTATUS_WIN_STATUS_INVALID_SIGNATURE,
  MD_NTSTATUS_WIN_STATUS_HMAC_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_AUTH_TAG_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_INVALID_STATE_TRANSITION,
  MD_NTSTATUS_WIN_STATUS_INVALID_KERNEL_INFO_VERSION,
  MD_NTSTATUS_WIN_STATUS_INVALID_PEP_INFO_VERSION,
  MD_NTSTATUS_WIN_STATUS_HANDLE_REVOKED,
  MD_NTSTATUS_WIN_STATUS_EOF_ON_GHOSTED_RANGE,
  MD_NTSTATUS_WIN_STATUS_CC_NEEDS_CALLBACK_SECTION_DRAIN,
  MD_NTSTATUS_WIN_STATUS_IPSEC_QUEUE_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_ND_QUEUE_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_HOPLIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_PROTOCOL_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_FASTPATH_REJECTED,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR,
  MD_NTSTATUS_WIN_STATUS_XML_PARSE_ERROR,
  MD_NTSTATUS_WIN_STATUS_XMLDSIG_ERROR,
  MD_NTSTATUS_WIN_STATUS_WRONG_COMPARTMENT,
  MD_NTSTATUS_WIN_STATUS_AUTHIP_FAILURE,
  MD_NTSTATUS_WIN_STATUS_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS,
  MD_NTSTATUS_WIN_STATUS_DS_OID_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_INCORRECT_ACCOUNT_TYPE,
  MD_NTSTATUS_WIN_STATUS_HASH_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_HASH_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_SECONDARY_IC_PROVIDER_NOT_REGISTERED,
  MD_NTSTATUS_WIN_STATUS_GPIO_CLIENT_INFORMATION_INVALID,
  MD_NTSTATUS_WIN_STATUS_GPIO_VERSION_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GPIO_INVALID_REGISTRATION_PACKET,
  MD_NTSTATUS_WIN_STATUS_GPIO_OPERATION_DENIED,
  MD_NTSTATUS_WIN_STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE,
  MD_NTSTATUS_WIN_STATUS_CANNOT_SWITCH_RUNLEVEL,
  MD_NTSTATUS_WIN_STATUS_INVALID_RUNLEVEL_SETTING,
  MD_NTSTATUS_WIN_STATUS_RUNLEVEL_SWITCH_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_RUNLEVEL_SWITCH_AGENT_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_RUNLEVEL_SWITCH_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_NOT_APPCONTAINER,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_IN_APPCONTAINER,
  MD_NTSTATUS_WIN_STATUS_INVALID_PACKAGE_SID_LENGTH,
  MD_NTSTATUS_WIN_STATUS_LPAC_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_ADMINLESS_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_REBOOT_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_READ_FLT_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_WRITE_FLT_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_READ_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_WRITE_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_WOF_WIM_HEADER_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_WOF_WIM_RESOURCE_TABLE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_WOF_FILE_RESOURCE_TABLE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_CIMFS_IMAGE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_BUSY,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROVIDER_NOT_RUNNING,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_METADATA_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_METADATA_TOO_LARGE,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_NOT_A_CLOUD_FILE,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NOT_IN_SYNC,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_ALREADY_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_INVALID_REQUEST,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_READ_ONLY_VOLUME,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_CONNECTED_PROVIDER_ONLY,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_VALIDATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_AUTHENTICATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_INSUFFICIENT_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NETWORK_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_UNSUCCESSFUL,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NOT_UNDER_SYNC_ROOT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_IN_USE,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PINNED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_REQUEST_ABORTED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_INCOMPATIBLE_HARDLINKS,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_LOCK_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_REQUEST_CANCELED,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROVIDER_TERMINATED,
  MD_NTSTATUS_WIN_STATUS_NOT_A_CLOUD_SYNC_ROOT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_REQUEST_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_DEHYDRATION_DISALLOWED,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_USER_SECTION_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_MODIFY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_IO_NOT_COORDINATED,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_UNEXPECTED_ERROR,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_INVALID_PARAMETER,
  MD_NTSTATUS_WIN_DBG_NO_STATE_CHANGE,
  MD_NTSTATUS_WIN_DBG_APP_NOT_IDLE,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_STRING_BINDING,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_KIND_OF_BINDING,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_BINDING,
  MD_NTSTATUS_WIN_RPC_NT_PROTSEQ_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_RPC_PROTSEQ,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_STRING_UUID,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ENDPOINT_FORMAT,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NET_ADDR,
  MD_NTSTATUS_WIN_RPC_NT_NO_ENDPOINT_FOUND,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_TIMEOUT,
  MD_NTSTATUS_WIN_RPC_NT_OBJECT_NOT_FOUND,
  MD_NTSTATUS_WIN_RPC_NT_ALREADY_REGISTERED,
  MD_NTSTATUS_WIN_RPC_NT_TYPE_ALREADY_REGISTERED,
  MD_NTSTATUS_WIN_RPC_NT_ALREADY_LISTENING,
  MD_NTSTATUS_WIN_RPC_NT_NO_PROTSEQS_REGISTERED,
  MD_NTSTATUS_WIN_RPC_NT_NOT_LISTENING,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_MGR_TYPE,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_IF,
  MD_NTSTATUS_WIN_RPC_NT_NO_BINDINGS,
  MD_NTSTATUS_WIN_RPC_NT_NO_PROTSEQS,
  MD_NTSTATUS_WIN_RPC_NT_CANT_CREATE_ENDPOINT,
  MD_NTSTATUS_WIN_RPC_NT_OUT_OF_RESOURCES,
  MD_NTSTATUS_WIN_RPC_NT_SERVER_UNAVAILABLE,
  MD_NTSTATUS_WIN_RPC_NT_SERVER_TOO_BUSY,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NETWORK_OPTIONS,
  MD_NTSTATUS_WIN_RPC_NT_NO_CALL_ACTIVE,
  MD_NTSTATUS_WIN_RPC_NT_CALL_FAILED,
  MD_NTSTATUS_WIN_RPC_NT_CALL_FAILED_DNE,
  MD_NTSTATUS_WIN_RPC_NT_PROTOCOL_ERROR,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_TRANS_SYN,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_TYPE,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_TAG,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_BOUND,
  MD_NTSTATUS_WIN_RPC_NT_NO_ENTRY_NAME,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NAME_SYNTAX,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_NAME_SYNTAX,
  MD_NTSTATUS_WIN_RPC_NT_UUID_NO_ADDRESS,
  MD_NTSTATUS_WIN_RPC_NT_DUPLICATE_ENDPOINT,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHN_TYPE,
  MD_NTSTATUS_WIN_RPC_NT_MAX_CALLS_TOO_SMALL,
  MD_NTSTATUS_WIN_RPC_NT_STRING_TOO_LONG,
  MD_NTSTATUS_WIN_RPC_NT_PROTSEQ_NOT_FOUND,
  MD_NTSTATUS_WIN_RPC_NT_PROCNUM_OUT_OF_RANGE,
  MD_NTSTATUS_WIN_RPC_NT_BINDING_HAS_NO_AUTH,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHN_SERVICE,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHN_LEVEL,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_AUTH_IDENTITY,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHZ_SERVICE,
  MD_NTSTATUS_WIN_EPT_NT_INVALID_ENTRY,
  MD_NTSTATUS_WIN_EPT_NT_CANT_PERFORM_OP,
  MD_NTSTATUS_WIN_EPT_NT_NOT_REGISTERED,
  MD_NTSTATUS_WIN_RPC_NT_NOTHING_TO_EXPORT,
  MD_NTSTATUS_WIN_RPC_NT_INCOMPLETE_NAME,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_VERS_OPTION,
  MD_NTSTATUS_WIN_RPC_NT_NO_MORE_MEMBERS,
  MD_NTSTATUS_WIN_RPC_NT_NOT_ALL_OBJS_UNEXPORTED,
  MD_NTSTATUS_WIN_RPC_NT_INTERFACE_NOT_FOUND,
  MD_NTSTATUS_WIN_RPC_NT_ENTRY_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_RPC_NT_ENTRY_NOT_FOUND,
  MD_NTSTATUS_WIN_RPC_NT_NAME_SERVICE_UNAVAILABLE,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NAF_ID,
  MD_NTSTATUS_WIN_RPC_NT_CANNOT_SUPPORT,
  MD_NTSTATUS_WIN_RPC_NT_NO_CONTEXT_AVAILABLE,
  MD_NTSTATUS_WIN_RPC_NT_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_RPC_NT_ZERO_DIVIDE,
  MD_NTSTATUS_WIN_RPC_NT_ADDRESS_ERROR,
  MD_NTSTATUS_WIN_RPC_NT_FP_DIV_ZERO,
  MD_NTSTATUS_WIN_RPC_NT_FP_UNDERFLOW,
  MD_NTSTATUS_WIN_RPC_NT_FP_OVERFLOW,
  MD_NTSTATUS_WIN_RPC_NT_CALL_IN_PROGRESS,
  MD_NTSTATUS_WIN_RPC_NT_NO_MORE_BINDINGS,
  MD_NTSTATUS_WIN_RPC_NT_GROUP_MEMBER_NOT_FOUND,
  MD_NTSTATUS_WIN_EPT_NT_CANT_CREATE,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_OBJECT,
  MD_NTSTATUS_WIN_RPC_NT_NO_INTERFACES,
  MD_NTSTATUS_WIN_RPC_NT_CALL_CANCELLED,
  MD_NTSTATUS_WIN_RPC_NT_BINDING_INCOMPLETE,
  MD_NTSTATUS_WIN_RPC_NT_COMM_FAILURE,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_AUTHN_LEVEL,
  MD_NTSTATUS_WIN_RPC_NT_NO_PRINC_NAME,
  MD_NTSTATUS_WIN_RPC_NT_NOT_RPC_ERROR,
  MD_NTSTATUS_WIN_RPC_NT_SEC_PKG_ERROR,
  MD_NTSTATUS_WIN_RPC_NT_NOT_CANCELLED,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ASYNC_HANDLE,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ASYNC_CALL,
  MD_NTSTATUS_WIN_RPC_NT_PROXY_ACCESS_DENIED,
  MD_NTSTATUS_WIN_RPC_NT_COOKIE_AUTH_FAILED,
  MD_NTSTATUS_WIN_RPC_NT_NO_MORE_ENTRIES,
  MD_NTSTATUS_WIN_RPC_NT_SS_CHAR_TRANS_OPEN_FAIL,
  MD_NTSTATUS_WIN_RPC_NT_SS_CHAR_TRANS_SHORT_FILE,
  MD_NTSTATUS_WIN_RPC_NT_SS_IN_NULL_CONTEXT,
  MD_NTSTATUS_WIN_RPC_NT_SS_CONTEXT_MISMATCH,
  MD_NTSTATUS_WIN_RPC_NT_SS_CONTEXT_DAMAGED,
  MD_NTSTATUS_WIN_RPC_NT_SS_HANDLES_MISMATCH,
  MD_NTSTATUS_WIN_RPC_NT_SS_CANNOT_GET_CALL_HANDLE,
  MD_NTSTATUS_WIN_RPC_NT_NULL_REF_POINTER,
  MD_NTSTATUS_WIN_RPC_NT_ENUM_VALUE_OUT_OF_RANGE,
  MD_NTSTATUS_WIN_RPC_NT_BYTE_COUNT_TOO_SMALL,
  MD_NTSTATUS_WIN_RPC_NT_BAD_STUB_DATA,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ES_ACTION,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_ES_VERSION,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_STUB_VERSION,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_PIPE_OBJECT,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_PIPE_OPERATION,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_PIPE_VERSION,
  MD_NTSTATUS_WIN_RPC_NT_PIPE_CLOSED,
  MD_NTSTATUS_WIN_RPC_NT_PIPE_DISCIPLINE_ERROR,
  MD_NTSTATUS_WIN_RPC_NT_PIPE_EMPTY,
  MD_NTSTATUS_WIN_STATUS_PNP_BAD_MPS_TABLE,
  MD_NTSTATUS_WIN_STATUS_PNP_TRANSLATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_PNP_IRQ_TRANSLATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_PNP_INVALID_ID,
  MD_NTSTATUS_WIN_STATUS_IO_REISSUE_AS_CACHED,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_NAME_INVALID,
  MD_NTSTATUS_WIN_STATUS_CTX_INVALID_PD,
  MD_NTSTATUS_WIN_STATUS_CTX_PD_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CTX_CLOSE_PENDING,
  MD_NTSTATUS_WIN_STATUS_CTX_NO_OUTBUF,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_INF_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CTX_INVALID_MODEMNAME,
  MD_NTSTATUS_WIN_STATUS_CTX_RESPONSE_ERROR,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_NO_CARRIER,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_BUSY,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_VOICE,
  MD_NTSTATUS_WIN_STATUS_CTX_TD_ERROR,
  MD_NTSTATUS_WIN_STATUS_CTX_LICENSE_CLIENT_INVALID,
  MD_NTSTATUS_WIN_STATUS_CTX_LICENSE_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_CTX_LICENSE_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_NAME_COLLISION,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_BUSY,
  MD_NTSTATUS_WIN_STATUS_CTX_BAD_VIDEO_MODE,
  MD_NTSTATUS_WIN_STATUS_CTX_GRAPHICS_INVALID,
  MD_NTSTATUS_WIN_STATUS_CTX_NOT_CONSOLE,
  MD_NTSTATUS_WIN_STATUS_CTX_CLIENT_QUERY_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_CTX_CONSOLE_DISCONNECT,
  MD_NTSTATUS_WIN_STATUS_CTX_CONSOLE_CONNECT,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_DENIED,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_CTX_INVALID_WD,
  MD_NTSTATUS_WIN_STATUS_CTX_WD_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_INVALID,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_DISABLED,
  MD_NTSTATUS_WIN_STATUS_RDP_PROTOCOL_ERROR,
  MD_NTSTATUS_WIN_STATUS_CTX_CLIENT_LICENSE_NOT_SET,
  MD_NTSTATUS_WIN_STATUS_CTX_CLIENT_LICENSE_IN_USE,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_NOT_RUNNING,
  MD_NTSTATUS_WIN_STATUS_CTX_LOGON_DISABLED,
  MD_NTSTATUS_WIN_STATUS_CTX_SECURITY_LAYER_ERROR,
  MD_NTSTATUS_WIN_STATUS_TS_INCOMPATIBLE_SESSIONS,
  MD_NTSTATUS_WIN_STATUS_TS_VIDEO_SUBSYSTEM_ERROR,
  MD_NTSTATUS_WIN_STATUS_MUI_FILE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_FILE,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_RC_CONFIG,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_LOCALE_NAME,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME,
  MD_NTSTATUS_WIN_STATUS_MUI_FILE_NOT_LOADED,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_ENUM_USER_STOP,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_NODE,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_EXISTS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_JOIN_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_EXISTS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETINTERFACE_EXISTS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETINTERFACE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_REQUEST,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_NETWORK_PROVIDER,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_DOWN,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_UNREACHABLE,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_NOT_MEMBER,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_NETWORK,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NO_NET_ADAPTERS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_UP,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_PAUSED,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_NOT_PAUSED,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NO_SECURITY_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_NOT_INTERNAL,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_POISONED,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NON_CSV_PATH,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_VOLUME_NOT_LOCAL,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_READ_OPLOCK_BREAK_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_AUTO_PAUSE_ERROR,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_REDIRECTED,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_NOT_REDIRECTED,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_VOLUME_DRAINING,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_SNAPSHOT_CREATION_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_VOLUME_DRAINING_SUCCEEDED_DOWNLEVEL,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_NO_SNAPSHOTS,
  MD_NTSTATUS_WIN_STATUS_CSV_IO_PAUSE_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_INVALID_HANDLE,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CAM_TICKET_REPLAY_DETECTED,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_OPCODE,
  MD_NTSTATUS_WIN_STATUS_ACPI_STACK_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_ACPI_ASSERT_FAILED,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_INDEX,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_ARGUMENT,
  MD_NTSTATUS_WIN_STATUS_ACPI_FATAL,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_SUPERNAME,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_ARGTYPE,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_OBJTYPE,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_TARGETTYPE,
  MD_NTSTATUS_WIN_STATUS_ACPI_INCORRECT_ARGUMENT_COUNT,
  MD_NTSTATUS_WIN_STATUS_ACPI_ADDRESS_NOT_MAPPED,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_EVENTTYPE,
  MD_NTSTATUS_WIN_STATUS_ACPI_HANDLER_COLLISION,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_DATA,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_REGION,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_ACCESS_SIZE,
  MD_NTSTATUS_WIN_STATUS_ACPI_ACQUIRE_GLOBAL_LOCK,
  MD_NTSTATUS_WIN_STATUS_ACPI_ALREADY_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_ACPI_NOT_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_MUTEX_LEVEL,
  MD_NTSTATUS_WIN_STATUS_ACPI_MUTEX_NOT_OWNED,
  MD_NTSTATUS_WIN_STATUS_ACPI_MUTEX_NOT_OWNER,
  MD_NTSTATUS_WIN_STATUS_ACPI_RS_ACCESS,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_TABLE,
  MD_NTSTATUS_WIN_STATUS_ACPI_REG_HANDLER_FAILED,
  MD_NTSTATUS_WIN_STATUS_ACPI_POWER_REQUEST_FAILED,
  MD_NTSTATUS_WIN_STATUS_SXS_SECTION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_SXS_CANT_GEN_ACTCTX,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_ACTCTXDATA_FORMAT,
  MD_NTSTATUS_WIN_STATUS_SXS_ASSEMBLY_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_FORMAT_ERROR,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_PARSE_ERROR,
  MD_NTSTATUS_WIN_STATUS_SXS_ACTIVATION_CONTEXT_DISABLED,
  MD_NTSTATUS_WIN_STATUS_SXS_KEY_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_SXS_VERSION_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_SXS_WRONG_SECTION_TYPE,
  MD_NTSTATUS_WIN_STATUS_SXS_THREAD_QUERIES_DISABLED,
  MD_NTSTATUS_WIN_STATUS_SXS_ASSEMBLY_MISSING,
  MD_NTSTATUS_WIN_STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET,
  MD_NTSTATUS_WIN_STATUS_SXS_EARLY_DEACTIVATION,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_DEACTIVATION,
  MD_NTSTATUS_WIN_STATUS_SXS_MULTIPLE_DEACTIVATION,
  MD_NTSTATUS_WIN_STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_SXS_PROCESS_TERMINATION_REQUESTED,
  MD_NTSTATUS_WIN_STATUS_SXS_CORRUPT_ACTIVATION_STACK,
  MD_NTSTATUS_WIN_STATUS_SXS_CORRUPTION,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME,
  MD_NTSTATUS_WIN_STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE,
  MD_NTSTATUS_WIN_STATUS_SXS_IDENTITY_PARSE_ERROR,
  MD_NTSTATUS_WIN_STATUS_SXS_COMPONENT_STORE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_SXS_FILE_HASH_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT,
  MD_NTSTATUS_WIN_STATUS_SXS_IDENTITIES_DIFFERENT,
  MD_NTSTATUS_WIN_STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT,
  MD_NTSTATUS_WIN_STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY,
  MD_NTSTATUS_WIN_STATUS_ADVANCED_INSTALLER_FAILED,
  MD_NTSTATUS_WIN_STATUS_XML_ENCODING_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_TOO_BIG,
  MD_NTSTATUS_WIN_STATUS_SXS_SETTING_NOT_REGISTERED,
  MD_NTSTATUS_WIN_STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE,
  MD_NTSTATUS_WIN_STATUS_SMI_PRIMITIVE_INSTALLER_FAILED,
  MD_NTSTATUS_WIN_STATUS_GENERIC_COMMAND_FAILED,
  MD_NTSTATUS_WIN_STATUS_SXS_FILE_HASH_MISSING,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONAL_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_INVALID_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_TM_INITIALIZATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_RM_NOT_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_RM_METADATA_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_JOINED,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_NOT_RM,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE,
  MD_NTSTATUS_WIN_STATUS_LOG_RESIZE_INVALID_SIZE,
  MD_NTSTATUS_WIN_STATUS_REMOTE_FILE_VERSION_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_CRM_PROTOCOL_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_PROPAGATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_CRM_PROTOCOL_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_SUPERIOR_EXISTS,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_REQUEST_NOT_VALID,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_REQUESTED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_ALREADY_ABORTED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_ALREADY_COMMITTED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER,
  MD_NTSTATUS_WIN_STATUS_CURRENT_TRANSACTION_NOT_VALID,
  MD_NTSTATUS_WIN_STATUS_LOG_GROWTH_FAILED,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NO_LONGER_EXISTS,
  MD_NTSTATUS_WIN_STATUS_STREAM_MINIVERSION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_STREAM_MINIVERSION_NOT_VALID,
  MD_NTSTATUS_WIN_STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT,
  MD_NTSTATUS_WIN_STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS,
  MD_NTSTATUS_WIN_STATUS_HANDLE_NO_LONGER_VALID,
  MD_NTSTATUS_WIN_STATUS_LOG_CORRUPTION_DETECTED,
  MD_NTSTATUS_WIN_STATUS_RM_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_ENLISTMENT_NOT_SUPERIOR,
  MD_NTSTATUS_WIN_STATUS_FILE_IDENTITY_NOT_PERSISTENT,
  MD_NTSTATUS_WIN_STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY,
  MD_NTSTATUS_WIN_STATUS_CANT_CROSS_RM_BOUNDARY,
  MD_NTSTATUS_WIN_STATUS_TXF_DIR_NOT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_INDOUBT_TRANSACTIONS_EXIST,
  MD_NTSTATUS_WIN_STATUS_TM_VOLATILE,
  MD_NTSTATUS_WIN_STATUS_ROLLBACK_TIMER_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_TXF_ATTRIBUTE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_REQUIRED_PROMOTION,
  MD_NTSTATUS_WIN_STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONS_NOT_FROZEN,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_FREEZE_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_NOT_SNAPSHOT_VOLUME,
  MD_NTSTATUS_WIN_STATUS_NO_SAVEPOINT_WITH_OPEN_FILES,
  MD_NTSTATUS_WIN_STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_TM_IDENTITY_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_FLOATED_SECTION,
  MD_NTSTATUS_WIN_STATUS_CANNOT_ACCEPT_TRANSACTED_WORK,
  MD_NTSTATUS_WIN_STATUS_CANNOT_ABORT_TRANSACTIONS,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_RESOURCEMANAGER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_ENLISTMENT_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_NOT_ONLINE,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_ROOT,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_OBJECT_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_RECORD_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_NO_LINK_TRACKING_IN_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INTEGRITY_VIOLATED,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_IDENTITY_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_MUST_WRITETHROUGH,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NO_SUPERIOR,
  MD_NTSTATUS_WIN_STATUS_EXPIRED_HANDLE,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_ENLISTED,
  MD_NTSTATUS_WIN_STATUS_LOG_SECTOR_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_SECTOR_PARITY_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_SECTOR_REMAPPED,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCK_INCOMPLETE,
  MD_NTSTATUS_WIN_STATUS_LOG_INVALID_RANGE,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCKS_EXHAUSTED,
  MD_NTSTATUS_WIN_STATUS_LOG_READ_CONTEXT_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_RESTART_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCK_VERSION,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCK_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_READ_MODE_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_INCONSISTENT,
  MD_NTSTATUS_WIN_STATUS_LOG_RESERVATION_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_CANT_DELETE,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_LOG_START_OF_LOG,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_ALREADY_INSTALLED,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_NOT_INSTALLED,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_LOG_PINNED_ARCHIVE_TAIL,
  MD_NTSTATUS_WIN_STATUS_LOG_RECORD_NONEXISTENT,
  MD_NTSTATUS_WIN_STATUS_LOG_RECORDS_RESERVED_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_SPACE_RESERVED_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_TAIL_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_FULL,
  MD_NTSTATUS_WIN_STATUS_LOG_MULTIPLEXED,
  MD_NTSTATUS_WIN_STATUS_LOG_DEDICATED,
  MD_NTSTATUS_WIN_STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_LOG_ARCHIVE_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_LOG_EPHEMERAL,
  MD_NTSTATUS_WIN_STATUS_LOG_NOT_ENOUGH_CONTAINERS,
  MD_NTSTATUS_WIN_STATUS_LOG_CLIENT_ALREADY_REGISTERED,
  MD_NTSTATUS_WIN_STATUS_LOG_CLIENT_NOT_REGISTERED,
  MD_NTSTATUS_WIN_STATUS_LOG_FULL_HANDLER_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_READ_FAILED,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_WRITE_FAILED,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_OPEN_FAILED,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_STATE_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_STATE_INVALID,
  MD_NTSTATUS_WIN_STATUS_LOG_PINNED,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_FLUSH_FAILED,
  MD_NTSTATUS_WIN_STATUS_LOG_INCONSISTENT_SECURITY,
  MD_NTSTATUS_WIN_STATUS_LOG_APPENDED_FLUSH_FAILED,
  MD_NTSTATUS_WIN_STATUS_LOG_PINNED_RESERVATION,
  MD_NTSTATUS_WIN_STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD,
  MD_NTSTATUS_WIN_STATUS_FLT_NO_HANDLER_DEFINED,
  MD_NTSTATUS_WIN_STATUS_FLT_CONTEXT_ALREADY_DEFINED,
  MD_NTSTATUS_WIN_STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST,
  MD_NTSTATUS_WIN_STATUS_FLT_DISALLOW_FAST_IO,
  MD_NTSTATUS_WIN_STATUS_FLT_INVALID_NAME_REQUEST,
  MD_NTSTATUS_WIN_STATUS_FLT_NOT_SAFE_TO_POST_OPERATION,
  MD_NTSTATUS_WIN_STATUS_FLT_NOT_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_FLT_FILTER_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_FLT_POST_OPERATION_CLEANUP,
  MD_NTSTATUS_WIN_STATUS_FLT_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_FLT_DELETING_OBJECT,
  MD_NTSTATUS_WIN_STATUS_FLT_MUST_BE_NONPAGED_POOL,
  MD_NTSTATUS_WIN_STATUS_FLT_DUPLICATE_ENTRY,
  MD_NTSTATUS_WIN_STATUS_FLT_CBDQ_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FLT_DO_NOT_ATTACH,
  MD_NTSTATUS_WIN_STATUS_FLT_DO_NOT_DETACH,
  MD_NTSTATUS_WIN_STATUS_FLT_INSTANCE_ALTITUDE_COLLISION,
  MD_NTSTATUS_WIN_STATUS_FLT_INSTANCE_NAME_COLLISION,
  MD_NTSTATUS_WIN_STATUS_FLT_FILTER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FLT_VOLUME_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FLT_INSTANCE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FLT_INVALID_CONTEXT_REGISTRATION,
  MD_NTSTATUS_WIN_STATUS_FLT_NAME_CACHE_MISS,
  MD_NTSTATUS_WIN_STATUS_FLT_NO_DEVICE_OBJECT,
  MD_NTSTATUS_WIN_STATUS_FLT_VOLUME_ALREADY_MOUNTED,
  MD_NTSTATUS_WIN_STATUS_FLT_ALREADY_ENLISTED,
  MD_NTSTATUS_WIN_STATUS_FLT_CONTEXT_ALREADY_LINKED,
  MD_NTSTATUS_WIN_STATUS_FLT_NO_WAITER_FOR_REPLY,
  MD_NTSTATUS_WIN_STATUS_FLT_REGISTRATION_BUSY,
  MD_NTSTATUS_WIN_STATUS_MONITOR_NO_DESCRIPTOR,
  MD_NTSTATUS_WIN_STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK,
  MD_NTSTATUS_WIN_STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK,
  MD_NTSTATUS_WIN_STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_MANUFACTURE_DATE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_WAS_RESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_DRIVER_MODEL,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_MODE_CHANGED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_OCCLUDED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_DENIED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANNOTCOLORCONVERT,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DRIVER_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_REDIRECTION_DISABLED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_UNOCCLUDED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_WINDOWDC_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_WINDOWLESS_PRESENT_DISABLED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_INVALID_WINDOW,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_BUFFER_NOT_BOUND,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VAIL_STATE_CHANGED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INDIRECT_DISPLAY_ABANDON_SWAPCHAIN,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INDIRECT_DISPLAY_DEVICE_STOPPED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_VIDEO_MEMORY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_LOCK_MEMORY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_BUSY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TOO_MANY_REFERENCES,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TRY_AGAIN_LATER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TRY_AGAIN_NOW,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_INVALID,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_CLOSED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_FREQUENCY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ACTIVE_REGION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_TOTAL_REGION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TARGET_ALREADY_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_STALE_MODESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_RESOURCES_NOT_RELATED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_VIDPNMGR,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_ACTIVE_VIDPN,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITOR_NOT_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_STRIDE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PIXELFORMAT,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_COLORBASIS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_GAMMA_RAMP,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_NOT_IN_MODESET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_CLIENT_TYPE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_COPP_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UAB_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_HANDLE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PVP_HFS_FAILED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_SRM,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_DATA,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MCA_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_POINTER,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_FVE_LOCKED_VOLUME,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ENCRYPTED,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_INFORMATION,
  MD_NTSTATUS_WIN_STATUS_FVE_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_FVE_FAILED_WRONG_FS,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_PARTITION_SIZE,
  MD_NTSTATUS_WIN_STATUS_FVE_FS_NOT_EXTENDED,
  MD_NTSTATUS_WIN_STATUS_FVE_FS_MOUNTED,
  MD_NTSTATUS_WIN_STATUS_FVE_NO_LICENSE,
  MD_NTSTATUS_WIN_STATUS_FVE_ACTION_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_DATA,
  MD_NTSTATUS_WIN_STATUS_FVE_VOLUME_NOT_BOUND,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_DATA_VOLUME,
  MD_NTSTATUS_WIN_STATUS_FVE_CONV_READ_ERROR,
  MD_NTSTATUS_WIN_STATUS_FVE_CONV_WRITE_ERROR,
  MD_NTSTATUS_WIN_STATUS_FVE_OVERLAPPED_UPDATE,
  MD_NTSTATUS_WIN_STATUS_FVE_FAILED_SECTOR_SIZE,
  MD_NTSTATUS_WIN_STATUS_FVE_FAILED_AUTHENTICATION,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_OS_VOLUME,
  MD_NTSTATUS_WIN_STATUS_FVE_KEYFILE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FVE_KEYFILE_INVALID,
  MD_NTSTATUS_WIN_STATUS_FVE_KEYFILE_NO_VMK,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_INVALID_PCR,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_NO_VMK,
  MD_NTSTATUS_WIN_STATUS_FVE_PIN_INVALID,
  MD_NTSTATUS_WIN_STATUS_FVE_AUTH_INVALID_APPLICATION,
  MD_NTSTATUS_WIN_STATUS_FVE_AUTH_INVALID_CONFIG,
  MD_NTSTATUS_WIN_STATUS_FVE_DEBUGGER_ENABLED,
  MD_NTSTATUS_WIN_STATUS_FVE_DRY_RUN_FAILED,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_METADATA_POINTER,
  MD_NTSTATUS_WIN_STATUS_FVE_OLD_METADATA_COPY,
  MD_NTSTATUS_WIN_STATUS_FVE_REBOOT_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_FVE_RAW_ACCESS,
  MD_NTSTATUS_WIN_STATUS_FVE_RAW_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_FVE_NO_AUTOUNLOCK_MASTER_KEY,
  MD_NTSTATUS_WIN_STATUS_FVE_MOR_FAILED,
  MD_NTSTATUS_WIN_STATUS_FVE_NO_FEATURE_LICENSE,
  MD_NTSTATUS_WIN_STATUS_FVE_POLICY_USER_DISABLE_RDV_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_FVE_CONV_RECOVERY_FAILED,
  MD_NTSTATUS_WIN_STATUS_FVE_VIRTUALIZED_SPACE_TOO_BIG,
  MD_NTSTATUS_WIN_STATUS_FVE_INVALID_DATUM_TYPE,
  MD_NTSTATUS_WIN_STATUS_FVE_VOLUME_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_FVE_ENH_PIN_INVALID,
  MD_NTSTATUS_WIN_STATUS_FVE_FULL_ENCRYPTION_NOT_ALLOWED_ON_TP_STORAGE,
  MD_NTSTATUS_WIN_STATUS_FVE_WIPE_NOT_ALLOWED_ON_TP_STORAGE,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ALLOWED_ON_CSV_STACK,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ALLOWED_ON_CLUSTER,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ALLOWED_TO_UPGRADE_WHILE_CONVERTING,
  MD_NTSTATUS_WIN_STATUS_FVE_WIPE_CANCEL_NOT_APPLICABLE,
  MD_NTSTATUS_WIN_STATUS_FVE_EDRIVE_DRY_RUN_FAILED,
  MD_NTSTATUS_WIN_STATUS_FVE_SECUREBOOT_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FVE_SECUREBOOT_CONFIG_CHANGE,
  MD_NTSTATUS_WIN_STATUS_FVE_DEVICE_LOCKEDOUT,
  MD_NTSTATUS_WIN_STATUS_FVE_VOLUME_EXTEND_PREVENTS_EOW_DECRYPT,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_DE_VOLUME,
  MD_NTSTATUS_WIN_STATUS_FVE_PROTECTION_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FVE_PROTECTION_CANNOT_BE_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FVE_OSV_KSR_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_FWP_CALLOUT_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_CONDITION_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_FILTER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_LAYER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_PROVIDER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_SUBLAYER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_FWP_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_FWP_IN_USE,
  MD_NTSTATUS_WIN_STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_FWP_WRONG_SESSION,
  MD_NTSTATUS_WIN_STATUS_FWP_NO_TXN_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_FWP_TXN_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_FWP_TXN_ABORTED,
  MD_NTSTATUS_WIN_STATUS_FWP_SESSION_ABORTED,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_TXN,
  MD_NTSTATUS_WIN_STATUS_FWP_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_FWP_NET_EVENTS_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_LAYER,
  MD_NTSTATUS_WIN_STATUS_FWP_KM_CLIENTS_ONLY,
  MD_NTSTATUS_WIN_STATUS_FWP_LIFETIME_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_FWP_BUILTIN_OBJECT,
  MD_NTSTATUS_WIN_STATUS_FWP_TOO_MANY_CALLOUTS,
  MD_NTSTATUS_WIN_STATUS_FWP_NOTIFICATION_DROPPED,
  MD_NTSTATUS_WIN_STATUS_FWP_TRAFFIC_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_SA_STATE,
  MD_NTSTATUS_WIN_STATUS_FWP_NULL_POINTER,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_ENUMERATOR,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_FLAGS,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_NET_MASK,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_RANGE,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_INTERVAL,
  MD_NTSTATUS_WIN_STATUS_FWP_ZERO_LENGTH_ARRAY,
  MD_NTSTATUS_WIN_STATUS_FWP_NULL_DISPLAY_NAME,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_ACTION_TYPE,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_WEIGHT,
  MD_NTSTATUS_WIN_STATUS_FWP_MATCH_TYPE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_FWP_TYPE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_FWP_OUT_OF_BOUNDS,
  MD_NTSTATUS_WIN_STATUS_FWP_RESERVED,
  MD_NTSTATUS_WIN_STATUS_FWP_DUPLICATE_CONDITION,
  MD_NTSTATUS_WIN_STATUS_FWP_DUPLICATE_KEYMOD,
  MD_NTSTATUS_WIN_STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER,
  MD_NTSTATUS_WIN_STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER,
  MD_NTSTATUS_WIN_STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER,
  MD_NTSTATUS_WIN_STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_AUTH_METHOD,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_DH_GROUP,
  MD_NTSTATUS_WIN_STATUS_FWP_EM_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_FWP_NEVER_MATCH,
  MD_NTSTATUS_WIN_STATUS_FWP_PROVIDER_CONTEXT_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_FWP_TOO_MANY_SUBLAYERS,
  MD_NTSTATUS_WIN_STATUS_FWP_CALLOUT_NOTIFICATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_AUTH_TRANSFORM,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_CIPHER_TRANSFORM,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_CIPHER_TRANSFORM,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_TRANSFORM_COMBINATION,
  MD_NTSTATUS_WIN_STATUS_FWP_DUPLICATE_AUTH_METHOD,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_TUNNEL_ENDPOINT,
  MD_NTSTATUS_WIN_STATUS_FWP_L2_DRIVER_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_FWP_KEY_DICTATOR_ALREADY_REGISTERED,
  MD_NTSTATUS_WIN_STATUS_FWP_KEY_DICTATION_INVALID_KEYING_MATERIAL,
  MD_NTSTATUS_WIN_STATUS_FWP_CONNECTIONS_DISABLED,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_DNS_NAME,
  MD_NTSTATUS_WIN_STATUS_FWP_STILL_ON,
  MD_NTSTATUS_WIN_STATUS_FWP_IKEEXT_NOT_RUNNING,
  MD_NTSTATUS_WIN_STATUS_FWP_TCPIP_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_FWP_INJECT_HANDLE_CLOSING,
  MD_NTSTATUS_WIN_STATUS_FWP_INJECT_HANDLE_STALE,
  MD_NTSTATUS_WIN_STATUS_FWP_CANNOT_PEND,
  MD_NTSTATUS_WIN_STATUS_FWP_DROP_NOICMP,
  MD_NTSTATUS_WIN_STATUS_NDIS_CLOSING,
  MD_NTSTATUS_WIN_STATUS_NDIS_BAD_VERSION,
  MD_NTSTATUS_WIN_STATUS_NDIS_BAD_CHARACTERISTICS,
  MD_NTSTATUS_WIN_STATUS_NDIS_ADAPTER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NDIS_OPEN_FAILED,
  MD_NTSTATUS_WIN_STATUS_NDIS_DEVICE_FAILED,
  MD_NTSTATUS_WIN_STATUS_NDIS_MULTICAST_FULL,
  MD_NTSTATUS_WIN_STATUS_NDIS_MULTICAST_EXISTS,
  MD_NTSTATUS_WIN_STATUS_NDIS_MULTICAST_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NDIS_REQUEST_ABORTED,
  MD_NTSTATUS_WIN_STATUS_NDIS_RESET_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_PACKET,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_DEVICE_REQUEST,
  MD_NTSTATUS_WIN_STATUS_NDIS_ADAPTER_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_LENGTH,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_DATA,
  MD_NTSTATUS_WIN_STATUS_NDIS_BUFFER_TOO_SHORT,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_OID,
  MD_NTSTATUS_WIN_STATUS_NDIS_ADAPTER_REMOVED,
  MD_NTSTATUS_WIN_STATUS_NDIS_UNSUPPORTED_MEDIA,
  MD_NTSTATUS_WIN_STATUS_NDIS_GROUP_ADDRESS_IN_USE,
  MD_NTSTATUS_WIN_STATUS_NDIS_FILE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NDIS_ERROR_READING_FILE,
  MD_NTSTATUS_WIN_STATUS_NDIS_ALREADY_MAPPED,
  MD_NTSTATUS_WIN_STATUS_NDIS_RESOURCE_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_NDIS_MEDIA_DISCONNECTED,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_ADDRESS,
  MD_NTSTATUS_WIN_STATUS_NDIS_PAUSED,
  MD_NTSTATUS_WIN_STATUS_NDIS_INTERFACE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_NDIS_UNSUPPORTED_REVISION,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_PORT,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_PORT_STATE,
  MD_NTSTATUS_WIN_STATUS_NDIS_LOW_POWER_STATE,
  MD_NTSTATUS_WIN_STATUS_NDIS_REINIT_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_NDIS_NO_QUEUES,
  MD_NTSTATUS_WIN_STATUS_NDIS_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_NDIS_OFFLOAD_POLICY,
  MD_NTSTATUS_WIN_STATUS_NDIS_OFFLOAD_CONNECTION_REJECTED,
  MD_NTSTATUS_WIN_STATUS_NDIS_OFFLOAD_PATH_REJECTED,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_MEDIA_IN_USE,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_POWER_STATE_INVALID,
  MD_NTSTATUS_WIN_STATUS_NDIS_PM_WOL_PATTERN_LIST_FULL,
  MD_NTSTATUS_WIN_STATUS_NDIS_PM_PROTOCOL_OFFLOAD_LIST_FULL,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_CHANNEL_CURRENTLY_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_BAND_CURRENTLY_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_CHANNEL_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_BAND_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_QUIC_HANDSHAKE_FAILURE,
  MD_NTSTATUS_WIN_STATUS_QUIC_VER_NEG_FAILURE,
  MD_NTSTATUS_WIN_STATUS_TPM_ERROR_MASK,
  MD_NTSTATUS_WIN_STATUS_TPM_AUTHFAIL,
  MD_NTSTATUS_WIN_STATUS_TPM_BADINDEX,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_TPM_AUDITFAILURE,
  MD_NTSTATUS_WIN_STATUS_TPM_CLEAR_DISABLED,
  MD_NTSTATUS_WIN_STATUS_TPM_DEACTIVATED,
  MD_NTSTATUS_WIN_STATUS_TPM_DISABLED,
  MD_NTSTATUS_WIN_STATUS_TPM_DISABLED_CMD,
  MD_NTSTATUS_WIN_STATUS_TPM_FAIL,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_ORDINAL,
  MD_NTSTATUS_WIN_STATUS_TPM_INSTALL_DISABLED,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_KEYHANDLE,
  MD_NTSTATUS_WIN_STATUS_TPM_KEYNOTFOUND,
  MD_NTSTATUS_WIN_STATUS_TPM_INAPPROPRIATE_ENC,
  MD_NTSTATUS_WIN_STATUS_TPM_MIGRATEFAIL,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_PCR_INFO,
  MD_NTSTATUS_WIN_STATUS_TPM_NOSPACE,
  MD_NTSTATUS_WIN_STATUS_TPM_NOSRK,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTSEALED_BLOB,
  MD_NTSTATUS_WIN_STATUS_TPM_OWNER_SET,
  MD_NTSTATUS_WIN_STATUS_TPM_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_TPM_SHORTRANDOM,
  MD_NTSTATUS_WIN_STATUS_TPM_SIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_WRONGPCRVAL,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_PARAM_SIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_SHA_THREAD,
  MD_NTSTATUS_WIN_STATUS_TPM_SHA_ERROR,
  MD_NTSTATUS_WIN_STATUS_TPM_FAILEDSELFTEST,
  MD_NTSTATUS_WIN_STATUS_TPM_AUTH2FAIL,
  MD_NTSTATUS_WIN_STATUS_TPM_BADTAG,
  MD_NTSTATUS_WIN_STATUS_TPM_IOERROR,
  MD_NTSTATUS_WIN_STATUS_TPM_ENCRYPT_ERROR,
  MD_NTSTATUS_WIN_STATUS_TPM_DECRYPT_ERROR,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_AUTHHANDLE,
  MD_NTSTATUS_WIN_STATUS_TPM_NO_ENDORSEMENT,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_KEYUSAGE,
  MD_NTSTATUS_WIN_STATUS_TPM_WRONG_ENTITYTYPE,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_POSTINIT,
  MD_NTSTATUS_WIN_STATUS_TPM_INAPPROPRIATE_SIG,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_KEY_PROPERTY,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_MIGRATION,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_SCHEME,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_DATASIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_MODE,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_PRESENCE,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_VERSION,
  MD_NTSTATUS_WIN_STATUS_TPM_NO_WRAP_TRANSPORT,
  MD_NTSTATUS_WIN_STATUS_TPM_AUDITFAIL_UNSUCCESSFUL,
  MD_NTSTATUS_WIN_STATUS_TPM_AUDITFAIL_SUCCESSFUL,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTRESETABLE,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTLOCAL,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_TYPE,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_RESOURCE,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTFIPS,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_FAMILY,
  MD_NTSTATUS_WIN_STATUS_TPM_NO_NV_PERMISSION,
  MD_NTSTATUS_WIN_STATUS_TPM_REQUIRES_SIGN,
  MD_NTSTATUS_WIN_STATUS_TPM_KEY_NOTSUPPORTED,
  MD_NTSTATUS_WIN_STATUS_TPM_AUTH_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_TPM_AREA_LOCKED,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_LOCALITY,
  MD_NTSTATUS_WIN_STATUS_TPM_READ_ONLY,
  MD_NTSTATUS_WIN_STATUS_TPM_PER_NOWRITE,
  MD_NTSTATUS_WIN_STATUS_TPM_FAMILYCOUNT,
  MD_NTSTATUS_WIN_STATUS_TPM_WRITE_LOCKED,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_ATTRIBUTES,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_STRUCTURE,
  MD_NTSTATUS_WIN_STATUS_TPM_KEY_OWNER_CONTROL,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_COUNTER,
  MD_NTSTATUS_WIN_STATUS_TPM_NOT_FULLWRITE,
  MD_NTSTATUS_WIN_STATUS_TPM_CONTEXT_GAP,
  MD_NTSTATUS_WIN_STATUS_TPM_MAXNVWRITES,
  MD_NTSTATUS_WIN_STATUS_TPM_NOOPERATOR,
  MD_NTSTATUS_WIN_STATUS_TPM_RESOURCEMISSING,
  MD_NTSTATUS_WIN_STATUS_TPM_DELEGATE_LOCK,
  MD_NTSTATUS_WIN_STATUS_TPM_DELEGATE_FAMILY,
  MD_NTSTATUS_WIN_STATUS_TPM_DELEGATE_ADMIN,
  MD_NTSTATUS_WIN_STATUS_TPM_TRANSPORT_NOTEXCLUSIVE,
  MD_NTSTATUS_WIN_STATUS_TPM_OWNER_CONTROL,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_INPUT_DATA0,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_INPUT_DATA1,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_ISSUER_SETTINGS,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_TPM_SETTINGS,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_STAGE,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_ISSUER_VALIDITY,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_WRONG_W,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_HANDLE,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_DELEGATE,
  MD_NTSTATUS_WIN_STATUS_TPM_BADCONTEXT,
  MD_NTSTATUS_WIN_STATUS_TPM_TOOMANYCONTEXTS,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_TICKET_SIGNATURE,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_DESTINATION,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_SOURCE,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_AUTHORITY,
  MD_NTSTATUS_WIN_STATUS_TPM_PERMANENTEK,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_SIGNATURE,
  MD_NTSTATUS_WIN_STATUS_TPM_NOCONTEXTSPACE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ASYMMETRIC,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ATTRIBUTES,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HASH,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_VALUE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HIERARCHY,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_KEY_SIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_MGF,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_MODE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TYPE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HANDLE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_KDF,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_RANGE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_FAIL,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NONCE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PP,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SCHEME,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SYMMETRIC,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TAG,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SELECTOR,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_INSUFFICIENT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SIGNATURE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_KEY,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_POLICY_FAIL,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_INTEGRITY,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TICKET,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_RESERVED_BITS,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_BAD_AUTH,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_POLICY_CC,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_BINDING,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_CURVE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ECC_POINT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_INITIALIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_FAILURE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SEQUENCE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PRIVATE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HMAC,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_DISABLED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_EXCLUSIVE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ECC_CURVE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_TYPE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_MISSING,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_POLICY,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PCR,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PCR_CHANGED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_UPGRADE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TOO_MANY_CONTEXTS,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_REBOOT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_UNBALANCED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_COMMAND_SIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_COMMAND_CODE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTHSIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_RANGE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_SIZE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_LOCKED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_AUTHORIZATION,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_UNINITIALIZED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_SPACE,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_DEFINED,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_BAD_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_CPHASH,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PARENT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NEEDS_TEST,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NO_RESULT,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SENSITIVE,
  MD_NTSTATUS_WIN_STATUS_TPM_COMMAND_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_HANDLE,
  MD_NTSTATUS_WIN_STATUS_TPM_DUPLICATE_VHANDLE,
  MD_NTSTATUS_WIN_STATUS_TPM_EMBEDDED_COMMAND_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_TPM_EMBEDDED_COMMAND_UNSUPPORTED,
  MD_NTSTATUS_WIN_STATUS_TPM_RETRY,
  MD_NTSTATUS_WIN_STATUS_TPM_NEEDS_SELFTEST,
  MD_NTSTATUS_WIN_STATUS_TPM_DOING_SELFTEST,
  MD_NTSTATUS_WIN_STATUS_TPM_DEFEND_LOCK_RUNNING,
  MD_NTSTATUS_WIN_STATUS_TPM_COMMAND_CANCELED,
  MD_NTSTATUS_WIN_STATUS_TPM_TOO_MANY_CONTEXTS,
  MD_NTSTATUS_WIN_STATUS_TPM_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_TPM_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_TPM_INSUFFICIENT_BUFFER,
  MD_NTSTATUS_WIN_STATUS_TPM_PPI_FUNCTION_UNSUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_ERROR_MASK,
  MD_NTSTATUS_WIN_STATUS_PCP_DEVICE_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_PCP_INVALID_HANDLE,
  MD_NTSTATUS_WIN_STATUS_PCP_INVALID_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_PCP_FLAG_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_BUFFER_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_PCP_INTERNAL_ERROR,
  MD_NTSTATUS_WIN_STATUS_PCP_AUTHENTICATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_PCP_AUTHENTICATION_IGNORED,
  MD_NTSTATUS_WIN_STATUS_PCP_POLICY_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PCP_PROFILE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PCP_VALIDATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_PCP_DEVICE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PCP_WRONG_PARENT,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_LOADED,
  MD_NTSTATUS_WIN_STATUS_PCP_NO_KEY_CERTIFICATION,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_FINALIZED,
  MD_NTSTATUS_WIN_STATUS_PCP_ATTESTATION_CHALLENGE_NOT_SET,
  MD_NTSTATUS_WIN_STATUS_PCP_NOT_PCR_BOUND,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_ALREADY_FINALIZED,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_USAGE_POLICY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_USAGE_POLICY_INVALID,
  MD_NTSTATUS_WIN_STATUS_PCP_SOFT_KEY_ERROR,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_AUTHENTICATED,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_AIK,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_SIGNING_KEY,
  MD_NTSTATUS_WIN_STATUS_PCP_LOCKED_OUT,
  MD_NTSTATUS_WIN_STATUS_PCP_CLAIM_TYPE_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_TPM_VERSION_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_BUFFER_LENGTH_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_PCP_IFX_RSA_KEY_CREATION_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_PCP_TICKET_MISSING,
  MD_NTSTATUS_WIN_STATUS_PCP_RAW_POLICY_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_HANDLE_INVALIDATED,
  MD_NTSTATUS_WIN_STATUS_RTPM_NO_RESULT,
  MD_NTSTATUS_WIN_STATUS_RTPM_PCR_READ_INCOMPLETE,
  MD_NTSTATUS_WIN_STATUS_RTPM_INVALID_CONTEXT,
  MD_NTSTATUS_WIN_STATUS_RTPM_UNSUPPORTED_CMD,
  MD_NTSTATUS_WIN_STATUS_TPM_ZERO_EXHAUST_ENABLED,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_HYPERCALL_CODE,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_HYPERCALL_INPUT,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_ALIGNMENT,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PARAMETER,
  MD_NTSTATUS_WIN_STATUS_HV_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PARTITION_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_OPERATION_DENIED,
  MD_NTSTATUS_WIN_STATUS_HV_UNKNOWN_PROPERTY,
  MD_NTSTATUS_WIN_STATUS_HV_PROPERTY_VALUE_OUT_OF_RANGE,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_MEMORY,
  MD_NTSTATUS_WIN_STATUS_HV_PARTITION_TOO_DEEP,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PARTITION_ID,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_VP_INDEX,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PORT_ID,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_CONNECTION_ID,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_BUFFERS,
  MD_NTSTATUS_WIN_STATUS_HV_NOT_ACKNOWLEDGED,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_VP_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_ACKNOWLEDGED,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_SAVE_RESTORE_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_SYNIC_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_OBJECT_IN_USE,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PROXIMITY_DOMAIN_INFO,
  MD_NTSTATUS_WIN_STATUS_HV_NO_DATA,
  MD_NTSTATUS_WIN_STATUS_HV_INACTIVE,
  MD_NTSTATUS_WIN_STATUS_HV_NO_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_HV_FEATURE_UNAVAILABLE,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_BUFFER,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_DEVICE_DOMAINS,
  MD_NTSTATUS_WIN_STATUS_HV_CPUID_FEATURE_VALIDATION_ERROR,
  MD_NTSTATUS_WIN_STATUS_HV_CPUID_XSAVE_FEATURE_VALIDATION_ERROR,
  MD_NTSTATUS_WIN_STATUS_HV_PROCESSOR_STARTUP_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_HV_SMX_ENABLED,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_LP_INDEX,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_REGISTER_VALUE,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_VTL_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_NX_NOT_DETECTED,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_DEVICE_ID,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_DEVICE_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_PAGE_REQUEST_INVALID,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_CPU_GROUP_ID,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_CPU_GROUP_STATE,
  MD_NTSTATUS_WIN_STATUS_HV_OPERATION_FAILED,
  MD_NTSTATUS_WIN_STATUS_HV_NOT_ALLOWED_WITH_NESTED_VIRT_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_ROOT_MEMORY,
  MD_NTSTATUS_WIN_STATUS_HV_EVENT_BUFFER_ALREADY_FREED,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_CONTIGUOUS_MEMORY,
  MD_NTSTATUS_WIN_STATUS_HV_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_IPSEC_BAD_SPI,
  MD_NTSTATUS_WIN_STATUS_IPSEC_SA_LIFETIME_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_IPSEC_WRONG_SA,
  MD_NTSTATUS_WIN_STATUS_IPSEC_REPLAY_CHECK_FAILED,
  MD_NTSTATUS_WIN_STATUS_IPSEC_INVALID_PACKET,
  MD_NTSTATUS_WIN_STATUS_IPSEC_INTEGRITY_CHECK_FAILED,
  MD_NTSTATUS_WIN_STATUS_IPSEC_CLEAR_TEXT_DROP,
  MD_NTSTATUS_WIN_STATUS_IPSEC_AUTH_FIREWALL_DROP,
  MD_NTSTATUS_WIN_STATUS_IPSEC_THROTTLE_DROP,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_BLOCK,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_RECEIVED_MULTICAST,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_INVALID_PACKET,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_STATE_LOOKUP_FAILED,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_MAX_ENTRIES,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_KEYMOD_NOT_ALLOWED,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES,
  MD_NTSTATUS_WIN_STATUS_VID_DUPLICATE_HANDLER,
  MD_NTSTATUS_WIN_STATUS_VID_TOO_MANY_HANDLERS,
  MD_NTSTATUS_WIN_STATUS_VID_QUEUE_FULL,
  MD_NTSTATUS_WIN_STATUS_VID_HANDLER_NOT_PRESENT,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_OBJECT_NAME,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_NAME_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_VID_MESSAGE_QUEUE_NAME_TOO_LONG,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_DOES_NOT_EXIST,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_NAME_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_VID_MESSAGE_QUEUE_ALREADY_EXISTS,
  MD_NTSTATUS_WIN_STATUS_VID_EXCEEDED_MBP_ENTRY_MAP_LIMIT,
  MD_NTSTATUS_WIN_STATUS_VID_MB_STILL_REFERENCED,
  MD_NTSTATUS_WIN_STATUS_VID_CHILD_GPA_PAGE_SET_CORRUPTED,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_NUMA_SETTINGS,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_NUMA_NODE_INDEX,
  MD_NTSTATUS_WIN_STATUS_VID_NOTIFICATION_QUEUE_ALREADY_ASSOCIATED,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_MEMORY_BLOCK_HANDLE,
  MD_NTSTATUS_WIN_STATUS_VID_PAGE_RANGE_OVERFLOW,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_MESSAGE_QUEUE_HANDLE,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_GPA_RANGE_HANDLE,
  MD_NTSTATUS_WIN_STATUS_VID_NO_MEMORY_BLOCK_NOTIFICATION_QUEUE,
  MD_NTSTATUS_WIN_STATUS_VID_MEMORY_BLOCK_LOCK_COUNT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_PPM_HANDLE,
  MD_NTSTATUS_WIN_STATUS_VID_MBPS_ARE_LOCKED,
  MD_NTSTATUS_WIN_STATUS_VID_MESSAGE_QUEUE_CLOSED,
  MD_NTSTATUS_WIN_STATUS_VID_VIRTUAL_PROCESSOR_LIMIT_EXCEEDED,
  MD_NTSTATUS_WIN_STATUS_VID_STOP_PENDING,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_PROCESSOR_STATE,
  MD_NTSTATUS_WIN_STATUS_VID_EXCEEDED_KM_CONTEXT_COUNT_LIMIT,
  MD_NTSTATUS_WIN_STATUS_VID_KM_INTERFACE_ALREADY_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_VID_MB_PROPERTY_ALREADY_SET_RESET,
  MD_NTSTATUS_WIN_STATUS_VID_MMIO_RANGE_DESTROYED,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_CHILD_GPA_PAGE_SET,
  MD_NTSTATUS_WIN_STATUS_VID_RESERVE_PAGE_SET_IS_BEING_USED,
  MD_NTSTATUS_WIN_STATUS_VID_RESERVE_PAGE_SET_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_VID_MBP_ALREADY_LOCKED_USING_RESERVED_PAGE,
  MD_NTSTATUS_WIN_STATUS_VID_MBP_COUNT_EXCEEDED_LIMIT,
  MD_NTSTATUS_WIN_STATUS_VID_SAVED_STATE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_VID_SAVED_STATE_UNRECOGNIZED_ITEM,
  MD_NTSTATUS_WIN_STATUS_VID_SAVED_STATE_INCOMPATIBLE,
  MD_NTSTATUS_WIN_STATUS_VID_VTL_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DATABASE_FULL,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_CONFIGURATION_CORRUPTED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_CONFIGURATION_NOT_IN_SYNC,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_CONFIG_UPDATE_FAILED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_CONTAINS_NON_SIMPLE_VOLUME,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_DUPLICATE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_DYNAMIC,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_ID_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAST_VOTER,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_NON_BASIC_BETWEEN_BASIC_PARTITIONS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_NOT_CYLINDER_ALIGNED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_PARTITIONS_TOO_SMALL,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_PRIMARY_BETWEEN_LOGICAL_PARTITIONS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_TOO_MANY_PARTITIONS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_MISSING,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_NOT_EMPTY,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_NOT_ENOUGH_SPACE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_REVECTORING_FAILED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_SECTOR_SIZE_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_SET_NOT_CONTAINED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_MEMBERS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_PLEXES,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DYNAMIC_DISK_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_ALREADY_USED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_NOT_CONTIGUOUS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_NOT_IN_PUBLIC_REGION,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_NOT_SECTOR_ALIGNED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_OVERLAPS_EBR_PARTITION,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_VOLUME_LENGTHS_DO_NOT_MATCH,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_FAULT_TOLERANT_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_INTERLEAVE_LENGTH_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MAXIMUM_REGISTERED_USERS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_IN_SYNC,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_INDEX_DUPLICATE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_INDEX_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_MISSING,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_NOT_DETACHED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_REGENERATING,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_ALL_DISKS_FAILED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NO_REGISTERED_USERS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NO_SUCH_USER,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NOTIFICATION_RESET,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_MEMBERS_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_PLEXES_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_DUPLICATE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_ID_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_NAME_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_OFFLINE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_HAS_QUORUM,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_WITHOUT_QUORUM,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PARTITION_STYLE_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PARTITION_UPDATE_FAILED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_IN_SYNC,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_INDEX_DUPLICATE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_INDEX_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_LAST_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_MISSING,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_REGENERATING,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_TYPE_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_NOT_RAID5,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_NOT_SIMPLE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_STRUCTURE_SIZE_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_TOO_MANY_NOTIFICATION_REQUESTS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_TRANSACTION_IN_PROGRESS,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_UNEXPECTED_DISK_LAYOUT_CHANGE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_CONTAINS_MISSING_DISK,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_ID_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_LENGTH_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_LENGTH_NOT_SECTOR_SIZE_MULTIPLE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_NOT_MIRRORED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_NOT_RETAINED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_OFFLINE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_RETAINED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_EXTENTS_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DIFFERENT_SECTOR_SIZE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_BAD_BOOT_DISK,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_CONFIG_OFFLINE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_CONFIG_ONLINE,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NOT_PRIMARY_PACK,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_LOG_UPDATE_FAILED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_DISKS_IN_PLEX_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_DISKS_IN_MEMBER_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_MIRRORED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_NOT_SIMPLE_SPANNED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NO_VALID_LOG_COPIES,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PRIMARY_PACK_PRESENT,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_DISKS_INVALID,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MIRROR_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_RAID5_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_BCD_TOO_MANY_ELEMENTS,
  MD_NTSTATUS_WIN_STATUS_VHD_DRIVE_FOOTER_MISSING,
  MD_NTSTATUS_WIN_STATUS_VHD_DRIVE_FOOTER_CHECKSUM_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_DRIVE_FOOTER_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_VHD_FORMAT_UNKNOWN,
  MD_NTSTATUS_WIN_STATUS_VHD_FORMAT_UNSUPPORTED_VERSION,
  MD_NTSTATUS_WIN_STATUS_VHD_SPARSE_HEADER_CHECKSUM_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_SPARSE_HEADER_UNSUPPORTED_VERSION,
  MD_NTSTATUS_WIN_STATUS_VHD_SPARSE_HEADER_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_VHD_BLOCK_ALLOCATION_FAILURE,
  MD_NTSTATUS_WIN_STATUS_VHD_BLOCK_ALLOCATION_TABLE_CORRUPT,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_BLOCK_SIZE,
  MD_NTSTATUS_WIN_STATUS_VHD_BITMAP_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_PARENT_VHD_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_VHD_CHILD_PARENT_ID_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_CHILD_PARENT_TIMESTAMP_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_METADATA_READ_FAILURE,
  MD_NTSTATUS_WIN_STATUS_VHD_METADATA_WRITE_FAILURE,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_SIZE,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_FILE_SIZE,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_PROVIDER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_NOT_VIRTUAL_DISK,
  MD_NTSTATUS_WIN_STATUS_VHD_PARENT_VHD_ACCESS_DENIED,
  MD_NTSTATUS_WIN_STATUS_VHD_CHILD_PARENT_SIZE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_DIFFERENCING_CHAIN_CYCLE_DETECTED,
  MD_NTSTATUS_WIN_STATUS_VHD_DIFFERENCING_CHAIN_ERROR_IN_PARENT,
  MD_NTSTATUS_WIN_STATUS_VIRTUAL_DISK_LIMITATION,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_TYPE,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_STATE,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_UNSUPPORTED_DISK_SECTOR_SIZE,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_DISK_ALREADY_OWNED,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_DISK_ONLINE_AND_WRITABLE,
  MD_NTSTATUS_WIN_STATUS_CTLOG_TRACKING_NOT_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_CTLOG_LOGFILE_SIZE_EXCEEDED_MAXSIZE,
  MD_NTSTATUS_WIN_STATUS_CTLOG_VHD_CHANGED_OFFLINE,
  MD_NTSTATUS_WIN_STATUS_CTLOG_INVALID_TRACKING_STATE,
  MD_NTSTATUS_WIN_STATUS_CTLOG_INCONSISTENT_TRACKING_FILE,
  MD_NTSTATUS_WIN_STATUS_VHD_METADATA_FULL,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_CHANGE_TRACKING_ID,
  MD_NTSTATUS_WIN_STATUS_VHD_CHANGE_TRACKING_DISABLED,
  MD_NTSTATUS_WIN_STATUS_VHD_MISSING_CHANGE_TRACKING_INFORMATION,
  MD_NTSTATUS_WIN_STATUS_VHD_RESIZE_WOULD_TRUNCATE_DATA,
  MD_NTSTATUS_WIN_STATUS_VHD_COULD_NOT_COMPUTE_MINIMUM_VIRTUAL_SIZE,
  MD_NTSTATUS_WIN_STATUS_VHD_ALREADY_AT_OR_BELOW_MINIMUM_VIRTUAL_SIZE,
  MD_NTSTATUS_WIN_STATUS_RKF_KEY_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_RKF_DUPLICATE_KEY,
  MD_NTSTATUS_WIN_STATUS_RKF_BLOB_FULL,
  MD_NTSTATUS_WIN_STATUS_RKF_STORE_FULL,
  MD_NTSTATUS_WIN_STATUS_RKF_FILE_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_RKF_ACTIVE_KEY,
  MD_NTSTATUS_WIN_STATUS_RDBSS_RESTART_OPERATION,
  MD_NTSTATUS_WIN_STATUS_RDBSS_CONTINUE_OPERATION,
  MD_NTSTATUS_WIN_STATUS_RDBSS_POST_OPERATION,
  MD_NTSTATUS_WIN_STATUS_RDBSS_RETRY_LOOKUP,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_HANDLE,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_READ_NOT_PERMITTED,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_WRITE_NOT_PERMITTED,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_PDU,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_AUTHENTICATION,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_REQUEST_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_OFFSET,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_AUTHORIZATION,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_PREPARE_QUEUE_FULL,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_ATTRIBUTE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_ATTRIBUTE_NOT_LONG,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION_KEY_SIZE,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_ATTRIBUTE_VALUE_LENGTH,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_UNLIKELY,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_UNSUPPORTED_GROUP_TYPE,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_RESOURCES,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_UNKNOWN_ERROR,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_ROLLBACK_DETECTED,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_INVALID_POLICY,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_NOT_SIGNED,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_FILE_REPLACED,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_NOT_AUTHORIZED,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_UNKNOWN,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_PLATFORM_ID_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_ROLLBACK_DETECTED,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_UPGRADE_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_NOT_BASE_POLICY,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY,
  MD_NTSTATUS_WIN_STATUS_AUDIO_ENGINE_NODE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_EMPTY_CONNECTION_LIST,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_CONNECTION_LIST_NOT_SUPPORTED,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_NO_LOGICAL_DEVICES_CREATED,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_NULL_LINKED_LIST_ENTRY,
  MD_NTSTATUS_WIN_STATUS_VSM_NOT_INITIALIZED,
  MD_NTSTATUS_WIN_STATUS_VSM_DMA_PROTECTION_NOT_IN_USE,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_BOOTFILE_NOT_VALID,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_ACTIVATION_TIMEOUT,
  MD_NTSTATUS_WIN_STATUS_IO_PREEMPTED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_ERROR_STORED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_ERROR_NOT_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_AVAILABLE,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_CAPACITY_DATA_CHANGED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_PREEMPTED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_RELEASED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_REGISTRATIONS_PREEMPTED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_OPERATING_DEFINITION_CHANGED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_RESERVATION_CONFLICT,
  MD_NTSTATUS_WIN_STATUS_SVHDX_WRONG_FILE_TYPE,
  MD_NTSTATUS_WIN_STATUS_SVHDX_VERSION_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_VHD_SHARED,
  MD_NTSTATUS_WIN_STATUS_SVHDX_NO_INITIATOR,
  MD_NTSTATUS_WIN_STATUS_VHDSET_BACKING_STORAGE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_SMB_NO_PREAUTH_INTEGRITY_HASH_OVERLAP,
  MD_NTSTATUS_WIN_STATUS_SMB_BAD_CLUSTER_DIALECT,
  MD_NTSTATUS_WIN_STATUS_SMB_GUEST_LOGON_BLOCKED,
  MD_NTSTATUS_WIN_STATUS_SPACES_FAULT_DOMAIN_TYPE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_RESILIENCY_TYPE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_SECTOR_SIZE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_REDUNDANCY_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_NUMBER_OF_DATA_COPIES_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_INTERLEAVE_LENGTH_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_NUMBER_OF_COLUMNS_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_NOT_ENOUGH_DRIVES,
  MD_NTSTATUS_WIN_STATUS_SPACES_EXTENDED_ERROR,
  MD_NTSTATUS_WIN_STATUS_SPACES_PROVISIONING_TYPE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_ALLOCATION_SIZE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_ENCLOSURE_AWARE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_WRITE_CACHE_SIZE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_NUMBER_OF_GROUPS_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_OPERATIONAL_STATE_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_UPDATE_COLUMN_STATE,
  MD_NTSTATUS_WIN_STATUS_SPACES_MAP_REQUIRED,
  MD_NTSTATUS_WIN_STATUS_SPACES_UNSUPPORTED_VERSION,
  MD_NTSTATUS_WIN_STATUS_SPACES_CORRUPT_METADATA,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRT_FULL,
  MD_NTSTATUS_WIN_STATUS_SPACES_INCONSISTENCY,
  MD_NTSTATUS_WIN_STATUS_SPACES_LOG_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_SPACES_NO_REDUNDANCY,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_NOT_READY,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_SPLIT,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_LOST_DATA,
  MD_NTSTATUS_WIN_STATUS_SPACES_ENTRY_INCOMPLETE,
  MD_NTSTATUS_WIN_STATUS_SPACES_ENTRY_INVALID,
  MD_NTSTATUS_WIN_STATUS_SPACES_MARK_DIRTY,
  MD_NTSTATUS_WIN_STATUS_SECCORE_INVALID_COMMAND,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_ROLLBACK_DETECTED,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_POLICY_VIOLATION,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_INVALID_POLICY,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_TOO_MANY_POLICIES,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED,
  MD_NTSTATUS_WIN_STATUS_NO_APPLICABLE_APP_LICENSES_FOUND,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_CLIP_DEVICE_LICENSE_MISSING,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_INVALID_SIGNATURE,
  MD_NTSTATUS_WIN_STATUS_CLIP_KEYHOLDER_LICENSE_MISSING_OR_INVALID,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_EXPIRED,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_SIGNED_BY_UNKNOWN_SOURCE,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_NOT_SIGNED,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_HARDWARE_ID_OUT_OF_TOLERANCE,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_DEVICE_ID_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_NOT_AUTHORIZED,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_INVALID,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_NOT_ACTIVE,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_NOT_SIGNED,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_CONDITION_NOT_SATISFIED,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_HANDLE_INVALIDATED,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_INVALID_HOST_GENERATION,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_INVALID_HOST_STATE,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_NO_DONOR,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_HOST_ID_MISMATCH,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_UNKNOWN_USER,
};

// The content of this array was created from ntstatus.h in the 10 SDK
// (version 10.0.19041.0) with
//
// egrep '#define [A-Z_0-9]+\s+\(\(NTSTATUS\)0x[048C][0-9A-F]+L\)' ntstatus.h
// | tr -d '\r'
// | sed -r 's@#define ([A-Z_0-9]+)\s+\(\(NTSTATUS\)(0x[048C][0-9A-F]+)L\).*@\2 \1@'
// | sort
// | sed -r 's@(0x[048C][0-9A-F]+) ([A-Z_0-9]+)@  "\2",@'
//
// With easy copy to clipboard with
// | xclip -selection c  # on linux
// | clip  # on windows
// | pbcopy  # on mac
static const char* kNTStatusStrings[] = {
  "STATUS_SUCCESS",
  "STATUS_WAIT_0",
  "STATUS_WAIT_1",
  "STATUS_WAIT_2",
  "STATUS_WAIT_3",
  "STATUS_WAIT_63",
  "STATUS_ABANDONED",
  "STATUS_ABANDONED_WAIT_0",
  "STATUS_ABANDONED_WAIT_63",
  "STATUS_USER_APC",
  "STATUS_ALREADY_COMPLETE",
  "STATUS_KERNEL_APC",
  "STATUS_ALERTED",
  "STATUS_TIMEOUT",
  "STATUS_PENDING",
  "STATUS_REPARSE",
  "STATUS_MORE_ENTRIES",
  "STATUS_NOT_ALL_ASSIGNED",
  "STATUS_SOME_NOT_MAPPED",
  "STATUS_OPLOCK_BREAK_IN_PROGRESS",
  "STATUS_VOLUME_MOUNTED",
  "STATUS_RXACT_COMMITTED",
  "STATUS_NOTIFY_CLEANUP",
  "STATUS_NOTIFY_ENUM_DIR",
  "STATUS_NO_QUOTAS_FOR_ACCOUNT",
  "STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED",
  "STATUS_PAGE_FAULT_TRANSITION",
  "STATUS_PAGE_FAULT_DEMAND_ZERO",
  "STATUS_PAGE_FAULT_COPY_ON_WRITE",
  "STATUS_PAGE_FAULT_GUARD_PAGE",
  "STATUS_PAGE_FAULT_PAGING_FILE",
  "STATUS_CACHE_PAGE_LOCKED",
  "STATUS_CRASH_DUMP",
  "STATUS_BUFFER_ALL_ZEROS",
  "STATUS_REPARSE_OBJECT",
  "STATUS_RESOURCE_REQUIREMENTS_CHANGED",
  "STATUS_TRANSLATION_COMPLETE",
  "STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY",
  "STATUS_NOTHING_TO_TERMINATE",
  "STATUS_PROCESS_NOT_IN_JOB",
  "STATUS_PROCESS_IN_JOB",
  "STATUS_VOLSNAP_HIBERNATE_READY",
  "STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY",
  "STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED",
  "STATUS_INTERRUPT_STILL_CONNECTED",
  "STATUS_PROCESS_CLONED",
  "STATUS_FILE_LOCKED_WITH_ONLY_READERS",
  "STATUS_FILE_LOCKED_WITH_WRITERS",
  "STATUS_VALID_IMAGE_HASH",
  "STATUS_VALID_CATALOG_HASH",
  "STATUS_VALID_STRONG_CODE_HASH",
  "STATUS_GHOSTED",
  "STATUS_DATA_OVERWRITTEN",
  "STATUS_RESOURCEMANAGER_READ_ONLY",
  "STATUS_RING_PREVIOUSLY_EMPTY",
  "STATUS_RING_PREVIOUSLY_FULL",
  "STATUS_RING_PREVIOUSLY_ABOVE_QUOTA",
  "STATUS_RING_NEWLY_EMPTY",
  "STATUS_RING_SIGNAL_OPPOSITE_ENDPOINT",
  "STATUS_OPLOCK_SWITCHED_TO_NEW_HANDLE",
  "STATUS_OPLOCK_HANDLE_CLOSED",
  "STATUS_WAIT_FOR_OPLOCK",
  "STATUS_REPARSE_GLOBAL",
  "DBG_EXCEPTION_HANDLED",
  "DBG_CONTINUE",
  "STATUS_FLT_IO_COMPLETE",
  "STATUS_RTPM_CONTEXT_CONTINUE",
  "STATUS_RTPM_CONTEXT_COMPLETE",
  "STATUS_HV_PENDING_PAGE_REQUESTS",
  "STATUS_SPACES_REPAIRED",
  "STATUS_SPACES_PAUSE",
  "STATUS_SPACES_COMPLETE",
  "STATUS_SPACES_REDIRECT",
  "STATUS_OBJECT_NAME_EXISTS",
  "STATUS_THREAD_WAS_SUSPENDED",
  "STATUS_WORKING_SET_LIMIT_RANGE",
  "STATUS_IMAGE_NOT_AT_BASE",
  "STATUS_RXACT_STATE_CREATED",
  "STATUS_SEGMENT_NOTIFICATION",
  "STATUS_LOCAL_USER_SESSION_KEY",
  "STATUS_BAD_CURRENT_DIRECTORY",
  "STATUS_SERIAL_MORE_WRITES",
  "STATUS_REGISTRY_RECOVERED",
  "STATUS_FT_READ_RECOVERY_FROM_BACKUP",
  "STATUS_FT_WRITE_RECOVERY",
  "STATUS_SERIAL_COUNTER_TIMEOUT",
  "STATUS_NULL_LM_PASSWORD",
  "STATUS_IMAGE_MACHINE_TYPE_MISMATCH",
  "STATUS_RECEIVE_PARTIAL",
  "STATUS_RECEIVE_EXPEDITED",
  "STATUS_RECEIVE_PARTIAL_EXPEDITED",
  "STATUS_EVENT_DONE",
  "STATUS_EVENT_PENDING",
  "STATUS_CHECKING_FILE_SYSTEM",
  "STATUS_FATAL_APP_EXIT",
  "STATUS_PREDEFINED_HANDLE",
  "STATUS_WAS_UNLOCKED",
  "STATUS_SERVICE_NOTIFICATION",
  "STATUS_WAS_LOCKED",
  "STATUS_LOG_HARD_ERROR",
  "STATUS_ALREADY_WIN32",
  "STATUS_WX86_UNSIMULATE",
  "STATUS_WX86_CONTINUE",
  "STATUS_WX86_SINGLE_STEP",
  "STATUS_WX86_BREAKPOINT",
  "STATUS_WX86_EXCEPTION_CONTINUE",
  "STATUS_WX86_EXCEPTION_LASTCHANCE",
  "STATUS_WX86_EXCEPTION_CHAIN",
  "STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE",
  "STATUS_NO_YIELD_PERFORMED",
  "STATUS_TIMER_RESUME_IGNORED",
  "STATUS_ARBITRATION_UNHANDLED",
  "STATUS_CARDBUS_NOT_SUPPORTED",
  "STATUS_WX86_CREATEWX86TIB",
  "STATUS_MP_PROCESSOR_MISMATCH",
  "STATUS_HIBERNATED",
  "STATUS_RESUME_HIBERNATION",
  "STATUS_FIRMWARE_UPDATED",
  "STATUS_DRIVERS_LEAKING_LOCKED_PAGES",
  "STATUS_MESSAGE_RETRIEVED",
  "STATUS_SYSTEM_POWERSTATE_TRANSITION",
  "STATUS_ALPC_CHECK_COMPLETION_LIST",
  "STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION",
  "STATUS_ACCESS_AUDIT_BY_POLICY",
  "STATUS_ABANDON_HIBERFILE",
  "STATUS_BIZRULES_NOT_ENABLED",
  "STATUS_FT_READ_FROM_COPY",
  "STATUS_IMAGE_AT_DIFFERENT_BASE",
  "STATUS_PATCH_DEFERRED",
  "STATUS_WAKE_SYSTEM",
  "STATUS_DS_SHUTTING_DOWN",
  "STATUS_DISK_REPAIR_REDIRECTED",
  "STATUS_SERVICES_FAILED_AUTOSTART",
  "DBG_REPLY_LATER",
  "DBG_UNABLE_TO_PROVIDE_HANDLE",
  "DBG_TERMINATE_THREAD",
  "DBG_TERMINATE_PROCESS",
  "DBG_CONTROL_C",
  "DBG_PRINTEXCEPTION_C",
  "DBG_RIPEXCEPTION",
  "DBG_CONTROL_BREAK",
  "DBG_COMMAND_EXCEPTION",
  "DBG_PRINTEXCEPTION_WIDE_C",
  "RPC_NT_UUID_LOCAL_ONLY",
  "RPC_NT_SEND_INCOMPLETE",
  "STATUS_CTX_CDM_CONNECT",
  "STATUS_CTX_CDM_DISCONNECT",
  "STATUS_SXS_RELEASE_ACTIVATION_CONTEXT",
  "STATUS_HEURISTIC_DAMAGE_POSSIBLE",
  "STATUS_RECOVERY_NOT_NEEDED",
  "STATUS_RM_ALREADY_STARTED",
  "STATUS_LOG_NO_RESTART",
  "STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST",
  "STATUS_GRAPHICS_PARTIAL_DATA_POPULATED",
  "STATUS_GRAPHICS_SKIP_ALLOCATION_PREPARATION",
  "STATUS_GRAPHICS_MODE_NOT_PINNED",
  "STATUS_GRAPHICS_NO_PREFERRED_MODE",
  "STATUS_GRAPHICS_DATASET_IS_EMPTY",
  "STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET",
  "STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED",
  "STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS",
  "STATUS_GRAPHICS_LEADLINK_START_DEFERRED",
  "STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY",
  "STATUS_GRAPHICS_START_DEFERRED",
  "STATUS_GRAPHICS_DEPENDABLE_CHILD_STATUS",
  "STATUS_NDIS_INDICATION_REQUIRED",
  "STATUS_PCP_UNSUPPORTED_PSS_SALT",
  "STATUS_GUARD_PAGE_VIOLATION",
  "STATUS_DATATYPE_MISALIGNMENT",
  "STATUS_BREAKPOINT",
  "STATUS_SINGLE_STEP",
  "STATUS_BUFFER_OVERFLOW",
  "STATUS_NO_MORE_FILES",
  "STATUS_WAKE_SYSTEM_DEBUGGER",
  "STATUS_HANDLES_CLOSED",
  "STATUS_NO_INHERITANCE",
  "STATUS_GUID_SUBSTITUTION_MADE",
  "STATUS_PARTIAL_COPY",
  "STATUS_DEVICE_PAPER_EMPTY",
  "STATUS_DEVICE_POWERED_OFF",
  "STATUS_DEVICE_OFF_LINE",
  "STATUS_DEVICE_BUSY",
  "STATUS_NO_MORE_EAS",
  "STATUS_INVALID_EA_NAME",
  "STATUS_EA_LIST_INCONSISTENT",
  "STATUS_INVALID_EA_FLAG",
  "STATUS_VERIFY_REQUIRED",
  "STATUS_EXTRANEOUS_INFORMATION",
  "STATUS_RXACT_COMMIT_NECESSARY",
  "STATUS_NO_MORE_ENTRIES",
  "STATUS_FILEMARK_DETECTED",
  "STATUS_MEDIA_CHANGED",
  "STATUS_BUS_RESET",
  "STATUS_END_OF_MEDIA",
  "STATUS_BEGINNING_OF_MEDIA",
  "STATUS_MEDIA_CHECK",
  "STATUS_SETMARK_DETECTED",
  "STATUS_NO_DATA_DETECTED",
  "STATUS_REDIRECTOR_HAS_OPEN_HANDLES",
  "STATUS_SERVER_HAS_OPEN_HANDLES",
  "STATUS_ALREADY_DISCONNECTED",
  "STATUS_LONGJUMP",
  "STATUS_CLEANER_CARTRIDGE_INSTALLED",
  "STATUS_PLUGPLAY_QUERY_VETOED",
  "STATUS_UNWIND_CONSOLIDATE",
  "STATUS_REGISTRY_HIVE_RECOVERED",
  "STATUS_DLL_MIGHT_BE_INSECURE",
  "STATUS_DLL_MIGHT_BE_INCOMPATIBLE",
  "STATUS_STOPPED_ON_SYMLINK",
  "STATUS_CANNOT_GRANT_REQUESTED_OPLOCK",
  "STATUS_NO_ACE_CONDITION",
  "STATUS_DEVICE_SUPPORT_IN_PROGRESS",
  "STATUS_DEVICE_POWER_CYCLE_REQUIRED",
  "STATUS_NO_WORK_DONE",
  "STATUS_RETURN_ADDRESS_HIJACK_ATTEMPT",
  "STATUS_DEVICE_REQUIRES_CLEANING",
  "STATUS_DEVICE_DOOR_OPEN",
  "STATUS_DATA_LOST_REPAIR",
  "STATUS_GPIO_INTERRUPT_ALREADY_UNMASKED",
  "STATUS_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH",
  "STATUS_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE",
  "STATUS_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS",
  "DBG_EXCEPTION_NOT_HANDLED",
  "STATUS_CLUSTER_NODE_ALREADY_UP",
  "STATUS_CLUSTER_NODE_ALREADY_DOWN",
  "STATUS_CLUSTER_NETWORK_ALREADY_ONLINE",
  "STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE",
  "STATUS_CLUSTER_NODE_ALREADY_MEMBER",
  "STATUS_COULD_NOT_RESIZE_LOG",
  "STATUS_NO_TXF_METADATA",
  "STATUS_CANT_RECOVER_WITH_HANDLE_OPEN",
  "STATUS_TXF_METADATA_ALREADY_PRESENT",
  "STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET",
  "STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED",
  "STATUS_FLT_BUFFER_TOO_SMALL",
  "STATUS_FVE_PARTIAL_METADATA",
  "STATUS_FVE_TRANSIENT_STATE",
  "STATUS_VID_REMOTE_NODE_PARENT_GPA_PAGES_USED",
  "STATUS_VOLMGR_INCOMPLETE_REGENERATION",
  "STATUS_VOLMGR_INCOMPLETE_DISK_MIGRATION",
  "STATUS_BCD_NOT_ALL_ENTRIES_IMPORTED",
  "STATUS_BCD_NOT_ALL_ENTRIES_SYNCHRONIZED",
  "STATUS_QUERY_STORAGE_ERROR",
  "STATUS_GDI_HANDLE_LEAK",
  "STATUS_SECUREBOOT_NOT_ENABLED",
  "STATUS_UNSUCCESSFUL",
  "STATUS_NOT_IMPLEMENTED",
  "STATUS_INVALID_INFO_CLASS",
  "STATUS_INFO_LENGTH_MISMATCH",
  "STATUS_ACCESS_VIOLATION",
  "STATUS_IN_PAGE_ERROR",
  "STATUS_PAGEFILE_QUOTA",
  "STATUS_INVALID_HANDLE",
  "STATUS_BAD_INITIAL_STACK",
  "STATUS_BAD_INITIAL_PC",
  "STATUS_INVALID_CID",
  "STATUS_TIMER_NOT_CANCELED",
  "STATUS_INVALID_PARAMETER",
  "STATUS_NO_SUCH_DEVICE",
  "STATUS_NO_SUCH_FILE",
  "STATUS_INVALID_DEVICE_REQUEST",
  "STATUS_END_OF_FILE",
  "STATUS_WRONG_VOLUME",
  "STATUS_NO_MEDIA_IN_DEVICE",
  "STATUS_UNRECOGNIZED_MEDIA",
  "STATUS_NONEXISTENT_SECTOR",
  "STATUS_MORE_PROCESSING_REQUIRED",
  "STATUS_NO_MEMORY",
  "STATUS_CONFLICTING_ADDRESSES",
  "STATUS_NOT_MAPPED_VIEW",
  "STATUS_UNABLE_TO_FREE_VM",
  "STATUS_UNABLE_TO_DELETE_SECTION",
  "STATUS_INVALID_SYSTEM_SERVICE",
  "STATUS_ILLEGAL_INSTRUCTION",
  "STATUS_INVALID_LOCK_SEQUENCE",
  "STATUS_INVALID_VIEW_SIZE",
  "STATUS_INVALID_FILE_FOR_SECTION",
  "STATUS_ALREADY_COMMITTED",
  "STATUS_ACCESS_DENIED",
  "STATUS_BUFFER_TOO_SMALL",
  "STATUS_OBJECT_TYPE_MISMATCH",
  "STATUS_NONCONTINUABLE_EXCEPTION",
  "STATUS_INVALID_DISPOSITION",
  "STATUS_UNWIND",
  "STATUS_BAD_STACK",
  "STATUS_INVALID_UNWIND_TARGET",
  "STATUS_NOT_LOCKED",
  "STATUS_PARITY_ERROR",
  "STATUS_UNABLE_TO_DECOMMIT_VM",
  "STATUS_NOT_COMMITTED",
  "STATUS_INVALID_PORT_ATTRIBUTES",
  "STATUS_PORT_MESSAGE_TOO_LONG",
  "STATUS_INVALID_PARAMETER_MIX",
  "STATUS_INVALID_QUOTA_LOWER",
  "STATUS_DISK_CORRUPT_ERROR",
  "STATUS_OBJECT_NAME_INVALID",
  "STATUS_OBJECT_NAME_NOT_FOUND",
  "STATUS_OBJECT_NAME_COLLISION",
  "STATUS_PORT_DO_NOT_DISTURB",
  "STATUS_PORT_DISCONNECTED",
  "STATUS_DEVICE_ALREADY_ATTACHED",
  "STATUS_OBJECT_PATH_INVALID",
  "STATUS_OBJECT_PATH_NOT_FOUND",
  "STATUS_OBJECT_PATH_SYNTAX_BAD",
  "STATUS_DATA_OVERRUN",
  "STATUS_DATA_LATE_ERROR",
  "STATUS_DATA_ERROR",
  "STATUS_CRC_ERROR",
  "STATUS_SECTION_TOO_BIG",
  "STATUS_PORT_CONNECTION_REFUSED",
  "STATUS_INVALID_PORT_HANDLE",
  "STATUS_SHARING_VIOLATION",
  "STATUS_QUOTA_EXCEEDED",
  "STATUS_INVALID_PAGE_PROTECTION",
  "STATUS_MUTANT_NOT_OWNED",
  "STATUS_SEMAPHORE_LIMIT_EXCEEDED",
  "STATUS_PORT_ALREADY_SET",
  "STATUS_SECTION_NOT_IMAGE",
  "STATUS_SUSPEND_COUNT_EXCEEDED",
  "STATUS_THREAD_IS_TERMINATING",
  "STATUS_BAD_WORKING_SET_LIMIT",
  "STATUS_INCOMPATIBLE_FILE_MAP",
  "STATUS_SECTION_PROTECTION",
  "STATUS_EAS_NOT_SUPPORTED",
  "STATUS_EA_TOO_LARGE",
  "STATUS_NONEXISTENT_EA_ENTRY",
  "STATUS_NO_EAS_ON_FILE",
  "STATUS_EA_CORRUPT_ERROR",
  "STATUS_FILE_LOCK_CONFLICT",
  "STATUS_LOCK_NOT_GRANTED",
  "STATUS_DELETE_PENDING",
  "STATUS_CTL_FILE_NOT_SUPPORTED",
  "STATUS_UNKNOWN_REVISION",
  "STATUS_REVISION_MISMATCH",
  "STATUS_INVALID_OWNER",
  "STATUS_INVALID_PRIMARY_GROUP",
  "STATUS_NO_IMPERSONATION_TOKEN",
  "STATUS_CANT_DISABLE_MANDATORY",
  "STATUS_NO_LOGON_SERVERS",
  "STATUS_NO_SUCH_LOGON_SESSION",
  "STATUS_NO_SUCH_PRIVILEGE",
  "STATUS_PRIVILEGE_NOT_HELD",
  "STATUS_INVALID_ACCOUNT_NAME",
  "STATUS_USER_EXISTS",
  "STATUS_NO_SUCH_USER",
  "STATUS_GROUP_EXISTS",
  "STATUS_NO_SUCH_GROUP",
  "STATUS_MEMBER_IN_GROUP",
  "STATUS_MEMBER_NOT_IN_GROUP",
  "STATUS_LAST_ADMIN",
  "STATUS_WRONG_PASSWORD",
  "STATUS_ILL_FORMED_PASSWORD",
  "STATUS_PASSWORD_RESTRICTION",
  "STATUS_LOGON_FAILURE",
  "STATUS_ACCOUNT_RESTRICTION",
  "STATUS_INVALID_LOGON_HOURS",
  "STATUS_INVALID_WORKSTATION",
  "STATUS_PASSWORD_EXPIRED",
  "STATUS_ACCOUNT_DISABLED",
  "STATUS_NONE_MAPPED",
  "STATUS_TOO_MANY_LUIDS_REQUESTED",
  "STATUS_LUIDS_EXHAUSTED",
  "STATUS_INVALID_SUB_AUTHORITY",
  "STATUS_INVALID_ACL",
  "STATUS_INVALID_SID",
  "STATUS_INVALID_SECURITY_DESCR",
  "STATUS_PROCEDURE_NOT_FOUND",
  "STATUS_INVALID_IMAGE_FORMAT",
  "STATUS_NO_TOKEN",
  "STATUS_BAD_INHERITANCE_ACL",
  "STATUS_RANGE_NOT_LOCKED",
  "STATUS_DISK_FULL",
  "STATUS_SERVER_DISABLED",
  "STATUS_SERVER_NOT_DISABLED",
  "STATUS_TOO_MANY_GUIDS_REQUESTED",
  "STATUS_GUIDS_EXHAUSTED",
  "STATUS_INVALID_ID_AUTHORITY",
  "STATUS_AGENTS_EXHAUSTED",
  "STATUS_INVALID_VOLUME_LABEL",
  "STATUS_SECTION_NOT_EXTENDED",
  "STATUS_NOT_MAPPED_DATA",
  "STATUS_RESOURCE_DATA_NOT_FOUND",
  "STATUS_RESOURCE_TYPE_NOT_FOUND",
  "STATUS_RESOURCE_NAME_NOT_FOUND",
  "STATUS_ARRAY_BOUNDS_EXCEEDED",
  "STATUS_FLOAT_DENORMAL_OPERAND",
  "STATUS_FLOAT_DIVIDE_BY_ZERO",
  "STATUS_FLOAT_INEXACT_RESULT",
  "STATUS_FLOAT_INVALID_OPERATION",
  "STATUS_FLOAT_OVERFLOW",
  "STATUS_FLOAT_STACK_CHECK",
  "STATUS_FLOAT_UNDERFLOW",
  "STATUS_INTEGER_DIVIDE_BY_ZERO",
  "STATUS_INTEGER_OVERFLOW",
  "STATUS_PRIVILEGED_INSTRUCTION",
  "STATUS_TOO_MANY_PAGING_FILES",
  "STATUS_FILE_INVALID",
  "STATUS_ALLOTTED_SPACE_EXCEEDED",
  "STATUS_INSUFFICIENT_RESOURCES",
  "STATUS_DFS_EXIT_PATH_FOUND",
  "STATUS_DEVICE_DATA_ERROR",
  "STATUS_DEVICE_NOT_CONNECTED",
  "STATUS_DEVICE_POWER_FAILURE",
  "STATUS_FREE_VM_NOT_AT_BASE",
  "STATUS_MEMORY_NOT_ALLOCATED",
  "STATUS_WORKING_SET_QUOTA",
  "STATUS_MEDIA_WRITE_PROTECTED",
  "STATUS_DEVICE_NOT_READY",
  "STATUS_INVALID_GROUP_ATTRIBUTES",
  "STATUS_BAD_IMPERSONATION_LEVEL",
  "STATUS_CANT_OPEN_ANONYMOUS",
  "STATUS_BAD_VALIDATION_CLASS",
  "STATUS_BAD_TOKEN_TYPE",
  "STATUS_BAD_MASTER_BOOT_RECORD",
  "STATUS_INSTRUCTION_MISALIGNMENT",
  "STATUS_INSTANCE_NOT_AVAILABLE",
  "STATUS_PIPE_NOT_AVAILABLE",
  "STATUS_INVALID_PIPE_STATE",
  "STATUS_PIPE_BUSY",
  "STATUS_ILLEGAL_FUNCTION",
  "STATUS_PIPE_DISCONNECTED",
  "STATUS_PIPE_CLOSING",
  "STATUS_PIPE_CONNECTED",
  "STATUS_PIPE_LISTENING",
  "STATUS_INVALID_READ_MODE",
  "STATUS_IO_TIMEOUT",
  "STATUS_FILE_FORCED_CLOSED",
  "STATUS_PROFILING_NOT_STARTED",
  "STATUS_PROFILING_NOT_STOPPED",
  "STATUS_COULD_NOT_INTERPRET",
  "STATUS_FILE_IS_A_DIRECTORY",
  "STATUS_NOT_SUPPORTED",
  "STATUS_REMOTE_NOT_LISTENING",
  "STATUS_DUPLICATE_NAME",
  "STATUS_BAD_NETWORK_PATH",
  "STATUS_NETWORK_BUSY",
  "STATUS_DEVICE_DOES_NOT_EXIST",
  "STATUS_TOO_MANY_COMMANDS",
  "STATUS_ADAPTER_HARDWARE_ERROR",
  "STATUS_INVALID_NETWORK_RESPONSE",
  "STATUS_UNEXPECTED_NETWORK_ERROR",
  "STATUS_BAD_REMOTE_ADAPTER",
  "STATUS_PRINT_QUEUE_FULL",
  "STATUS_NO_SPOOL_SPACE",
  "STATUS_PRINT_CANCELLED",
  "STATUS_NETWORK_NAME_DELETED",
  "STATUS_NETWORK_ACCESS_DENIED",
  "STATUS_BAD_DEVICE_TYPE",
  "STATUS_BAD_NETWORK_NAME",
  "STATUS_TOO_MANY_NAMES",
  "STATUS_TOO_MANY_SESSIONS",
  "STATUS_SHARING_PAUSED",
  "STATUS_REQUEST_NOT_ACCEPTED",
  "STATUS_REDIRECTOR_PAUSED",
  "STATUS_NET_WRITE_FAULT",
  "STATUS_PROFILING_AT_LIMIT",
  "STATUS_NOT_SAME_DEVICE",
  "STATUS_FILE_RENAMED",
  "STATUS_VIRTUAL_CIRCUIT_CLOSED",
  "STATUS_NO_SECURITY_ON_OBJECT",
  "STATUS_CANT_WAIT",
  "STATUS_PIPE_EMPTY",
  "STATUS_CANT_ACCESS_DOMAIN_INFO",
  "STATUS_CANT_TERMINATE_SELF",
  "STATUS_INVALID_SERVER_STATE",
  "STATUS_INVALID_DOMAIN_STATE",
  "STATUS_INVALID_DOMAIN_ROLE",
  "STATUS_NO_SUCH_DOMAIN",
  "STATUS_DOMAIN_EXISTS",
  "STATUS_DOMAIN_LIMIT_EXCEEDED",
  "STATUS_OPLOCK_NOT_GRANTED",
  "STATUS_INVALID_OPLOCK_PROTOCOL",
  "STATUS_INTERNAL_DB_CORRUPTION",
  "STATUS_INTERNAL_ERROR",
  "STATUS_GENERIC_NOT_MAPPED",
  "STATUS_BAD_DESCRIPTOR_FORMAT",
  "STATUS_INVALID_USER_BUFFER",
  "STATUS_UNEXPECTED_IO_ERROR",
  "STATUS_UNEXPECTED_MM_CREATE_ERR",
  "STATUS_UNEXPECTED_MM_MAP_ERROR",
  "STATUS_UNEXPECTED_MM_EXTEND_ERR",
  "STATUS_NOT_LOGON_PROCESS",
  "STATUS_LOGON_SESSION_EXISTS",
  "STATUS_INVALID_PARAMETER_1",
  "STATUS_INVALID_PARAMETER_2",
  "STATUS_INVALID_PARAMETER_3",
  "STATUS_INVALID_PARAMETER_4",
  "STATUS_INVALID_PARAMETER_5",
  "STATUS_INVALID_PARAMETER_6",
  "STATUS_INVALID_PARAMETER_7",
  "STATUS_INVALID_PARAMETER_8",
  "STATUS_INVALID_PARAMETER_9",
  "STATUS_INVALID_PARAMETER_10",
  "STATUS_INVALID_PARAMETER_11",
  "STATUS_INVALID_PARAMETER_12",
  "STATUS_REDIRECTOR_NOT_STARTED",
  "STATUS_REDIRECTOR_STARTED",
  "STATUS_STACK_OVERFLOW",
  "STATUS_NO_SUCH_PACKAGE",
  "STATUS_BAD_FUNCTION_TABLE",
  "STATUS_VARIABLE_NOT_FOUND",
  "STATUS_DIRECTORY_NOT_EMPTY",
  "STATUS_FILE_CORRUPT_ERROR",
  "STATUS_NOT_A_DIRECTORY",
  "STATUS_BAD_LOGON_SESSION_STATE",
  "STATUS_LOGON_SESSION_COLLISION",
  "STATUS_NAME_TOO_LONG",
  "STATUS_FILES_OPEN",
  "STATUS_CONNECTION_IN_USE",
  "STATUS_MESSAGE_NOT_FOUND",
  "STATUS_PROCESS_IS_TERMINATING",
  "STATUS_INVALID_LOGON_TYPE",
  "STATUS_NO_GUID_TRANSLATION",
  "STATUS_CANNOT_IMPERSONATE",
  "STATUS_IMAGE_ALREADY_LOADED",
  "STATUS_ABIOS_NOT_PRESENT",
  "STATUS_ABIOS_LID_NOT_EXIST",
  "STATUS_ABIOS_LID_ALREADY_OWNED",
  "STATUS_ABIOS_NOT_LID_OWNER",
  "STATUS_ABIOS_INVALID_COMMAND",
  "STATUS_ABIOS_INVALID_LID",
  "STATUS_ABIOS_SELECTOR_NOT_AVAILABLE",
  "STATUS_ABIOS_INVALID_SELECTOR",
  "STATUS_NO_LDT",
  "STATUS_INVALID_LDT_SIZE",
  "STATUS_INVALID_LDT_OFFSET",
  "STATUS_INVALID_LDT_DESCRIPTOR",
  "STATUS_INVALID_IMAGE_NE_FORMAT",
  "STATUS_RXACT_INVALID_STATE",
  "STATUS_RXACT_COMMIT_FAILURE",
  "STATUS_MAPPED_FILE_SIZE_ZERO",
  "STATUS_TOO_MANY_OPENED_FILES",
  "STATUS_CANCELLED",
  "STATUS_CANNOT_DELETE",
  "STATUS_INVALID_COMPUTER_NAME",
  "STATUS_FILE_DELETED",
  "STATUS_SPECIAL_ACCOUNT",
  "STATUS_SPECIAL_GROUP",
  "STATUS_SPECIAL_USER",
  "STATUS_MEMBERS_PRIMARY_GROUP",
  "STATUS_FILE_CLOSED",
  "STATUS_TOO_MANY_THREADS",
  "STATUS_THREAD_NOT_IN_PROCESS",
  "STATUS_TOKEN_ALREADY_IN_USE",
  "STATUS_PAGEFILE_QUOTA_EXCEEDED",
  "STATUS_COMMITMENT_LIMIT",
  "STATUS_INVALID_IMAGE_LE_FORMAT",
  "STATUS_INVALID_IMAGE_NOT_MZ",
  "STATUS_INVALID_IMAGE_PROTECT",
  "STATUS_INVALID_IMAGE_WIN_16",
  "STATUS_LOGON_SERVER_CONFLICT",
  "STATUS_TIME_DIFFERENCE_AT_DC",
  "STATUS_SYNCHRONIZATION_REQUIRED",
  "STATUS_DLL_NOT_FOUND",
  "STATUS_OPEN_FAILED",
  "STATUS_IO_PRIVILEGE_FAILED",
  "STATUS_ORDINAL_NOT_FOUND",
  "STATUS_ENTRYPOINT_NOT_FOUND",
  "STATUS_CONTROL_C_EXIT",
  "STATUS_LOCAL_DISCONNECT",
  "STATUS_REMOTE_DISCONNECT",
  "STATUS_REMOTE_RESOURCES",
  "STATUS_LINK_FAILED",
  "STATUS_LINK_TIMEOUT",
  "STATUS_INVALID_CONNECTION",
  "STATUS_INVALID_ADDRESS",
  "STATUS_DLL_INIT_FAILED",
  "STATUS_MISSING_SYSTEMFILE",
  "STATUS_UNHANDLED_EXCEPTION",
  "STATUS_APP_INIT_FAILURE",
  "STATUS_PAGEFILE_CREATE_FAILED",
  "STATUS_NO_PAGEFILE",
  "STATUS_INVALID_LEVEL",
  "STATUS_WRONG_PASSWORD_CORE",
  "STATUS_ILLEGAL_FLOAT_CONTEXT",
  "STATUS_PIPE_BROKEN",
  "STATUS_REGISTRY_CORRUPT",
  "STATUS_REGISTRY_IO_FAILED",
  "STATUS_NO_EVENT_PAIR",
  "STATUS_UNRECOGNIZED_VOLUME",
  "STATUS_SERIAL_NO_DEVICE_INITED",
  "STATUS_NO_SUCH_ALIAS",
  "STATUS_MEMBER_NOT_IN_ALIAS",
  "STATUS_MEMBER_IN_ALIAS",
  "STATUS_ALIAS_EXISTS",
  "STATUS_LOGON_NOT_GRANTED",
  "STATUS_TOO_MANY_SECRETS",
  "STATUS_SECRET_TOO_LONG",
  "STATUS_INTERNAL_DB_ERROR",
  "STATUS_FULLSCREEN_MODE",
  "STATUS_TOO_MANY_CONTEXT_IDS",
  "STATUS_LOGON_TYPE_NOT_GRANTED",
  "STATUS_NOT_REGISTRY_FILE",
  "STATUS_NT_CROSS_ENCRYPTION_REQUIRED",
  "STATUS_DOMAIN_CTRLR_CONFIG_ERROR",
  "STATUS_FT_MISSING_MEMBER",
  "STATUS_ILL_FORMED_SERVICE_ENTRY",
  "STATUS_ILLEGAL_CHARACTER",
  "STATUS_UNMAPPABLE_CHARACTER",
  "STATUS_UNDEFINED_CHARACTER",
  "STATUS_FLOPPY_VOLUME",
  "STATUS_FLOPPY_ID_MARK_NOT_FOUND",
  "STATUS_FLOPPY_WRONG_CYLINDER",
  "STATUS_FLOPPY_UNKNOWN_ERROR",
  "STATUS_FLOPPY_BAD_REGISTERS",
  "STATUS_DISK_RECALIBRATE_FAILED",
  "STATUS_DISK_OPERATION_FAILED",
  "STATUS_DISK_RESET_FAILED",
  "STATUS_SHARED_IRQ_BUSY",
  "STATUS_FT_ORPHANING",
  "STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT",
  "STATUS_PARTITION_FAILURE",
  "STATUS_INVALID_BLOCK_LENGTH",
  "STATUS_DEVICE_NOT_PARTITIONED",
  "STATUS_UNABLE_TO_LOCK_MEDIA",
  "STATUS_UNABLE_TO_UNLOAD_MEDIA",
  "STATUS_EOM_OVERFLOW",
  "STATUS_NO_MEDIA",
  "STATUS_NO_SUCH_MEMBER",
  "STATUS_INVALID_MEMBER",
  "STATUS_KEY_DELETED",
  "STATUS_NO_LOG_SPACE",
  "STATUS_TOO_MANY_SIDS",
  "STATUS_LM_CROSS_ENCRYPTION_REQUIRED",
  "STATUS_KEY_HAS_CHILDREN",
  "STATUS_CHILD_MUST_BE_VOLATILE",
  "STATUS_DEVICE_CONFIGURATION_ERROR",
  "STATUS_DRIVER_INTERNAL_ERROR",
  "STATUS_INVALID_DEVICE_STATE",
  "STATUS_IO_DEVICE_ERROR",
  "STATUS_DEVICE_PROTOCOL_ERROR",
  "STATUS_BACKUP_CONTROLLER",
  "STATUS_LOG_FILE_FULL",
  "STATUS_TOO_LATE",
  "STATUS_NO_TRUST_LSA_SECRET",
  "STATUS_NO_TRUST_SAM_ACCOUNT",
  "STATUS_TRUSTED_DOMAIN_FAILURE",
  "STATUS_TRUSTED_RELATIONSHIP_FAILURE",
  "STATUS_EVENTLOG_FILE_CORRUPT",
  "STATUS_EVENTLOG_CANT_START",
  "STATUS_TRUST_FAILURE",
  "STATUS_MUTANT_LIMIT_EXCEEDED",
  "STATUS_NETLOGON_NOT_STARTED",
  "STATUS_ACCOUNT_EXPIRED",
  "STATUS_POSSIBLE_DEADLOCK",
  "STATUS_NETWORK_CREDENTIAL_CONFLICT",
  "STATUS_REMOTE_SESSION_LIMIT",
  "STATUS_EVENTLOG_FILE_CHANGED",
  "STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT",
  "STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT",
  "STATUS_NOLOGON_SERVER_TRUST_ACCOUNT",
  "STATUS_DOMAIN_TRUST_INCONSISTENT",
  "STATUS_FS_DRIVER_REQUIRED",
  "STATUS_IMAGE_ALREADY_LOADED_AS_DLL",
  "STATUS_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING",
  "STATUS_SHORT_NAMES_NOT_ENABLED_ON_VOLUME",
  "STATUS_SECURITY_STREAM_IS_INCONSISTENT",
  "STATUS_INVALID_LOCK_RANGE",
  "STATUS_INVALID_ACE_CONDITION",
  "STATUS_IMAGE_SUBSYSTEM_NOT_PRESENT",
  "STATUS_NOTIFICATION_GUID_ALREADY_DEFINED",
  "STATUS_INVALID_EXCEPTION_HANDLER",
  "STATUS_DUPLICATE_PRIVILEGES",
  "STATUS_NOT_ALLOWED_ON_SYSTEM_FILE",
  "STATUS_REPAIR_NEEDED",
  "STATUS_QUOTA_NOT_ENABLED",
  "STATUS_NO_APPLICATION_PACKAGE",
  "STATUS_FILE_METADATA_OPTIMIZATION_IN_PROGRESS",
  "STATUS_NOT_SAME_OBJECT",
  "STATUS_FATAL_MEMORY_EXHAUSTION",
  "STATUS_ERROR_PROCESS_NOT_IN_JOB",
  "STATUS_CPU_SET_INVALID",
  "STATUS_IO_DEVICE_INVALID_DATA",
  "STATUS_IO_UNALIGNED_WRITE",
  "STATUS_CONTROL_STACK_VIOLATION",
  "STATUS_NETWORK_OPEN_RESTRICTION",
  "STATUS_NO_USER_SESSION_KEY",
  "STATUS_USER_SESSION_DELETED",
  "STATUS_RESOURCE_LANG_NOT_FOUND",
  "STATUS_INSUFF_SERVER_RESOURCES",
  "STATUS_INVALID_BUFFER_SIZE",
  "STATUS_INVALID_ADDRESS_COMPONENT",
  "STATUS_INVALID_ADDRESS_WILDCARD",
  "STATUS_TOO_MANY_ADDRESSES",
  "STATUS_ADDRESS_ALREADY_EXISTS",
  "STATUS_ADDRESS_CLOSED",
  "STATUS_CONNECTION_DISCONNECTED",
  "STATUS_CONNECTION_RESET",
  "STATUS_TOO_MANY_NODES",
  "STATUS_TRANSACTION_ABORTED",
  "STATUS_TRANSACTION_TIMED_OUT",
  "STATUS_TRANSACTION_NO_RELEASE",
  "STATUS_TRANSACTION_NO_MATCH",
  "STATUS_TRANSACTION_RESPONDED",
  "STATUS_TRANSACTION_INVALID_ID",
  "STATUS_TRANSACTION_INVALID_TYPE",
  "STATUS_NOT_SERVER_SESSION",
  "STATUS_NOT_CLIENT_SESSION",
  "STATUS_CANNOT_LOAD_REGISTRY_FILE",
  "STATUS_DEBUG_ATTACH_FAILED",
  "STATUS_SYSTEM_PROCESS_TERMINATED",
  "STATUS_DATA_NOT_ACCEPTED",
  "STATUS_NO_BROWSER_SERVERS_FOUND",
  "STATUS_VDM_HARD_ERROR",
  "STATUS_DRIVER_CANCEL_TIMEOUT",
  "STATUS_REPLY_MESSAGE_MISMATCH",
  "STATUS_MAPPED_ALIGNMENT",
  "STATUS_IMAGE_CHECKSUM_MISMATCH",
  "STATUS_LOST_WRITEBEHIND_DATA",
  "STATUS_CLIENT_SERVER_PARAMETERS_INVALID",
  "STATUS_PASSWORD_MUST_CHANGE",
  "STATUS_NOT_FOUND",
  "STATUS_NOT_TINY_STREAM",
  "STATUS_RECOVERY_FAILURE",
  "STATUS_STACK_OVERFLOW_READ",
  "STATUS_FAIL_CHECK",
  "STATUS_DUPLICATE_OBJECTID",
  "STATUS_OBJECTID_EXISTS",
  "STATUS_CONVERT_TO_LARGE",
  "STATUS_RETRY",
  "STATUS_FOUND_OUT_OF_SCOPE",
  "STATUS_ALLOCATE_BUCKET",
  "STATUS_PROPSET_NOT_FOUND",
  "STATUS_MARSHALL_OVERFLOW",
  "STATUS_INVALID_VARIANT",
  "STATUS_DOMAIN_CONTROLLER_NOT_FOUND",
  "STATUS_ACCOUNT_LOCKED_OUT",
  "STATUS_HANDLE_NOT_CLOSABLE",
  "STATUS_CONNECTION_REFUSED",
  "STATUS_GRACEFUL_DISCONNECT",
  "STATUS_ADDRESS_ALREADY_ASSOCIATED",
  "STATUS_ADDRESS_NOT_ASSOCIATED",
  "STATUS_CONNECTION_INVALID",
  "STATUS_CONNECTION_ACTIVE",
  "STATUS_NETWORK_UNREACHABLE",
  "STATUS_HOST_UNREACHABLE",
  "STATUS_PROTOCOL_UNREACHABLE",
  "STATUS_PORT_UNREACHABLE",
  "STATUS_REQUEST_ABORTED",
  "STATUS_CONNECTION_ABORTED",
  "STATUS_BAD_COMPRESSION_BUFFER",
  "STATUS_USER_MAPPED_FILE",
  "STATUS_AUDIT_FAILED",
  "STATUS_TIMER_RESOLUTION_NOT_SET",
  "STATUS_CONNECTION_COUNT_LIMIT",
  "STATUS_LOGIN_TIME_RESTRICTION",
  "STATUS_LOGIN_WKSTA_RESTRICTION",
  "STATUS_IMAGE_MP_UP_MISMATCH",
  "STATUS_INSUFFICIENT_LOGON_INFO",
  "STATUS_BAD_DLL_ENTRYPOINT",
  "STATUS_BAD_SERVICE_ENTRYPOINT",
  "STATUS_LPC_REPLY_LOST",
  "STATUS_IP_ADDRESS_CONFLICT1",
  "STATUS_IP_ADDRESS_CONFLICT2",
  "STATUS_REGISTRY_QUOTA_LIMIT",
  "STATUS_PATH_NOT_COVERED",
  "STATUS_NO_CALLBACK_ACTIVE",
  "STATUS_LICENSE_QUOTA_EXCEEDED",
  "STATUS_PWD_TOO_SHORT",
  "STATUS_PWD_TOO_RECENT",
  "STATUS_PWD_HISTORY_CONFLICT",
  "STATUS_PLUGPLAY_NO_DEVICE",
  "STATUS_UNSUPPORTED_COMPRESSION",
  "STATUS_INVALID_HW_PROFILE",
  "STATUS_INVALID_PLUGPLAY_DEVICE_PATH",
  "STATUS_DRIVER_ORDINAL_NOT_FOUND",
  "STATUS_DRIVER_ENTRYPOINT_NOT_FOUND",
  "STATUS_RESOURCE_NOT_OWNED",
  "STATUS_TOO_MANY_LINKS",
  "STATUS_QUOTA_LIST_INCONSISTENT",
  "STATUS_FILE_IS_OFFLINE",
  "STATUS_EVALUATION_EXPIRATION",
  "STATUS_ILLEGAL_DLL_RELOCATION",
  "STATUS_LICENSE_VIOLATION",
  "STATUS_DLL_INIT_FAILED_LOGOFF",
  "STATUS_DRIVER_UNABLE_TO_LOAD",
  "STATUS_DFS_UNAVAILABLE",
  "STATUS_VOLUME_DISMOUNTED",
  "STATUS_WX86_INTERNAL_ERROR",
  "STATUS_WX86_FLOAT_STACK_CHECK",
  "STATUS_VALIDATE_CONTINUE",
  "STATUS_NO_MATCH",
  "STATUS_NO_MORE_MATCHES",
  "STATUS_NOT_A_REPARSE_POINT",
  "STATUS_IO_REPARSE_TAG_INVALID",
  "STATUS_IO_REPARSE_TAG_MISMATCH",
  "STATUS_IO_REPARSE_DATA_INVALID",
  "STATUS_IO_REPARSE_TAG_NOT_HANDLED",
  "STATUS_PWD_TOO_LONG",
  "STATUS_STOWED_EXCEPTION",
  "STATUS_CONTEXT_STOWED_EXCEPTION",
  "STATUS_REPARSE_POINT_NOT_RESOLVED",
  "STATUS_DIRECTORY_IS_A_REPARSE_POINT",
  "STATUS_RANGE_LIST_CONFLICT",
  "STATUS_SOURCE_ELEMENT_EMPTY",
  "STATUS_DESTINATION_ELEMENT_FULL",
  "STATUS_ILLEGAL_ELEMENT_ADDRESS",
  "STATUS_MAGAZINE_NOT_PRESENT",
  "STATUS_REINITIALIZATION_NEEDED",
  "STATUS_ENCRYPTION_FAILED",
  "STATUS_DECRYPTION_FAILED",
  "STATUS_RANGE_NOT_FOUND",
  "STATUS_NO_RECOVERY_POLICY",
  "STATUS_NO_EFS",
  "STATUS_WRONG_EFS",
  "STATUS_NO_USER_KEYS",
  "STATUS_FILE_NOT_ENCRYPTED",
  "STATUS_NOT_EXPORT_FORMAT",
  "STATUS_FILE_ENCRYPTED",
  "STATUS_WMI_GUID_NOT_FOUND",
  "STATUS_WMI_INSTANCE_NOT_FOUND",
  "STATUS_WMI_ITEMID_NOT_FOUND",
  "STATUS_WMI_TRY_AGAIN",
  "STATUS_SHARED_POLICY",
  "STATUS_POLICY_OBJECT_NOT_FOUND",
  "STATUS_POLICY_ONLY_IN_DS",
  "STATUS_VOLUME_NOT_UPGRADED",
  "STATUS_REMOTE_STORAGE_NOT_ACTIVE",
  "STATUS_REMOTE_STORAGE_MEDIA_ERROR",
  "STATUS_NO_TRACKING_SERVICE",
  "STATUS_SERVER_SID_MISMATCH",
  "STATUS_DS_NO_ATTRIBUTE_OR_VALUE",
  "STATUS_DS_INVALID_ATTRIBUTE_SYNTAX",
  "STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED",
  "STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS",
  "STATUS_DS_BUSY",
  "STATUS_DS_UNAVAILABLE",
  "STATUS_DS_NO_RIDS_ALLOCATED",
  "STATUS_DS_NO_MORE_RIDS",
  "STATUS_DS_INCORRECT_ROLE_OWNER",
  "STATUS_DS_RIDMGR_INIT_ERROR",
  "STATUS_DS_OBJ_CLASS_VIOLATION",
  "STATUS_DS_CANT_ON_NON_LEAF",
  "STATUS_DS_CANT_ON_RDN",
  "STATUS_DS_CANT_MOD_OBJ_CLASS",
  "STATUS_DS_CROSS_DOM_MOVE_FAILED",
  "STATUS_DS_GC_NOT_AVAILABLE",
  "STATUS_DIRECTORY_SERVICE_REQUIRED",
  "STATUS_REPARSE_ATTRIBUTE_CONFLICT",
  "STATUS_CANT_ENABLE_DENY_ONLY",
  "STATUS_FLOAT_MULTIPLE_FAULTS",
  "STATUS_FLOAT_MULTIPLE_TRAPS",
  "STATUS_DEVICE_REMOVED",
  "STATUS_JOURNAL_DELETE_IN_PROGRESS",
  "STATUS_JOURNAL_NOT_ACTIVE",
  "STATUS_NOINTERFACE",
  "STATUS_DS_RIDMGR_DISABLED",
  "STATUS_DS_ADMIN_LIMIT_EXCEEDED",
  "STATUS_DRIVER_FAILED_SLEEP",
  "STATUS_MUTUAL_AUTHENTICATION_FAILED",
  "STATUS_CORRUPT_SYSTEM_FILE",
  "STATUS_DATATYPE_MISALIGNMENT_ERROR",
  "STATUS_WMI_READ_ONLY",
  "STATUS_WMI_SET_FAILURE",
  "STATUS_COMMITMENT_MINIMUM",
  "STATUS_REG_NAT_CONSUMPTION",
  "STATUS_TRANSPORT_FULL",
  "STATUS_DS_SAM_INIT_FAILURE",
  "STATUS_ONLY_IF_CONNECTED",
  "STATUS_DS_SENSITIVE_GROUP_VIOLATION",
  "STATUS_PNP_RESTART_ENUMERATION",
  "STATUS_JOURNAL_ENTRY_DELETED",
  "STATUS_DS_CANT_MOD_PRIMARYGROUPID",
  "STATUS_SYSTEM_IMAGE_BAD_SIGNATURE",
  "STATUS_PNP_REBOOT_REQUIRED",
  "STATUS_POWER_STATE_INVALID",
  "STATUS_DS_INVALID_GROUP_TYPE",
  "STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN",
  "STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN",
  "STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER",
  "STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER",
  "STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER",
  "STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER",
  "STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER",
  "STATUS_DS_HAVE_PRIMARY_MEMBERS",
  "STATUS_WMI_NOT_SUPPORTED",
  "STATUS_INSUFFICIENT_POWER",
  "STATUS_SAM_NEED_BOOTKEY_PASSWORD",
  "STATUS_SAM_NEED_BOOTKEY_FLOPPY",
  "STATUS_DS_CANT_START",
  "STATUS_DS_INIT_FAILURE",
  "STATUS_SAM_INIT_FAILURE",
  "STATUS_DS_GC_REQUIRED",
  "STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY",
  "STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS",
  "STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED",
  "STATUS_MULTIPLE_FAULT_VIOLATION",
  "STATUS_CURRENT_DOMAIN_NOT_ALLOWED",
  "STATUS_CANNOT_MAKE",
  "STATUS_SYSTEM_SHUTDOWN",
  "STATUS_DS_INIT_FAILURE_CONSOLE",
  "STATUS_DS_SAM_INIT_FAILURE_CONSOLE",
  "STATUS_UNFINISHED_CONTEXT_DELETED",
  "STATUS_NO_TGT_REPLY",
  "STATUS_OBJECTID_NOT_FOUND",
  "STATUS_NO_IP_ADDRESSES",
  "STATUS_WRONG_CREDENTIAL_HANDLE",
  "STATUS_CRYPTO_SYSTEM_INVALID",
  "STATUS_MAX_REFERRALS_EXCEEDED",
  "STATUS_MUST_BE_KDC",
  "STATUS_STRONG_CRYPTO_NOT_SUPPORTED",
  "STATUS_TOO_MANY_PRINCIPALS",
  "STATUS_NO_PA_DATA",
  "STATUS_PKINIT_NAME_MISMATCH",
  "STATUS_SMARTCARD_LOGON_REQUIRED",
  "STATUS_KDC_INVALID_REQUEST",
  "STATUS_KDC_UNABLE_TO_REFER",
  "STATUS_KDC_UNKNOWN_ETYPE",
  "STATUS_SHUTDOWN_IN_PROGRESS",
  "STATUS_SERVER_SHUTDOWN_IN_PROGRESS",
  "STATUS_NOT_SUPPORTED_ON_SBS",
  "STATUS_WMI_GUID_DISCONNECTED",
  "STATUS_WMI_ALREADY_DISABLED",
  "STATUS_WMI_ALREADY_ENABLED",
  "STATUS_MFT_TOO_FRAGMENTED",
  "STATUS_COPY_PROTECTION_FAILURE",
  "STATUS_CSS_AUTHENTICATION_FAILURE",
  "STATUS_CSS_KEY_NOT_PRESENT",
  "STATUS_CSS_KEY_NOT_ESTABLISHED",
  "STATUS_CSS_SCRAMBLED_SECTOR",
  "STATUS_CSS_REGION_MISMATCH",
  "STATUS_CSS_RESETS_EXHAUSTED",
  "STATUS_PASSWORD_CHANGE_REQUIRED",
  "STATUS_LOST_MODE_LOGON_RESTRICTION",
  "STATUS_PKINIT_FAILURE",
  "STATUS_SMARTCARD_SUBSYSTEM_FAILURE",
  "STATUS_NO_KERB_KEY",
  "STATUS_HOST_DOWN",
  "STATUS_UNSUPPORTED_PREAUTH",
  "STATUS_EFS_ALG_BLOB_TOO_BIG",
  "STATUS_PORT_NOT_SET",
  "STATUS_DEBUGGER_INACTIVE",
  "STATUS_DS_VERSION_CHECK_FAILURE",
  "STATUS_AUDITING_DISABLED",
  "STATUS_PRENT4_MACHINE_ACCOUNT",
  "STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER",
  "STATUS_INVALID_IMAGE_WIN_32",
  "STATUS_INVALID_IMAGE_WIN_64",
  "STATUS_BAD_BINDINGS",
  "STATUS_NETWORK_SESSION_EXPIRED",
  "STATUS_APPHELP_BLOCK",
  "STATUS_ALL_SIDS_FILTERED",
  "STATUS_NOT_SAFE_MODE_DRIVER",
  "STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT",
  "STATUS_ACCESS_DISABLED_BY_POLICY_PATH",
  "STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER",
  "STATUS_ACCESS_DISABLED_BY_POLICY_OTHER",
  "STATUS_FAILED_DRIVER_ENTRY",
  "STATUS_DEVICE_ENUMERATION_ERROR",
  "STATUS_MOUNT_POINT_NOT_RESOLVED",
  "STATUS_INVALID_DEVICE_OBJECT_PARAMETER",
  "STATUS_MCA_OCCURED",
  "STATUS_DRIVER_BLOCKED_CRITICAL",
  "STATUS_DRIVER_BLOCKED",
  "STATUS_DRIVER_DATABASE_ERROR",
  "STATUS_SYSTEM_HIVE_TOO_LARGE",
  "STATUS_INVALID_IMPORT_OF_NON_DLL",
  "STATUS_NO_SECRETS",
  "STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY",
  "STATUS_FAILED_STACK_SWITCH",
  "STATUS_HEAP_CORRUPTION",
  "STATUS_SMARTCARD_WRONG_PIN",
  "STATUS_SMARTCARD_CARD_BLOCKED",
  "STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED",
  "STATUS_SMARTCARD_NO_CARD",
  "STATUS_SMARTCARD_NO_KEY_CONTAINER",
  "STATUS_SMARTCARD_NO_CERTIFICATE",
  "STATUS_SMARTCARD_NO_KEYSET",
  "STATUS_SMARTCARD_IO_ERROR",
  "STATUS_DOWNGRADE_DETECTED",
  "STATUS_SMARTCARD_CERT_REVOKED",
  "STATUS_ISSUING_CA_UNTRUSTED",
  "STATUS_REVOCATION_OFFLINE_C",
  "STATUS_PKINIT_CLIENT_FAILURE",
  "STATUS_SMARTCARD_CERT_EXPIRED",
  "STATUS_DRIVER_FAILED_PRIOR_UNLOAD",
  "STATUS_SMARTCARD_SILENT_CONTEXT",
  "STATUS_PER_USER_TRUST_QUOTA_EXCEEDED",
  "STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED",
  "STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED",
  "STATUS_DS_NAME_NOT_UNIQUE",
  "STATUS_DS_DUPLICATE_ID_FOUND",
  "STATUS_DS_GROUP_CONVERSION_ERROR",
  "STATUS_VOLSNAP_PREPARE_HIBERNATE",
  "STATUS_USER2USER_REQUIRED",
  "STATUS_STACK_BUFFER_OVERRUN",
  "STATUS_NO_S4U_PROT_SUPPORT",
  "STATUS_CROSSREALM_DELEGATION_FAILURE",
  "STATUS_REVOCATION_OFFLINE_KDC",
  "STATUS_ISSUING_CA_UNTRUSTED_KDC",
  "STATUS_KDC_CERT_EXPIRED",
  "STATUS_KDC_CERT_REVOKED",
  "STATUS_PARAMETER_QUOTA_EXCEEDED",
  "STATUS_HIBERNATION_FAILURE",
  "STATUS_DELAY_LOAD_FAILED",
  "STATUS_AUTHENTICATION_FIREWALL_FAILED",
  "STATUS_VDM_DISALLOWED",
  "STATUS_HUNG_DISPLAY_DRIVER_THREAD",
  "STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE",
  "STATUS_INVALID_CRUNTIME_PARAMETER",
  "STATUS_NTLM_BLOCKED",
  "STATUS_DS_SRC_SID_EXISTS_IN_FOREST",
  "STATUS_DS_DOMAIN_NAME_EXISTS_IN_FOREST",
  "STATUS_DS_FLAT_NAME_EXISTS_IN_FOREST",
  "STATUS_INVALID_USER_PRINCIPAL_NAME",
  "STATUS_FATAL_USER_CALLBACK_EXCEPTION",
  "STATUS_ASSERTION_FAILURE",
  "STATUS_VERIFIER_STOP",
  "STATUS_CALLBACK_POP_STACK",
  "STATUS_INCOMPATIBLE_DRIVER_BLOCKED",
  "STATUS_HIVE_UNLOADED",
  "STATUS_COMPRESSION_DISABLED",
  "STATUS_FILE_SYSTEM_LIMITATION",
  "STATUS_INVALID_IMAGE_HASH",
  "STATUS_NOT_CAPABLE",
  "STATUS_REQUEST_OUT_OF_SEQUENCE",
  "STATUS_IMPLEMENTATION_LIMIT",
  "STATUS_ELEVATION_REQUIRED",
  "STATUS_NO_SECURITY_CONTEXT",
  "STATUS_PKU2U_CERT_FAILURE",
  "STATUS_BEYOND_VDL",
  "STATUS_ENCOUNTERED_WRITE_IN_PROGRESS",
  "STATUS_PTE_CHANGED",
  "STATUS_PURGE_FAILED",
  "STATUS_CRED_REQUIRES_CONFIRMATION",
  "STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE",
  "STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER",
  "STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE",
  "STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE",
  "STATUS_CS_ENCRYPTION_FILE_NOT_CSE",
  "STATUS_INVALID_LABEL",
  "STATUS_DRIVER_PROCESS_TERMINATED",
  "STATUS_AMBIGUOUS_SYSTEM_DEVICE",
  "STATUS_SYSTEM_DEVICE_NOT_FOUND",
  "STATUS_RESTART_BOOT_APPLICATION",
  "STATUS_INSUFFICIENT_NVRAM_RESOURCES",
  "STATUS_INVALID_SESSION",
  "STATUS_THREAD_ALREADY_IN_SESSION",
  "STATUS_THREAD_NOT_IN_SESSION",
  "STATUS_INVALID_WEIGHT",
  "STATUS_REQUEST_PAUSED",
  "STATUS_NO_RANGES_PROCESSED",
  "STATUS_DISK_RESOURCES_EXHAUSTED",
  "STATUS_NEEDS_REMEDIATION",
  "STATUS_DEVICE_FEATURE_NOT_SUPPORTED",
  "STATUS_DEVICE_UNREACHABLE",
  "STATUS_INVALID_TOKEN",
  "STATUS_SERVER_UNAVAILABLE",
  "STATUS_FILE_NOT_AVAILABLE",
  "STATUS_DEVICE_INSUFFICIENT_RESOURCES",
  "STATUS_PACKAGE_UPDATING",
  "STATUS_NOT_READ_FROM_COPY",
  "STATUS_FT_WRITE_FAILURE",
  "STATUS_FT_DI_SCAN_REQUIRED",
  "STATUS_OBJECT_NOT_EXTERNALLY_BACKED",
  "STATUS_EXTERNAL_BACKING_PROVIDER_UNKNOWN",
  "STATUS_COMPRESSION_NOT_BENEFICIAL",
  "STATUS_DATA_CHECKSUM_ERROR",
  "STATUS_INTERMIXED_KERNEL_EA_OPERATION",
  "STATUS_TRIM_READ_ZERO_NOT_SUPPORTED",
  "STATUS_TOO_MANY_SEGMENT_DESCRIPTORS",
  "STATUS_INVALID_OFFSET_ALIGNMENT",
  "STATUS_INVALID_FIELD_IN_PARAMETER_LIST",
  "STATUS_OPERATION_IN_PROGRESS",
  "STATUS_INVALID_INITIATOR_TARGET_PATH",
  "STATUS_SCRUB_DATA_DISABLED",
  "STATUS_NOT_REDUNDANT_STORAGE",
  "STATUS_RESIDENT_FILE_NOT_SUPPORTED",
  "STATUS_COMPRESSED_FILE_NOT_SUPPORTED",
  "STATUS_DIRECTORY_NOT_SUPPORTED",
  "STATUS_IO_OPERATION_TIMEOUT",
  "STATUS_SYSTEM_NEEDS_REMEDIATION",
  "STATUS_APPX_INTEGRITY_FAILURE_CLR_NGEN",
  "STATUS_SHARE_UNAVAILABLE",
  "STATUS_APISET_NOT_HOSTED",
  "STATUS_APISET_NOT_PRESENT",
  "STATUS_DEVICE_HARDWARE_ERROR",
  "STATUS_FIRMWARE_SLOT_INVALID",
  "STATUS_FIRMWARE_IMAGE_INVALID",
  "STATUS_STORAGE_TOPOLOGY_ID_MISMATCH",
  "STATUS_WIM_NOT_BOOTABLE",
  "STATUS_BLOCKED_BY_PARENTAL_CONTROLS",
  "STATUS_NEEDS_REGISTRATION",
  "STATUS_QUOTA_ACTIVITY",
  "STATUS_CALLBACK_INVOKE_INLINE",
  "STATUS_BLOCK_TOO_MANY_REFERENCES",
  "STATUS_MARKED_TO_DISALLOW_WRITES",
  "STATUS_NETWORK_ACCESS_DENIED_EDP",
  "STATUS_ENCLAVE_FAILURE",
  "STATUS_PNP_NO_COMPAT_DRIVERS",
  "STATUS_PNP_DRIVER_PACKAGE_NOT_FOUND",
  "STATUS_PNP_DRIVER_CONFIGURATION_NOT_FOUND",
  "STATUS_PNP_DRIVER_CONFIGURATION_INCOMPLETE",
  "STATUS_PNP_FUNCTION_DRIVER_REQUIRED",
  "STATUS_PNP_DEVICE_CONFIGURATION_PENDING",
  "STATUS_DEVICE_HINT_NAME_BUFFER_TOO_SMALL",
  "STATUS_PACKAGE_NOT_AVAILABLE",
  "STATUS_DEVICE_IN_MAINTENANCE",
  "STATUS_NOT_SUPPORTED_ON_DAX",
  "STATUS_FREE_SPACE_TOO_FRAGMENTED",
  "STATUS_DAX_MAPPING_EXISTS",
  "STATUS_CHILD_PROCESS_BLOCKED",
  "STATUS_STORAGE_LOST_DATA_PERSISTENCE",
  "STATUS_VRF_CFG_AND_IO_ENABLED",
  "STATUS_PARTITION_TERMINATING",
  "STATUS_EXTERNAL_SYSKEY_NOT_SUPPORTED",
  "STATUS_ENCLAVE_VIOLATION",
  "STATUS_FILE_PROTECTED_UNDER_DPL",
  "STATUS_VOLUME_NOT_CLUSTER_ALIGNED",
  "STATUS_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND",
  "STATUS_APPX_FILE_NOT_ENCRYPTED",
  "STATUS_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED",
  "STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET",
  "STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE",
  "STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER",
  "STATUS_FT_READ_FAILURE",
  "STATUS_PATCH_CONFLICT",
  "STATUS_STORAGE_RESERVE_ID_INVALID",
  "STATUS_STORAGE_RESERVE_DOES_NOT_EXIST",
  "STATUS_STORAGE_RESERVE_ALREADY_EXISTS",
  "STATUS_STORAGE_RESERVE_NOT_EMPTY",
  "STATUS_NOT_A_DAX_VOLUME",
  "STATUS_NOT_DAX_MAPPABLE",
  "STATUS_CASE_DIFFERING_NAMES_IN_DIR",
  "STATUS_FILE_NOT_SUPPORTED",
  "STATUS_NOT_SUPPORTED_WITH_BTT",
  "STATUS_ENCRYPTION_DISABLED",
  "STATUS_ENCRYPTING_METADATA_DISALLOWED",
  "STATUS_CANT_CLEAR_ENCRYPTION_FLAG",
  "STATUS_UNSATISFIED_DEPENDENCIES",
  "STATUS_CASE_SENSITIVE_PATH",
  "STATUS_HAS_SYSTEM_CRITICAL_FILES",
  "STATUS_INVALID_TASK_NAME",
  "STATUS_INVALID_TASK_INDEX",
  "STATUS_THREAD_ALREADY_IN_TASK",
  "STATUS_CALLBACK_BYPASS",
  "STATUS_UNDEFINED_SCOPE",
  "STATUS_INVALID_CAP",
  "STATUS_NOT_GUI_PROCESS",
  "STATUS_DEVICE_HUNG",
  "STATUS_CONTAINER_ASSIGNED",
  "STATUS_JOB_NO_CONTAINER",
  "STATUS_DEVICE_UNRESPONSIVE",
  "STATUS_REPARSE_POINT_ENCOUNTERED",
  "STATUS_ATTRIBUTE_NOT_PRESENT",
  "STATUS_NOT_A_TIERED_VOLUME",
  "STATUS_ALREADY_HAS_STREAM_ID",
  "STATUS_JOB_NOT_EMPTY",
  "STATUS_ALREADY_INITIALIZED",
  "STATUS_ENCLAVE_NOT_TERMINATED",
  "STATUS_ENCLAVE_IS_TERMINATING",
  "STATUS_SMB1_NOT_AVAILABLE",
  "STATUS_SMR_GARBAGE_COLLECTION_REQUIRED",
  "STATUS_INTERRUPTED",
  "STATUS_THREAD_NOT_RUNNING",
  "STATUS_FAIL_FAST_EXCEPTION",
  "STATUS_IMAGE_CERT_REVOKED",
  "STATUS_DYNAMIC_CODE_BLOCKED",
  "STATUS_IMAGE_CERT_EXPIRED",
  "STATUS_STRICT_CFG_VIOLATION",
  "STATUS_SET_CONTEXT_DENIED",
  "STATUS_CROSS_PARTITION_VIOLATION",
  "STATUS_PORT_CLOSED",
  "STATUS_MESSAGE_LOST",
  "STATUS_INVALID_MESSAGE",
  "STATUS_REQUEST_CANCELED",
  "STATUS_RECURSIVE_DISPATCH",
  "STATUS_LPC_RECEIVE_BUFFER_EXPECTED",
  "STATUS_LPC_INVALID_CONNECTION_USAGE",
  "STATUS_LPC_REQUESTS_NOT_ALLOWED",
  "STATUS_RESOURCE_IN_USE",
  "STATUS_HARDWARE_MEMORY_ERROR",
  "STATUS_THREADPOOL_HANDLE_EXCEPTION",
  "STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED",
  "STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED",
  "STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED",
  "STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED",
  "STATUS_THREADPOOL_RELEASED_DURING_OPERATION",
  "STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING",
  "STATUS_APC_RETURNED_WHILE_IMPERSONATING",
  "STATUS_PROCESS_IS_PROTECTED",
  "STATUS_MCA_EXCEPTION",
  "STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE",
  "STATUS_SYMLINK_CLASS_DISABLED",
  "STATUS_INVALID_IDN_NORMALIZATION",
  "STATUS_NO_UNICODE_TRANSLATION",
  "STATUS_ALREADY_REGISTERED",
  "STATUS_CONTEXT_MISMATCH",
  "STATUS_PORT_ALREADY_HAS_COMPLETION_LIST",
  "STATUS_CALLBACK_RETURNED_THREAD_PRIORITY",
  "STATUS_INVALID_THREAD",
  "STATUS_CALLBACK_RETURNED_TRANSACTION",
  "STATUS_CALLBACK_RETURNED_LDR_LOCK",
  "STATUS_CALLBACK_RETURNED_LANG",
  "STATUS_CALLBACK_RETURNED_PRI_BACK",
  "STATUS_CALLBACK_RETURNED_THREAD_AFFINITY",
  "STATUS_LPC_HANDLE_COUNT_EXCEEDED",
  "STATUS_EXECUTABLE_MEMORY_WRITE",
  "STATUS_KERNEL_EXECUTABLE_MEMORY_WRITE",
  "STATUS_ATTACHED_EXECUTABLE_MEMORY_WRITE",
  "STATUS_TRIGGERED_EXECUTABLE_MEMORY_WRITE",
  "STATUS_DISK_REPAIR_DISABLED",
  "STATUS_DS_DOMAIN_RENAME_IN_PROGRESS",
  "STATUS_DISK_QUOTA_EXCEEDED",
  "STATUS_CONTENT_BLOCKED",
  "STATUS_BAD_CLUSTERS",
  "STATUS_VOLUME_DIRTY",
  "STATUS_DISK_REPAIR_UNSUCCESSFUL",
  "STATUS_CORRUPT_LOG_OVERFULL",
  "STATUS_CORRUPT_LOG_CORRUPTED",
  "STATUS_CORRUPT_LOG_UNAVAILABLE",
  "STATUS_CORRUPT_LOG_DELETED_FULL",
  "STATUS_CORRUPT_LOG_CLEARED",
  "STATUS_ORPHAN_NAME_EXHAUSTED",
  "STATUS_PROACTIVE_SCAN_IN_PROGRESS",
  "STATUS_ENCRYPTED_IO_NOT_POSSIBLE",
  "STATUS_CORRUPT_LOG_UPLEVEL_RECORDS",
  "STATUS_FILE_CHECKED_OUT",
  "STATUS_CHECKOUT_REQUIRED",
  "STATUS_BAD_FILE_TYPE",
  "STATUS_FILE_TOO_LARGE",
  "STATUS_FORMS_AUTH_REQUIRED",
  "STATUS_VIRUS_INFECTED",
  "STATUS_VIRUS_DELETED",
  "STATUS_BAD_MCFG_TABLE",
  "STATUS_CANNOT_BREAK_OPLOCK",
  "STATUS_BAD_KEY",
  "STATUS_BAD_DATA",
  "STATUS_NO_KEY",
  "STATUS_FILE_HANDLE_REVOKED",
  "STATUS_WOW_ASSERTION",
  "STATUS_INVALID_SIGNATURE",
  "STATUS_HMAC_NOT_SUPPORTED",
  "STATUS_AUTH_TAG_MISMATCH",
  "STATUS_INVALID_STATE_TRANSITION",
  "STATUS_INVALID_KERNEL_INFO_VERSION",
  "STATUS_INVALID_PEP_INFO_VERSION",
  "STATUS_HANDLE_REVOKED",
  "STATUS_EOF_ON_GHOSTED_RANGE",
  "STATUS_CC_NEEDS_CALLBACK_SECTION_DRAIN",
  "STATUS_IPSEC_QUEUE_OVERFLOW",
  "STATUS_ND_QUEUE_OVERFLOW",
  "STATUS_HOPLIMIT_EXCEEDED",
  "STATUS_PROTOCOL_NOT_SUPPORTED",
  "STATUS_FASTPATH_REJECTED",
  "STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED",
  "STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR",
  "STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR",
  "STATUS_XML_PARSE_ERROR",
  "STATUS_XMLDSIG_ERROR",
  "STATUS_WRONG_COMPARTMENT",
  "STATUS_AUTHIP_FAILURE",
  "STATUS_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS",
  "STATUS_DS_OID_NOT_FOUND",
  "STATUS_INCORRECT_ACCOUNT_TYPE",
  "STATUS_HASH_NOT_SUPPORTED",
  "STATUS_HASH_NOT_PRESENT",
  "STATUS_SECONDARY_IC_PROVIDER_NOT_REGISTERED",
  "STATUS_GPIO_CLIENT_INFORMATION_INVALID",
  "STATUS_GPIO_VERSION_NOT_SUPPORTED",
  "STATUS_GPIO_INVALID_REGISTRATION_PACKET",
  "STATUS_GPIO_OPERATION_DENIED",
  "STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE",
  "STATUS_CANNOT_SWITCH_RUNLEVEL",
  "STATUS_INVALID_RUNLEVEL_SETTING",
  "STATUS_RUNLEVEL_SWITCH_TIMEOUT",
  "STATUS_RUNLEVEL_SWITCH_AGENT_TIMEOUT",
  "STATUS_RUNLEVEL_SWITCH_IN_PROGRESS",
  "STATUS_NOT_APPCONTAINER",
  "STATUS_NOT_SUPPORTED_IN_APPCONTAINER",
  "STATUS_INVALID_PACKAGE_SID_LENGTH",
  "STATUS_LPAC_ACCESS_DENIED",
  "STATUS_ADMINLESS_ACCESS_DENIED",
  "STATUS_APP_DATA_NOT_FOUND",
  "STATUS_APP_DATA_EXPIRED",
  "STATUS_APP_DATA_CORRUPT",
  "STATUS_APP_DATA_LIMIT_EXCEEDED",
  "STATUS_APP_DATA_REBOOT_REQUIRED",
  "STATUS_OFFLOAD_READ_FLT_NOT_SUPPORTED",
  "STATUS_OFFLOAD_WRITE_FLT_NOT_SUPPORTED",
  "STATUS_OFFLOAD_READ_FILE_NOT_SUPPORTED",
  "STATUS_OFFLOAD_WRITE_FILE_NOT_SUPPORTED",
  "STATUS_WOF_WIM_HEADER_CORRUPT",
  "STATUS_WOF_WIM_RESOURCE_TABLE_CORRUPT",
  "STATUS_WOF_FILE_RESOURCE_TABLE_CORRUPT",
  "STATUS_CIMFS_IMAGE_CORRUPT",
  "STATUS_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE",
  "STATUS_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT",
  "STATUS_FILE_SYSTEM_VIRTUALIZATION_BUSY",
  "STATUS_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN",
  "STATUS_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION",
  "STATUS_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT",
  "STATUS_CLOUD_FILE_PROVIDER_NOT_RUNNING",
  "STATUS_CLOUD_FILE_METADATA_CORRUPT",
  "STATUS_CLOUD_FILE_METADATA_TOO_LARGE",
  "STATUS_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED",
  "STATUS_NOT_A_CLOUD_FILE",
  "STATUS_CLOUD_FILE_NOT_IN_SYNC",
  "STATUS_CLOUD_FILE_ALREADY_CONNECTED",
  "STATUS_CLOUD_FILE_NOT_SUPPORTED",
  "STATUS_CLOUD_FILE_INVALID_REQUEST",
  "STATUS_CLOUD_FILE_READ_ONLY_VOLUME",
  "STATUS_CLOUD_FILE_CONNECTED_PROVIDER_ONLY",
  "STATUS_CLOUD_FILE_VALIDATION_FAILED",
  "STATUS_CLOUD_FILE_AUTHENTICATION_FAILED",
  "STATUS_CLOUD_FILE_INSUFFICIENT_RESOURCES",
  "STATUS_CLOUD_FILE_NETWORK_UNAVAILABLE",
  "STATUS_CLOUD_FILE_UNSUCCESSFUL",
  "STATUS_CLOUD_FILE_NOT_UNDER_SYNC_ROOT",
  "STATUS_CLOUD_FILE_IN_USE",
  "STATUS_CLOUD_FILE_PINNED",
  "STATUS_CLOUD_FILE_REQUEST_ABORTED",
  "STATUS_CLOUD_FILE_PROPERTY_CORRUPT",
  "STATUS_CLOUD_FILE_ACCESS_DENIED",
  "STATUS_CLOUD_FILE_INCOMPATIBLE_HARDLINKS",
  "STATUS_CLOUD_FILE_PROPERTY_LOCK_CONFLICT",
  "STATUS_CLOUD_FILE_REQUEST_CANCELED",
  "STATUS_CLOUD_FILE_PROVIDER_TERMINATED",
  "STATUS_NOT_A_CLOUD_SYNC_ROOT",
  "STATUS_CLOUD_FILE_REQUEST_TIMEOUT",
  "STATUS_CLOUD_FILE_DEHYDRATION_DISALLOWED",
  "STATUS_FILE_SNAP_IN_PROGRESS",
  "STATUS_FILE_SNAP_USER_SECTION_NOT_SUPPORTED",
  "STATUS_FILE_SNAP_MODIFY_NOT_SUPPORTED",
  "STATUS_FILE_SNAP_IO_NOT_COORDINATED",
  "STATUS_FILE_SNAP_UNEXPECTED_ERROR",
  "STATUS_FILE_SNAP_INVALID_PARAMETER",
  "DBG_NO_STATE_CHANGE",
  "DBG_APP_NOT_IDLE",
  "RPC_NT_INVALID_STRING_BINDING",
  "RPC_NT_WRONG_KIND_OF_BINDING",
  "RPC_NT_INVALID_BINDING",
  "RPC_NT_PROTSEQ_NOT_SUPPORTED",
  "RPC_NT_INVALID_RPC_PROTSEQ",
  "RPC_NT_INVALID_STRING_UUID",
  "RPC_NT_INVALID_ENDPOINT_FORMAT",
  "RPC_NT_INVALID_NET_ADDR",
  "RPC_NT_NO_ENDPOINT_FOUND",
  "RPC_NT_INVALID_TIMEOUT",
  "RPC_NT_OBJECT_NOT_FOUND",
  "RPC_NT_ALREADY_REGISTERED",
  "RPC_NT_TYPE_ALREADY_REGISTERED",
  "RPC_NT_ALREADY_LISTENING",
  "RPC_NT_NO_PROTSEQS_REGISTERED",
  "RPC_NT_NOT_LISTENING",
  "RPC_NT_UNKNOWN_MGR_TYPE",
  "RPC_NT_UNKNOWN_IF",
  "RPC_NT_NO_BINDINGS",
  "RPC_NT_NO_PROTSEQS",
  "RPC_NT_CANT_CREATE_ENDPOINT",
  "RPC_NT_OUT_OF_RESOURCES",
  "RPC_NT_SERVER_UNAVAILABLE",
  "RPC_NT_SERVER_TOO_BUSY",
  "RPC_NT_INVALID_NETWORK_OPTIONS",
  "RPC_NT_NO_CALL_ACTIVE",
  "RPC_NT_CALL_FAILED",
  "RPC_NT_CALL_FAILED_DNE",
  "RPC_NT_PROTOCOL_ERROR",
  "RPC_NT_UNSUPPORTED_TRANS_SYN",
  "RPC_NT_UNSUPPORTED_TYPE",
  "RPC_NT_INVALID_TAG",
  "RPC_NT_INVALID_BOUND",
  "RPC_NT_NO_ENTRY_NAME",
  "RPC_NT_INVALID_NAME_SYNTAX",
  "RPC_NT_UNSUPPORTED_NAME_SYNTAX",
  "RPC_NT_UUID_NO_ADDRESS",
  "RPC_NT_DUPLICATE_ENDPOINT",
  "RPC_NT_UNKNOWN_AUTHN_TYPE",
  "RPC_NT_MAX_CALLS_TOO_SMALL",
  "RPC_NT_STRING_TOO_LONG",
  "RPC_NT_PROTSEQ_NOT_FOUND",
  "RPC_NT_PROCNUM_OUT_OF_RANGE",
  "RPC_NT_BINDING_HAS_NO_AUTH",
  "RPC_NT_UNKNOWN_AUTHN_SERVICE",
  "RPC_NT_UNKNOWN_AUTHN_LEVEL",
  "RPC_NT_INVALID_AUTH_IDENTITY",
  "RPC_NT_UNKNOWN_AUTHZ_SERVICE",
  "EPT_NT_INVALID_ENTRY",
  "EPT_NT_CANT_PERFORM_OP",
  "EPT_NT_NOT_REGISTERED",
  "RPC_NT_NOTHING_TO_EXPORT",
  "RPC_NT_INCOMPLETE_NAME",
  "RPC_NT_INVALID_VERS_OPTION",
  "RPC_NT_NO_MORE_MEMBERS",
  "RPC_NT_NOT_ALL_OBJS_UNEXPORTED",
  "RPC_NT_INTERFACE_NOT_FOUND",
  "RPC_NT_ENTRY_ALREADY_EXISTS",
  "RPC_NT_ENTRY_NOT_FOUND",
  "RPC_NT_NAME_SERVICE_UNAVAILABLE",
  "RPC_NT_INVALID_NAF_ID",
  "RPC_NT_CANNOT_SUPPORT",
  "RPC_NT_NO_CONTEXT_AVAILABLE",
  "RPC_NT_INTERNAL_ERROR",
  "RPC_NT_ZERO_DIVIDE",
  "RPC_NT_ADDRESS_ERROR",
  "RPC_NT_FP_DIV_ZERO",
  "RPC_NT_FP_UNDERFLOW",
  "RPC_NT_FP_OVERFLOW",
  "RPC_NT_CALL_IN_PROGRESS",
  "RPC_NT_NO_MORE_BINDINGS",
  "RPC_NT_GROUP_MEMBER_NOT_FOUND",
  "EPT_NT_CANT_CREATE",
  "RPC_NT_INVALID_OBJECT",
  "RPC_NT_NO_INTERFACES",
  "RPC_NT_CALL_CANCELLED",
  "RPC_NT_BINDING_INCOMPLETE",
  "RPC_NT_COMM_FAILURE",
  "RPC_NT_UNSUPPORTED_AUTHN_LEVEL",
  "RPC_NT_NO_PRINC_NAME",
  "RPC_NT_NOT_RPC_ERROR",
  "RPC_NT_SEC_PKG_ERROR",
  "RPC_NT_NOT_CANCELLED",
  "RPC_NT_INVALID_ASYNC_HANDLE",
  "RPC_NT_INVALID_ASYNC_CALL",
  "RPC_NT_PROXY_ACCESS_DENIED",
  "RPC_NT_COOKIE_AUTH_FAILED",
  "RPC_NT_NO_MORE_ENTRIES",
  "RPC_NT_SS_CHAR_TRANS_OPEN_FAIL",
  "RPC_NT_SS_CHAR_TRANS_SHORT_FILE",
  "RPC_NT_SS_IN_NULL_CONTEXT",
  "RPC_NT_SS_CONTEXT_MISMATCH",
  "RPC_NT_SS_CONTEXT_DAMAGED",
  "RPC_NT_SS_HANDLES_MISMATCH",
  "RPC_NT_SS_CANNOT_GET_CALL_HANDLE",
  "RPC_NT_NULL_REF_POINTER",
  "RPC_NT_ENUM_VALUE_OUT_OF_RANGE",
  "RPC_NT_BYTE_COUNT_TOO_SMALL",
  "RPC_NT_BAD_STUB_DATA",
  "RPC_NT_INVALID_ES_ACTION",
  "RPC_NT_WRONG_ES_VERSION",
  "RPC_NT_WRONG_STUB_VERSION",
  "RPC_NT_INVALID_PIPE_OBJECT",
  "RPC_NT_INVALID_PIPE_OPERATION",
  "RPC_NT_WRONG_PIPE_VERSION",
  "RPC_NT_PIPE_CLOSED",
  "RPC_NT_PIPE_DISCIPLINE_ERROR",
  "RPC_NT_PIPE_EMPTY",
  "STATUS_PNP_BAD_MPS_TABLE",
  "STATUS_PNP_TRANSLATION_FAILED",
  "STATUS_PNP_IRQ_TRANSLATION_FAILED",
  "STATUS_PNP_INVALID_ID",
  "STATUS_IO_REISSUE_AS_CACHED",
  "STATUS_CTX_WINSTATION_NAME_INVALID",
  "STATUS_CTX_INVALID_PD",
  "STATUS_CTX_PD_NOT_FOUND",
  "STATUS_CTX_CLOSE_PENDING",
  "STATUS_CTX_NO_OUTBUF",
  "STATUS_CTX_MODEM_INF_NOT_FOUND",
  "STATUS_CTX_INVALID_MODEMNAME",
  "STATUS_CTX_RESPONSE_ERROR",
  "STATUS_CTX_MODEM_RESPONSE_TIMEOUT",
  "STATUS_CTX_MODEM_RESPONSE_NO_CARRIER",
  "STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE",
  "STATUS_CTX_MODEM_RESPONSE_BUSY",
  "STATUS_CTX_MODEM_RESPONSE_VOICE",
  "STATUS_CTX_TD_ERROR",
  "STATUS_CTX_LICENSE_CLIENT_INVALID",
  "STATUS_CTX_LICENSE_NOT_AVAILABLE",
  "STATUS_CTX_LICENSE_EXPIRED",
  "STATUS_CTX_WINSTATION_NOT_FOUND",
  "STATUS_CTX_WINSTATION_NAME_COLLISION",
  "STATUS_CTX_WINSTATION_BUSY",
  "STATUS_CTX_BAD_VIDEO_MODE",
  "STATUS_CTX_GRAPHICS_INVALID",
  "STATUS_CTX_NOT_CONSOLE",
  "STATUS_CTX_CLIENT_QUERY_TIMEOUT",
  "STATUS_CTX_CONSOLE_DISCONNECT",
  "STATUS_CTX_CONSOLE_CONNECT",
  "STATUS_CTX_SHADOW_DENIED",
  "STATUS_CTX_WINSTATION_ACCESS_DENIED",
  "STATUS_CTX_INVALID_WD",
  "STATUS_CTX_WD_NOT_FOUND",
  "STATUS_CTX_SHADOW_INVALID",
  "STATUS_CTX_SHADOW_DISABLED",
  "STATUS_RDP_PROTOCOL_ERROR",
  "STATUS_CTX_CLIENT_LICENSE_NOT_SET",
  "STATUS_CTX_CLIENT_LICENSE_IN_USE",
  "STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE",
  "STATUS_CTX_SHADOW_NOT_RUNNING",
  "STATUS_CTX_LOGON_DISABLED",
  "STATUS_CTX_SECURITY_LAYER_ERROR",
  "STATUS_TS_INCOMPATIBLE_SESSIONS",
  "STATUS_TS_VIDEO_SUBSYSTEM_ERROR",
  "STATUS_MUI_FILE_NOT_FOUND",
  "STATUS_MUI_INVALID_FILE",
  "STATUS_MUI_INVALID_RC_CONFIG",
  "STATUS_MUI_INVALID_LOCALE_NAME",
  "STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME",
  "STATUS_MUI_FILE_NOT_LOADED",
  "STATUS_RESOURCE_ENUM_USER_STOP",
  "STATUS_CLUSTER_INVALID_NODE",
  "STATUS_CLUSTER_NODE_EXISTS",
  "STATUS_CLUSTER_JOIN_IN_PROGRESS",
  "STATUS_CLUSTER_NODE_NOT_FOUND",
  "STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND",
  "STATUS_CLUSTER_NETWORK_EXISTS",
  "STATUS_CLUSTER_NETWORK_NOT_FOUND",
  "STATUS_CLUSTER_NETINTERFACE_EXISTS",
  "STATUS_CLUSTER_NETINTERFACE_NOT_FOUND",
  "STATUS_CLUSTER_INVALID_REQUEST",
  "STATUS_CLUSTER_INVALID_NETWORK_PROVIDER",
  "STATUS_CLUSTER_NODE_DOWN",
  "STATUS_CLUSTER_NODE_UNREACHABLE",
  "STATUS_CLUSTER_NODE_NOT_MEMBER",
  "STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS",
  "STATUS_CLUSTER_INVALID_NETWORK",
  "STATUS_CLUSTER_NO_NET_ADAPTERS",
  "STATUS_CLUSTER_NODE_UP",
  "STATUS_CLUSTER_NODE_PAUSED",
  "STATUS_CLUSTER_NODE_NOT_PAUSED",
  "STATUS_CLUSTER_NO_SECURITY_CONTEXT",
  "STATUS_CLUSTER_NETWORK_NOT_INTERNAL",
  "STATUS_CLUSTER_POISONED",
  "STATUS_CLUSTER_NON_CSV_PATH",
  "STATUS_CLUSTER_CSV_VOLUME_NOT_LOCAL",
  "STATUS_CLUSTER_CSV_READ_OPLOCK_BREAK_IN_PROGRESS",
  "STATUS_CLUSTER_CSV_AUTO_PAUSE_ERROR",
  "STATUS_CLUSTER_CSV_REDIRECTED",
  "STATUS_CLUSTER_CSV_NOT_REDIRECTED",
  "STATUS_CLUSTER_CSV_VOLUME_DRAINING",
  "STATUS_CLUSTER_CSV_SNAPSHOT_CREATION_IN_PROGRESS",
  "STATUS_CLUSTER_CSV_VOLUME_DRAINING_SUCCEEDED_DOWNLEVEL",
  "STATUS_CLUSTER_CSV_NO_SNAPSHOTS",
  "STATUS_CSV_IO_PAUSE_TIMEOUT",
  "STATUS_CLUSTER_CSV_INVALID_HANDLE",
  "STATUS_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR",
  "STATUS_CLUSTER_CAM_TICKET_REPLAY_DETECTED",
  "STATUS_ACPI_INVALID_OPCODE",
  "STATUS_ACPI_STACK_OVERFLOW",
  "STATUS_ACPI_ASSERT_FAILED",
  "STATUS_ACPI_INVALID_INDEX",
  "STATUS_ACPI_INVALID_ARGUMENT",
  "STATUS_ACPI_FATAL",
  "STATUS_ACPI_INVALID_SUPERNAME",
  "STATUS_ACPI_INVALID_ARGTYPE",
  "STATUS_ACPI_INVALID_OBJTYPE",
  "STATUS_ACPI_INVALID_TARGETTYPE",
  "STATUS_ACPI_INCORRECT_ARGUMENT_COUNT",
  "STATUS_ACPI_ADDRESS_NOT_MAPPED",
  "STATUS_ACPI_INVALID_EVENTTYPE",
  "STATUS_ACPI_HANDLER_COLLISION",
  "STATUS_ACPI_INVALID_DATA",
  "STATUS_ACPI_INVALID_REGION",
  "STATUS_ACPI_INVALID_ACCESS_SIZE",
  "STATUS_ACPI_ACQUIRE_GLOBAL_LOCK",
  "STATUS_ACPI_ALREADY_INITIALIZED",
  "STATUS_ACPI_NOT_INITIALIZED",
  "STATUS_ACPI_INVALID_MUTEX_LEVEL",
  "STATUS_ACPI_MUTEX_NOT_OWNED",
  "STATUS_ACPI_MUTEX_NOT_OWNER",
  "STATUS_ACPI_RS_ACCESS",
  "STATUS_ACPI_INVALID_TABLE",
  "STATUS_ACPI_REG_HANDLER_FAILED",
  "STATUS_ACPI_POWER_REQUEST_FAILED",
  "STATUS_SXS_SECTION_NOT_FOUND",
  "STATUS_SXS_CANT_GEN_ACTCTX",
  "STATUS_SXS_INVALID_ACTCTXDATA_FORMAT",
  "STATUS_SXS_ASSEMBLY_NOT_FOUND",
  "STATUS_SXS_MANIFEST_FORMAT_ERROR",
  "STATUS_SXS_MANIFEST_PARSE_ERROR",
  "STATUS_SXS_ACTIVATION_CONTEXT_DISABLED",
  "STATUS_SXS_KEY_NOT_FOUND",
  "STATUS_SXS_VERSION_CONFLICT",
  "STATUS_SXS_WRONG_SECTION_TYPE",
  "STATUS_SXS_THREAD_QUERIES_DISABLED",
  "STATUS_SXS_ASSEMBLY_MISSING",
  "STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET",
  "STATUS_SXS_EARLY_DEACTIVATION",
  "STATUS_SXS_INVALID_DEACTIVATION",
  "STATUS_SXS_MULTIPLE_DEACTIVATION",
  "STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY",
  "STATUS_SXS_PROCESS_TERMINATION_REQUESTED",
  "STATUS_SXS_CORRUPT_ACTIVATION_STACK",
  "STATUS_SXS_CORRUPTION",
  "STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE",
  "STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME",
  "STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE",
  "STATUS_SXS_IDENTITY_PARSE_ERROR",
  "STATUS_SXS_COMPONENT_STORE_CORRUPT",
  "STATUS_SXS_FILE_HASH_MISMATCH",
  "STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT",
  "STATUS_SXS_IDENTITIES_DIFFERENT",
  "STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT",
  "STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY",
  "STATUS_ADVANCED_INSTALLER_FAILED",
  "STATUS_XML_ENCODING_MISMATCH",
  "STATUS_SXS_MANIFEST_TOO_BIG",
  "STATUS_SXS_SETTING_NOT_REGISTERED",
  "STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE",
  "STATUS_SMI_PRIMITIVE_INSTALLER_FAILED",
  "STATUS_GENERIC_COMMAND_FAILED",
  "STATUS_SXS_FILE_HASH_MISSING",
  "STATUS_TRANSACTIONAL_CONFLICT",
  "STATUS_INVALID_TRANSACTION",
  "STATUS_TRANSACTION_NOT_ACTIVE",
  "STATUS_TM_INITIALIZATION_FAILED",
  "STATUS_RM_NOT_ACTIVE",
  "STATUS_RM_METADATA_CORRUPT",
  "STATUS_TRANSACTION_NOT_JOINED",
  "STATUS_DIRECTORY_NOT_RM",
  "STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE",
  "STATUS_LOG_RESIZE_INVALID_SIZE",
  "STATUS_REMOTE_FILE_VERSION_MISMATCH",
  "STATUS_CRM_PROTOCOL_ALREADY_EXISTS",
  "STATUS_TRANSACTION_PROPAGATION_FAILED",
  "STATUS_CRM_PROTOCOL_NOT_FOUND",
  "STATUS_TRANSACTION_SUPERIOR_EXISTS",
  "STATUS_TRANSACTION_REQUEST_NOT_VALID",
  "STATUS_TRANSACTION_NOT_REQUESTED",
  "STATUS_TRANSACTION_ALREADY_ABORTED",
  "STATUS_TRANSACTION_ALREADY_COMMITTED",
  "STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER",
  "STATUS_CURRENT_TRANSACTION_NOT_VALID",
  "STATUS_LOG_GROWTH_FAILED",
  "STATUS_OBJECT_NO_LONGER_EXISTS",
  "STATUS_STREAM_MINIVERSION_NOT_FOUND",
  "STATUS_STREAM_MINIVERSION_NOT_VALID",
  "STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION",
  "STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT",
  "STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS",
  "STATUS_HANDLE_NO_LONGER_VALID",
  "STATUS_LOG_CORRUPTION_DETECTED",
  "STATUS_RM_DISCONNECTED",
  "STATUS_ENLISTMENT_NOT_SUPERIOR",
  "STATUS_FILE_IDENTITY_NOT_PERSISTENT",
  "STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY",
  "STATUS_CANT_CROSS_RM_BOUNDARY",
  "STATUS_TXF_DIR_NOT_EMPTY",
  "STATUS_INDOUBT_TRANSACTIONS_EXIST",
  "STATUS_TM_VOLATILE",
  "STATUS_ROLLBACK_TIMER_EXPIRED",
  "STATUS_TXF_ATTRIBUTE_CORRUPT",
  "STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION",
  "STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED",
  "STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE",
  "STATUS_TRANSACTION_REQUIRED_PROMOTION",
  "STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION",
  "STATUS_TRANSACTIONS_NOT_FROZEN",
  "STATUS_TRANSACTION_FREEZE_IN_PROGRESS",
  "STATUS_NOT_SNAPSHOT_VOLUME",
  "STATUS_NO_SAVEPOINT_WITH_OPEN_FILES",
  "STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION",
  "STATUS_TM_IDENTITY_MISMATCH",
  "STATUS_FLOATED_SECTION",
  "STATUS_CANNOT_ACCEPT_TRANSACTED_WORK",
  "STATUS_CANNOT_ABORT_TRANSACTIONS",
  "STATUS_TRANSACTION_NOT_FOUND",
  "STATUS_RESOURCEMANAGER_NOT_FOUND",
  "STATUS_ENLISTMENT_NOT_FOUND",
  "STATUS_TRANSACTIONMANAGER_NOT_FOUND",
  "STATUS_TRANSACTIONMANAGER_NOT_ONLINE",
  "STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION",
  "STATUS_TRANSACTION_NOT_ROOT",
  "STATUS_TRANSACTION_OBJECT_EXPIRED",
  "STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION",
  "STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED",
  "STATUS_TRANSACTION_RECORD_TOO_LONG",
  "STATUS_NO_LINK_TRACKING_IN_TRANSACTION",
  "STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION",
  "STATUS_TRANSACTION_INTEGRITY_VIOLATED",
  "STATUS_TRANSACTIONMANAGER_IDENTITY_MISMATCH",
  "STATUS_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT",
  "STATUS_TRANSACTION_MUST_WRITETHROUGH",
  "STATUS_TRANSACTION_NO_SUPERIOR",
  "STATUS_EXPIRED_HANDLE",
  "STATUS_TRANSACTION_NOT_ENLISTED",
  "STATUS_LOG_SECTOR_INVALID",
  "STATUS_LOG_SECTOR_PARITY_INVALID",
  "STATUS_LOG_SECTOR_REMAPPED",
  "STATUS_LOG_BLOCK_INCOMPLETE",
  "STATUS_LOG_INVALID_RANGE",
  "STATUS_LOG_BLOCKS_EXHAUSTED",
  "STATUS_LOG_READ_CONTEXT_INVALID",
  "STATUS_LOG_RESTART_INVALID",
  "STATUS_LOG_BLOCK_VERSION",
  "STATUS_LOG_BLOCK_INVALID",
  "STATUS_LOG_READ_MODE_INVALID",
  "STATUS_LOG_METADATA_CORRUPT",
  "STATUS_LOG_METADATA_INVALID",
  "STATUS_LOG_METADATA_INCONSISTENT",
  "STATUS_LOG_RESERVATION_INVALID",
  "STATUS_LOG_CANT_DELETE",
  "STATUS_LOG_CONTAINER_LIMIT_EXCEEDED",
  "STATUS_LOG_START_OF_LOG",
  "STATUS_LOG_POLICY_ALREADY_INSTALLED",
  "STATUS_LOG_POLICY_NOT_INSTALLED",
  "STATUS_LOG_POLICY_INVALID",
  "STATUS_LOG_POLICY_CONFLICT",
  "STATUS_LOG_PINNED_ARCHIVE_TAIL",
  "STATUS_LOG_RECORD_NONEXISTENT",
  "STATUS_LOG_RECORDS_RESERVED_INVALID",
  "STATUS_LOG_SPACE_RESERVED_INVALID",
  "STATUS_LOG_TAIL_INVALID",
  "STATUS_LOG_FULL",
  "STATUS_LOG_MULTIPLEXED",
  "STATUS_LOG_DEDICATED",
  "STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS",
  "STATUS_LOG_ARCHIVE_IN_PROGRESS",
  "STATUS_LOG_EPHEMERAL",
  "STATUS_LOG_NOT_ENOUGH_CONTAINERS",
  "STATUS_LOG_CLIENT_ALREADY_REGISTERED",
  "STATUS_LOG_CLIENT_NOT_REGISTERED",
  "STATUS_LOG_FULL_HANDLER_IN_PROGRESS",
  "STATUS_LOG_CONTAINER_READ_FAILED",
  "STATUS_LOG_CONTAINER_WRITE_FAILED",
  "STATUS_LOG_CONTAINER_OPEN_FAILED",
  "STATUS_LOG_CONTAINER_STATE_INVALID",
  "STATUS_LOG_STATE_INVALID",
  "STATUS_LOG_PINNED",
  "STATUS_LOG_METADATA_FLUSH_FAILED",
  "STATUS_LOG_INCONSISTENT_SECURITY",
  "STATUS_LOG_APPENDED_FLUSH_FAILED",
  "STATUS_LOG_PINNED_RESERVATION",
  "STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD",
  "STATUS_FLT_NO_HANDLER_DEFINED",
  "STATUS_FLT_CONTEXT_ALREADY_DEFINED",
  "STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST",
  "STATUS_FLT_DISALLOW_FAST_IO",
  "STATUS_FLT_INVALID_NAME_REQUEST",
  "STATUS_FLT_NOT_SAFE_TO_POST_OPERATION",
  "STATUS_FLT_NOT_INITIALIZED",
  "STATUS_FLT_FILTER_NOT_READY",
  "STATUS_FLT_POST_OPERATION_CLEANUP",
  "STATUS_FLT_INTERNAL_ERROR",
  "STATUS_FLT_DELETING_OBJECT",
  "STATUS_FLT_MUST_BE_NONPAGED_POOL",
  "STATUS_FLT_DUPLICATE_ENTRY",
  "STATUS_FLT_CBDQ_DISABLED",
  "STATUS_FLT_DO_NOT_ATTACH",
  "STATUS_FLT_DO_NOT_DETACH",
  "STATUS_FLT_INSTANCE_ALTITUDE_COLLISION",
  "STATUS_FLT_INSTANCE_NAME_COLLISION",
  "STATUS_FLT_FILTER_NOT_FOUND",
  "STATUS_FLT_VOLUME_NOT_FOUND",
  "STATUS_FLT_INSTANCE_NOT_FOUND",
  "STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND",
  "STATUS_FLT_INVALID_CONTEXT_REGISTRATION",
  "STATUS_FLT_NAME_CACHE_MISS",
  "STATUS_FLT_NO_DEVICE_OBJECT",
  "STATUS_FLT_VOLUME_ALREADY_MOUNTED",
  "STATUS_FLT_ALREADY_ENLISTED",
  "STATUS_FLT_CONTEXT_ALREADY_LINKED",
  "STATUS_FLT_NO_WAITER_FOR_REPLY",
  "STATUS_FLT_REGISTRATION_BUSY",
  "STATUS_MONITOR_NO_DESCRIPTOR",
  "STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT",
  "STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM",
  "STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK",
  "STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED",
  "STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK",
  "STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK",
  "STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA",
  "STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK",
  "STATUS_MONITOR_INVALID_MANUFACTURE_DATE",
  "STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER",
  "STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER",
  "STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER",
  "STATUS_GRAPHICS_ADAPTER_WAS_RESET",
  "STATUS_GRAPHICS_INVALID_DRIVER_MODEL",
  "STATUS_GRAPHICS_PRESENT_MODE_CHANGED",
  "STATUS_GRAPHICS_PRESENT_OCCLUDED",
  "STATUS_GRAPHICS_PRESENT_DENIED",
  "STATUS_GRAPHICS_CANNOTCOLORCONVERT",
  "STATUS_GRAPHICS_DRIVER_MISMATCH",
  "STATUS_GRAPHICS_PRESENT_REDIRECTION_DISABLED",
  "STATUS_GRAPHICS_PRESENT_UNOCCLUDED",
  "STATUS_GRAPHICS_WINDOWDC_NOT_AVAILABLE",
  "STATUS_GRAPHICS_WINDOWLESS_PRESENT_DISABLED",
  "STATUS_GRAPHICS_PRESENT_INVALID_WINDOW",
  "STATUS_GRAPHICS_PRESENT_BUFFER_NOT_BOUND",
  "STATUS_GRAPHICS_VAIL_STATE_CHANGED",
  "STATUS_GRAPHICS_INDIRECT_DISPLAY_ABANDON_SWAPCHAIN",
  "STATUS_GRAPHICS_INDIRECT_DISPLAY_DEVICE_STOPPED",
  "STATUS_GRAPHICS_NO_VIDEO_MEMORY",
  "STATUS_GRAPHICS_CANT_LOCK_MEMORY",
  "STATUS_GRAPHICS_ALLOCATION_BUSY",
  "STATUS_GRAPHICS_TOO_MANY_REFERENCES",
  "STATUS_GRAPHICS_TRY_AGAIN_LATER",
  "STATUS_GRAPHICS_TRY_AGAIN_NOW",
  "STATUS_GRAPHICS_ALLOCATION_INVALID",
  "STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE",
  "STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED",
  "STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION",
  "STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE",
  "STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION",
  "STATUS_GRAPHICS_ALLOCATION_CLOSED",
  "STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE",
  "STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE",
  "STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE",
  "STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST",
  "STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE",
  "STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY",
  "STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED",
  "STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED",
  "STATUS_GRAPHICS_INVALID_VIDPN",
  "STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE",
  "STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET",
  "STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED",
  "STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET",
  "STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET",
  "STATUS_GRAPHICS_INVALID_FREQUENCY",
  "STATUS_GRAPHICS_INVALID_ACTIVE_REGION",
  "STATUS_GRAPHICS_INVALID_TOTAL_REGION",
  "STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE",
  "STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE",
  "STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET",
  "STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY",
  "STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET",
  "STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET",
  "STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET",
  "STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET",
  "STATUS_GRAPHICS_TARGET_ALREADY_IN_SET",
  "STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH",
  "STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY",
  "STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET",
  "STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE",
  "STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET",
  "STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET",
  "STATUS_GRAPHICS_STALE_MODESET",
  "STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET",
  "STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE",
  "STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN",
  "STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE",
  "STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION",
  "STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES",
  "STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY",
  "STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE",
  "STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET",
  "STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET",
  "STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR",
  "STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET",
  "STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET",
  "STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE",
  "STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE",
  "STATUS_GRAPHICS_RESOURCES_NOT_RELATED",
  "STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE",
  "STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE",
  "STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET",
  "STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER",
  "STATUS_GRAPHICS_NO_VIDPNMGR",
  "STATUS_GRAPHICS_NO_ACTIVE_VIDPN",
  "STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY",
  "STATUS_GRAPHICS_MONITOR_NOT_CONNECTED",
  "STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY",
  "STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE",
  "STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE",
  "STATUS_GRAPHICS_INVALID_STRIDE",
  "STATUS_GRAPHICS_INVALID_PIXELFORMAT",
  "STATUS_GRAPHICS_INVALID_COLORBASIS",
  "STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE",
  "STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY",
  "STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT",
  "STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE",
  "STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN",
  "STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL",
  "STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION",
  "STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED",
  "STATUS_GRAPHICS_INVALID_GAMMA_RAMP",
  "STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED",
  "STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED",
  "STATUS_GRAPHICS_MODE_NOT_IN_MODESET",
  "STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON",
  "STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE",
  "STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE",
  "STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS",
  "STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING",
  "STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED",
  "STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS",
  "STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT",
  "STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM",
  "STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN",
  "STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT",
  "STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED",
  "STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION",
  "STATUS_GRAPHICS_INVALID_CLIENT_TYPE",
  "STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET",
  "STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED",
  "STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED",
  "STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER",
  "STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED",
  "STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED",
  "STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY",
  "STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED",
  "STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON",
  "STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE",
  "STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER",
  "STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED",
  "STATUS_GRAPHICS_OPM_NOT_SUPPORTED",
  "STATUS_GRAPHICS_COPP_NOT_SUPPORTED",
  "STATUS_GRAPHICS_UAB_NOT_SUPPORTED",
  "STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS",
  "STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST",
  "STATUS_GRAPHICS_OPM_INTERNAL_ERROR",
  "STATUS_GRAPHICS_OPM_INVALID_HANDLE",
  "STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH",
  "STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED",
  "STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED",
  "STATUS_GRAPHICS_PVP_HFS_FAILED",
  "STATUS_GRAPHICS_OPM_INVALID_SRM",
  "STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP",
  "STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP",
  "STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA",
  "STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET",
  "STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH",
  "STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE",
  "STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS",
  "STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS",
  "STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST",
  "STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR",
  "STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS",
  "STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED",
  "STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST",
  "STATUS_GRAPHICS_I2C_NOT_SUPPORTED",
  "STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST",
  "STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA",
  "STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA",
  "STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED",
  "STATUS_GRAPHICS_DDCCI_INVALID_DATA",
  "STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE",
  "STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING",
  "STATUS_GRAPHICS_MCA_INTERNAL_ERROR",
  "STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND",
  "STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH",
  "STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM",
  "STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE",
  "STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS",
  "STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED",
  "STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME",
  "STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP",
  "STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED",
  "STATUS_GRAPHICS_INVALID_POINTER",
  "STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE",
  "STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL",
  "STATUS_GRAPHICS_INTERNAL_ERROR",
  "STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS",
  "STATUS_FVE_LOCKED_VOLUME",
  "STATUS_FVE_NOT_ENCRYPTED",
  "STATUS_FVE_BAD_INFORMATION",
  "STATUS_FVE_TOO_SMALL",
  "STATUS_FVE_FAILED_WRONG_FS",
  "STATUS_FVE_BAD_PARTITION_SIZE",
  "STATUS_FVE_FS_NOT_EXTENDED",
  "STATUS_FVE_FS_MOUNTED",
  "STATUS_FVE_NO_LICENSE",
  "STATUS_FVE_ACTION_NOT_ALLOWED",
  "STATUS_FVE_BAD_DATA",
  "STATUS_FVE_VOLUME_NOT_BOUND",
  "STATUS_FVE_NOT_DATA_VOLUME",
  "STATUS_FVE_CONV_READ_ERROR",
  "STATUS_FVE_CONV_WRITE_ERROR",
  "STATUS_FVE_OVERLAPPED_UPDATE",
  "STATUS_FVE_FAILED_SECTOR_SIZE",
  "STATUS_FVE_FAILED_AUTHENTICATION",
  "STATUS_FVE_NOT_OS_VOLUME",
  "STATUS_FVE_KEYFILE_NOT_FOUND",
  "STATUS_FVE_KEYFILE_INVALID",
  "STATUS_FVE_KEYFILE_NO_VMK",
  "STATUS_FVE_TPM_DISABLED",
  "STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO",
  "STATUS_FVE_TPM_INVALID_PCR",
  "STATUS_FVE_TPM_NO_VMK",
  "STATUS_FVE_PIN_INVALID",
  "STATUS_FVE_AUTH_INVALID_APPLICATION",
  "STATUS_FVE_AUTH_INVALID_CONFIG",
  "STATUS_FVE_DEBUGGER_ENABLED",
  "STATUS_FVE_DRY_RUN_FAILED",
  "STATUS_FVE_BAD_METADATA_POINTER",
  "STATUS_FVE_OLD_METADATA_COPY",
  "STATUS_FVE_REBOOT_REQUIRED",
  "STATUS_FVE_RAW_ACCESS",
  "STATUS_FVE_RAW_BLOCKED",
  "STATUS_FVE_NO_AUTOUNLOCK_MASTER_KEY",
  "STATUS_FVE_MOR_FAILED",
  "STATUS_FVE_NO_FEATURE_LICENSE",
  "STATUS_FVE_POLICY_USER_DISABLE_RDV_NOT_ALLOWED",
  "STATUS_FVE_CONV_RECOVERY_FAILED",
  "STATUS_FVE_VIRTUALIZED_SPACE_TOO_BIG",
  "STATUS_FVE_INVALID_DATUM_TYPE",
  "STATUS_FVE_VOLUME_TOO_SMALL",
  "STATUS_FVE_ENH_PIN_INVALID",
  "STATUS_FVE_FULL_ENCRYPTION_NOT_ALLOWED_ON_TP_STORAGE",
  "STATUS_FVE_WIPE_NOT_ALLOWED_ON_TP_STORAGE",
  "STATUS_FVE_NOT_ALLOWED_ON_CSV_STACK",
  "STATUS_FVE_NOT_ALLOWED_ON_CLUSTER",
  "STATUS_FVE_NOT_ALLOWED_TO_UPGRADE_WHILE_CONVERTING",
  "STATUS_FVE_WIPE_CANCEL_NOT_APPLICABLE",
  "STATUS_FVE_EDRIVE_DRY_RUN_FAILED",
  "STATUS_FVE_SECUREBOOT_DISABLED",
  "STATUS_FVE_SECUREBOOT_CONFIG_CHANGE",
  "STATUS_FVE_DEVICE_LOCKEDOUT",
  "STATUS_FVE_VOLUME_EXTEND_PREVENTS_EOW_DECRYPT",
  "STATUS_FVE_NOT_DE_VOLUME",
  "STATUS_FVE_PROTECTION_DISABLED",
  "STATUS_FVE_PROTECTION_CANNOT_BE_DISABLED",
  "STATUS_FVE_OSV_KSR_NOT_ALLOWED",
  "STATUS_FWP_CALLOUT_NOT_FOUND",
  "STATUS_FWP_CONDITION_NOT_FOUND",
  "STATUS_FWP_FILTER_NOT_FOUND",
  "STATUS_FWP_LAYER_NOT_FOUND",
  "STATUS_FWP_PROVIDER_NOT_FOUND",
  "STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND",
  "STATUS_FWP_SUBLAYER_NOT_FOUND",
  "STATUS_FWP_NOT_FOUND",
  "STATUS_FWP_ALREADY_EXISTS",
  "STATUS_FWP_IN_USE",
  "STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS",
  "STATUS_FWP_WRONG_SESSION",
  "STATUS_FWP_NO_TXN_IN_PROGRESS",
  "STATUS_FWP_TXN_IN_PROGRESS",
  "STATUS_FWP_TXN_ABORTED",
  "STATUS_FWP_SESSION_ABORTED",
  "STATUS_FWP_INCOMPATIBLE_TXN",
  "STATUS_FWP_TIMEOUT",
  "STATUS_FWP_NET_EVENTS_DISABLED",
  "STATUS_FWP_INCOMPATIBLE_LAYER",
  "STATUS_FWP_KM_CLIENTS_ONLY",
  "STATUS_FWP_LIFETIME_MISMATCH",
  "STATUS_FWP_BUILTIN_OBJECT",
  "STATUS_FWP_TOO_MANY_CALLOUTS",
  "STATUS_FWP_NOTIFICATION_DROPPED",
  "STATUS_FWP_TRAFFIC_MISMATCH",
  "STATUS_FWP_INCOMPATIBLE_SA_STATE",
  "STATUS_FWP_NULL_POINTER",
  "STATUS_FWP_INVALID_ENUMERATOR",
  "STATUS_FWP_INVALID_FLAGS",
  "STATUS_FWP_INVALID_NET_MASK",
  "STATUS_FWP_INVALID_RANGE",
  "STATUS_FWP_INVALID_INTERVAL",
  "STATUS_FWP_ZERO_LENGTH_ARRAY",
  "STATUS_FWP_NULL_DISPLAY_NAME",
  "STATUS_FWP_INVALID_ACTION_TYPE",
  "STATUS_FWP_INVALID_WEIGHT",
  "STATUS_FWP_MATCH_TYPE_MISMATCH",
  "STATUS_FWP_TYPE_MISMATCH",
  "STATUS_FWP_OUT_OF_BOUNDS",
  "STATUS_FWP_RESERVED",
  "STATUS_FWP_DUPLICATE_CONDITION",
  "STATUS_FWP_DUPLICATE_KEYMOD",
  "STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER",
  "STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER",
  "STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER",
  "STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT",
  "STATUS_FWP_INCOMPATIBLE_AUTH_METHOD",
  "STATUS_FWP_INCOMPATIBLE_DH_GROUP",
  "STATUS_FWP_EM_NOT_SUPPORTED",
  "STATUS_FWP_NEVER_MATCH",
  "STATUS_FWP_PROVIDER_CONTEXT_MISMATCH",
  "STATUS_FWP_INVALID_PARAMETER",
  "STATUS_FWP_TOO_MANY_SUBLAYERS",
  "STATUS_FWP_CALLOUT_NOTIFICATION_FAILED",
  "STATUS_FWP_INVALID_AUTH_TRANSFORM",
  "STATUS_FWP_INVALID_CIPHER_TRANSFORM",
  "STATUS_FWP_INCOMPATIBLE_CIPHER_TRANSFORM",
  "STATUS_FWP_INVALID_TRANSFORM_COMBINATION",
  "STATUS_FWP_DUPLICATE_AUTH_METHOD",
  "STATUS_FWP_INVALID_TUNNEL_ENDPOINT",
  "STATUS_FWP_L2_DRIVER_NOT_READY",
  "STATUS_FWP_KEY_DICTATOR_ALREADY_REGISTERED",
  "STATUS_FWP_KEY_DICTATION_INVALID_KEYING_MATERIAL",
  "STATUS_FWP_CONNECTIONS_DISABLED",
  "STATUS_FWP_INVALID_DNS_NAME",
  "STATUS_FWP_STILL_ON",
  "STATUS_FWP_IKEEXT_NOT_RUNNING",
  "STATUS_FWP_TCPIP_NOT_READY",
  "STATUS_FWP_INJECT_HANDLE_CLOSING",
  "STATUS_FWP_INJECT_HANDLE_STALE",
  "STATUS_FWP_CANNOT_PEND",
  "STATUS_FWP_DROP_NOICMP",
  "STATUS_NDIS_CLOSING",
  "STATUS_NDIS_BAD_VERSION",
  "STATUS_NDIS_BAD_CHARACTERISTICS",
  "STATUS_NDIS_ADAPTER_NOT_FOUND",
  "STATUS_NDIS_OPEN_FAILED",
  "STATUS_NDIS_DEVICE_FAILED",
  "STATUS_NDIS_MULTICAST_FULL",
  "STATUS_NDIS_MULTICAST_EXISTS",
  "STATUS_NDIS_MULTICAST_NOT_FOUND",
  "STATUS_NDIS_REQUEST_ABORTED",
  "STATUS_NDIS_RESET_IN_PROGRESS",
  "STATUS_NDIS_INVALID_PACKET",
  "STATUS_NDIS_INVALID_DEVICE_REQUEST",
  "STATUS_NDIS_ADAPTER_NOT_READY",
  "STATUS_NDIS_INVALID_LENGTH",
  "STATUS_NDIS_INVALID_DATA",
  "STATUS_NDIS_BUFFER_TOO_SHORT",
  "STATUS_NDIS_INVALID_OID",
  "STATUS_NDIS_ADAPTER_REMOVED",
  "STATUS_NDIS_UNSUPPORTED_MEDIA",
  "STATUS_NDIS_GROUP_ADDRESS_IN_USE",
  "STATUS_NDIS_FILE_NOT_FOUND",
  "STATUS_NDIS_ERROR_READING_FILE",
  "STATUS_NDIS_ALREADY_MAPPED",
  "STATUS_NDIS_RESOURCE_CONFLICT",
  "STATUS_NDIS_MEDIA_DISCONNECTED",
  "STATUS_NDIS_INVALID_ADDRESS",
  "STATUS_NDIS_PAUSED",
  "STATUS_NDIS_INTERFACE_NOT_FOUND",
  "STATUS_NDIS_UNSUPPORTED_REVISION",
  "STATUS_NDIS_INVALID_PORT",
  "STATUS_NDIS_INVALID_PORT_STATE",
  "STATUS_NDIS_LOW_POWER_STATE",
  "STATUS_NDIS_REINIT_REQUIRED",
  "STATUS_NDIS_NO_QUEUES",
  "STATUS_NDIS_NOT_SUPPORTED",
  "STATUS_NDIS_OFFLOAD_POLICY",
  "STATUS_NDIS_OFFLOAD_CONNECTION_REJECTED",
  "STATUS_NDIS_OFFLOAD_PATH_REJECTED",
  "STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED",
  "STATUS_NDIS_DOT11_MEDIA_IN_USE",
  "STATUS_NDIS_DOT11_POWER_STATE_INVALID",
  "STATUS_NDIS_PM_WOL_PATTERN_LIST_FULL",
  "STATUS_NDIS_PM_PROTOCOL_OFFLOAD_LIST_FULL",
  "STATUS_NDIS_DOT11_AP_CHANNEL_CURRENTLY_NOT_AVAILABLE",
  "STATUS_NDIS_DOT11_AP_BAND_CURRENTLY_NOT_AVAILABLE",
  "STATUS_NDIS_DOT11_AP_CHANNEL_NOT_ALLOWED",
  "STATUS_NDIS_DOT11_AP_BAND_NOT_ALLOWED",
  "STATUS_QUIC_HANDSHAKE_FAILURE",
  "STATUS_QUIC_VER_NEG_FAILURE",
  "STATUS_TPM_ERROR_MASK",
  "STATUS_TPM_AUTHFAIL",
  "STATUS_TPM_BADINDEX",
  "STATUS_TPM_BAD_PARAMETER",
  "STATUS_TPM_AUDITFAILURE",
  "STATUS_TPM_CLEAR_DISABLED",
  "STATUS_TPM_DEACTIVATED",
  "STATUS_TPM_DISABLED",
  "STATUS_TPM_DISABLED_CMD",
  "STATUS_TPM_FAIL",
  "STATUS_TPM_BAD_ORDINAL",
  "STATUS_TPM_INSTALL_DISABLED",
  "STATUS_TPM_INVALID_KEYHANDLE",
  "STATUS_TPM_KEYNOTFOUND",
  "STATUS_TPM_INAPPROPRIATE_ENC",
  "STATUS_TPM_MIGRATEFAIL",
  "STATUS_TPM_INVALID_PCR_INFO",
  "STATUS_TPM_NOSPACE",
  "STATUS_TPM_NOSRK",
  "STATUS_TPM_NOTSEALED_BLOB",
  "STATUS_TPM_OWNER_SET",
  "STATUS_TPM_RESOURCES",
  "STATUS_TPM_SHORTRANDOM",
  "STATUS_TPM_SIZE",
  "STATUS_TPM_WRONGPCRVAL",
  "STATUS_TPM_BAD_PARAM_SIZE",
  "STATUS_TPM_SHA_THREAD",
  "STATUS_TPM_SHA_ERROR",
  "STATUS_TPM_FAILEDSELFTEST",
  "STATUS_TPM_AUTH2FAIL",
  "STATUS_TPM_BADTAG",
  "STATUS_TPM_IOERROR",
  "STATUS_TPM_ENCRYPT_ERROR",
  "STATUS_TPM_DECRYPT_ERROR",
  "STATUS_TPM_INVALID_AUTHHANDLE",
  "STATUS_TPM_NO_ENDORSEMENT",
  "STATUS_TPM_INVALID_KEYUSAGE",
  "STATUS_TPM_WRONG_ENTITYTYPE",
  "STATUS_TPM_INVALID_POSTINIT",
  "STATUS_TPM_INAPPROPRIATE_SIG",
  "STATUS_TPM_BAD_KEY_PROPERTY",
  "STATUS_TPM_BAD_MIGRATION",
  "STATUS_TPM_BAD_SCHEME",
  "STATUS_TPM_BAD_DATASIZE",
  "STATUS_TPM_BAD_MODE",
  "STATUS_TPM_BAD_PRESENCE",
  "STATUS_TPM_BAD_VERSION",
  "STATUS_TPM_NO_WRAP_TRANSPORT",
  "STATUS_TPM_AUDITFAIL_UNSUCCESSFUL",
  "STATUS_TPM_AUDITFAIL_SUCCESSFUL",
  "STATUS_TPM_NOTRESETABLE",
  "STATUS_TPM_NOTLOCAL",
  "STATUS_TPM_BAD_TYPE",
  "STATUS_TPM_INVALID_RESOURCE",
  "STATUS_TPM_NOTFIPS",
  "STATUS_TPM_INVALID_FAMILY",
  "STATUS_TPM_NO_NV_PERMISSION",
  "STATUS_TPM_REQUIRES_SIGN",
  "STATUS_TPM_KEY_NOTSUPPORTED",
  "STATUS_TPM_AUTH_CONFLICT",
  "STATUS_TPM_AREA_LOCKED",
  "STATUS_TPM_BAD_LOCALITY",
  "STATUS_TPM_READ_ONLY",
  "STATUS_TPM_PER_NOWRITE",
  "STATUS_TPM_FAMILYCOUNT",
  "STATUS_TPM_WRITE_LOCKED",
  "STATUS_TPM_BAD_ATTRIBUTES",
  "STATUS_TPM_INVALID_STRUCTURE",
  "STATUS_TPM_KEY_OWNER_CONTROL",
  "STATUS_TPM_BAD_COUNTER",
  "STATUS_TPM_NOT_FULLWRITE",
  "STATUS_TPM_CONTEXT_GAP",
  "STATUS_TPM_MAXNVWRITES",
  "STATUS_TPM_NOOPERATOR",
  "STATUS_TPM_RESOURCEMISSING",
  "STATUS_TPM_DELEGATE_LOCK",
  "STATUS_TPM_DELEGATE_FAMILY",
  "STATUS_TPM_DELEGATE_ADMIN",
  "STATUS_TPM_TRANSPORT_NOTEXCLUSIVE",
  "STATUS_TPM_OWNER_CONTROL",
  "STATUS_TPM_DAA_RESOURCES",
  "STATUS_TPM_DAA_INPUT_DATA0",
  "STATUS_TPM_DAA_INPUT_DATA1",
  "STATUS_TPM_DAA_ISSUER_SETTINGS",
  "STATUS_TPM_DAA_TPM_SETTINGS",
  "STATUS_TPM_DAA_STAGE",
  "STATUS_TPM_DAA_ISSUER_VALIDITY",
  "STATUS_TPM_DAA_WRONG_W",
  "STATUS_TPM_BAD_HANDLE",
  "STATUS_TPM_BAD_DELEGATE",
  "STATUS_TPM_BADCONTEXT",
  "STATUS_TPM_TOOMANYCONTEXTS",
  "STATUS_TPM_MA_TICKET_SIGNATURE",
  "STATUS_TPM_MA_DESTINATION",
  "STATUS_TPM_MA_SOURCE",
  "STATUS_TPM_MA_AUTHORITY",
  "STATUS_TPM_PERMANENTEK",
  "STATUS_TPM_BAD_SIGNATURE",
  "STATUS_TPM_NOCONTEXTSPACE",
  "STATUS_TPM_20_E_ASYMMETRIC",
  "STATUS_TPM_20_E_ATTRIBUTES",
  "STATUS_TPM_20_E_HASH",
  "STATUS_TPM_20_E_VALUE",
  "STATUS_TPM_20_E_HIERARCHY",
  "STATUS_TPM_20_E_KEY_SIZE",
  "STATUS_TPM_20_E_MGF",
  "STATUS_TPM_20_E_MODE",
  "STATUS_TPM_20_E_TYPE",
  "STATUS_TPM_20_E_HANDLE",
  "STATUS_TPM_20_E_KDF",
  "STATUS_TPM_20_E_RANGE",
  "STATUS_TPM_20_E_AUTH_FAIL",
  "STATUS_TPM_20_E_NONCE",
  "STATUS_TPM_20_E_PP",
  "STATUS_TPM_20_E_SCHEME",
  "STATUS_TPM_20_E_SIZE",
  "STATUS_TPM_20_E_SYMMETRIC",
  "STATUS_TPM_20_E_TAG",
  "STATUS_TPM_20_E_SELECTOR",
  "STATUS_TPM_20_E_INSUFFICIENT",
  "STATUS_TPM_20_E_SIGNATURE",
  "STATUS_TPM_20_E_KEY",
  "STATUS_TPM_20_E_POLICY_FAIL",
  "STATUS_TPM_20_E_INTEGRITY",
  "STATUS_TPM_20_E_TICKET",
  "STATUS_TPM_20_E_RESERVED_BITS",
  "STATUS_TPM_20_E_BAD_AUTH",
  "STATUS_TPM_20_E_EXPIRED",
  "STATUS_TPM_20_E_POLICY_CC",
  "STATUS_TPM_20_E_BINDING",
  "STATUS_TPM_20_E_CURVE",
  "STATUS_TPM_20_E_ECC_POINT",
  "STATUS_TPM_20_E_INITIALIZE",
  "STATUS_TPM_20_E_FAILURE",
  "STATUS_TPM_20_E_SEQUENCE",
  "STATUS_TPM_20_E_PRIVATE",
  "STATUS_TPM_20_E_HMAC",
  "STATUS_TPM_20_E_DISABLED",
  "STATUS_TPM_20_E_EXCLUSIVE",
  "STATUS_TPM_20_E_ECC_CURVE",
  "STATUS_TPM_20_E_AUTH_TYPE",
  "STATUS_TPM_20_E_AUTH_MISSING",
  "STATUS_TPM_20_E_POLICY",
  "STATUS_TPM_20_E_PCR",
  "STATUS_TPM_20_E_PCR_CHANGED",
  "STATUS_TPM_20_E_UPGRADE",
  "STATUS_TPM_20_E_TOO_MANY_CONTEXTS",
  "STATUS_TPM_20_E_AUTH_UNAVAILABLE",
  "STATUS_TPM_20_E_REBOOT",
  "STATUS_TPM_20_E_UNBALANCED",
  "STATUS_TPM_20_E_COMMAND_SIZE",
  "STATUS_TPM_20_E_COMMAND_CODE",
  "STATUS_TPM_20_E_AUTHSIZE",
  "STATUS_TPM_20_E_AUTH_CONTEXT",
  "STATUS_TPM_20_E_NV_RANGE",
  "STATUS_TPM_20_E_NV_SIZE",
  "STATUS_TPM_20_E_NV_LOCKED",
  "STATUS_TPM_20_E_NV_AUTHORIZATION",
  "STATUS_TPM_20_E_NV_UNINITIALIZED",
  "STATUS_TPM_20_E_NV_SPACE",
  "STATUS_TPM_20_E_NV_DEFINED",
  "STATUS_TPM_20_E_BAD_CONTEXT",
  "STATUS_TPM_20_E_CPHASH",
  "STATUS_TPM_20_E_PARENT",
  "STATUS_TPM_20_E_NEEDS_TEST",
  "STATUS_TPM_20_E_NO_RESULT",
  "STATUS_TPM_20_E_SENSITIVE",
  "STATUS_TPM_COMMAND_BLOCKED",
  "STATUS_TPM_INVALID_HANDLE",
  "STATUS_TPM_DUPLICATE_VHANDLE",
  "STATUS_TPM_EMBEDDED_COMMAND_BLOCKED",
  "STATUS_TPM_EMBEDDED_COMMAND_UNSUPPORTED",
  "STATUS_TPM_RETRY",
  "STATUS_TPM_NEEDS_SELFTEST",
  "STATUS_TPM_DOING_SELFTEST",
  "STATUS_TPM_DEFEND_LOCK_RUNNING",
  "STATUS_TPM_COMMAND_CANCELED",
  "STATUS_TPM_TOO_MANY_CONTEXTS",
  "STATUS_TPM_NOT_FOUND",
  "STATUS_TPM_ACCESS_DENIED",
  "STATUS_TPM_INSUFFICIENT_BUFFER",
  "STATUS_TPM_PPI_FUNCTION_UNSUPPORTED",
  "STATUS_PCP_ERROR_MASK",
  "STATUS_PCP_DEVICE_NOT_READY",
  "STATUS_PCP_INVALID_HANDLE",
  "STATUS_PCP_INVALID_PARAMETER",
  "STATUS_PCP_FLAG_NOT_SUPPORTED",
  "STATUS_PCP_NOT_SUPPORTED",
  "STATUS_PCP_BUFFER_TOO_SMALL",
  "STATUS_PCP_INTERNAL_ERROR",
  "STATUS_PCP_AUTHENTICATION_FAILED",
  "STATUS_PCP_AUTHENTICATION_IGNORED",
  "STATUS_PCP_POLICY_NOT_FOUND",
  "STATUS_PCP_PROFILE_NOT_FOUND",
  "STATUS_PCP_VALIDATION_FAILED",
  "STATUS_PCP_DEVICE_NOT_FOUND",
  "STATUS_PCP_WRONG_PARENT",
  "STATUS_PCP_KEY_NOT_LOADED",
  "STATUS_PCP_NO_KEY_CERTIFICATION",
  "STATUS_PCP_KEY_NOT_FINALIZED",
  "STATUS_PCP_ATTESTATION_CHALLENGE_NOT_SET",
  "STATUS_PCP_NOT_PCR_BOUND",
  "STATUS_PCP_KEY_ALREADY_FINALIZED",
  "STATUS_PCP_KEY_USAGE_POLICY_NOT_SUPPORTED",
  "STATUS_PCP_KEY_USAGE_POLICY_INVALID",
  "STATUS_PCP_SOFT_KEY_ERROR",
  "STATUS_PCP_KEY_NOT_AUTHENTICATED",
  "STATUS_PCP_KEY_NOT_AIK",
  "STATUS_PCP_KEY_NOT_SIGNING_KEY",
  "STATUS_PCP_LOCKED_OUT",
  "STATUS_PCP_CLAIM_TYPE_NOT_SUPPORTED",
  "STATUS_PCP_TPM_VERSION_NOT_SUPPORTED",
  "STATUS_PCP_BUFFER_LENGTH_MISMATCH",
  "STATUS_PCP_IFX_RSA_KEY_CREATION_BLOCKED",
  "STATUS_PCP_TICKET_MISSING",
  "STATUS_PCP_RAW_POLICY_NOT_SUPPORTED",
  "STATUS_PCP_KEY_HANDLE_INVALIDATED",
  "STATUS_RTPM_NO_RESULT",
  "STATUS_RTPM_PCR_READ_INCOMPLETE",
  "STATUS_RTPM_INVALID_CONTEXT",
  "STATUS_RTPM_UNSUPPORTED_CMD",
  "STATUS_TPM_ZERO_EXHAUST_ENABLED",
  "STATUS_HV_INVALID_HYPERCALL_CODE",
  "STATUS_HV_INVALID_HYPERCALL_INPUT",
  "STATUS_HV_INVALID_ALIGNMENT",
  "STATUS_HV_INVALID_PARAMETER",
  "STATUS_HV_ACCESS_DENIED",
  "STATUS_HV_INVALID_PARTITION_STATE",
  "STATUS_HV_OPERATION_DENIED",
  "STATUS_HV_UNKNOWN_PROPERTY",
  "STATUS_HV_PROPERTY_VALUE_OUT_OF_RANGE",
  "STATUS_HV_INSUFFICIENT_MEMORY",
  "STATUS_HV_PARTITION_TOO_DEEP",
  "STATUS_HV_INVALID_PARTITION_ID",
  "STATUS_HV_INVALID_VP_INDEX",
  "STATUS_HV_INVALID_PORT_ID",
  "STATUS_HV_INVALID_CONNECTION_ID",
  "STATUS_HV_INSUFFICIENT_BUFFERS",
  "STATUS_HV_NOT_ACKNOWLEDGED",
  "STATUS_HV_INVALID_VP_STATE",
  "STATUS_HV_ACKNOWLEDGED",
  "STATUS_HV_INVALID_SAVE_RESTORE_STATE",
  "STATUS_HV_INVALID_SYNIC_STATE",
  "STATUS_HV_OBJECT_IN_USE",
  "STATUS_HV_INVALID_PROXIMITY_DOMAIN_INFO",
  "STATUS_HV_NO_DATA",
  "STATUS_HV_INACTIVE",
  "STATUS_HV_NO_RESOURCES",
  "STATUS_HV_FEATURE_UNAVAILABLE",
  "STATUS_HV_INSUFFICIENT_BUFFER",
  "STATUS_HV_INSUFFICIENT_DEVICE_DOMAINS",
  "STATUS_HV_CPUID_FEATURE_VALIDATION_ERROR",
  "STATUS_HV_CPUID_XSAVE_FEATURE_VALIDATION_ERROR",
  "STATUS_HV_PROCESSOR_STARTUP_TIMEOUT",
  "STATUS_HV_SMX_ENABLED",
  "STATUS_HV_INVALID_LP_INDEX",
  "STATUS_HV_INVALID_REGISTER_VALUE",
  "STATUS_HV_INVALID_VTL_STATE",
  "STATUS_HV_NX_NOT_DETECTED",
  "STATUS_HV_INVALID_DEVICE_ID",
  "STATUS_HV_INVALID_DEVICE_STATE",
  "STATUS_HV_PAGE_REQUEST_INVALID",
  "STATUS_HV_INVALID_CPU_GROUP_ID",
  "STATUS_HV_INVALID_CPU_GROUP_STATE",
  "STATUS_HV_OPERATION_FAILED",
  "STATUS_HV_NOT_ALLOWED_WITH_NESTED_VIRT_ACTIVE",
  "STATUS_HV_INSUFFICIENT_ROOT_MEMORY",
  "STATUS_HV_EVENT_BUFFER_ALREADY_FREED",
  "STATUS_HV_INSUFFICIENT_CONTIGUOUS_MEMORY",
  "STATUS_HV_NOT_PRESENT",
  "STATUS_IPSEC_BAD_SPI",
  "STATUS_IPSEC_SA_LIFETIME_EXPIRED",
  "STATUS_IPSEC_WRONG_SA",
  "STATUS_IPSEC_REPLAY_CHECK_FAILED",
  "STATUS_IPSEC_INVALID_PACKET",
  "STATUS_IPSEC_INTEGRITY_CHECK_FAILED",
  "STATUS_IPSEC_CLEAR_TEXT_DROP",
  "STATUS_IPSEC_AUTH_FIREWALL_DROP",
  "STATUS_IPSEC_THROTTLE_DROP",
  "STATUS_IPSEC_DOSP_BLOCK",
  "STATUS_IPSEC_DOSP_RECEIVED_MULTICAST",
  "STATUS_IPSEC_DOSP_INVALID_PACKET",
  "STATUS_IPSEC_DOSP_STATE_LOOKUP_FAILED",
  "STATUS_IPSEC_DOSP_MAX_ENTRIES",
  "STATUS_IPSEC_DOSP_KEYMOD_NOT_ALLOWED",
  "STATUS_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES",
  "STATUS_VID_DUPLICATE_HANDLER",
  "STATUS_VID_TOO_MANY_HANDLERS",
  "STATUS_VID_QUEUE_FULL",
  "STATUS_VID_HANDLER_NOT_PRESENT",
  "STATUS_VID_INVALID_OBJECT_NAME",
  "STATUS_VID_PARTITION_NAME_TOO_LONG",
  "STATUS_VID_MESSAGE_QUEUE_NAME_TOO_LONG",
  "STATUS_VID_PARTITION_ALREADY_EXISTS",
  "STATUS_VID_PARTITION_DOES_NOT_EXIST",
  "STATUS_VID_PARTITION_NAME_NOT_FOUND",
  "STATUS_VID_MESSAGE_QUEUE_ALREADY_EXISTS",
  "STATUS_VID_EXCEEDED_MBP_ENTRY_MAP_LIMIT",
  "STATUS_VID_MB_STILL_REFERENCED",
  "STATUS_VID_CHILD_GPA_PAGE_SET_CORRUPTED",
  "STATUS_VID_INVALID_NUMA_SETTINGS",
  "STATUS_VID_INVALID_NUMA_NODE_INDEX",
  "STATUS_VID_NOTIFICATION_QUEUE_ALREADY_ASSOCIATED",
  "STATUS_VID_INVALID_MEMORY_BLOCK_HANDLE",
  "STATUS_VID_PAGE_RANGE_OVERFLOW",
  "STATUS_VID_INVALID_MESSAGE_QUEUE_HANDLE",
  "STATUS_VID_INVALID_GPA_RANGE_HANDLE",
  "STATUS_VID_NO_MEMORY_BLOCK_NOTIFICATION_QUEUE",
  "STATUS_VID_MEMORY_BLOCK_LOCK_COUNT_EXCEEDED",
  "STATUS_VID_INVALID_PPM_HANDLE",
  "STATUS_VID_MBPS_ARE_LOCKED",
  "STATUS_VID_MESSAGE_QUEUE_CLOSED",
  "STATUS_VID_VIRTUAL_PROCESSOR_LIMIT_EXCEEDED",
  "STATUS_VID_STOP_PENDING",
  "STATUS_VID_INVALID_PROCESSOR_STATE",
  "STATUS_VID_EXCEEDED_KM_CONTEXT_COUNT_LIMIT",
  "STATUS_VID_KM_INTERFACE_ALREADY_INITIALIZED",
  "STATUS_VID_MB_PROPERTY_ALREADY_SET_RESET",
  "STATUS_VID_MMIO_RANGE_DESTROYED",
  "STATUS_VID_INVALID_CHILD_GPA_PAGE_SET",
  "STATUS_VID_RESERVE_PAGE_SET_IS_BEING_USED",
  "STATUS_VID_RESERVE_PAGE_SET_TOO_SMALL",
  "STATUS_VID_MBP_ALREADY_LOCKED_USING_RESERVED_PAGE",
  "STATUS_VID_MBP_COUNT_EXCEEDED_LIMIT",
  "STATUS_VID_SAVED_STATE_CORRUPT",
  "STATUS_VID_SAVED_STATE_UNRECOGNIZED_ITEM",
  "STATUS_VID_SAVED_STATE_INCOMPATIBLE",
  "STATUS_VID_VTL_ACCESS_DENIED",
  "STATUS_VOLMGR_DATABASE_FULL",
  "STATUS_VOLMGR_DISK_CONFIGURATION_CORRUPTED",
  "STATUS_VOLMGR_DISK_CONFIGURATION_NOT_IN_SYNC",
  "STATUS_VOLMGR_PACK_CONFIG_UPDATE_FAILED",
  "STATUS_VOLMGR_DISK_CONTAINS_NON_SIMPLE_VOLUME",
  "STATUS_VOLMGR_DISK_DUPLICATE",
  "STATUS_VOLMGR_DISK_DYNAMIC",
  "STATUS_VOLMGR_DISK_ID_INVALID",
  "STATUS_VOLMGR_DISK_INVALID",
  "STATUS_VOLMGR_DISK_LAST_VOTER",
  "STATUS_VOLMGR_DISK_LAYOUT_INVALID",
  "STATUS_VOLMGR_DISK_LAYOUT_NON_BASIC_BETWEEN_BASIC_PARTITIONS",
  "STATUS_VOLMGR_DISK_LAYOUT_NOT_CYLINDER_ALIGNED",
  "STATUS_VOLMGR_DISK_LAYOUT_PARTITIONS_TOO_SMALL",
  "STATUS_VOLMGR_DISK_LAYOUT_PRIMARY_BETWEEN_LOGICAL_PARTITIONS",
  "STATUS_VOLMGR_DISK_LAYOUT_TOO_MANY_PARTITIONS",
  "STATUS_VOLMGR_DISK_MISSING",
  "STATUS_VOLMGR_DISK_NOT_EMPTY",
  "STATUS_VOLMGR_DISK_NOT_ENOUGH_SPACE",
  "STATUS_VOLMGR_DISK_REVECTORING_FAILED",
  "STATUS_VOLMGR_DISK_SECTOR_SIZE_INVALID",
  "STATUS_VOLMGR_DISK_SET_NOT_CONTAINED",
  "STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_MEMBERS",
  "STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_PLEXES",
  "STATUS_VOLMGR_DYNAMIC_DISK_NOT_SUPPORTED",
  "STATUS_VOLMGR_EXTENT_ALREADY_USED",
  "STATUS_VOLMGR_EXTENT_NOT_CONTIGUOUS",
  "STATUS_VOLMGR_EXTENT_NOT_IN_PUBLIC_REGION",
  "STATUS_VOLMGR_EXTENT_NOT_SECTOR_ALIGNED",
  "STATUS_VOLMGR_EXTENT_OVERLAPS_EBR_PARTITION",
  "STATUS_VOLMGR_EXTENT_VOLUME_LENGTHS_DO_NOT_MATCH",
  "STATUS_VOLMGR_FAULT_TOLERANT_NOT_SUPPORTED",
  "STATUS_VOLMGR_INTERLEAVE_LENGTH_INVALID",
  "STATUS_VOLMGR_MAXIMUM_REGISTERED_USERS",
  "STATUS_VOLMGR_MEMBER_IN_SYNC",
  "STATUS_VOLMGR_MEMBER_INDEX_DUPLICATE",
  "STATUS_VOLMGR_MEMBER_INDEX_INVALID",
  "STATUS_VOLMGR_MEMBER_MISSING",
  "STATUS_VOLMGR_MEMBER_NOT_DETACHED",
  "STATUS_VOLMGR_MEMBER_REGENERATING",
  "STATUS_VOLMGR_ALL_DISKS_FAILED",
  "STATUS_VOLMGR_NO_REGISTERED_USERS",
  "STATUS_VOLMGR_NO_SUCH_USER",
  "STATUS_VOLMGR_NOTIFICATION_RESET",
  "STATUS_VOLMGR_NUMBER_OF_MEMBERS_INVALID",
  "STATUS_VOLMGR_NUMBER_OF_PLEXES_INVALID",
  "STATUS_VOLMGR_PACK_DUPLICATE",
  "STATUS_VOLMGR_PACK_ID_INVALID",
  "STATUS_VOLMGR_PACK_INVALID",
  "STATUS_VOLMGR_PACK_NAME_INVALID",
  "STATUS_VOLMGR_PACK_OFFLINE",
  "STATUS_VOLMGR_PACK_HAS_QUORUM",
  "STATUS_VOLMGR_PACK_WITHOUT_QUORUM",
  "STATUS_VOLMGR_PARTITION_STYLE_INVALID",
  "STATUS_VOLMGR_PARTITION_UPDATE_FAILED",
  "STATUS_VOLMGR_PLEX_IN_SYNC",
  "STATUS_VOLMGR_PLEX_INDEX_DUPLICATE",
  "STATUS_VOLMGR_PLEX_INDEX_INVALID",
  "STATUS_VOLMGR_PLEX_LAST_ACTIVE",
  "STATUS_VOLMGR_PLEX_MISSING",
  "STATUS_VOLMGR_PLEX_REGENERATING",
  "STATUS_VOLMGR_PLEX_TYPE_INVALID",
  "STATUS_VOLMGR_PLEX_NOT_RAID5",
  "STATUS_VOLMGR_PLEX_NOT_SIMPLE",
  "STATUS_VOLMGR_STRUCTURE_SIZE_INVALID",
  "STATUS_VOLMGR_TOO_MANY_NOTIFICATION_REQUESTS",
  "STATUS_VOLMGR_TRANSACTION_IN_PROGRESS",
  "STATUS_VOLMGR_UNEXPECTED_DISK_LAYOUT_CHANGE",
  "STATUS_VOLMGR_VOLUME_CONTAINS_MISSING_DISK",
  "STATUS_VOLMGR_VOLUME_ID_INVALID",
  "STATUS_VOLMGR_VOLUME_LENGTH_INVALID",
  "STATUS_VOLMGR_VOLUME_LENGTH_NOT_SECTOR_SIZE_MULTIPLE",
  "STATUS_VOLMGR_VOLUME_NOT_MIRRORED",
  "STATUS_VOLMGR_VOLUME_NOT_RETAINED",
  "STATUS_VOLMGR_VOLUME_OFFLINE",
  "STATUS_VOLMGR_VOLUME_RETAINED",
  "STATUS_VOLMGR_NUMBER_OF_EXTENTS_INVALID",
  "STATUS_VOLMGR_DIFFERENT_SECTOR_SIZE",
  "STATUS_VOLMGR_BAD_BOOT_DISK",
  "STATUS_VOLMGR_PACK_CONFIG_OFFLINE",
  "STATUS_VOLMGR_PACK_CONFIG_ONLINE",
  "STATUS_VOLMGR_NOT_PRIMARY_PACK",
  "STATUS_VOLMGR_PACK_LOG_UPDATE_FAILED",
  "STATUS_VOLMGR_NUMBER_OF_DISKS_IN_PLEX_INVALID",
  "STATUS_VOLMGR_NUMBER_OF_DISKS_IN_MEMBER_INVALID",
  "STATUS_VOLMGR_VOLUME_MIRRORED",
  "STATUS_VOLMGR_PLEX_NOT_SIMPLE_SPANNED",
  "STATUS_VOLMGR_NO_VALID_LOG_COPIES",
  "STATUS_VOLMGR_PRIMARY_PACK_PRESENT",
  "STATUS_VOLMGR_NUMBER_OF_DISKS_INVALID",
  "STATUS_VOLMGR_MIRROR_NOT_SUPPORTED",
  "STATUS_VOLMGR_RAID5_NOT_SUPPORTED",
  "STATUS_BCD_TOO_MANY_ELEMENTS",
  "STATUS_VHD_DRIVE_FOOTER_MISSING",
  "STATUS_VHD_DRIVE_FOOTER_CHECKSUM_MISMATCH",
  "STATUS_VHD_DRIVE_FOOTER_CORRUPT",
  "STATUS_VHD_FORMAT_UNKNOWN",
  "STATUS_VHD_FORMAT_UNSUPPORTED_VERSION",
  "STATUS_VHD_SPARSE_HEADER_CHECKSUM_MISMATCH",
  "STATUS_VHD_SPARSE_HEADER_UNSUPPORTED_VERSION",
  "STATUS_VHD_SPARSE_HEADER_CORRUPT",
  "STATUS_VHD_BLOCK_ALLOCATION_FAILURE",
  "STATUS_VHD_BLOCK_ALLOCATION_TABLE_CORRUPT",
  "STATUS_VHD_INVALID_BLOCK_SIZE",
  "STATUS_VHD_BITMAP_MISMATCH",
  "STATUS_VHD_PARENT_VHD_NOT_FOUND",
  "STATUS_VHD_CHILD_PARENT_ID_MISMATCH",
  "STATUS_VHD_CHILD_PARENT_TIMESTAMP_MISMATCH",
  "STATUS_VHD_METADATA_READ_FAILURE",
  "STATUS_VHD_METADATA_WRITE_FAILURE",
  "STATUS_VHD_INVALID_SIZE",
  "STATUS_VHD_INVALID_FILE_SIZE",
  "STATUS_VIRTDISK_PROVIDER_NOT_FOUND",
  "STATUS_VIRTDISK_NOT_VIRTUAL_DISK",
  "STATUS_VHD_PARENT_VHD_ACCESS_DENIED",
  "STATUS_VHD_CHILD_PARENT_SIZE_MISMATCH",
  "STATUS_VHD_DIFFERENCING_CHAIN_CYCLE_DETECTED",
  "STATUS_VHD_DIFFERENCING_CHAIN_ERROR_IN_PARENT",
  "STATUS_VIRTUAL_DISK_LIMITATION",
  "STATUS_VHD_INVALID_TYPE",
  "STATUS_VHD_INVALID_STATE",
  "STATUS_VIRTDISK_UNSUPPORTED_DISK_SECTOR_SIZE",
  "STATUS_VIRTDISK_DISK_ALREADY_OWNED",
  "STATUS_VIRTDISK_DISK_ONLINE_AND_WRITABLE",
  "STATUS_CTLOG_TRACKING_NOT_INITIALIZED",
  "STATUS_CTLOG_LOGFILE_SIZE_EXCEEDED_MAXSIZE",
  "STATUS_CTLOG_VHD_CHANGED_OFFLINE",
  "STATUS_CTLOG_INVALID_TRACKING_STATE",
  "STATUS_CTLOG_INCONSISTENT_TRACKING_FILE",
  "STATUS_VHD_METADATA_FULL",
  "STATUS_VHD_INVALID_CHANGE_TRACKING_ID",
  "STATUS_VHD_CHANGE_TRACKING_DISABLED",
  "STATUS_VHD_MISSING_CHANGE_TRACKING_INFORMATION",
  "STATUS_VHD_RESIZE_WOULD_TRUNCATE_DATA",
  "STATUS_VHD_COULD_NOT_COMPUTE_MINIMUM_VIRTUAL_SIZE",
  "STATUS_VHD_ALREADY_AT_OR_BELOW_MINIMUM_VIRTUAL_SIZE",
  "STATUS_RKF_KEY_NOT_FOUND",
  "STATUS_RKF_DUPLICATE_KEY",
  "STATUS_RKF_BLOB_FULL",
  "STATUS_RKF_STORE_FULL",
  "STATUS_RKF_FILE_BLOCKED",
  "STATUS_RKF_ACTIVE_KEY",
  "STATUS_RDBSS_RESTART_OPERATION",
  "STATUS_RDBSS_CONTINUE_OPERATION",
  "STATUS_RDBSS_POST_OPERATION",
  "STATUS_RDBSS_RETRY_LOOKUP",
  "STATUS_BTH_ATT_INVALID_HANDLE",
  "STATUS_BTH_ATT_READ_NOT_PERMITTED",
  "STATUS_BTH_ATT_WRITE_NOT_PERMITTED",
  "STATUS_BTH_ATT_INVALID_PDU",
  "STATUS_BTH_ATT_INSUFFICIENT_AUTHENTICATION",
  "STATUS_BTH_ATT_REQUEST_NOT_SUPPORTED",
  "STATUS_BTH_ATT_INVALID_OFFSET",
  "STATUS_BTH_ATT_INSUFFICIENT_AUTHORIZATION",
  "STATUS_BTH_ATT_PREPARE_QUEUE_FULL",
  "STATUS_BTH_ATT_ATTRIBUTE_NOT_FOUND",
  "STATUS_BTH_ATT_ATTRIBUTE_NOT_LONG",
  "STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION_KEY_SIZE",
  "STATUS_BTH_ATT_INVALID_ATTRIBUTE_VALUE_LENGTH",
  "STATUS_BTH_ATT_UNLIKELY",
  "STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION",
  "STATUS_BTH_ATT_UNSUPPORTED_GROUP_TYPE",
  "STATUS_BTH_ATT_INSUFFICIENT_RESOURCES",
  "STATUS_BTH_ATT_UNKNOWN_ERROR",
  "STATUS_SECUREBOOT_ROLLBACK_DETECTED",
  "STATUS_SECUREBOOT_POLICY_VIOLATION",
  "STATUS_SECUREBOOT_INVALID_POLICY",
  "STATUS_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND",
  "STATUS_SECUREBOOT_POLICY_NOT_SIGNED",
  "STATUS_SECUREBOOT_FILE_REPLACED",
  "STATUS_SECUREBOOT_POLICY_NOT_AUTHORIZED",
  "STATUS_SECUREBOOT_POLICY_UNKNOWN",
  "STATUS_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION",
  "STATUS_SECUREBOOT_PLATFORM_ID_MISMATCH",
  "STATUS_SECUREBOOT_POLICY_ROLLBACK_DETECTED",
  "STATUS_SECUREBOOT_POLICY_UPGRADE_MISMATCH",
  "STATUS_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING",
  "STATUS_SECUREBOOT_NOT_BASE_POLICY",
  "STATUS_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY",
  "STATUS_AUDIO_ENGINE_NODE_NOT_FOUND",
  "STATUS_HDAUDIO_EMPTY_CONNECTION_LIST",
  "STATUS_HDAUDIO_CONNECTION_LIST_NOT_SUPPORTED",
  "STATUS_HDAUDIO_NO_LOGICAL_DEVICES_CREATED",
  "STATUS_HDAUDIO_NULL_LINKED_LIST_ENTRY",
  "STATUS_VSM_NOT_INITIALIZED",
  "STATUS_VSM_DMA_PROTECTION_NOT_IN_USE",
  "STATUS_VOLSNAP_BOOTFILE_NOT_VALID",
  "STATUS_VOLSNAP_ACTIVATION_TIMEOUT",
  "STATUS_IO_PREEMPTED",
  "STATUS_SVHDX_ERROR_STORED",
  "STATUS_SVHDX_ERROR_NOT_AVAILABLE",
  "STATUS_SVHDX_UNIT_ATTENTION_AVAILABLE",
  "STATUS_SVHDX_UNIT_ATTENTION_CAPACITY_DATA_CHANGED",
  "STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_PREEMPTED",
  "STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_RELEASED",
  "STATUS_SVHDX_UNIT_ATTENTION_REGISTRATIONS_PREEMPTED",
  "STATUS_SVHDX_UNIT_ATTENTION_OPERATING_DEFINITION_CHANGED",
  "STATUS_SVHDX_RESERVATION_CONFLICT",
  "STATUS_SVHDX_WRONG_FILE_TYPE",
  "STATUS_SVHDX_VERSION_MISMATCH",
  "STATUS_VHD_SHARED",
  "STATUS_SVHDX_NO_INITIATOR",
  "STATUS_VHDSET_BACKING_STORAGE_NOT_FOUND",
  "STATUS_SMB_NO_PREAUTH_INTEGRITY_HASH_OVERLAP",
  "STATUS_SMB_BAD_CLUSTER_DIALECT",
  "STATUS_SMB_GUEST_LOGON_BLOCKED",
  "STATUS_SPACES_FAULT_DOMAIN_TYPE_INVALID",
  "STATUS_SPACES_RESILIENCY_TYPE_INVALID",
  "STATUS_SPACES_DRIVE_SECTOR_SIZE_INVALID",
  "STATUS_SPACES_DRIVE_REDUNDANCY_INVALID",
  "STATUS_SPACES_NUMBER_OF_DATA_COPIES_INVALID",
  "STATUS_SPACES_INTERLEAVE_LENGTH_INVALID",
  "STATUS_SPACES_NUMBER_OF_COLUMNS_INVALID",
  "STATUS_SPACES_NOT_ENOUGH_DRIVES",
  "STATUS_SPACES_EXTENDED_ERROR",
  "STATUS_SPACES_PROVISIONING_TYPE_INVALID",
  "STATUS_SPACES_ALLOCATION_SIZE_INVALID",
  "STATUS_SPACES_ENCLOSURE_AWARE_INVALID",
  "STATUS_SPACES_WRITE_CACHE_SIZE_INVALID",
  "STATUS_SPACES_NUMBER_OF_GROUPS_INVALID",
  "STATUS_SPACES_DRIVE_OPERATIONAL_STATE_INVALID",
  "STATUS_SPACES_UPDATE_COLUMN_STATE",
  "STATUS_SPACES_MAP_REQUIRED",
  "STATUS_SPACES_UNSUPPORTED_VERSION",
  "STATUS_SPACES_CORRUPT_METADATA",
  "STATUS_SPACES_DRT_FULL",
  "STATUS_SPACES_INCONSISTENCY",
  "STATUS_SPACES_LOG_NOT_READY",
  "STATUS_SPACES_NO_REDUNDANCY",
  "STATUS_SPACES_DRIVE_NOT_READY",
  "STATUS_SPACES_DRIVE_SPLIT",
  "STATUS_SPACES_DRIVE_LOST_DATA",
  "STATUS_SPACES_ENTRY_INCOMPLETE",
  "STATUS_SPACES_ENTRY_INVALID",
  "STATUS_SPACES_MARK_DIRTY",
  "STATUS_SECCORE_INVALID_COMMAND",
  "STATUS_SYSTEM_INTEGRITY_ROLLBACK_DETECTED",
  "STATUS_SYSTEM_INTEGRITY_POLICY_VIOLATION",
  "STATUS_SYSTEM_INTEGRITY_INVALID_POLICY",
  "STATUS_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED",
  "STATUS_SYSTEM_INTEGRITY_TOO_MANY_POLICIES",
  "STATUS_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED",
  "STATUS_NO_APPLICABLE_APP_LICENSES_FOUND",
  "STATUS_CLIP_LICENSE_NOT_FOUND",
  "STATUS_CLIP_DEVICE_LICENSE_MISSING",
  "STATUS_CLIP_LICENSE_INVALID_SIGNATURE",
  "STATUS_CLIP_KEYHOLDER_LICENSE_MISSING_OR_INVALID",
  "STATUS_CLIP_LICENSE_EXPIRED",
  "STATUS_CLIP_LICENSE_SIGNED_BY_UNKNOWN_SOURCE",
  "STATUS_CLIP_LICENSE_NOT_SIGNED",
  "STATUS_CLIP_LICENSE_HARDWARE_ID_OUT_OF_TOLERANCE",
  "STATUS_CLIP_LICENSE_DEVICE_ID_MISMATCH",
  "STATUS_PLATFORM_MANIFEST_NOT_AUTHORIZED",
  "STATUS_PLATFORM_MANIFEST_INVALID",
  "STATUS_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED",
  "STATUS_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED",
  "STATUS_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND",
  "STATUS_PLATFORM_MANIFEST_NOT_ACTIVE",
  "STATUS_PLATFORM_MANIFEST_NOT_SIGNED",
  "STATUS_APPEXEC_CONDITION_NOT_SATISFIED",
  "STATUS_APPEXEC_HANDLE_INVALIDATED",
  "STATUS_APPEXEC_INVALID_HOST_GENERATION",
  "STATUS_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION",
  "STATUS_APPEXEC_INVALID_HOST_STATE",
  "STATUS_APPEXEC_NO_DONOR",
  "STATUS_APPEXEC_HOST_ID_MISMATCH",
  "STATUS_APPEXEC_UNKNOWN_USER",
};

const char* NTStatusToString(uint32_t ntstatus) {
  auto itr = std::find(std::begin(kNTStatusValues),
                       std::end(kNTStatusValues),
                       ntstatus);
  if (itr == std::end(kNTStatusValues)) {
    return nullptr;
  }

  return kNTStatusStrings[itr - std::begin(kNTStatusValues)];
}

// The content of this array was created from winnt.h in the 10 SDK
// (version 10.0.19041.0) with the following script:
//
// egrep '#define FAST_FAIL_[A-Z_0-9]+\s+[0-9]' winnt.h
// | tr -d '\r'
// | sed -r 's@#define FAST_FAIL_([A-Z_0-9]+)\s+([0-9]+).*@\2 \1@'
// | sed -r 's@([0-9]+) ([A-Z_0-9]+)@  MD_FAST_FAIL_WIN_\2,@'
// | head -n -1
static const uint8_t kFastFailValues[] = {
  MD_FAST_FAIL_WIN_LEGACY_GS_VIOLATION,
  MD_FAST_FAIL_WIN_VTGUARD_CHECK_FAILURE,
  MD_FAST_FAIL_WIN_STACK_COOKIE_CHECK_FAILURE,
  MD_FAST_FAIL_WIN_CORRUPT_LIST_ENTRY,
  MD_FAST_FAIL_WIN_INCORRECT_STACK,
  MD_FAST_FAIL_WIN_INVALID_ARG,
  MD_FAST_FAIL_WIN_GS_COOKIE_INIT,
  MD_FAST_FAIL_WIN_FATAL_APP_EXIT,
  MD_FAST_FAIL_WIN_RANGE_CHECK_FAILURE,
  MD_FAST_FAIL_WIN_UNSAFE_REGISTRY_ACCESS,
  MD_FAST_FAIL_WIN_GUARD_ICALL_CHECK_FAILURE,
  MD_FAST_FAIL_WIN_GUARD_WRITE_CHECK_FAILURE,
  MD_FAST_FAIL_WIN_INVALID_FIBER_SWITCH,
  MD_FAST_FAIL_WIN_INVALID_SET_OF_CONTEXT,
  MD_FAST_FAIL_WIN_INVALID_REFERENCE_COUNT,
  MD_FAST_FAIL_WIN_INVALID_JUMP_BUFFER,
  MD_FAST_FAIL_WIN_MRDATA_MODIFIED,
  MD_FAST_FAIL_WIN_CERTIFICATION_FAILURE,
  MD_FAST_FAIL_WIN_INVALID_EXCEPTION_CHAIN,
  MD_FAST_FAIL_WIN_CRYPTO_LIBRARY,
  MD_FAST_FAIL_WIN_INVALID_CALL_IN_DLL_CALLOUT,
  MD_FAST_FAIL_WIN_INVALID_IMAGE_BASE,
  MD_FAST_FAIL_WIN_DLOAD_PROTECTION_FAILURE,
  MD_FAST_FAIL_WIN_UNSAFE_EXTENSION_CALL,
  MD_FAST_FAIL_WIN_DEPRECATED_SERVICE_INVOKED,
  MD_FAST_FAIL_WIN_INVALID_BUFFER_ACCESS,
  MD_FAST_FAIL_WIN_INVALID_BALANCED_TREE,
  MD_FAST_FAIL_WIN_INVALID_NEXT_THREAD,
  MD_FAST_FAIL_WIN_GUARD_ICALL_CHECK_SUPPRESSED,
  MD_FAST_FAIL_WIN_APCS_DISABLED,
  MD_FAST_FAIL_WIN_INVALID_IDLE_STATE,
  MD_FAST_FAIL_WIN_MRDATA_PROTECTION_FAILURE,
  MD_FAST_FAIL_WIN_UNEXPECTED_HEAP_EXCEPTION,
  MD_FAST_FAIL_WIN_INVALID_LOCK_STATE,
  MD_FAST_FAIL_WIN_GUARD_JUMPTABLE,
  MD_FAST_FAIL_WIN_INVALID_LONGJUMP_TARGET,
  MD_FAST_FAIL_WIN_INVALID_DISPATCH_CONTEXT,
  MD_FAST_FAIL_WIN_INVALID_THREAD,
  MD_FAST_FAIL_WIN_INVALID_SYSCALL_NUMBER,
  MD_FAST_FAIL_WIN_INVALID_FILE_OPERATION,
  MD_FAST_FAIL_WIN_LPAC_ACCESS_DENIED,
  MD_FAST_FAIL_WIN_GUARD_SS_FAILURE,
  MD_FAST_FAIL_WIN_LOADER_CONTINUITY_FAILURE,
  MD_FAST_FAIL_WIN_GUARD_EXPORT_SUPPRESSION_FAILURE,
  MD_FAST_FAIL_WIN_INVALID_CONTROL_STACK,
  MD_FAST_FAIL_WIN_SET_CONTEXT_DENIED,
  MD_FAST_FAIL_WIN_INVALID_IAT,
  MD_FAST_FAIL_WIN_HEAP_METADATA_CORRUPTION,
  MD_FAST_FAIL_WIN_PAYLOAD_RESTRICTION_VIOLATION,
  MD_FAST_FAIL_WIN_LOW_LABEL_ACCESS_DENIED,
  MD_FAST_FAIL_WIN_ENCLAVE_CALL_FAILURE,
  MD_FAST_FAIL_WIN_UNHANDLED_LSS_EXCEPTON,
  MD_FAST_FAIL_WIN_ADMINLESS_ACCESS_DENIED,
  MD_FAST_FAIL_WIN_UNEXPECTED_CALL,
  MD_FAST_FAIL_WIN_CONTROL_INVALID_RETURN_ADDRESS,
  MD_FAST_FAIL_WIN_UNEXPECTED_HOST_BEHAVIOR,
  MD_FAST_FAIL_WIN_FLAGS_CORRUPTION,
  MD_FAST_FAIL_WIN_VEH_CORRUPTION,
  MD_FAST_FAIL_WIN_ETW_CORRUPTION,
  MD_FAST_FAIL_WIN_RIO_ABORT,
  MD_FAST_FAIL_WIN_INVALID_PFN,
};

// The content of this array was created from winnt.h in the 10 SDK
// (version 10.0.19041.0) with the following script:
//
// egrep '#define FAST_FAIL_[A-Z_0-9]+\s+[0-9]' winnt.h
// | tr -d '\r'
// | sed -r 's@#define FAST_FAIL_([A-Z_0-9]+)\s+([0-9]+).*@\2 \1@'
// | sed -r 's@([0-9]+) ([A-Z_0-9]+)@  "FAST_FAIL_\2",@'
// | head -n -1
static const char* kFastFailStrings[] = {
  "FAST_FAIL_LEGACY_GS_VIOLATION",
  "FAST_FAIL_VTGUARD_CHECK_FAILURE",
  "FAST_FAIL_STACK_COOKIE_CHECK_FAILURE",
  "FAST_FAIL_CORRUPT_LIST_ENTRY",
  "FAST_FAIL_INCORRECT_STACK",
  "FAST_FAIL_INVALID_ARG",
  "FAST_FAIL_GS_COOKIE_INIT",
  "FAST_FAIL_FATAL_APP_EXIT",
  "FAST_FAIL_RANGE_CHECK_FAILURE",
  "FAST_FAIL_UNSAFE_REGISTRY_ACCESS",
  "FAST_FAIL_GUARD_ICALL_CHECK_FAILURE",
  "FAST_FAIL_GUARD_WRITE_CHECK_FAILURE",
  "FAST_FAIL_INVALID_FIBER_SWITCH",
  "FAST_FAIL_INVALID_SET_OF_CONTEXT",
  "FAST_FAIL_INVALID_REFERENCE_COUNT",
  "FAST_FAIL_INVALID_JUMP_BUFFER",
  "FAST_FAIL_MRDATA_MODIFIED",
  "FAST_FAIL_CERTIFICATION_FAILURE",
  "FAST_FAIL_INVALID_EXCEPTION_CHAIN",
  "FAST_FAIL_CRYPTO_LIBRARY",
  "FAST_FAIL_INVALID_CALL_IN_DLL_CALLOUT",
  "FAST_FAIL_INVALID_IMAGE_BASE",
  "FAST_FAIL_DLOAD_PROTECTION_FAILURE",
  "FAST_FAIL_UNSAFE_EXTENSION_CALL",
  "FAST_FAIL_DEPRECATED_SERVICE_INVOKED",
  "FAST_FAIL_INVALID_BUFFER_ACCESS",
  "FAST_FAIL_INVALID_BALANCED_TREE",
  "FAST_FAIL_INVALID_NEXT_THREAD",
  "FAST_FAIL_GUARD_ICALL_CHECK_SUPPRESSED",
  "FAST_FAIL_APCS_DISABLED",
  "FAST_FAIL_INVALID_IDLE_STATE",
  "FAST_FAIL_MRDATA_PROTECTION_FAILURE",
  "FAST_FAIL_UNEXPECTED_HEAP_EXCEPTION",
  "FAST_FAIL_INVALID_LOCK_STATE",
  "FAST_FAIL_GUARD_JUMPTABLE",
  "FAST_FAIL_INVALID_LONGJUMP_TARGET",
  "FAST_FAIL_INVALID_DISPATCH_CONTEXT",
  "FAST_FAIL_INVALID_THREAD",
  "FAST_FAIL_INVALID_SYSCALL_NUMBER",
  "FAST_FAIL_INVALID_FILE_OPERATION",
  "FAST_FAIL_LPAC_ACCESS_DENIED",
  "FAST_FAIL_GUARD_SS_FAILURE",
  "FAST_FAIL_LOADER_CONTINUITY_FAILURE",
  "FAST_FAIL_GUARD_EXPORT_SUPPRESSION_FAILURE",
  "FAST_FAIL_INVALID_CONTROL_STACK",
  "FAST_FAIL_SET_CONTEXT_DENIED",
  "FAST_FAIL_INVALID_IAT",
  "FAST_FAIL_HEAP_METADATA_CORRUPTION",
  "FAST_FAIL_PAYLOAD_RESTRICTION_VIOLATION",
  "FAST_FAIL_LOW_LABEL_ACCESS_DENIED",
  "FAST_FAIL_ENCLAVE_CALL_FAILURE",
  "FAST_FAIL_UNHANDLED_LSS_EXCEPTON",
  "FAST_FAIL_ADMINLESS_ACCESS_DENIED",
  "FAST_FAIL_UNEXPECTED_CALL",
  "FAST_FAIL_CONTROL_INVALID_RETURN_ADDRESS",
  "FAST_FAIL_UNEXPECTED_HOST_BEHAVIOR",
  "FAST_FAIL_FLAGS_CORRUPTION",
  "FAST_FAIL_VEH_CORRUPTION",
  "FAST_FAIL_ETW_CORRUPTION",
  "FAST_FAIL_RIO_ABORT",
  "FAST_FAIL_INVALID_PFN",
};

const char* FastFailToString(uint32_t fast_fail_code) {
  auto itr = std::find(std::begin(kFastFailValues),
                       std::end(kFastFailValues),
                       fast_fail_code);
  if (itr == std::end(kFastFailValues)) {
    return nullptr;
  }

  return kFastFailStrings[itr - std::begin(kFastFailValues)];
}

// The content of this array was created from winerror.h in the 10 SDK
// (version 10.0.19041.0) with the following script:
//
// egrep -o '#define ERROR_[A-Z_0-9]+\s+[0-9]+L' winerror.h
// | tr -d '\r'
// | sed -r 's@#define ERROR_([A-Z_0-9]+)\s+([0-9]+)L@\2 \1@'
// | sort -n
// | sed -r 's@([0-9]+) ([A-Z_0-9]+)@  MD_ERROR_WIN_\2,@'
static const uint16_t kWinErrorValues[] = {
  MD_ERROR_WIN_SUCCESS,
  MD_ERROR_WIN_INVALID_FUNCTION,
  MD_ERROR_WIN_FILE_NOT_FOUND,
  MD_ERROR_WIN_PATH_NOT_FOUND,
  MD_ERROR_WIN_TOO_MANY_OPEN_FILES,
  MD_ERROR_WIN_ACCESS_DENIED,
  MD_ERROR_WIN_INVALID_HANDLE,
  MD_ERROR_WIN_ARENA_TRASHED,
  MD_ERROR_WIN_NOT_ENOUGH_MEMORY,
  MD_ERROR_WIN_INVALID_BLOCK,
  MD_ERROR_WIN_BAD_ENVIRONMENT,
  MD_ERROR_WIN_BAD_FORMAT,
  MD_ERROR_WIN_INVALID_ACCESS,
  MD_ERROR_WIN_INVALID_DATA,
  MD_ERROR_WIN_OUTOFMEMORY,
  MD_ERROR_WIN_INVALID_DRIVE,
  MD_ERROR_WIN_CURRENT_DIRECTORY,
  MD_ERROR_WIN_NOT_SAME_DEVICE,
  MD_ERROR_WIN_NO_MORE_FILES,
  MD_ERROR_WIN_WRITE_PROTECT,
  MD_ERROR_WIN_BAD_UNIT,
  MD_ERROR_WIN_NOT_READY,
  MD_ERROR_WIN_BAD_COMMAND,
  MD_ERROR_WIN_CRC,
  MD_ERROR_WIN_BAD_LENGTH,
  MD_ERROR_WIN_SEEK,
  MD_ERROR_WIN_NOT_DOS_DISK,
  MD_ERROR_WIN_SECTOR_NOT_FOUND,
  MD_ERROR_WIN_OUT_OF_PAPER,
  MD_ERROR_WIN_WRITE_FAULT,
  MD_ERROR_WIN_READ_FAULT,
  MD_ERROR_WIN_GEN_FAILURE,
  MD_ERROR_WIN_SHARING_VIOLATION,
  MD_ERROR_WIN_LOCK_VIOLATION,
  MD_ERROR_WIN_WRONG_DISK,
  MD_ERROR_WIN_SHARING_BUFFER_EXCEEDED,
  MD_ERROR_WIN_HANDLE_EOF,
  MD_ERROR_WIN_HANDLE_DISK_FULL,
  MD_ERROR_WIN_NOT_SUPPORTED,
  MD_ERROR_WIN_REM_NOT_LIST,
  MD_ERROR_WIN_DUP_NAME,
  MD_ERROR_WIN_BAD_NETPATH,
  MD_ERROR_WIN_NETWORK_BUSY,
  MD_ERROR_WIN_DEV_NOT_EXIST,
  MD_ERROR_WIN_TOO_MANY_CMDS,
  MD_ERROR_WIN_ADAP_HDW_ERR,
  MD_ERROR_WIN_BAD_NET_RESP,
  MD_ERROR_WIN_UNEXP_NET_ERR,
  MD_ERROR_WIN_BAD_REM_ADAP,
  MD_ERROR_WIN_PRINTQ_FULL,
  MD_ERROR_WIN_NO_SPOOL_SPACE,
  MD_ERROR_WIN_PRINT_CANCELLED,
  MD_ERROR_WIN_NETNAME_DELETED,
  MD_ERROR_WIN_NETWORK_ACCESS_DENIED,
  MD_ERROR_WIN_BAD_DEV_TYPE,
  MD_ERROR_WIN_BAD_NET_NAME,
  MD_ERROR_WIN_TOO_MANY_NAMES,
  MD_ERROR_WIN_TOO_MANY_SESS,
  MD_ERROR_WIN_SHARING_PAUSED,
  MD_ERROR_WIN_REQ_NOT_ACCEP,
  MD_ERROR_WIN_REDIR_PAUSED,
  MD_ERROR_WIN_FILE_EXISTS,
  MD_ERROR_WIN_CANNOT_MAKE,
  MD_ERROR_WIN_FAIL_I24,
  MD_ERROR_WIN_OUT_OF_STRUCTURES,
  MD_ERROR_WIN_ALREADY_ASSIGNED,
  MD_ERROR_WIN_INVALID_PASSWORD,
  MD_ERROR_WIN_INVALID_PARAMETER,
  MD_ERROR_WIN_NET_WRITE_FAULT,
  MD_ERROR_WIN_NO_PROC_SLOTS,
  MD_ERROR_WIN_TOO_MANY_SEMAPHORES,
  MD_ERROR_WIN_EXCL_SEM_ALREADY_OWNED,
  MD_ERROR_WIN_SEM_IS_SET,
  MD_ERROR_WIN_TOO_MANY_SEM_REQUESTS,
  MD_ERROR_WIN_INVALID_AT_INTERRUPT_TIME,
  MD_ERROR_WIN_SEM_OWNER_DIED,
  MD_ERROR_WIN_SEM_USER_LIMIT,
  MD_ERROR_WIN_DISK_CHANGE,
  MD_ERROR_WIN_DRIVE_LOCKED,
  MD_ERROR_WIN_BROKEN_PIPE,
  MD_ERROR_WIN_OPEN_FAILED,
  MD_ERROR_WIN_BUFFER_OVERFLOW,
  MD_ERROR_WIN_DISK_FULL,
  MD_ERROR_WIN_NO_MORE_SEARCH_HANDLES,
  MD_ERROR_WIN_INVALID_TARGET_HANDLE,
  MD_ERROR_WIN_INVALID_CATEGORY,
  MD_ERROR_WIN_INVALID_VERIFY_SWITCH,
  MD_ERROR_WIN_BAD_DRIVER_LEVEL,
  MD_ERROR_WIN_CALL_NOT_IMPLEMENTED,
  MD_ERROR_WIN_SEM_TIMEOUT,
  MD_ERROR_WIN_INSUFFICIENT_BUFFER,
  MD_ERROR_WIN_INVALID_NAME,
  MD_ERROR_WIN_INVALID_LEVEL,
  MD_ERROR_WIN_NO_VOLUME_LABEL,
  MD_ERROR_WIN_MOD_NOT_FOUND,
  MD_ERROR_WIN_PROC_NOT_FOUND,
  MD_ERROR_WIN_WAIT_NO_CHILDREN,
  MD_ERROR_WIN_CHILD_NOT_COMPLETE,
  MD_ERROR_WIN_DIRECT_ACCESS_HANDLE,
  MD_ERROR_WIN_NEGATIVE_SEEK,
  MD_ERROR_WIN_SEEK_ON_DEVICE,
  MD_ERROR_WIN_IS_JOIN_TARGET,
  MD_ERROR_WIN_IS_JOINED,
  MD_ERROR_WIN_IS_SUBSTED,
  MD_ERROR_WIN_NOT_JOINED,
  MD_ERROR_WIN_NOT_SUBSTED,
  MD_ERROR_WIN_JOIN_TO_JOIN,
  MD_ERROR_WIN_SUBST_TO_SUBST,
  MD_ERROR_WIN_JOIN_TO_SUBST,
  MD_ERROR_WIN_SUBST_TO_JOIN,
  MD_ERROR_WIN_BUSY_DRIVE,
  MD_ERROR_WIN_SAME_DRIVE,
  MD_ERROR_WIN_DIR_NOT_ROOT,
  MD_ERROR_WIN_DIR_NOT_EMPTY,
  MD_ERROR_WIN_IS_SUBST_PATH,
  MD_ERROR_WIN_IS_JOIN_PATH,
  MD_ERROR_WIN_PATH_BUSY,
  MD_ERROR_WIN_IS_SUBST_TARGET,
  MD_ERROR_WIN_SYSTEM_TRACE,
  MD_ERROR_WIN_INVALID_EVENT_COUNT,
  MD_ERROR_WIN_TOO_MANY_MUXWAITERS,
  MD_ERROR_WIN_INVALID_LIST_FORMAT,
  MD_ERROR_WIN_LABEL_TOO_LONG,
  MD_ERROR_WIN_TOO_MANY_TCBS,
  MD_ERROR_WIN_SIGNAL_REFUSED,
  MD_ERROR_WIN_DISCARDED,
  MD_ERROR_WIN_NOT_LOCKED,
  MD_ERROR_WIN_BAD_THREADID_ADDR,
  MD_ERROR_WIN_BAD_ARGUMENTS,
  MD_ERROR_WIN_BAD_PATHNAME,
  MD_ERROR_WIN_SIGNAL_PENDING,
  MD_ERROR_WIN_MAX_THRDS_REACHED,
  MD_ERROR_WIN_LOCK_FAILED,
  MD_ERROR_WIN_BUSY,
  MD_ERROR_WIN_DEVICE_SUPPORT_IN_PROGRESS,
  MD_ERROR_WIN_CANCEL_VIOLATION,
  MD_ERROR_WIN_ATOMIC_LOCKS_NOT_SUPPORTED,
  MD_ERROR_WIN_INVALID_SEGMENT_NUMBER,
  MD_ERROR_WIN_INVALID_ORDINAL,
  MD_ERROR_WIN_ALREADY_EXISTS,
  MD_ERROR_WIN_INVALID_FLAG_NUMBER,
  MD_ERROR_WIN_SEM_NOT_FOUND,
  MD_ERROR_WIN_INVALID_STARTING_CODESEG,
  MD_ERROR_WIN_INVALID_STACKSEG,
  MD_ERROR_WIN_INVALID_MODULETYPE,
  MD_ERROR_WIN_INVALID_EXE_SIGNATURE,
  MD_ERROR_WIN_EXE_MARKED_INVALID,
  MD_ERROR_WIN_BAD_EXE_FORMAT,
  MD_ERROR_WIN_INVALID_MINALLOCSIZE,
  MD_ERROR_WIN_DYNLINK_FROM_INVALID_RING,
  MD_ERROR_WIN_IOPL_NOT_ENABLED,
  MD_ERROR_WIN_INVALID_SEGDPL,
  MD_ERROR_WIN_RING2SEG_MUST_BE_MOVABLE,
  MD_ERROR_WIN_RELOC_CHAIN_XEEDS_SEGLIM,
  MD_ERROR_WIN_INFLOOP_IN_RELOC_CHAIN,
  MD_ERROR_WIN_ENVVAR_NOT_FOUND,
  MD_ERROR_WIN_NO_SIGNAL_SENT,
  MD_ERROR_WIN_FILENAME_EXCED_RANGE,
  MD_ERROR_WIN_RING2_STACK_IN_USE,
  MD_ERROR_WIN_META_EXPANSION_TOO_LONG,
  MD_ERROR_WIN_INVALID_SIGNAL_NUMBER,
  MD_ERROR_WIN_THREAD_1_INACTIVE,
  MD_ERROR_WIN_LOCKED,
  MD_ERROR_WIN_TOO_MANY_MODULES,
  MD_ERROR_WIN_NESTING_NOT_ALLOWED,
  MD_ERROR_WIN_EXE_MACHINE_TYPE_MISMATCH,
  MD_ERROR_WIN_EXE_CANNOT_MODIFY_SIGNED_BINARY,
  MD_ERROR_WIN_EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY,
  MD_ERROR_WIN_FILE_CHECKED_OUT,
  MD_ERROR_WIN_CHECKOUT_REQUIRED,
  MD_ERROR_WIN_BAD_FILE_TYPE,
  MD_ERROR_WIN_FILE_TOO_LARGE,
  MD_ERROR_WIN_FORMS_AUTH_REQUIRED,
  MD_ERROR_WIN_VIRUS_INFECTED,
  MD_ERROR_WIN_VIRUS_DELETED,
  MD_ERROR_WIN_PIPE_LOCAL,
  MD_ERROR_WIN_BAD_PIPE,
  MD_ERROR_WIN_PIPE_BUSY,
  MD_ERROR_WIN_NO_DATA,
  MD_ERROR_WIN_PIPE_NOT_CONNECTED,
  MD_ERROR_WIN_MORE_DATA,
  MD_ERROR_WIN_NO_WORK_DONE,
  MD_ERROR_WIN_VC_DISCONNECTED,
  MD_ERROR_WIN_INVALID_EA_NAME,
  MD_ERROR_WIN_EA_LIST_INCONSISTENT,
  MD_ERROR_WIN_NO_MORE_ITEMS,
  MD_ERROR_WIN_CANNOT_COPY,
  MD_ERROR_WIN_DIRECTORY,
  MD_ERROR_WIN_EAS_DIDNT_FIT,
  MD_ERROR_WIN_EA_FILE_CORRUPT,
  MD_ERROR_WIN_EA_TABLE_FULL,
  MD_ERROR_WIN_INVALID_EA_HANDLE,
  MD_ERROR_WIN_EAS_NOT_SUPPORTED,
  MD_ERROR_WIN_NOT_OWNER,
  MD_ERROR_WIN_TOO_MANY_POSTS,
  MD_ERROR_WIN_PARTIAL_COPY,
  MD_ERROR_WIN_OPLOCK_NOT_GRANTED,
  MD_ERROR_WIN_INVALID_OPLOCK_PROTOCOL,
  MD_ERROR_WIN_DISK_TOO_FRAGMENTED,
  MD_ERROR_WIN_DELETE_PENDING,
  MD_ERROR_WIN_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING,
  MD_ERROR_WIN_SHORT_NAMES_NOT_ENABLED_ON_VOLUME,
  MD_ERROR_WIN_SECURITY_STREAM_IS_INCONSISTENT,
  MD_ERROR_WIN_INVALID_LOCK_RANGE,
  MD_ERROR_WIN_IMAGE_SUBSYSTEM_NOT_PRESENT,
  MD_ERROR_WIN_NOTIFICATION_GUID_ALREADY_DEFINED,
  MD_ERROR_WIN_INVALID_EXCEPTION_HANDLER,
  MD_ERROR_WIN_DUPLICATE_PRIVILEGES,
  MD_ERROR_WIN_NO_RANGES_PROCESSED,
  MD_ERROR_WIN_NOT_ALLOWED_ON_SYSTEM_FILE,
  MD_ERROR_WIN_DISK_RESOURCES_EXHAUSTED,
  MD_ERROR_WIN_INVALID_TOKEN,
  MD_ERROR_WIN_DEVICE_FEATURE_NOT_SUPPORTED,
  MD_ERROR_WIN_MR_MID_NOT_FOUND,
  MD_ERROR_WIN_SCOPE_NOT_FOUND,
  MD_ERROR_WIN_UNDEFINED_SCOPE,
  MD_ERROR_WIN_INVALID_CAP,
  MD_ERROR_WIN_DEVICE_UNREACHABLE,
  MD_ERROR_WIN_DEVICE_NO_RESOURCES,
  MD_ERROR_WIN_DATA_CHECKSUM_ERROR,
  MD_ERROR_WIN_INTERMIXED_KERNEL_EA_OPERATION,
  MD_ERROR_WIN_FILE_LEVEL_TRIM_NOT_SUPPORTED,
  MD_ERROR_WIN_OFFSET_ALIGNMENT_VIOLATION,
  MD_ERROR_WIN_INVALID_FIELD_IN_PARAMETER_LIST,
  MD_ERROR_WIN_OPERATION_IN_PROGRESS,
  MD_ERROR_WIN_BAD_DEVICE_PATH,
  MD_ERROR_WIN_TOO_MANY_DESCRIPTORS,
  MD_ERROR_WIN_SCRUB_DATA_DISABLED,
  MD_ERROR_WIN_NOT_REDUNDANT_STORAGE,
  MD_ERROR_WIN_RESIDENT_FILE_NOT_SUPPORTED,
  MD_ERROR_WIN_COMPRESSED_FILE_NOT_SUPPORTED,
  MD_ERROR_WIN_DIRECTORY_NOT_SUPPORTED,
  MD_ERROR_WIN_NOT_READ_FROM_COPY,
  MD_ERROR_WIN_FT_WRITE_FAILURE,
  MD_ERROR_WIN_FT_DI_SCAN_REQUIRED,
  MD_ERROR_WIN_INVALID_KERNEL_INFO_VERSION,
  MD_ERROR_WIN_INVALID_PEP_INFO_VERSION,
  MD_ERROR_WIN_OBJECT_NOT_EXTERNALLY_BACKED,
  MD_ERROR_WIN_EXTERNAL_BACKING_PROVIDER_UNKNOWN,
  MD_ERROR_WIN_COMPRESSION_NOT_BENEFICIAL,
  MD_ERROR_WIN_STORAGE_TOPOLOGY_ID_MISMATCH,
  MD_ERROR_WIN_BLOCKED_BY_PARENTAL_CONTROLS,
  MD_ERROR_WIN_BLOCK_TOO_MANY_REFERENCES,
  MD_ERROR_WIN_MARKED_TO_DISALLOW_WRITES,
  MD_ERROR_WIN_ENCLAVE_FAILURE,
  MD_ERROR_WIN_FAIL_NOACTION_REBOOT,
  MD_ERROR_WIN_FAIL_SHUTDOWN,
  MD_ERROR_WIN_FAIL_RESTART,
  MD_ERROR_WIN_MAX_SESSIONS_REACHED,
  MD_ERROR_WIN_NETWORK_ACCESS_DENIED_EDP,
  MD_ERROR_WIN_DEVICE_HINT_NAME_BUFFER_TOO_SMALL,
  MD_ERROR_WIN_EDP_POLICY_DENIES_OPERATION,
  MD_ERROR_WIN_EDP_DPL_POLICY_CANT_BE_SATISFIED,
  MD_ERROR_WIN_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT,
  MD_ERROR_WIN_DEVICE_IN_MAINTENANCE,
  MD_ERROR_WIN_NOT_SUPPORTED_ON_DAX,
  MD_ERROR_WIN_DAX_MAPPING_EXISTS,
  MD_ERROR_WIN_CLOUD_FILE_PROVIDER_NOT_RUNNING,
  MD_ERROR_WIN_CLOUD_FILE_METADATA_CORRUPT,
  MD_ERROR_WIN_CLOUD_FILE_METADATA_TOO_LARGE,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH,
  MD_ERROR_WIN_CHILD_PROCESS_BLOCKED,
  MD_ERROR_WIN_STORAGE_LOST_DATA_PERSISTENCE,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_BUSY,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN,
  MD_ERROR_WIN_GDI_HANDLE_LEAK,
  MD_ERROR_WIN_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED,
  MD_ERROR_WIN_NOT_A_CLOUD_FILE,
  MD_ERROR_WIN_CLOUD_FILE_NOT_IN_SYNC,
  MD_ERROR_WIN_CLOUD_FILE_ALREADY_CONNECTED,
  MD_ERROR_WIN_CLOUD_FILE_NOT_SUPPORTED,
  MD_ERROR_WIN_CLOUD_FILE_INVALID_REQUEST,
  MD_ERROR_WIN_CLOUD_FILE_READ_ONLY_VOLUME,
  MD_ERROR_WIN_CLOUD_FILE_CONNECTED_PROVIDER_ONLY,
  MD_ERROR_WIN_CLOUD_FILE_VALIDATION_FAILED,
  MD_ERROR_WIN_SMB1_NOT_AVAILABLE,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION,
  MD_ERROR_WIN_CLOUD_FILE_AUTHENTICATION_FAILED,
  MD_ERROR_WIN_CLOUD_FILE_INSUFFICIENT_RESOURCES,
  MD_ERROR_WIN_CLOUD_FILE_NETWORK_UNAVAILABLE,
  MD_ERROR_WIN_CLOUD_FILE_UNSUCCESSFUL,
  MD_ERROR_WIN_CLOUD_FILE_NOT_UNDER_SYNC_ROOT,
  MD_ERROR_WIN_CLOUD_FILE_IN_USE,
  MD_ERROR_WIN_CLOUD_FILE_PINNED,
  MD_ERROR_WIN_CLOUD_FILE_REQUEST_ABORTED,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_CORRUPT,
  MD_ERROR_WIN_CLOUD_FILE_ACCESS_DENIED,
  MD_ERROR_WIN_CLOUD_FILE_INCOMPATIBLE_HARDLINKS,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_LOCK_CONFLICT,
  MD_ERROR_WIN_CLOUD_FILE_REQUEST_CANCELED,
  MD_ERROR_WIN_EXTERNAL_SYSKEY_NOT_SUPPORTED,
  MD_ERROR_WIN_THREAD_MODE_ALREADY_BACKGROUND,
  MD_ERROR_WIN_THREAD_MODE_NOT_BACKGROUND,
  MD_ERROR_WIN_PROCESS_MODE_ALREADY_BACKGROUND,
  MD_ERROR_WIN_PROCESS_MODE_NOT_BACKGROUND,
  MD_ERROR_WIN_CLOUD_FILE_PROVIDER_TERMINATED,
  MD_ERROR_WIN_NOT_A_CLOUD_SYNC_ROOT,
  MD_ERROR_WIN_FILE_PROTECTED_UNDER_DPL,
  MD_ERROR_WIN_VOLUME_NOT_CLUSTER_ALIGNED,
  MD_ERROR_WIN_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND,
  MD_ERROR_WIN_APPX_FILE_NOT_ENCRYPTED,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER,
  MD_ERROR_WIN_LINUX_SUBSYSTEM_NOT_PRESENT,
  MD_ERROR_WIN_FT_READ_FAILURE,
  MD_ERROR_WIN_STORAGE_RESERVE_ID_INVALID,
  MD_ERROR_WIN_STORAGE_RESERVE_DOES_NOT_EXIST,
  MD_ERROR_WIN_STORAGE_RESERVE_ALREADY_EXISTS,
  MD_ERROR_WIN_STORAGE_RESERVE_NOT_EMPTY,
  MD_ERROR_WIN_NOT_A_DAX_VOLUME,
  MD_ERROR_WIN_NOT_DAX_MAPPABLE,
  MD_ERROR_WIN_TIME_SENSITIVE_THREAD,
  MD_ERROR_WIN_DPL_NOT_SUPPORTED_FOR_USER,
  MD_ERROR_WIN_CASE_DIFFERING_NAMES_IN_DIR,
  MD_ERROR_WIN_FILE_NOT_SUPPORTED,
  MD_ERROR_WIN_CLOUD_FILE_REQUEST_TIMEOUT,
  MD_ERROR_WIN_NO_TASK_QUEUE,
  MD_ERROR_WIN_SRC_SRV_DLL_LOAD_FAILED,
  MD_ERROR_WIN_NOT_SUPPORTED_WITH_BTT,
  MD_ERROR_WIN_ENCRYPTION_DISABLED,
  MD_ERROR_WIN_ENCRYPTING_METADATA_DISALLOWED,
  MD_ERROR_WIN_CANT_CLEAR_ENCRYPTION_FLAG,
  MD_ERROR_WIN_NO_SUCH_DEVICE,
  MD_ERROR_WIN_CLOUD_FILE_DEHYDRATION_DISALLOWED,
  MD_ERROR_WIN_FILE_SNAP_IN_PROGRESS,
  MD_ERROR_WIN_FILE_SNAP_USER_SECTION_NOT_SUPPORTED,
  MD_ERROR_WIN_FILE_SNAP_MODIFY_NOT_SUPPORTED,
  MD_ERROR_WIN_FILE_SNAP_IO_NOT_COORDINATED,
  MD_ERROR_WIN_FILE_SNAP_UNEXPECTED_ERROR,
  MD_ERROR_WIN_FILE_SNAP_INVALID_PARAMETER,
  MD_ERROR_WIN_UNSATISFIED_DEPENDENCIES,
  MD_ERROR_WIN_CASE_SENSITIVE_PATH,
  MD_ERROR_WIN_UNEXPECTED_NTCACHEMANAGER_ERROR,
  MD_ERROR_WIN_LINUX_SUBSYSTEM_UPDATE_REQUIRED,
  MD_ERROR_WIN_DLP_POLICY_WARNS_AGAINST_OPERATION,
  MD_ERROR_WIN_DLP_POLICY_DENIES_OPERATION,
  MD_ERROR_WIN_DLP_POLICY_SILENTLY_FAIL,
  MD_ERROR_WIN_CAPAUTHZ_NOT_DEVUNLOCKED,
  MD_ERROR_WIN_CAPAUTHZ_CHANGE_TYPE,
  MD_ERROR_WIN_CAPAUTHZ_NOT_PROVISIONED,
  MD_ERROR_WIN_CAPAUTHZ_NOT_AUTHORIZED,
  MD_ERROR_WIN_CAPAUTHZ_NO_POLICY,
  MD_ERROR_WIN_CAPAUTHZ_DB_CORRUPTED,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_INVALID_CATALOG,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_NO_AUTH_ENTITY,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_PARSE_ERROR,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_DEV_MODE_REQUIRED,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_NO_CAPABILITY_MATCH,
  MD_ERROR_WIN_CIMFS_IMAGE_CORRUPT,
  MD_ERROR_WIN_PNP_QUERY_REMOVE_DEVICE_TIMEOUT,
  MD_ERROR_WIN_PNP_QUERY_REMOVE_RELATED_DEVICE_TIMEOUT,
  MD_ERROR_WIN_PNP_QUERY_REMOVE_UNRELATED_DEVICE_TIMEOUT,
  MD_ERROR_WIN_DEVICE_HARDWARE_ERROR,
  MD_ERROR_WIN_INVALID_ADDRESS,
  MD_ERROR_WIN_HAS_SYSTEM_CRITICAL_FILES,
  MD_ERROR_WIN_USER_PROFILE_LOAD,
  MD_ERROR_WIN_ARITHMETIC_OVERFLOW,
  MD_ERROR_WIN_PIPE_CONNECTED,
  MD_ERROR_WIN_PIPE_LISTENING,
  MD_ERROR_WIN_VERIFIER_STOP,
  MD_ERROR_WIN_ABIOS_ERROR,
  MD_ERROR_WIN_WX86_WARNING,
  MD_ERROR_WIN_WX86_ERROR,
  MD_ERROR_WIN_TIMER_NOT_CANCELED,
  MD_ERROR_WIN_UNWIND,
  MD_ERROR_WIN_BAD_STACK,
  MD_ERROR_WIN_INVALID_UNWIND_TARGET,
  MD_ERROR_WIN_INVALID_PORT_ATTRIBUTES,
  MD_ERROR_WIN_PORT_MESSAGE_TOO_LONG,
  MD_ERROR_WIN_INVALID_QUOTA_LOWER,
  MD_ERROR_WIN_DEVICE_ALREADY_ATTACHED,
  MD_ERROR_WIN_INSTRUCTION_MISALIGNMENT,
  MD_ERROR_WIN_PROFILING_NOT_STARTED,
  MD_ERROR_WIN_PROFILING_NOT_STOPPED,
  MD_ERROR_WIN_COULD_NOT_INTERPRET,
  MD_ERROR_WIN_PROFILING_AT_LIMIT,
  MD_ERROR_WIN_CANT_WAIT,
  MD_ERROR_WIN_CANT_TERMINATE_SELF,
  MD_ERROR_WIN_UNEXPECTED_MM_CREATE_ERR,
  MD_ERROR_WIN_UNEXPECTED_MM_MAP_ERROR,
  MD_ERROR_WIN_UNEXPECTED_MM_EXTEND_ERR,
  MD_ERROR_WIN_BAD_FUNCTION_TABLE,
  MD_ERROR_WIN_NO_GUID_TRANSLATION,
  MD_ERROR_WIN_INVALID_LDT_SIZE,
  MD_ERROR_WIN_INVALID_LDT_OFFSET,
  MD_ERROR_WIN_INVALID_LDT_DESCRIPTOR,
  MD_ERROR_WIN_TOO_MANY_THREADS,
  MD_ERROR_WIN_THREAD_NOT_IN_PROCESS,
  MD_ERROR_WIN_PAGEFILE_QUOTA_EXCEEDED,
  MD_ERROR_WIN_LOGON_SERVER_CONFLICT,
  MD_ERROR_WIN_SYNCHRONIZATION_REQUIRED,
  MD_ERROR_WIN_NET_OPEN_FAILED,
  MD_ERROR_WIN_IO_PRIVILEGE_FAILED,
  MD_ERROR_WIN_CONTROL_C_EXIT,
  MD_ERROR_WIN_MISSING_SYSTEMFILE,
  MD_ERROR_WIN_UNHANDLED_EXCEPTION,
  MD_ERROR_WIN_APP_INIT_FAILURE,
  MD_ERROR_WIN_PAGEFILE_CREATE_FAILED,
  MD_ERROR_WIN_INVALID_IMAGE_HASH,
  MD_ERROR_WIN_NO_PAGEFILE,
  MD_ERROR_WIN_ILLEGAL_FLOAT_CONTEXT,
  MD_ERROR_WIN_NO_EVENT_PAIR,
  MD_ERROR_WIN_DOMAIN_CTRLR_CONFIG_ERROR,
  MD_ERROR_WIN_ILLEGAL_CHARACTER,
  MD_ERROR_WIN_UNDEFINED_CHARACTER,
  MD_ERROR_WIN_FLOPPY_VOLUME,
  MD_ERROR_WIN_BIOS_FAILED_TO_CONNECT_INTERRUPT,
  MD_ERROR_WIN_BACKUP_CONTROLLER,
  MD_ERROR_WIN_MUTANT_LIMIT_EXCEEDED,
  MD_ERROR_WIN_FS_DRIVER_REQUIRED,
  MD_ERROR_WIN_CANNOT_LOAD_REGISTRY_FILE,
  MD_ERROR_WIN_DEBUG_ATTACH_FAILED,
  MD_ERROR_WIN_SYSTEM_PROCESS_TERMINATED,
  MD_ERROR_WIN_DATA_NOT_ACCEPTED,
  MD_ERROR_WIN_VDM_HARD_ERROR,
  MD_ERROR_WIN_DRIVER_CANCEL_TIMEOUT,
  MD_ERROR_WIN_REPLY_MESSAGE_MISMATCH,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA,
  MD_ERROR_WIN_CLIENT_SERVER_PARAMETERS_INVALID,
  MD_ERROR_WIN_NOT_TINY_STREAM,
  MD_ERROR_WIN_STACK_OVERFLOW_READ,
  MD_ERROR_WIN_CONVERT_TO_LARGE,
  MD_ERROR_WIN_FOUND_OUT_OF_SCOPE,
  MD_ERROR_WIN_ALLOCATE_BUCKET,
  MD_ERROR_WIN_MARSHALL_OVERFLOW,
  MD_ERROR_WIN_INVALID_VARIANT,
  MD_ERROR_WIN_BAD_COMPRESSION_BUFFER,
  MD_ERROR_WIN_AUDIT_FAILED,
  MD_ERROR_WIN_TIMER_RESOLUTION_NOT_SET,
  MD_ERROR_WIN_INSUFFICIENT_LOGON_INFO,
  MD_ERROR_WIN_BAD_DLL_ENTRYPOINT,
  MD_ERROR_WIN_BAD_SERVICE_ENTRYPOINT,
  MD_ERROR_WIN_IP_ADDRESS_CONFLICT1,
  MD_ERROR_WIN_IP_ADDRESS_CONFLICT2,
  MD_ERROR_WIN_REGISTRY_QUOTA_LIMIT,
  MD_ERROR_WIN_NO_CALLBACK_ACTIVE,
  MD_ERROR_WIN_PWD_TOO_SHORT,
  MD_ERROR_WIN_PWD_TOO_RECENT,
  MD_ERROR_WIN_PWD_HISTORY_CONFLICT,
  MD_ERROR_WIN_UNSUPPORTED_COMPRESSION,
  MD_ERROR_WIN_INVALID_HW_PROFILE,
  MD_ERROR_WIN_INVALID_PLUGPLAY_DEVICE_PATH,
  MD_ERROR_WIN_QUOTA_LIST_INCONSISTENT,
  MD_ERROR_WIN_EVALUATION_EXPIRATION,
  MD_ERROR_WIN_ILLEGAL_DLL_RELOCATION,
  MD_ERROR_WIN_DLL_INIT_FAILED_LOGOFF,
  MD_ERROR_WIN_VALIDATE_CONTINUE,
  MD_ERROR_WIN_NO_MORE_MATCHES,
  MD_ERROR_WIN_RANGE_LIST_CONFLICT,
  MD_ERROR_WIN_SERVER_SID_MISMATCH,
  MD_ERROR_WIN_CANT_ENABLE_DENY_ONLY,
  MD_ERROR_WIN_FLOAT_MULTIPLE_FAULTS,
  MD_ERROR_WIN_FLOAT_MULTIPLE_TRAPS,
  MD_ERROR_WIN_NOINTERFACE,
  MD_ERROR_WIN_DRIVER_FAILED_SLEEP,
  MD_ERROR_WIN_CORRUPT_SYSTEM_FILE,
  MD_ERROR_WIN_COMMITMENT_MINIMUM,
  MD_ERROR_WIN_PNP_RESTART_ENUMERATION,
  MD_ERROR_WIN_SYSTEM_IMAGE_BAD_SIGNATURE,
  MD_ERROR_WIN_PNP_REBOOT_REQUIRED,
  MD_ERROR_WIN_INSUFFICIENT_POWER,
  MD_ERROR_WIN_MULTIPLE_FAULT_VIOLATION,
  MD_ERROR_WIN_SYSTEM_SHUTDOWN,
  MD_ERROR_WIN_PORT_NOT_SET,
  MD_ERROR_WIN_DS_VERSION_CHECK_FAILURE,
  MD_ERROR_WIN_RANGE_NOT_FOUND,
  MD_ERROR_WIN_NOT_SAFE_MODE_DRIVER,
  MD_ERROR_WIN_FAILED_DRIVER_ENTRY,
  MD_ERROR_WIN_DEVICE_ENUMERATION_ERROR,
  MD_ERROR_WIN_MOUNT_POINT_NOT_RESOLVED,
  MD_ERROR_WIN_INVALID_DEVICE_OBJECT_PARAMETER,
  MD_ERROR_WIN_MCA_OCCURED,
  MD_ERROR_WIN_DRIVER_DATABASE_ERROR,
  MD_ERROR_WIN_SYSTEM_HIVE_TOO_LARGE,
  MD_ERROR_WIN_DRIVER_FAILED_PRIOR_UNLOAD,
  MD_ERROR_WIN_VOLSNAP_PREPARE_HIBERNATE,
  MD_ERROR_WIN_HIBERNATION_FAILURE,
  MD_ERROR_WIN_PWD_TOO_LONG,
  MD_ERROR_WIN_FILE_SYSTEM_LIMITATION,
  MD_ERROR_WIN_ASSERTION_FAILURE,
  MD_ERROR_WIN_ACPI_ERROR,
  MD_ERROR_WIN_WOW_ASSERTION,
  MD_ERROR_WIN_PNP_BAD_MPS_TABLE,
  MD_ERROR_WIN_PNP_TRANSLATION_FAILED,
  MD_ERROR_WIN_PNP_IRQ_TRANSLATION_FAILED,
  MD_ERROR_WIN_PNP_INVALID_ID,
  MD_ERROR_WIN_WAKE_SYSTEM_DEBUGGER,
  MD_ERROR_WIN_HANDLES_CLOSED,
  MD_ERROR_WIN_EXTRANEOUS_INFORMATION,
  MD_ERROR_WIN_RXACT_COMMIT_NECESSARY,
  MD_ERROR_WIN_MEDIA_CHECK,
  MD_ERROR_WIN_GUID_SUBSTITUTION_MADE,
  MD_ERROR_WIN_STOPPED_ON_SYMLINK,
  MD_ERROR_WIN_LONGJUMP,
  MD_ERROR_WIN_PLUGPLAY_QUERY_VETOED,
  MD_ERROR_WIN_UNWIND_CONSOLIDATE,
  MD_ERROR_WIN_REGISTRY_HIVE_RECOVERED,
  MD_ERROR_WIN_DLL_MIGHT_BE_INSECURE,
  MD_ERROR_WIN_DLL_MIGHT_BE_INCOMPATIBLE,
  MD_ERROR_WIN_DBG_EXCEPTION_NOT_HANDLED,
  MD_ERROR_WIN_DBG_REPLY_LATER,
  MD_ERROR_WIN_DBG_UNABLE_TO_PROVIDE_HANDLE,
  MD_ERROR_WIN_DBG_TERMINATE_THREAD,
  MD_ERROR_WIN_DBG_TERMINATE_PROCESS,
  MD_ERROR_WIN_DBG_CONTROL_C,
  MD_ERROR_WIN_DBG_PRINTEXCEPTION_C,
  MD_ERROR_WIN_DBG_RIPEXCEPTION,
  MD_ERROR_WIN_DBG_CONTROL_BREAK,
  MD_ERROR_WIN_DBG_COMMAND_EXCEPTION,
  MD_ERROR_WIN_OBJECT_NAME_EXISTS,
  MD_ERROR_WIN_THREAD_WAS_SUSPENDED,
  MD_ERROR_WIN_IMAGE_NOT_AT_BASE,
  MD_ERROR_WIN_RXACT_STATE_CREATED,
  MD_ERROR_WIN_SEGMENT_NOTIFICATION,
  MD_ERROR_WIN_BAD_CURRENT_DIRECTORY,
  MD_ERROR_WIN_FT_READ_RECOVERY_FROM_BACKUP,
  MD_ERROR_WIN_FT_WRITE_RECOVERY,
  MD_ERROR_WIN_IMAGE_MACHINE_TYPE_MISMATCH,
  MD_ERROR_WIN_RECEIVE_PARTIAL,
  MD_ERROR_WIN_RECEIVE_EXPEDITED,
  MD_ERROR_WIN_RECEIVE_PARTIAL_EXPEDITED,
  MD_ERROR_WIN_EVENT_DONE,
  MD_ERROR_WIN_EVENT_PENDING,
  MD_ERROR_WIN_CHECKING_FILE_SYSTEM,
  MD_ERROR_WIN_FATAL_APP_EXIT,
  MD_ERROR_WIN_PREDEFINED_HANDLE,
  MD_ERROR_WIN_WAS_UNLOCKED,
  MD_ERROR_WIN_SERVICE_NOTIFICATION,
  MD_ERROR_WIN_WAS_LOCKED,
  MD_ERROR_WIN_LOG_HARD_ERROR,
  MD_ERROR_WIN_ALREADY_WIN32,
  MD_ERROR_WIN_IMAGE_MACHINE_TYPE_MISMATCH_EXE,
  MD_ERROR_WIN_NO_YIELD_PERFORMED,
  MD_ERROR_WIN_TIMER_RESUME_IGNORED,
  MD_ERROR_WIN_ARBITRATION_UNHANDLED,
  MD_ERROR_WIN_CARDBUS_NOT_SUPPORTED,
  MD_ERROR_WIN_MP_PROCESSOR_MISMATCH,
  MD_ERROR_WIN_HIBERNATED,
  MD_ERROR_WIN_RESUME_HIBERNATION,
  MD_ERROR_WIN_FIRMWARE_UPDATED,
  MD_ERROR_WIN_DRIVERS_LEAKING_LOCKED_PAGES,
  MD_ERROR_WIN_WAKE_SYSTEM,
  MD_ERROR_WIN_WAIT_1,
  MD_ERROR_WIN_WAIT_2,
  MD_ERROR_WIN_WAIT_3,
  MD_ERROR_WIN_WAIT_63,
  MD_ERROR_WIN_ABANDONED_WAIT_0,
  MD_ERROR_WIN_ABANDONED_WAIT_63,
  MD_ERROR_WIN_USER_APC,
  MD_ERROR_WIN_KERNEL_APC,
  MD_ERROR_WIN_ALERTED,
  MD_ERROR_WIN_ELEVATION_REQUIRED,
  MD_ERROR_WIN_REPARSE,
  MD_ERROR_WIN_OPLOCK_BREAK_IN_PROGRESS,
  MD_ERROR_WIN_VOLUME_MOUNTED,
  MD_ERROR_WIN_RXACT_COMMITTED,
  MD_ERROR_WIN_NOTIFY_CLEANUP,
  MD_ERROR_WIN_PRIMARY_TRANSPORT_CONNECT_FAILED,
  MD_ERROR_WIN_PAGE_FAULT_TRANSITION,
  MD_ERROR_WIN_PAGE_FAULT_DEMAND_ZERO,
  MD_ERROR_WIN_PAGE_FAULT_COPY_ON_WRITE,
  MD_ERROR_WIN_PAGE_FAULT_GUARD_PAGE,
  MD_ERROR_WIN_PAGE_FAULT_PAGING_FILE,
  MD_ERROR_WIN_CACHE_PAGE_LOCKED,
  MD_ERROR_WIN_CRASH_DUMP,
  MD_ERROR_WIN_BUFFER_ALL_ZEROS,
  MD_ERROR_WIN_REPARSE_OBJECT,
  MD_ERROR_WIN_RESOURCE_REQUIREMENTS_CHANGED,
  MD_ERROR_WIN_TRANSLATION_COMPLETE,
  MD_ERROR_WIN_NOTHING_TO_TERMINATE,
  MD_ERROR_WIN_PROCESS_NOT_IN_JOB,
  MD_ERROR_WIN_PROCESS_IN_JOB,
  MD_ERROR_WIN_VOLSNAP_HIBERNATE_READY,
  MD_ERROR_WIN_FSFILTER_OP_COMPLETED_SUCCESSFULLY,
  MD_ERROR_WIN_INTERRUPT_VECTOR_ALREADY_CONNECTED,
  MD_ERROR_WIN_INTERRUPT_STILL_CONNECTED,
  MD_ERROR_WIN_WAIT_FOR_OPLOCK,
  MD_ERROR_WIN_DBG_EXCEPTION_HANDLED,
  MD_ERROR_WIN_DBG_CONTINUE,
  MD_ERROR_WIN_CALLBACK_POP_STACK,
  MD_ERROR_WIN_COMPRESSION_DISABLED,
  MD_ERROR_WIN_CANTFETCHBACKWARDS,
  MD_ERROR_WIN_CANTSCROLLBACKWARDS,
  MD_ERROR_WIN_ROWSNOTRELEASED,
  MD_ERROR_WIN_BAD_ACCESSOR_FLAGS,
  MD_ERROR_WIN_ERRORS_ENCOUNTERED,
  MD_ERROR_WIN_NOT_CAPABLE,
  MD_ERROR_WIN_REQUEST_OUT_OF_SEQUENCE,
  MD_ERROR_WIN_VERSION_PARSE_ERROR,
  MD_ERROR_WIN_BADSTARTPOSITION,
  MD_ERROR_WIN_MEMORY_HARDWARE,
  MD_ERROR_WIN_DISK_REPAIR_DISABLED,
  MD_ERROR_WIN_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE,
  MD_ERROR_WIN_SYSTEM_POWERSTATE_TRANSITION,
  MD_ERROR_WIN_SYSTEM_POWERSTATE_COMPLEX_TRANSITION,
  MD_ERROR_WIN_MCA_EXCEPTION,
  MD_ERROR_WIN_ACCESS_AUDIT_BY_POLICY,
  MD_ERROR_WIN_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY,
  MD_ERROR_WIN_ABANDON_HIBERFILE,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR,
  MD_ERROR_WIN_BAD_MCFG_TABLE,
  MD_ERROR_WIN_DISK_REPAIR_REDIRECTED,
  MD_ERROR_WIN_DISK_REPAIR_UNSUCCESSFUL,
  MD_ERROR_WIN_CORRUPT_LOG_OVERFULL,
  MD_ERROR_WIN_CORRUPT_LOG_CORRUPTED,
  MD_ERROR_WIN_CORRUPT_LOG_UNAVAILABLE,
  MD_ERROR_WIN_CORRUPT_LOG_DELETED_FULL,
  MD_ERROR_WIN_CORRUPT_LOG_CLEARED,
  MD_ERROR_WIN_ORPHAN_NAME_EXHAUSTED,
  MD_ERROR_WIN_OPLOCK_SWITCHED_TO_NEW_HANDLE,
  MD_ERROR_WIN_CANNOT_GRANT_REQUESTED_OPLOCK,
  MD_ERROR_WIN_CANNOT_BREAK_OPLOCK,
  MD_ERROR_WIN_OPLOCK_HANDLE_CLOSED,
  MD_ERROR_WIN_NO_ACE_CONDITION,
  MD_ERROR_WIN_INVALID_ACE_CONDITION,
  MD_ERROR_WIN_FILE_HANDLE_REVOKED,
  MD_ERROR_WIN_IMAGE_AT_DIFFERENT_BASE,
  MD_ERROR_WIN_ENCRYPTED_IO_NOT_POSSIBLE,
  MD_ERROR_WIN_FILE_METADATA_OPTIMIZATION_IN_PROGRESS,
  MD_ERROR_WIN_QUOTA_ACTIVITY,
  MD_ERROR_WIN_HANDLE_REVOKED,
  MD_ERROR_WIN_CALLBACK_INVOKE_INLINE,
  MD_ERROR_WIN_CPU_SET_INVALID,
  MD_ERROR_WIN_ENCLAVE_NOT_TERMINATED,
  MD_ERROR_WIN_ENCLAVE_VIOLATION,
  MD_ERROR_WIN_EA_ACCESS_DENIED,
  MD_ERROR_WIN_OPERATION_ABORTED,
  MD_ERROR_WIN_IO_INCOMPLETE,
  MD_ERROR_WIN_IO_PENDING,
  MD_ERROR_WIN_NOACCESS,
  MD_ERROR_WIN_SWAPERROR,
  MD_ERROR_WIN_STACK_OVERFLOW,
  MD_ERROR_WIN_INVALID_MESSAGE,
  MD_ERROR_WIN_CAN_NOT_COMPLETE,
  MD_ERROR_WIN_INVALID_FLAGS,
  MD_ERROR_WIN_UNRECOGNIZED_VOLUME,
  MD_ERROR_WIN_FILE_INVALID,
  MD_ERROR_WIN_FULLSCREEN_MODE,
  MD_ERROR_WIN_NO_TOKEN,
  MD_ERROR_WIN_BADDB,
  MD_ERROR_WIN_BADKEY,
  MD_ERROR_WIN_CANTOPEN,
  MD_ERROR_WIN_CANTREAD,
  MD_ERROR_WIN_CANTWRITE,
  MD_ERROR_WIN_REGISTRY_RECOVERED,
  MD_ERROR_WIN_REGISTRY_CORRUPT,
  MD_ERROR_WIN_REGISTRY_IO_FAILED,
  MD_ERROR_WIN_NOT_REGISTRY_FILE,
  MD_ERROR_WIN_KEY_DELETED,
  MD_ERROR_WIN_NO_LOG_SPACE,
  MD_ERROR_WIN_KEY_HAS_CHILDREN,
  MD_ERROR_WIN_CHILD_MUST_BE_VOLATILE,
  MD_ERROR_WIN_NOTIFY_ENUM_DIR,
  MD_ERROR_WIN_DEPENDENT_SERVICES_RUNNING,
  MD_ERROR_WIN_INVALID_SERVICE_CONTROL,
  MD_ERROR_WIN_SERVICE_REQUEST_TIMEOUT,
  MD_ERROR_WIN_SERVICE_NO_THREAD,
  MD_ERROR_WIN_SERVICE_DATABASE_LOCKED,
  MD_ERROR_WIN_SERVICE_ALREADY_RUNNING,
  MD_ERROR_WIN_INVALID_SERVICE_ACCOUNT,
  MD_ERROR_WIN_SERVICE_DISABLED,
  MD_ERROR_WIN_CIRCULAR_DEPENDENCY,
  MD_ERROR_WIN_SERVICE_DOES_NOT_EXIST,
  MD_ERROR_WIN_SERVICE_CANNOT_ACCEPT_CTRL,
  MD_ERROR_WIN_SERVICE_NOT_ACTIVE,
  MD_ERROR_WIN_FAILED_SERVICE_CONTROLLER_CONNECT,
  MD_ERROR_WIN_EXCEPTION_IN_SERVICE,
  MD_ERROR_WIN_DATABASE_DOES_NOT_EXIST,
  MD_ERROR_WIN_SERVICE_SPECIFIC_ERROR,
  MD_ERROR_WIN_PROCESS_ABORTED,
  MD_ERROR_WIN_SERVICE_DEPENDENCY_FAIL,
  MD_ERROR_WIN_SERVICE_LOGON_FAILED,
  MD_ERROR_WIN_SERVICE_START_HANG,
  MD_ERROR_WIN_INVALID_SERVICE_LOCK,
  MD_ERROR_WIN_SERVICE_MARKED_FOR_DELETE,
  MD_ERROR_WIN_SERVICE_EXISTS,
  MD_ERROR_WIN_ALREADY_RUNNING_LKG,
  MD_ERROR_WIN_SERVICE_DEPENDENCY_DELETED,
  MD_ERROR_WIN_BOOT_ALREADY_ACCEPTED,
  MD_ERROR_WIN_SERVICE_NEVER_STARTED,
  MD_ERROR_WIN_DUPLICATE_SERVICE_NAME,
  MD_ERROR_WIN_DIFFERENT_SERVICE_ACCOUNT,
  MD_ERROR_WIN_CANNOT_DETECT_DRIVER_FAILURE,
  MD_ERROR_WIN_CANNOT_DETECT_PROCESS_ABORT,
  MD_ERROR_WIN_NO_RECOVERY_PROGRAM,
  MD_ERROR_WIN_SERVICE_NOT_IN_EXE,
  MD_ERROR_WIN_NOT_SAFEBOOT_SERVICE,
  MD_ERROR_WIN_END_OF_MEDIA,
  MD_ERROR_WIN_FILEMARK_DETECTED,
  MD_ERROR_WIN_BEGINNING_OF_MEDIA,
  MD_ERROR_WIN_SETMARK_DETECTED,
  MD_ERROR_WIN_NO_DATA_DETECTED,
  MD_ERROR_WIN_PARTITION_FAILURE,
  MD_ERROR_WIN_INVALID_BLOCK_LENGTH,
  MD_ERROR_WIN_DEVICE_NOT_PARTITIONED,
  MD_ERROR_WIN_UNABLE_TO_LOCK_MEDIA,
  MD_ERROR_WIN_UNABLE_TO_UNLOAD_MEDIA,
  MD_ERROR_WIN_MEDIA_CHANGED,
  MD_ERROR_WIN_BUS_RESET,
  MD_ERROR_WIN_NO_MEDIA_IN_DRIVE,
  MD_ERROR_WIN_NO_UNICODE_TRANSLATION,
  MD_ERROR_WIN_DLL_INIT_FAILED,
  MD_ERROR_WIN_SHUTDOWN_IN_PROGRESS,
  MD_ERROR_WIN_NO_SHUTDOWN_IN_PROGRESS,
  MD_ERROR_WIN_IO_DEVICE,
  MD_ERROR_WIN_SERIAL_NO_DEVICE,
  MD_ERROR_WIN_IRQ_BUSY,
  MD_ERROR_WIN_MORE_WRITES,
  MD_ERROR_WIN_COUNTER_TIMEOUT,
  MD_ERROR_WIN_FLOPPY_ID_MARK_NOT_FOUND,
  MD_ERROR_WIN_FLOPPY_WRONG_CYLINDER,
  MD_ERROR_WIN_FLOPPY_UNKNOWN_ERROR,
  MD_ERROR_WIN_FLOPPY_BAD_REGISTERS,
  MD_ERROR_WIN_DISK_RECALIBRATE_FAILED,
  MD_ERROR_WIN_DISK_OPERATION_FAILED,
  MD_ERROR_WIN_DISK_RESET_FAILED,
  MD_ERROR_WIN_EOM_OVERFLOW,
  MD_ERROR_WIN_NOT_ENOUGH_SERVER_MEMORY,
  MD_ERROR_WIN_POSSIBLE_DEADLOCK,
  MD_ERROR_WIN_MAPPED_ALIGNMENT,
  MD_ERROR_WIN_SET_POWER_STATE_VETOED,
  MD_ERROR_WIN_SET_POWER_STATE_FAILED,
  MD_ERROR_WIN_TOO_MANY_LINKS,
  MD_ERROR_WIN_OLD_WIN_VERSION,
  MD_ERROR_WIN_APP_WRONG_OS,
  MD_ERROR_WIN_SINGLE_INSTANCE_APP,
  MD_ERROR_WIN_RMODE_APP,
  MD_ERROR_WIN_INVALID_DLL,
  MD_ERROR_WIN_NO_ASSOCIATION,
  MD_ERROR_WIN_DDE_FAIL,
  MD_ERROR_WIN_DLL_NOT_FOUND,
  MD_ERROR_WIN_NO_MORE_USER_HANDLES,
  MD_ERROR_WIN_MESSAGE_SYNC_ONLY,
  MD_ERROR_WIN_SOURCE_ELEMENT_EMPTY,
  MD_ERROR_WIN_DESTINATION_ELEMENT_FULL,
  MD_ERROR_WIN_ILLEGAL_ELEMENT_ADDRESS,
  MD_ERROR_WIN_MAGAZINE_NOT_PRESENT,
  MD_ERROR_WIN_DEVICE_REINITIALIZATION_NEEDED,
  MD_ERROR_WIN_DEVICE_REQUIRES_CLEANING,
  MD_ERROR_WIN_DEVICE_DOOR_OPEN,
  MD_ERROR_WIN_DEVICE_NOT_CONNECTED,
  MD_ERROR_WIN_NOT_FOUND,
  MD_ERROR_WIN_NO_MATCH,
  MD_ERROR_WIN_SET_NOT_FOUND,
  MD_ERROR_WIN_POINT_NOT_FOUND,
  MD_ERROR_WIN_NO_TRACKING_SERVICE,
  MD_ERROR_WIN_NO_VOLUME_ID,
  MD_ERROR_WIN_UNABLE_TO_REMOVE_REPLACED,
  MD_ERROR_WIN_UNABLE_TO_MOVE_REPLACEMENT,
  MD_ERROR_WIN_UNABLE_TO_MOVE_REPLACEMENT_2,
  MD_ERROR_WIN_JOURNAL_DELETE_IN_PROGRESS,
  MD_ERROR_WIN_JOURNAL_NOT_ACTIVE,
  MD_ERROR_WIN_POTENTIAL_FILE_FOUND,
  MD_ERROR_WIN_JOURNAL_ENTRY_DELETED,
  MD_ERROR_WIN_VRF_CFG_AND_IO_ENABLED,
  MD_ERROR_WIN_PARTITION_TERMINATING,
  MD_ERROR_WIN_SHUTDOWN_IS_SCHEDULED,
  MD_ERROR_WIN_SHUTDOWN_USERS_LOGGED_ON,
  MD_ERROR_WIN_BAD_DEVICE,
  MD_ERROR_WIN_CONNECTION_UNAVAIL,
  MD_ERROR_WIN_DEVICE_ALREADY_REMEMBERED,
  MD_ERROR_WIN_NO_NET_OR_BAD_PATH,
  MD_ERROR_WIN_BAD_PROVIDER,
  MD_ERROR_WIN_CANNOT_OPEN_PROFILE,
  MD_ERROR_WIN_BAD_PROFILE,
  MD_ERROR_WIN_NOT_CONTAINER,
  MD_ERROR_WIN_EXTENDED_ERROR,
  MD_ERROR_WIN_INVALID_GROUPNAME,
  MD_ERROR_WIN_INVALID_COMPUTERNAME,
  MD_ERROR_WIN_INVALID_EVENTNAME,
  MD_ERROR_WIN_INVALID_DOMAINNAME,
  MD_ERROR_WIN_INVALID_SERVICENAME,
  MD_ERROR_WIN_INVALID_NETNAME,
  MD_ERROR_WIN_INVALID_SHARENAME,
  MD_ERROR_WIN_INVALID_PASSWORDNAME,
  MD_ERROR_WIN_INVALID_MESSAGENAME,
  MD_ERROR_WIN_INVALID_MESSAGEDEST,
  MD_ERROR_WIN_SESSION_CREDENTIAL_CONFLICT,
  MD_ERROR_WIN_REMOTE_SESSION_LIMIT_EXCEEDED,
  MD_ERROR_WIN_DUP_DOMAINNAME,
  MD_ERROR_WIN_NO_NETWORK,
  MD_ERROR_WIN_CANCELLED,
  MD_ERROR_WIN_USER_MAPPED_FILE,
  MD_ERROR_WIN_CONNECTION_REFUSED,
  MD_ERROR_WIN_GRACEFUL_DISCONNECT,
  MD_ERROR_WIN_ADDRESS_ALREADY_ASSOCIATED,
  MD_ERROR_WIN_ADDRESS_NOT_ASSOCIATED,
  MD_ERROR_WIN_CONNECTION_INVALID,
  MD_ERROR_WIN_CONNECTION_ACTIVE,
  MD_ERROR_WIN_NETWORK_UNREACHABLE,
  MD_ERROR_WIN_HOST_UNREACHABLE,
  MD_ERROR_WIN_PROTOCOL_UNREACHABLE,
  MD_ERROR_WIN_PORT_UNREACHABLE,
  MD_ERROR_WIN_REQUEST_ABORTED,
  MD_ERROR_WIN_CONNECTION_ABORTED,
  MD_ERROR_WIN_RETRY,
  MD_ERROR_WIN_CONNECTION_COUNT_LIMIT,
  MD_ERROR_WIN_LOGIN_TIME_RESTRICTION,
  MD_ERROR_WIN_LOGIN_WKSTA_RESTRICTION,
  MD_ERROR_WIN_INCORRECT_ADDRESS,
  MD_ERROR_WIN_ALREADY_REGISTERED,
  MD_ERROR_WIN_SERVICE_NOT_FOUND,
  MD_ERROR_WIN_NOT_AUTHENTICATED,
  MD_ERROR_WIN_NOT_LOGGED_ON,
  MD_ERROR_WIN_CONTINUE,
  MD_ERROR_WIN_ALREADY_INITIALIZED,
  MD_ERROR_WIN_NO_MORE_DEVICES,
  MD_ERROR_WIN_NO_SUCH_SITE,
  MD_ERROR_WIN_DOMAIN_CONTROLLER_EXISTS,
  MD_ERROR_WIN_ONLY_IF_CONNECTED,
  MD_ERROR_WIN_OVERRIDE_NOCHANGES,
  MD_ERROR_WIN_BAD_USER_PROFILE,
  MD_ERROR_WIN_NOT_SUPPORTED_ON_SBS,
  MD_ERROR_WIN_SERVER_SHUTDOWN_IN_PROGRESS,
  MD_ERROR_WIN_HOST_DOWN,
  MD_ERROR_WIN_NON_ACCOUNT_SID,
  MD_ERROR_WIN_NON_DOMAIN_SID,
  MD_ERROR_WIN_APPHELP_BLOCK,
  MD_ERROR_WIN_ACCESS_DISABLED_BY_POLICY,
  MD_ERROR_WIN_REG_NAT_CONSUMPTION,
  MD_ERROR_WIN_CSCSHARE_OFFLINE,
  MD_ERROR_WIN_PKINIT_FAILURE,
  MD_ERROR_WIN_SMARTCARD_SUBSYSTEM_FAILURE,
  MD_ERROR_WIN_DOWNGRADE_DETECTED,
  MD_ERROR_WIN_MACHINE_LOCKED,
  MD_ERROR_WIN_SMB_GUEST_LOGON_BLOCKED,
  MD_ERROR_WIN_CALLBACK_SUPPLIED_INVALID_DATA,
  MD_ERROR_WIN_SYNC_FOREGROUND_REFRESH_REQUIRED,
  MD_ERROR_WIN_DRIVER_BLOCKED,
  MD_ERROR_WIN_INVALID_IMPORT_OF_NON_DLL,
  MD_ERROR_WIN_ACCESS_DISABLED_WEBBLADE,
  MD_ERROR_WIN_ACCESS_DISABLED_WEBBLADE_TAMPER,
  MD_ERROR_WIN_RECOVERY_FAILURE,
  MD_ERROR_WIN_ALREADY_FIBER,
  MD_ERROR_WIN_ALREADY_THREAD,
  MD_ERROR_WIN_STACK_BUFFER_OVERRUN,
  MD_ERROR_WIN_PARAMETER_QUOTA_EXCEEDED,
  MD_ERROR_WIN_DEBUGGER_INACTIVE,
  MD_ERROR_WIN_DELAY_LOAD_FAILED,
  MD_ERROR_WIN_VDM_DISALLOWED,
  MD_ERROR_WIN_UNIDENTIFIED_ERROR,
  MD_ERROR_WIN_INVALID_CRUNTIME_PARAMETER,
  MD_ERROR_WIN_BEYOND_VDL,
  MD_ERROR_WIN_INCOMPATIBLE_SERVICE_SID_TYPE,
  MD_ERROR_WIN_DRIVER_PROCESS_TERMINATED,
  MD_ERROR_WIN_IMPLEMENTATION_LIMIT,
  MD_ERROR_WIN_PROCESS_IS_PROTECTED,
  MD_ERROR_WIN_SERVICE_NOTIFY_CLIENT_LAGGING,
  MD_ERROR_WIN_DISK_QUOTA_EXCEEDED,
  MD_ERROR_WIN_CONTENT_BLOCKED,
  MD_ERROR_WIN_INCOMPATIBLE_SERVICE_PRIVILEGE,
  MD_ERROR_WIN_APP_HANG,
  MD_ERROR_WIN_INVALID_LABEL,
  MD_ERROR_WIN_NOT_ALL_ASSIGNED,
  MD_ERROR_WIN_SOME_NOT_MAPPED,
  MD_ERROR_WIN_NO_QUOTAS_FOR_ACCOUNT,
  MD_ERROR_WIN_LOCAL_USER_SESSION_KEY,
  MD_ERROR_WIN_NULL_LM_PASSWORD,
  MD_ERROR_WIN_UNKNOWN_REVISION,
  MD_ERROR_WIN_REVISION_MISMATCH,
  MD_ERROR_WIN_INVALID_OWNER,
  MD_ERROR_WIN_INVALID_PRIMARY_GROUP,
  MD_ERROR_WIN_NO_IMPERSONATION_TOKEN,
  MD_ERROR_WIN_CANT_DISABLE_MANDATORY,
  MD_ERROR_WIN_NO_LOGON_SERVERS,
  MD_ERROR_WIN_NO_SUCH_LOGON_SESSION,
  MD_ERROR_WIN_NO_SUCH_PRIVILEGE,
  MD_ERROR_WIN_PRIVILEGE_NOT_HELD,
  MD_ERROR_WIN_INVALID_ACCOUNT_NAME,
  MD_ERROR_WIN_USER_EXISTS,
  MD_ERROR_WIN_NO_SUCH_USER,
  MD_ERROR_WIN_GROUP_EXISTS,
  MD_ERROR_WIN_NO_SUCH_GROUP,
  MD_ERROR_WIN_MEMBER_IN_GROUP,
  MD_ERROR_WIN_MEMBER_NOT_IN_GROUP,
  MD_ERROR_WIN_LAST_ADMIN,
  MD_ERROR_WIN_WRONG_PASSWORD,
  MD_ERROR_WIN_ILL_FORMED_PASSWORD,
  MD_ERROR_WIN_PASSWORD_RESTRICTION,
  MD_ERROR_WIN_LOGON_FAILURE,
  MD_ERROR_WIN_ACCOUNT_RESTRICTION,
  MD_ERROR_WIN_INVALID_LOGON_HOURS,
  MD_ERROR_WIN_INVALID_WORKSTATION,
  MD_ERROR_WIN_PASSWORD_EXPIRED,
  MD_ERROR_WIN_ACCOUNT_DISABLED,
  MD_ERROR_WIN_NONE_MAPPED,
  MD_ERROR_WIN_TOO_MANY_LUIDS_REQUESTED,
  MD_ERROR_WIN_LUIDS_EXHAUSTED,
  MD_ERROR_WIN_INVALID_SUB_AUTHORITY,
  MD_ERROR_WIN_INVALID_ACL,
  MD_ERROR_WIN_INVALID_SID,
  MD_ERROR_WIN_INVALID_SECURITY_DESCR,
  MD_ERROR_WIN_BAD_INHERITANCE_ACL,
  MD_ERROR_WIN_SERVER_DISABLED,
  MD_ERROR_WIN_SERVER_NOT_DISABLED,
  MD_ERROR_WIN_INVALID_ID_AUTHORITY,
  MD_ERROR_WIN_ALLOTTED_SPACE_EXCEEDED,
  MD_ERROR_WIN_INVALID_GROUP_ATTRIBUTES,
  MD_ERROR_WIN_BAD_IMPERSONATION_LEVEL,
  MD_ERROR_WIN_CANT_OPEN_ANONYMOUS,
  MD_ERROR_WIN_BAD_VALIDATION_CLASS,
  MD_ERROR_WIN_BAD_TOKEN_TYPE,
  MD_ERROR_WIN_NO_SECURITY_ON_OBJECT,
  MD_ERROR_WIN_CANT_ACCESS_DOMAIN_INFO,
  MD_ERROR_WIN_INVALID_SERVER_STATE,
  MD_ERROR_WIN_INVALID_DOMAIN_STATE,
  MD_ERROR_WIN_INVALID_DOMAIN_ROLE,
  MD_ERROR_WIN_NO_SUCH_DOMAIN,
  MD_ERROR_WIN_DOMAIN_EXISTS,
  MD_ERROR_WIN_DOMAIN_LIMIT_EXCEEDED,
  MD_ERROR_WIN_INTERNAL_DB_CORRUPTION,
  MD_ERROR_WIN_INTERNAL_ERROR,
  MD_ERROR_WIN_GENERIC_NOT_MAPPED,
  MD_ERROR_WIN_BAD_DESCRIPTOR_FORMAT,
  MD_ERROR_WIN_NOT_LOGON_PROCESS,
  MD_ERROR_WIN_LOGON_SESSION_EXISTS,
  MD_ERROR_WIN_NO_SUCH_PACKAGE,
  MD_ERROR_WIN_BAD_LOGON_SESSION_STATE,
  MD_ERROR_WIN_LOGON_SESSION_COLLISION,
  MD_ERROR_WIN_INVALID_LOGON_TYPE,
  MD_ERROR_WIN_CANNOT_IMPERSONATE,
  MD_ERROR_WIN_RXACT_INVALID_STATE,
  MD_ERROR_WIN_RXACT_COMMIT_FAILURE,
  MD_ERROR_WIN_SPECIAL_ACCOUNT,
  MD_ERROR_WIN_SPECIAL_GROUP,
  MD_ERROR_WIN_SPECIAL_USER,
  MD_ERROR_WIN_MEMBERS_PRIMARY_GROUP,
  MD_ERROR_WIN_TOKEN_ALREADY_IN_USE,
  MD_ERROR_WIN_NO_SUCH_ALIAS,
  MD_ERROR_WIN_MEMBER_NOT_IN_ALIAS,
  MD_ERROR_WIN_MEMBER_IN_ALIAS,
  MD_ERROR_WIN_ALIAS_EXISTS,
  MD_ERROR_WIN_LOGON_NOT_GRANTED,
  MD_ERROR_WIN_TOO_MANY_SECRETS,
  MD_ERROR_WIN_SECRET_TOO_LONG,
  MD_ERROR_WIN_INTERNAL_DB_ERROR,
  MD_ERROR_WIN_TOO_MANY_CONTEXT_IDS,
  MD_ERROR_WIN_LOGON_TYPE_NOT_GRANTED,
  MD_ERROR_WIN_NT_CROSS_ENCRYPTION_REQUIRED,
  MD_ERROR_WIN_NO_SUCH_MEMBER,
  MD_ERROR_WIN_INVALID_MEMBER,
  MD_ERROR_WIN_TOO_MANY_SIDS,
  MD_ERROR_WIN_LM_CROSS_ENCRYPTION_REQUIRED,
  MD_ERROR_WIN_NO_INHERITANCE,
  MD_ERROR_WIN_FILE_CORRUPT,
  MD_ERROR_WIN_DISK_CORRUPT,
  MD_ERROR_WIN_NO_USER_SESSION_KEY,
  MD_ERROR_WIN_LICENSE_QUOTA_EXCEEDED,
  MD_ERROR_WIN_WRONG_TARGET_NAME,
  MD_ERROR_WIN_MUTUAL_AUTH_FAILED,
  MD_ERROR_WIN_TIME_SKEW,
  MD_ERROR_WIN_CURRENT_DOMAIN_NOT_ALLOWED,
  MD_ERROR_WIN_INVALID_WINDOW_HANDLE,
  MD_ERROR_WIN_INVALID_MENU_HANDLE,
  MD_ERROR_WIN_INVALID_CURSOR_HANDLE,
  MD_ERROR_WIN_INVALID_ACCEL_HANDLE,
  MD_ERROR_WIN_INVALID_HOOK_HANDLE,
  MD_ERROR_WIN_INVALID_DWP_HANDLE,
  MD_ERROR_WIN_TLW_WITH_WSCHILD,
  MD_ERROR_WIN_CANNOT_FIND_WND_CLASS,
  MD_ERROR_WIN_WINDOW_OF_OTHER_THREAD,
  MD_ERROR_WIN_HOTKEY_ALREADY_REGISTERED,
  MD_ERROR_WIN_CLASS_ALREADY_EXISTS,
  MD_ERROR_WIN_CLASS_DOES_NOT_EXIST,
  MD_ERROR_WIN_CLASS_HAS_WINDOWS,
  MD_ERROR_WIN_INVALID_INDEX,
  MD_ERROR_WIN_INVALID_ICON_HANDLE,
  MD_ERROR_WIN_PRIVATE_DIALOG_INDEX,
  MD_ERROR_WIN_LISTBOX_ID_NOT_FOUND,
  MD_ERROR_WIN_NO_WILDCARD_CHARACTERS,
  MD_ERROR_WIN_CLIPBOARD_NOT_OPEN,
  MD_ERROR_WIN_HOTKEY_NOT_REGISTERED,
  MD_ERROR_WIN_WINDOW_NOT_DIALOG,
  MD_ERROR_WIN_CONTROL_ID_NOT_FOUND,
  MD_ERROR_WIN_INVALID_COMBOBOX_MESSAGE,
  MD_ERROR_WIN_WINDOW_NOT_COMBOBOX,
  MD_ERROR_WIN_INVALID_EDIT_HEIGHT,
  MD_ERROR_WIN_DC_NOT_FOUND,
  MD_ERROR_WIN_INVALID_HOOK_FILTER,
  MD_ERROR_WIN_INVALID_FILTER_PROC,
  MD_ERROR_WIN_HOOK_NEEDS_HMOD,
  MD_ERROR_WIN_GLOBAL_ONLY_HOOK,
  MD_ERROR_WIN_JOURNAL_HOOK_SET,
  MD_ERROR_WIN_HOOK_NOT_INSTALLED,
  MD_ERROR_WIN_INVALID_LB_MESSAGE,
  MD_ERROR_WIN_SETCOUNT_ON_BAD_LB,
  MD_ERROR_WIN_LB_WITHOUT_TABSTOPS,
  MD_ERROR_WIN_DESTROY_OBJECT_OF_OTHER_THREAD,
  MD_ERROR_WIN_CHILD_WINDOW_MENU,
  MD_ERROR_WIN_NO_SYSTEM_MENU,
  MD_ERROR_WIN_INVALID_MSGBOX_STYLE,
  MD_ERROR_WIN_INVALID_SPI_VALUE,
  MD_ERROR_WIN_SCREEN_ALREADY_LOCKED,
  MD_ERROR_WIN_HWNDS_HAVE_DIFF_PARENT,
  MD_ERROR_WIN_NOT_CHILD_WINDOW,
  MD_ERROR_WIN_INVALID_GW_COMMAND,
  MD_ERROR_WIN_INVALID_THREAD_ID,
  MD_ERROR_WIN_NON_MDICHILD_WINDOW,
  MD_ERROR_WIN_POPUP_ALREADY_ACTIVE,
  MD_ERROR_WIN_NO_SCROLLBARS,
  MD_ERROR_WIN_INVALID_SCROLLBAR_RANGE,
  MD_ERROR_WIN_INVALID_SHOWWIN_COMMAND,
  MD_ERROR_WIN_NO_SYSTEM_RESOURCES,
  MD_ERROR_WIN_NONPAGED_SYSTEM_RESOURCES,
  MD_ERROR_WIN_PAGED_SYSTEM_RESOURCES,
  MD_ERROR_WIN_WORKING_SET_QUOTA,
  MD_ERROR_WIN_PAGEFILE_QUOTA,
  MD_ERROR_WIN_COMMITMENT_LIMIT,
  MD_ERROR_WIN_MENU_ITEM_NOT_FOUND,
  MD_ERROR_WIN_INVALID_KEYBOARD_HANDLE,
  MD_ERROR_WIN_HOOK_TYPE_NOT_ALLOWED,
  MD_ERROR_WIN_REQUIRES_INTERACTIVE_WINDOWSTATION,
  MD_ERROR_WIN_TIMEOUT,
  MD_ERROR_WIN_INVALID_MONITOR_HANDLE,
  MD_ERROR_WIN_INCORRECT_SIZE,
  MD_ERROR_WIN_SYMLINK_CLASS_DISABLED,
  MD_ERROR_WIN_SYMLINK_NOT_SUPPORTED,
  MD_ERROR_WIN_XML_PARSE_ERROR,
  MD_ERROR_WIN_XMLDSIG_ERROR,
  MD_ERROR_WIN_RESTART_APPLICATION,
  MD_ERROR_WIN_WRONG_COMPARTMENT,
  MD_ERROR_WIN_AUTHIP_FAILURE,
  MD_ERROR_WIN_NO_NVRAM_RESOURCES,
  MD_ERROR_WIN_NOT_GUI_PROCESS,
  MD_ERROR_WIN_EVENTLOG_FILE_CORRUPT,
  MD_ERROR_WIN_EVENTLOG_CANT_START,
  MD_ERROR_WIN_LOG_FILE_FULL,
  MD_ERROR_WIN_EVENTLOG_FILE_CHANGED,
  MD_ERROR_WIN_CONTAINER_ASSIGNED,
  MD_ERROR_WIN_JOB_NO_CONTAINER,
  MD_ERROR_WIN_INVALID_TASK_NAME,
  MD_ERROR_WIN_INVALID_TASK_INDEX,
  MD_ERROR_WIN_THREAD_ALREADY_IN_TASK,
  MD_ERROR_WIN_INSTALL_SERVICE_FAILURE,
  MD_ERROR_WIN_INSTALL_USEREXIT,
  MD_ERROR_WIN_INSTALL_FAILURE,
  MD_ERROR_WIN_INSTALL_SUSPEND,
  MD_ERROR_WIN_UNKNOWN_PRODUCT,
  MD_ERROR_WIN_UNKNOWN_FEATURE,
  MD_ERROR_WIN_UNKNOWN_COMPONENT,
  MD_ERROR_WIN_UNKNOWN_PROPERTY,
  MD_ERROR_WIN_INVALID_HANDLE_STATE,
  MD_ERROR_WIN_BAD_CONFIGURATION,
  MD_ERROR_WIN_INDEX_ABSENT,
  MD_ERROR_WIN_INSTALL_SOURCE_ABSENT,
  MD_ERROR_WIN_INSTALL_PACKAGE_VERSION,
  MD_ERROR_WIN_PRODUCT_UNINSTALLED,
  MD_ERROR_WIN_BAD_QUERY_SYNTAX,
  MD_ERROR_WIN_INVALID_FIELD,
  MD_ERROR_WIN_DEVICE_REMOVED,
  MD_ERROR_WIN_INSTALL_ALREADY_RUNNING,
  MD_ERROR_WIN_INSTALL_PACKAGE_OPEN_FAILED,
  MD_ERROR_WIN_INSTALL_PACKAGE_INVALID,
  MD_ERROR_WIN_INSTALL_UI_FAILURE,
  MD_ERROR_WIN_INSTALL_LOG_FAILURE,
  MD_ERROR_WIN_INSTALL_LANGUAGE_UNSUPPORTED,
  MD_ERROR_WIN_INSTALL_TRANSFORM_FAILURE,
  MD_ERROR_WIN_INSTALL_PACKAGE_REJECTED,
  MD_ERROR_WIN_FUNCTION_NOT_CALLED,
  MD_ERROR_WIN_FUNCTION_FAILED,
  MD_ERROR_WIN_INVALID_TABLE,
  MD_ERROR_WIN_DATATYPE_MISMATCH,
  MD_ERROR_WIN_UNSUPPORTED_TYPE,
  MD_ERROR_WIN_CREATE_FAILED,
  MD_ERROR_WIN_INSTALL_TEMP_UNWRITABLE,
  MD_ERROR_WIN_INSTALL_PLATFORM_UNSUPPORTED,
  MD_ERROR_WIN_INSTALL_NOTUSED,
  MD_ERROR_WIN_PATCH_PACKAGE_OPEN_FAILED,
  MD_ERROR_WIN_PATCH_PACKAGE_INVALID,
  MD_ERROR_WIN_PATCH_PACKAGE_UNSUPPORTED,
  MD_ERROR_WIN_PRODUCT_VERSION,
  MD_ERROR_WIN_INVALID_COMMAND_LINE,
  MD_ERROR_WIN_INSTALL_REMOTE_DISALLOWED,
  MD_ERROR_WIN_SUCCESS_REBOOT_INITIATED,
  MD_ERROR_WIN_PATCH_TARGET_NOT_FOUND,
  MD_ERROR_WIN_PATCH_PACKAGE_REJECTED,
  MD_ERROR_WIN_INSTALL_TRANSFORM_REJECTED,
  MD_ERROR_WIN_INSTALL_REMOTE_PROHIBITED,
  MD_ERROR_WIN_PATCH_REMOVAL_UNSUPPORTED,
  MD_ERROR_WIN_UNKNOWN_PATCH,
  MD_ERROR_WIN_PATCH_NO_SEQUENCE,
  MD_ERROR_WIN_PATCH_REMOVAL_DISALLOWED,
  MD_ERROR_WIN_INVALID_PATCH_XML,
  MD_ERROR_WIN_PATCH_MANAGED_ADVERTISED_PRODUCT,
  MD_ERROR_WIN_INSTALL_SERVICE_SAFEBOOT,
  MD_ERROR_WIN_FAIL_FAST_EXCEPTION,
  MD_ERROR_WIN_INSTALL_REJECTED,
  MD_ERROR_WIN_DYNAMIC_CODE_BLOCKED,
  MD_ERROR_WIN_NOT_SAME_OBJECT,
  MD_ERROR_WIN_STRICT_CFG_VIOLATION,
  MD_ERROR_WIN_SET_CONTEXT_DENIED,
  MD_ERROR_WIN_CROSS_PARTITION_VIOLATION,
  MD_ERROR_WIN_RETURN_ADDRESS_HIJACK_ATTEMPT,
  MD_ERROR_WIN_INVALID_USER_BUFFER,
  MD_ERROR_WIN_UNRECOGNIZED_MEDIA,
  MD_ERROR_WIN_NO_TRUST_LSA_SECRET,
  MD_ERROR_WIN_NO_TRUST_SAM_ACCOUNT,
  MD_ERROR_WIN_TRUSTED_DOMAIN_FAILURE,
  MD_ERROR_WIN_TRUSTED_RELATIONSHIP_FAILURE,
  MD_ERROR_WIN_TRUST_FAILURE,
  MD_ERROR_WIN_NETLOGON_NOT_STARTED,
  MD_ERROR_WIN_ACCOUNT_EXPIRED,
  MD_ERROR_WIN_REDIRECTOR_HAS_OPEN_HANDLES,
  MD_ERROR_WIN_PRINTER_DRIVER_ALREADY_INSTALLED,
  MD_ERROR_WIN_UNKNOWN_PORT,
  MD_ERROR_WIN_UNKNOWN_PRINTER_DRIVER,
  MD_ERROR_WIN_UNKNOWN_PRINTPROCESSOR,
  MD_ERROR_WIN_INVALID_SEPARATOR_FILE,
  MD_ERROR_WIN_INVALID_PRIORITY,
  MD_ERROR_WIN_INVALID_PRINTER_NAME,
  MD_ERROR_WIN_PRINTER_ALREADY_EXISTS,
  MD_ERROR_WIN_INVALID_PRINTER_COMMAND,
  MD_ERROR_WIN_INVALID_DATATYPE,
  MD_ERROR_WIN_INVALID_ENVIRONMENT,
  MD_ERROR_WIN_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT,
  MD_ERROR_WIN_NOLOGON_WORKSTATION_TRUST_ACCOUNT,
  MD_ERROR_WIN_NOLOGON_SERVER_TRUST_ACCOUNT,
  MD_ERROR_WIN_DOMAIN_TRUST_INCONSISTENT,
  MD_ERROR_WIN_SERVER_HAS_OPEN_HANDLES,
  MD_ERROR_WIN_RESOURCE_DATA_NOT_FOUND,
  MD_ERROR_WIN_RESOURCE_TYPE_NOT_FOUND,
  MD_ERROR_WIN_RESOURCE_NAME_NOT_FOUND,
  MD_ERROR_WIN_RESOURCE_LANG_NOT_FOUND,
  MD_ERROR_WIN_NOT_ENOUGH_QUOTA,
  MD_ERROR_WIN_INVALID_TIME,
  MD_ERROR_WIN_INVALID_FORM_NAME,
  MD_ERROR_WIN_INVALID_FORM_SIZE,
  MD_ERROR_WIN_ALREADY_WAITING,
  MD_ERROR_WIN_PRINTER_DELETED,
  MD_ERROR_WIN_INVALID_PRINTER_STATE,
  MD_ERROR_WIN_PASSWORD_MUST_CHANGE,
  MD_ERROR_WIN_DOMAIN_CONTROLLER_NOT_FOUND,
  MD_ERROR_WIN_ACCOUNT_LOCKED_OUT,
  MD_ERROR_WIN_NO_SITENAME,
  MD_ERROR_WIN_CANT_ACCESS_FILE,
  MD_ERROR_WIN_CANT_RESOLVE_FILENAME,
  MD_ERROR_WIN_KM_DRIVER_BLOCKED,
  MD_ERROR_WIN_CONTEXT_EXPIRED,
  MD_ERROR_WIN_PER_USER_TRUST_QUOTA_EXCEEDED,
  MD_ERROR_WIN_ALL_USER_TRUST_QUOTA_EXCEEDED,
  MD_ERROR_WIN_USER_DELETE_TRUST_QUOTA_EXCEEDED,
  MD_ERROR_WIN_AUTHENTICATION_FIREWALL_FAILED,
  MD_ERROR_WIN_REMOTE_PRINT_CONNECTIONS_BLOCKED,
  MD_ERROR_WIN_NTLM_BLOCKED,
  MD_ERROR_WIN_PASSWORD_CHANGE_REQUIRED,
  MD_ERROR_WIN_LOST_MODE_LOGON_RESTRICTION,
  MD_ERROR_WIN_INVALID_PIXEL_FORMAT,
  MD_ERROR_WIN_BAD_DRIVER,
  MD_ERROR_WIN_INVALID_WINDOW_STYLE,
  MD_ERROR_WIN_METAFILE_NOT_SUPPORTED,
  MD_ERROR_WIN_TRANSFORM_NOT_SUPPORTED,
  MD_ERROR_WIN_CLIPPING_NOT_SUPPORTED,
  MD_ERROR_WIN_INVALID_CMM,
  MD_ERROR_WIN_INVALID_PROFILE,
  MD_ERROR_WIN_TAG_NOT_FOUND,
  MD_ERROR_WIN_TAG_NOT_PRESENT,
  MD_ERROR_WIN_DUPLICATE_TAG,
  MD_ERROR_WIN_PROFILE_NOT_ASSOCIATED_WITH_DEVICE,
  MD_ERROR_WIN_PROFILE_NOT_FOUND,
  MD_ERROR_WIN_INVALID_COLORSPACE,
  MD_ERROR_WIN_ICM_NOT_ENABLED,
  MD_ERROR_WIN_DELETING_ICM_XFORM,
  MD_ERROR_WIN_INVALID_TRANSFORM,
  MD_ERROR_WIN_COLORSPACE_MISMATCH,
  MD_ERROR_WIN_INVALID_COLORINDEX,
  MD_ERROR_WIN_PROFILE_DOES_NOT_MATCH_DEVICE,
  MD_ERROR_WIN_CONNECTED_OTHER_PASSWORD,
  MD_ERROR_WIN_CONNECTED_OTHER_PASSWORD_DEFAULT,
  MD_ERROR_WIN_BAD_USERNAME,
  MD_ERROR_WIN_NOT_CONNECTED,
  MD_ERROR_WIN_OPEN_FILES,
  MD_ERROR_WIN_ACTIVE_CONNECTIONS,
  MD_ERROR_WIN_DEVICE_IN_USE,
  MD_ERROR_WIN_UNKNOWN_PRINT_MONITOR,
  MD_ERROR_WIN_PRINTER_DRIVER_IN_USE,
  MD_ERROR_WIN_SPOOL_FILE_NOT_FOUND,
  MD_ERROR_WIN_SPL_NO_STARTDOC,
  MD_ERROR_WIN_SPL_NO_ADDJOB,
  MD_ERROR_WIN_PRINT_PROCESSOR_ALREADY_INSTALLED,
  MD_ERROR_WIN_PRINT_MONITOR_ALREADY_INSTALLED,
  MD_ERROR_WIN_INVALID_PRINT_MONITOR,
  MD_ERROR_WIN_PRINT_MONITOR_IN_USE,
  MD_ERROR_WIN_PRINTER_HAS_JOBS_QUEUED,
  MD_ERROR_WIN_SUCCESS_REBOOT_REQUIRED,
  MD_ERROR_WIN_SUCCESS_RESTART_REQUIRED,
  MD_ERROR_WIN_PRINTER_NOT_FOUND,
  MD_ERROR_WIN_PRINTER_DRIVER_WARNED,
  MD_ERROR_WIN_PRINTER_DRIVER_BLOCKED,
  MD_ERROR_WIN_PRINTER_DRIVER_PACKAGE_IN_USE,
  MD_ERROR_WIN_CORE_DRIVER_PACKAGE_NOT_FOUND,
  MD_ERROR_WIN_FAIL_REBOOT_REQUIRED,
  MD_ERROR_WIN_FAIL_REBOOT_INITIATED,
  MD_ERROR_WIN_PRINTER_DRIVER_DOWNLOAD_NEEDED,
  MD_ERROR_WIN_PRINT_JOB_RESTART_REQUIRED,
  MD_ERROR_WIN_INVALID_PRINTER_DRIVER_MANIFEST,
  MD_ERROR_WIN_PRINTER_NOT_SHAREABLE,
  MD_ERROR_WIN_REQUEST_PAUSED,
  MD_ERROR_WIN_APPEXEC_CONDITION_NOT_SATISFIED,
  MD_ERROR_WIN_APPEXEC_HANDLE_INVALIDATED,
  MD_ERROR_WIN_APPEXEC_INVALID_HOST_GENERATION,
  MD_ERROR_WIN_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION,
  MD_ERROR_WIN_APPEXEC_INVALID_HOST_STATE,
  MD_ERROR_WIN_APPEXEC_NO_DONOR,
  MD_ERROR_WIN_APPEXEC_HOST_ID_MISMATCH,
  MD_ERROR_WIN_APPEXEC_UNKNOWN_USER,
  MD_ERROR_WIN_IO_REISSUE_AS_CACHED,
  MD_ERROR_WIN_WINS_INTERNAL,
  MD_ERROR_WIN_CAN_NOT_DEL_LOCAL_WINS,
  MD_ERROR_WIN_STATIC_INIT,
  MD_ERROR_WIN_INC_BACKUP,
  MD_ERROR_WIN_FULL_BACKUP,
  MD_ERROR_WIN_REC_NON_EXISTENT,
  MD_ERROR_WIN_RPL_NOT_ALLOWED,
  MD_ERROR_WIN_DHCP_ADDRESS_CONFLICT,
  MD_ERROR_WIN_WMI_GUID_NOT_FOUND,
  MD_ERROR_WIN_WMI_INSTANCE_NOT_FOUND,
  MD_ERROR_WIN_WMI_ITEMID_NOT_FOUND,
  MD_ERROR_WIN_WMI_TRY_AGAIN,
  MD_ERROR_WIN_WMI_DP_NOT_FOUND,
  MD_ERROR_WIN_WMI_UNRESOLVED_INSTANCE_REF,
  MD_ERROR_WIN_WMI_ALREADY_ENABLED,
  MD_ERROR_WIN_WMI_GUID_DISCONNECTED,
  MD_ERROR_WIN_WMI_SERVER_UNAVAILABLE,
  MD_ERROR_WIN_WMI_DP_FAILED,
  MD_ERROR_WIN_WMI_INVALID_MOF,
  MD_ERROR_WIN_WMI_INVALID_REGINFO,
  MD_ERROR_WIN_WMI_ALREADY_DISABLED,
  MD_ERROR_WIN_WMI_READ_ONLY,
  MD_ERROR_WIN_WMI_SET_FAILURE,
  MD_ERROR_WIN_NOT_APPCONTAINER,
  MD_ERROR_WIN_APPCONTAINER_REQUIRED,
  MD_ERROR_WIN_NOT_SUPPORTED_IN_APPCONTAINER,
  MD_ERROR_WIN_INVALID_PACKAGE_SID_LENGTH,
  MD_ERROR_WIN_INVALID_MEDIA,
  MD_ERROR_WIN_INVALID_LIBRARY,
  MD_ERROR_WIN_INVALID_MEDIA_POOL,
  MD_ERROR_WIN_DRIVE_MEDIA_MISMATCH,
  MD_ERROR_WIN_MEDIA_OFFLINE,
  MD_ERROR_WIN_LIBRARY_OFFLINE,
  MD_ERROR_WIN_EMPTY,
  MD_ERROR_WIN_NOT_EMPTY,
  MD_ERROR_WIN_MEDIA_UNAVAILABLE,
  MD_ERROR_WIN_RESOURCE_DISABLED,
  MD_ERROR_WIN_INVALID_CLEANER,
  MD_ERROR_WIN_UNABLE_TO_CLEAN,
  MD_ERROR_WIN_OBJECT_NOT_FOUND,
  MD_ERROR_WIN_DATABASE_FAILURE,
  MD_ERROR_WIN_DATABASE_FULL,
  MD_ERROR_WIN_MEDIA_INCOMPATIBLE,
  MD_ERROR_WIN_RESOURCE_NOT_PRESENT,
  MD_ERROR_WIN_INVALID_OPERATION,
  MD_ERROR_WIN_MEDIA_NOT_AVAILABLE,
  MD_ERROR_WIN_DEVICE_NOT_AVAILABLE,
  MD_ERROR_WIN_REQUEST_REFUSED,
  MD_ERROR_WIN_INVALID_DRIVE_OBJECT,
  MD_ERROR_WIN_LIBRARY_FULL,
  MD_ERROR_WIN_MEDIUM_NOT_ACCESSIBLE,
  MD_ERROR_WIN_UNABLE_TO_LOAD_MEDIUM,
  MD_ERROR_WIN_UNABLE_TO_INVENTORY_DRIVE,
  MD_ERROR_WIN_UNABLE_TO_INVENTORY_SLOT,
  MD_ERROR_WIN_UNABLE_TO_INVENTORY_TRANSPORT,
  MD_ERROR_WIN_TRANSPORT_FULL,
  MD_ERROR_WIN_CONTROLLING_IEPORT,
  MD_ERROR_WIN_UNABLE_TO_EJECT_MOUNTED_MEDIA,
  MD_ERROR_WIN_CLEANER_SLOT_SET,
  MD_ERROR_WIN_CLEANER_SLOT_NOT_SET,
  MD_ERROR_WIN_CLEANER_CARTRIDGE_SPENT,
  MD_ERROR_WIN_UNEXPECTED_OMID,
  MD_ERROR_WIN_CANT_DELETE_LAST_ITEM,
  MD_ERROR_WIN_MESSAGE_EXCEEDS_MAX_SIZE,
  MD_ERROR_WIN_VOLUME_CONTAINS_SYS_FILES,
  MD_ERROR_WIN_INDIGENOUS_TYPE,
  MD_ERROR_WIN_NO_SUPPORTING_DRIVES,
  MD_ERROR_WIN_CLEANER_CARTRIDGE_INSTALLED,
  MD_ERROR_WIN_IEPORT_FULL,
  MD_ERROR_WIN_FILE_OFFLINE,
  MD_ERROR_WIN_REMOTE_STORAGE_NOT_ACTIVE,
  MD_ERROR_WIN_REMOTE_STORAGE_MEDIA_ERROR,
  MD_ERROR_WIN_NOT_A_REPARSE_POINT,
  MD_ERROR_WIN_REPARSE_ATTRIBUTE_CONFLICT,
  MD_ERROR_WIN_INVALID_REPARSE_DATA,
  MD_ERROR_WIN_REPARSE_TAG_INVALID,
  MD_ERROR_WIN_REPARSE_TAG_MISMATCH,
  MD_ERROR_WIN_REPARSE_POINT_ENCOUNTERED,
  MD_ERROR_WIN_APP_DATA_NOT_FOUND,
  MD_ERROR_WIN_APP_DATA_EXPIRED,
  MD_ERROR_WIN_APP_DATA_CORRUPT,
  MD_ERROR_WIN_APP_DATA_LIMIT_EXCEEDED,
  MD_ERROR_WIN_APP_DATA_REBOOT_REQUIRED,
  MD_ERROR_WIN_SECUREBOOT_ROLLBACK_DETECTED,
  MD_ERROR_WIN_SECUREBOOT_POLICY_VIOLATION,
  MD_ERROR_WIN_SECUREBOOT_INVALID_POLICY,
  MD_ERROR_WIN_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND,
  MD_ERROR_WIN_SECUREBOOT_POLICY_NOT_SIGNED,
  MD_ERROR_WIN_SECUREBOOT_NOT_ENABLED,
  MD_ERROR_WIN_SECUREBOOT_FILE_REPLACED,
  MD_ERROR_WIN_SECUREBOOT_POLICY_NOT_AUTHORIZED,
  MD_ERROR_WIN_SECUREBOOT_POLICY_UNKNOWN,
  MD_ERROR_WIN_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION,
  MD_ERROR_WIN_SECUREBOOT_PLATFORM_ID_MISMATCH,
  MD_ERROR_WIN_SECUREBOOT_POLICY_ROLLBACK_DETECTED,
  MD_ERROR_WIN_SECUREBOOT_POLICY_UPGRADE_MISMATCH,
  MD_ERROR_WIN_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING,
  MD_ERROR_WIN_SECUREBOOT_NOT_BASE_POLICY,
  MD_ERROR_WIN_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY,
  MD_ERROR_WIN_OFFLOAD_READ_FLT_NOT_SUPPORTED,
  MD_ERROR_WIN_OFFLOAD_WRITE_FLT_NOT_SUPPORTED,
  MD_ERROR_WIN_OFFLOAD_READ_FILE_NOT_SUPPORTED,
  MD_ERROR_WIN_OFFLOAD_WRITE_FILE_NOT_SUPPORTED,
  MD_ERROR_WIN_ALREADY_HAS_STREAM_ID,
  MD_ERROR_WIN_SMR_GARBAGE_COLLECTION_REQUIRED,
  MD_ERROR_WIN_WOF_WIM_HEADER_CORRUPT,
  MD_ERROR_WIN_WOF_WIM_RESOURCE_TABLE_CORRUPT,
  MD_ERROR_WIN_WOF_FILE_RESOURCE_TABLE_CORRUPT,
  MD_ERROR_WIN_VOLUME_NOT_SIS_ENABLED,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_ROLLBACK_DETECTED,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_POLICY_VIOLATION,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_INVALID_POLICY,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_TOO_MANY_POLICIES,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED,
  MD_ERROR_WIN_VSM_NOT_INITIALIZED,
  MD_ERROR_WIN_VSM_DMA_PROTECTION_NOT_IN_USE,
  MD_ERROR_WIN_PLATFORM_MANIFEST_NOT_AUTHORIZED,
  MD_ERROR_WIN_PLATFORM_MANIFEST_INVALID,
  MD_ERROR_WIN_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED,
  MD_ERROR_WIN_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED,
  MD_ERROR_WIN_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND,
  MD_ERROR_WIN_PLATFORM_MANIFEST_NOT_ACTIVE,
  MD_ERROR_WIN_PLATFORM_MANIFEST_NOT_SIGNED,
  MD_ERROR_WIN_DEPENDENT_RESOURCE_EXISTS,
  MD_ERROR_WIN_DEPENDENCY_NOT_FOUND,
  MD_ERROR_WIN_DEPENDENCY_ALREADY_EXISTS,
  MD_ERROR_WIN_RESOURCE_NOT_ONLINE,
  MD_ERROR_WIN_HOST_NODE_NOT_AVAILABLE,
  MD_ERROR_WIN_RESOURCE_NOT_AVAILABLE,
  MD_ERROR_WIN_RESOURCE_NOT_FOUND,
  MD_ERROR_WIN_SHUTDOWN_CLUSTER,
  MD_ERROR_WIN_CANT_EVICT_ACTIVE_NODE,
  MD_ERROR_WIN_OBJECT_ALREADY_EXISTS,
  MD_ERROR_WIN_OBJECT_IN_LIST,
  MD_ERROR_WIN_GROUP_NOT_AVAILABLE,
  MD_ERROR_WIN_GROUP_NOT_FOUND,
  MD_ERROR_WIN_GROUP_NOT_ONLINE,
  MD_ERROR_WIN_HOST_NODE_NOT_RESOURCE_OWNER,
  MD_ERROR_WIN_HOST_NODE_NOT_GROUP_OWNER,
  MD_ERROR_WIN_RESMON_CREATE_FAILED,
  MD_ERROR_WIN_RESMON_ONLINE_FAILED,
  MD_ERROR_WIN_RESOURCE_ONLINE,
  MD_ERROR_WIN_QUORUM_RESOURCE,
  MD_ERROR_WIN_NOT_QUORUM_CAPABLE,
  MD_ERROR_WIN_CLUSTER_SHUTTING_DOWN,
  MD_ERROR_WIN_INVALID_STATE,
  MD_ERROR_WIN_RESOURCE_PROPERTIES_STORED,
  MD_ERROR_WIN_NOT_QUORUM_CLASS,
  MD_ERROR_WIN_CORE_RESOURCE,
  MD_ERROR_WIN_QUORUM_RESOURCE_ONLINE_FAILED,
  MD_ERROR_WIN_QUORUMLOG_OPEN_FAILED,
  MD_ERROR_WIN_CLUSTERLOG_CORRUPT,
  MD_ERROR_WIN_CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE,
  MD_ERROR_WIN_CLUSTERLOG_EXCEEDS_MAXSIZE,
  MD_ERROR_WIN_CLUSTERLOG_CHKPOINT_NOT_FOUND,
  MD_ERROR_WIN_CLUSTERLOG_NOT_ENOUGH_SPACE,
  MD_ERROR_WIN_QUORUM_OWNER_ALIVE,
  MD_ERROR_WIN_NETWORK_NOT_AVAILABLE,
  MD_ERROR_WIN_NODE_NOT_AVAILABLE,
  MD_ERROR_WIN_ALL_NODES_NOT_AVAILABLE,
  MD_ERROR_WIN_RESOURCE_FAILED,
  MD_ERROR_WIN_CLUSTER_INVALID_NODE,
  MD_ERROR_WIN_CLUSTER_NODE_EXISTS,
  MD_ERROR_WIN_CLUSTER_JOIN_IN_PROGRESS,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_LOCAL_NODE_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_NETWORK_EXISTS,
  MD_ERROR_WIN_CLUSTER_NETWORK_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_NETINTERFACE_EXISTS,
  MD_ERROR_WIN_CLUSTER_NETINTERFACE_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_INVALID_REQUEST,
  MD_ERROR_WIN_CLUSTER_INVALID_NETWORK_PROVIDER,
  MD_ERROR_WIN_CLUSTER_NODE_DOWN,
  MD_ERROR_WIN_CLUSTER_NODE_UNREACHABLE,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_MEMBER,
  MD_ERROR_WIN_CLUSTER_JOIN_NOT_IN_PROGRESS,
  MD_ERROR_WIN_CLUSTER_INVALID_NETWORK,
  MD_ERROR_WIN_CLUSTER_NODE_UP,
  MD_ERROR_WIN_CLUSTER_IPADDR_IN_USE,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_PAUSED,
  MD_ERROR_WIN_CLUSTER_NO_SECURITY_CONTEXT,
  MD_ERROR_WIN_CLUSTER_NETWORK_NOT_INTERNAL,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_UP,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_DOWN,
  MD_ERROR_WIN_CLUSTER_NETWORK_ALREADY_ONLINE,
  MD_ERROR_WIN_CLUSTER_NETWORK_ALREADY_OFFLINE,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_MEMBER,
  MD_ERROR_WIN_CLUSTER_LAST_INTERNAL_NETWORK,
  MD_ERROR_WIN_CLUSTER_NETWORK_HAS_DEPENDENTS,
  MD_ERROR_WIN_INVALID_OPERATION_ON_QUORUM,
  MD_ERROR_WIN_DEPENDENCY_NOT_ALLOWED,
  MD_ERROR_WIN_CLUSTER_NODE_PAUSED,
  MD_ERROR_WIN_NODE_CANT_HOST_RESOURCE,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_READY,
  MD_ERROR_WIN_CLUSTER_NODE_SHUTTING_DOWN,
  MD_ERROR_WIN_CLUSTER_JOIN_ABORTED,
  MD_ERROR_WIN_CLUSTER_INCOMPATIBLE_VERSIONS,
  MD_ERROR_WIN_CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED,
  MD_ERROR_WIN_CLUSTER_SYSTEM_CONFIG_CHANGED,
  MD_ERROR_WIN_CLUSTER_RESOURCE_TYPE_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_RESTYPE_NOT_SUPPORTED,
  MD_ERROR_WIN_CLUSTER_RESNAME_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_NO_RPC_PACKAGES_REGISTERED,
  MD_ERROR_WIN_CLUSTER_OWNER_NOT_IN_PREFLIST,
  MD_ERROR_WIN_CLUSTER_DATABASE_SEQMISMATCH,
  MD_ERROR_WIN_RESMON_INVALID_STATE,
  MD_ERROR_WIN_CLUSTER_GUM_NOT_LOCKER,
  MD_ERROR_WIN_QUORUM_DISK_NOT_FOUND,
  MD_ERROR_WIN_DATABASE_BACKUP_CORRUPT,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_HAS_DFS_ROOT,
  MD_ERROR_WIN_RESOURCE_PROPERTY_UNCHANGEABLE,
  MD_ERROR_WIN_NO_ADMIN_ACCESS_POINT,
  MD_ERROR_WIN_CLUSTER_MEMBERSHIP_INVALID_STATE,
  MD_ERROR_WIN_CLUSTER_QUORUMLOG_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_MEMBERSHIP_HALT,
  MD_ERROR_WIN_CLUSTER_INSTANCE_ID_MISMATCH,
  MD_ERROR_WIN_CLUSTER_NETWORK_NOT_FOUND_FOR_IP,
  MD_ERROR_WIN_CLUSTER_PROPERTY_DATA_TYPE_MISMATCH,
  MD_ERROR_WIN_CLUSTER_EVICT_WITHOUT_CLEANUP,
  MD_ERROR_WIN_CLUSTER_PARAMETER_MISMATCH,
  MD_ERROR_WIN_NODE_CANNOT_BE_CLUSTERED,
  MD_ERROR_WIN_CLUSTER_WRONG_OS_VERSION,
  MD_ERROR_WIN_CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME,
  MD_ERROR_WIN_CLUSCFG_ALREADY_COMMITTED,
  MD_ERROR_WIN_CLUSCFG_ROLLBACK_FAILED,
  MD_ERROR_WIN_CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT,
  MD_ERROR_WIN_CLUSTER_OLD_VERSION,
  MD_ERROR_WIN_CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME,
  MD_ERROR_WIN_CLUSTER_NO_NET_ADAPTERS,
  MD_ERROR_WIN_CLUSTER_POISONED,
  MD_ERROR_WIN_CLUSTER_GROUP_MOVING,
  MD_ERROR_WIN_CLUSTER_RESOURCE_TYPE_BUSY,
  MD_ERROR_WIN_RESOURCE_CALL_TIMED_OUT,
  MD_ERROR_WIN_INVALID_CLUSTER_IPV6_ADDRESS,
  MD_ERROR_WIN_CLUSTER_INTERNAL_INVALID_FUNCTION,
  MD_ERROR_WIN_CLUSTER_PARAMETER_OUT_OF_BOUNDS,
  MD_ERROR_WIN_CLUSTER_PARTIAL_SEND,
  MD_ERROR_WIN_CLUSTER_REGISTRY_INVALID_FUNCTION,
  MD_ERROR_WIN_CLUSTER_INVALID_STRING_TERMINATION,
  MD_ERROR_WIN_CLUSTER_INVALID_STRING_FORMAT,
  MD_ERROR_WIN_CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS,
  MD_ERROR_WIN_CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS,
  MD_ERROR_WIN_CLUSTER_NULL_DATA,
  MD_ERROR_WIN_CLUSTER_PARTIAL_READ,
  MD_ERROR_WIN_CLUSTER_PARTIAL_WRITE,
  MD_ERROR_WIN_CLUSTER_CANT_DESERIALIZE_DATA,
  MD_ERROR_WIN_DEPENDENT_RESOURCE_PROPERTY_CONFLICT,
  MD_ERROR_WIN_CLUSTER_NO_QUORUM,
  MD_ERROR_WIN_CLUSTER_INVALID_IPV6_NETWORK,
  MD_ERROR_WIN_CLUSTER_INVALID_IPV6_TUNNEL_NETWORK,
  MD_ERROR_WIN_QUORUM_NOT_ALLOWED_IN_THIS_GROUP,
  MD_ERROR_WIN_DEPENDENCY_TREE_TOO_COMPLEX,
  MD_ERROR_WIN_EXCEPTION_IN_RESOURCE_CALL,
  MD_ERROR_WIN_CLUSTER_RHS_FAILED_INITIALIZATION,
  MD_ERROR_WIN_CLUSTER_NOT_INSTALLED,
  MD_ERROR_WIN_CLUSTER_RESOURCES_MUST_BE_ONLINE_ON_THE_SAME_NODE,
  MD_ERROR_WIN_CLUSTER_MAX_NODES_IN_CLUSTER,
  MD_ERROR_WIN_CLUSTER_TOO_MANY_NODES,
  MD_ERROR_WIN_CLUSTER_OBJECT_ALREADY_USED,
  MD_ERROR_WIN_NONCORE_GROUPS_FOUND,
  MD_ERROR_WIN_FILE_SHARE_RESOURCE_CONFLICT,
  MD_ERROR_WIN_CLUSTER_EVICT_INVALID_REQUEST,
  MD_ERROR_WIN_CLUSTER_SINGLETON_RESOURCE,
  MD_ERROR_WIN_CLUSTER_GROUP_SINGLETON_RESOURCE,
  MD_ERROR_WIN_CLUSTER_RESOURCE_PROVIDER_FAILED,
  MD_ERROR_WIN_CLUSTER_RESOURCE_CONFIGURATION_ERROR,
  MD_ERROR_WIN_CLUSTER_GROUP_BUSY,
  MD_ERROR_WIN_CLUSTER_NOT_SHARED_VOLUME,
  MD_ERROR_WIN_CLUSTER_INVALID_SECURITY_DESCRIPTOR,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUMES_IN_USE,
  MD_ERROR_WIN_CLUSTER_USE_SHARED_VOLUMES_API,
  MD_ERROR_WIN_CLUSTER_BACKUP_IN_PROGRESS,
  MD_ERROR_WIN_NON_CSV_PATH,
  MD_ERROR_WIN_CSV_VOLUME_NOT_LOCAL,
  MD_ERROR_WIN_CLUSTER_WATCHDOG_TERMINATING,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_MOVE_INCOMPATIBLE_NODES,
  MD_ERROR_WIN_CLUSTER_INVALID_NODE_WEIGHT,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_CALL,
  MD_ERROR_WIN_RESMON_SYSTEM_RESOURCES_LACKING,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_DESTINATION,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_SOURCE,
  MD_ERROR_WIN_CLUSTER_GROUP_QUEUED,
  MD_ERROR_WIN_CLUSTER_RESOURCE_LOCKED_STATUS,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUME_FAILOVER_NOT_ALLOWED,
  MD_ERROR_WIN_CLUSTER_NODE_DRAIN_IN_PROGRESS,
  MD_ERROR_WIN_CLUSTER_DISK_NOT_CONNECTED,
  MD_ERROR_WIN_DISK_NOT_CSV_CAPABLE,
  MD_ERROR_WIN_RESOURCE_NOT_IN_AVAILABLE_STORAGE,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUME_REDIRECTED,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUME_NOT_REDIRECTED,
  MD_ERROR_WIN_CLUSTER_CANNOT_RETURN_PROPERTIES,
  MD_ERROR_WIN_CLUSTER_RESOURCE_CONTAINS_UNSUPPORTED_DIFF_AREA_FOR_SHARED_VOLUMES,
  MD_ERROR_WIN_CLUSTER_RESOURCE_IS_IN_MAINTENANCE_MODE,
  MD_ERROR_WIN_CLUSTER_AFFINITY_CONFLICT,
  MD_ERROR_WIN_CLUSTER_RESOURCE_IS_REPLICA_VIRTUAL_MACHINE,
  MD_ERROR_WIN_CLUSTER_UPGRADE_INCOMPATIBLE_VERSIONS,
  MD_ERROR_WIN_CLUSTER_UPGRADE_FIX_QUORUM_NOT_SUPPORTED,
  MD_ERROR_WIN_CLUSTER_UPGRADE_RESTART_REQUIRED,
  MD_ERROR_WIN_CLUSTER_UPGRADE_IN_PROGRESS,
  MD_ERROR_WIN_CLUSTER_UPGRADE_INCOMPLETE,
  MD_ERROR_WIN_CLUSTER_NODE_IN_GRACE_PERIOD,
  MD_ERROR_WIN_CLUSTER_CSV_IO_PAUSE_TIMEOUT,
  MD_ERROR_WIN_NODE_NOT_ACTIVE_CLUSTER_MEMBER,
  MD_ERROR_WIN_CLUSTER_RESOURCE_NOT_MONITORED,
  MD_ERROR_WIN_CLUSTER_RESOURCE_DOES_NOT_SUPPORT_UNMONITORED,
  MD_ERROR_WIN_CLUSTER_RESOURCE_IS_REPLICATED,
  MD_ERROR_WIN_CLUSTER_NODE_ISOLATED,
  MD_ERROR_WIN_CLUSTER_NODE_QUARANTINED,
  MD_ERROR_WIN_CLUSTER_DATABASE_UPDATE_CONDITION_FAILED,
  MD_ERROR_WIN_CLUSTER_SPACE_DEGRADED,
  MD_ERROR_WIN_CLUSTER_TOKEN_DELEGATION_NOT_SUPPORTED,
  MD_ERROR_WIN_CLUSTER_CSV_INVALID_HANDLE,
  MD_ERROR_WIN_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR,
  MD_ERROR_WIN_GROUPSET_NOT_AVAILABLE,
  MD_ERROR_WIN_GROUPSET_NOT_FOUND,
  MD_ERROR_WIN_GROUPSET_CANT_PROVIDE,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_PARENT_NOT_FOUND,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_INVALID_HIERARCHY,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_FAILED_S2D_VALIDATION,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_S2D_CONNECTIVITY_LOSS,
  MD_ERROR_WIN_CLUSTER_INVALID_INFRASTRUCTURE_FILESERVER_NAME,
  MD_ERROR_WIN_CLUSTERSET_MANAGEMENT_CLUSTER_UNREACHABLE,
  MD_ERROR_WIN_ENCRYPTION_FAILED,
  MD_ERROR_WIN_DECRYPTION_FAILED,
  MD_ERROR_WIN_FILE_ENCRYPTED,
  MD_ERROR_WIN_NO_RECOVERY_POLICY,
  MD_ERROR_WIN_NO_EFS,
  MD_ERROR_WIN_WRONG_EFS,
  MD_ERROR_WIN_NO_USER_KEYS,
  MD_ERROR_WIN_FILE_NOT_ENCRYPTED,
  MD_ERROR_WIN_NOT_EXPORT_FORMAT,
  MD_ERROR_WIN_FILE_READ_ONLY,
  MD_ERROR_WIN_DIR_EFS_DISALLOWED,
  MD_ERROR_WIN_EFS_SERVER_NOT_TRUSTED,
  MD_ERROR_WIN_BAD_RECOVERY_POLICY,
  MD_ERROR_WIN_EFS_ALG_BLOB_TOO_BIG,
  MD_ERROR_WIN_VOLUME_NOT_SUPPORT_EFS,
  MD_ERROR_WIN_EFS_DISABLED,
  MD_ERROR_WIN_EFS_VERSION_NOT_SUPPORT,
  MD_ERROR_WIN_CS_ENCRYPTION_INVALID_SERVER_RESPONSE,
  MD_ERROR_WIN_CS_ENCRYPTION_UNSUPPORTED_SERVER,
  MD_ERROR_WIN_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE,
  MD_ERROR_WIN_CS_ENCRYPTION_NEW_ENCRYPTED_FILE,
  MD_ERROR_WIN_CS_ENCRYPTION_FILE_NOT_CSE,
  MD_ERROR_WIN_ENCRYPTION_POLICY_DENIES_OPERATION,
  MD_ERROR_WIN_WIP_ENCRYPTION_FAILED,
  MD_ERROR_WIN_NO_BROWSER_SERVERS_FOUND,
  MD_ERROR_WIN_CLUSTER_OBJECT_IS_CLUSTER_SET_VM,
  MD_ERROR_WIN_LOG_SECTOR_INVALID,
  MD_ERROR_WIN_LOG_SECTOR_PARITY_INVALID,
  MD_ERROR_WIN_LOG_SECTOR_REMAPPED,
  MD_ERROR_WIN_LOG_BLOCK_INCOMPLETE,
  MD_ERROR_WIN_LOG_INVALID_RANGE,
  MD_ERROR_WIN_LOG_BLOCKS_EXHAUSTED,
  MD_ERROR_WIN_LOG_READ_CONTEXT_INVALID,
  MD_ERROR_WIN_LOG_RESTART_INVALID,
  MD_ERROR_WIN_LOG_BLOCK_VERSION,
  MD_ERROR_WIN_LOG_BLOCK_INVALID,
  MD_ERROR_WIN_LOG_READ_MODE_INVALID,
  MD_ERROR_WIN_LOG_NO_RESTART,
  MD_ERROR_WIN_LOG_METADATA_CORRUPT,
  MD_ERROR_WIN_LOG_METADATA_INVALID,
  MD_ERROR_WIN_LOG_METADATA_INCONSISTENT,
  MD_ERROR_WIN_LOG_RESERVATION_INVALID,
  MD_ERROR_WIN_LOG_CANT_DELETE,
  MD_ERROR_WIN_LOG_CONTAINER_LIMIT_EXCEEDED,
  MD_ERROR_WIN_LOG_START_OF_LOG,
  MD_ERROR_WIN_LOG_POLICY_ALREADY_INSTALLED,
  MD_ERROR_WIN_LOG_POLICY_NOT_INSTALLED,
  MD_ERROR_WIN_LOG_POLICY_INVALID,
  MD_ERROR_WIN_LOG_POLICY_CONFLICT,
  MD_ERROR_WIN_LOG_PINNED_ARCHIVE_TAIL,
  MD_ERROR_WIN_LOG_RECORD_NONEXISTENT,
  MD_ERROR_WIN_LOG_RECORDS_RESERVED_INVALID,
  MD_ERROR_WIN_LOG_SPACE_RESERVED_INVALID,
  MD_ERROR_WIN_LOG_TAIL_INVALID,
  MD_ERROR_WIN_LOG_FULL,
  MD_ERROR_WIN_COULD_NOT_RESIZE_LOG,
  MD_ERROR_WIN_LOG_MULTIPLEXED,
  MD_ERROR_WIN_LOG_DEDICATED,
  MD_ERROR_WIN_LOG_ARCHIVE_NOT_IN_PROGRESS,
  MD_ERROR_WIN_LOG_ARCHIVE_IN_PROGRESS,
  MD_ERROR_WIN_LOG_EPHEMERAL,
  MD_ERROR_WIN_LOG_NOT_ENOUGH_CONTAINERS,
  MD_ERROR_WIN_LOG_CLIENT_ALREADY_REGISTERED,
  MD_ERROR_WIN_LOG_CLIENT_NOT_REGISTERED,
  MD_ERROR_WIN_LOG_FULL_HANDLER_IN_PROGRESS,
  MD_ERROR_WIN_LOG_CONTAINER_READ_FAILED,
  MD_ERROR_WIN_LOG_CONTAINER_WRITE_FAILED,
  MD_ERROR_WIN_LOG_CONTAINER_OPEN_FAILED,
  MD_ERROR_WIN_LOG_CONTAINER_STATE_INVALID,
  MD_ERROR_WIN_LOG_STATE_INVALID,
  MD_ERROR_WIN_LOG_PINNED,
  MD_ERROR_WIN_LOG_METADATA_FLUSH_FAILED,
  MD_ERROR_WIN_LOG_INCONSISTENT_SECURITY,
  MD_ERROR_WIN_LOG_APPENDED_FLUSH_FAILED,
  MD_ERROR_WIN_LOG_PINNED_RESERVATION,
  MD_ERROR_WIN_INVALID_TRANSACTION,
  MD_ERROR_WIN_TRANSACTION_NOT_ACTIVE,
  MD_ERROR_WIN_TRANSACTION_REQUEST_NOT_VALID,
  MD_ERROR_WIN_TRANSACTION_NOT_REQUESTED,
  MD_ERROR_WIN_TRANSACTION_ALREADY_ABORTED,
  MD_ERROR_WIN_TRANSACTION_ALREADY_COMMITTED,
  MD_ERROR_WIN_TM_INITIALIZATION_FAILED,
  MD_ERROR_WIN_RESOURCEMANAGER_READ_ONLY,
  MD_ERROR_WIN_TRANSACTION_NOT_JOINED,
  MD_ERROR_WIN_TRANSACTION_SUPERIOR_EXISTS,
  MD_ERROR_WIN_CRM_PROTOCOL_ALREADY_EXISTS,
  MD_ERROR_WIN_TRANSACTION_PROPAGATION_FAILED,
  MD_ERROR_WIN_CRM_PROTOCOL_NOT_FOUND,
  MD_ERROR_WIN_TRANSACTION_INVALID_MARSHALL_BUFFER,
  MD_ERROR_WIN_CURRENT_TRANSACTION_NOT_VALID,
  MD_ERROR_WIN_TRANSACTION_NOT_FOUND,
  MD_ERROR_WIN_RESOURCEMANAGER_NOT_FOUND,
  MD_ERROR_WIN_ENLISTMENT_NOT_FOUND,
  MD_ERROR_WIN_TRANSACTIONMANAGER_NOT_FOUND,
  MD_ERROR_WIN_TRANSACTIONMANAGER_NOT_ONLINE,
  MD_ERROR_WIN_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION,
  MD_ERROR_WIN_TRANSACTION_NOT_ROOT,
  MD_ERROR_WIN_TRANSACTION_OBJECT_EXPIRED,
  MD_ERROR_WIN_TRANSACTION_RESPONSE_NOT_ENLISTED,
  MD_ERROR_WIN_TRANSACTION_RECORD_TOO_LONG,
  MD_ERROR_WIN_IMPLICIT_TRANSACTION_NOT_SUPPORTED,
  MD_ERROR_WIN_TRANSACTION_INTEGRITY_VIOLATED,
  MD_ERROR_WIN_TRANSACTIONMANAGER_IDENTITY_MISMATCH,
  MD_ERROR_WIN_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT,
  MD_ERROR_WIN_TRANSACTION_MUST_WRITETHROUGH,
  MD_ERROR_WIN_TRANSACTION_NO_SUPERIOR,
  MD_ERROR_WIN_HEURISTIC_DAMAGE_POSSIBLE,
  MD_ERROR_WIN_TRANSACTIONAL_CONFLICT,
  MD_ERROR_WIN_RM_NOT_ACTIVE,
  MD_ERROR_WIN_RM_METADATA_CORRUPT,
  MD_ERROR_WIN_DIRECTORY_NOT_RM,
  MD_ERROR_WIN_TRANSACTIONS_UNSUPPORTED_REMOTE,
  MD_ERROR_WIN_LOG_RESIZE_INVALID_SIZE,
  MD_ERROR_WIN_OBJECT_NO_LONGER_EXISTS,
  MD_ERROR_WIN_STREAM_MINIVERSION_NOT_FOUND,
  MD_ERROR_WIN_STREAM_MINIVERSION_NOT_VALID,
  MD_ERROR_WIN_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION,
  MD_ERROR_WIN_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT,
  MD_ERROR_WIN_CANT_CREATE_MORE_STREAM_MINIVERSIONS,
  MD_ERROR_WIN_REMOTE_FILE_VERSION_MISMATCH,
  MD_ERROR_WIN_HANDLE_NO_LONGER_VALID,
  MD_ERROR_WIN_NO_TXF_METADATA,
  MD_ERROR_WIN_LOG_CORRUPTION_DETECTED,
  MD_ERROR_WIN_CANT_RECOVER_WITH_HANDLE_OPEN,
  MD_ERROR_WIN_RM_DISCONNECTED,
  MD_ERROR_WIN_ENLISTMENT_NOT_SUPERIOR,
  MD_ERROR_WIN_RECOVERY_NOT_NEEDED,
  MD_ERROR_WIN_RM_ALREADY_STARTED,
  MD_ERROR_WIN_FILE_IDENTITY_NOT_PERSISTENT,
  MD_ERROR_WIN_CANT_BREAK_TRANSACTIONAL_DEPENDENCY,
  MD_ERROR_WIN_CANT_CROSS_RM_BOUNDARY,
  MD_ERROR_WIN_TXF_DIR_NOT_EMPTY,
  MD_ERROR_WIN_INDOUBT_TRANSACTIONS_EXIST,
  MD_ERROR_WIN_TM_VOLATILE,
  MD_ERROR_WIN_ROLLBACK_TIMER_EXPIRED,
  MD_ERROR_WIN_TXF_ATTRIBUTE_CORRUPT,
  MD_ERROR_WIN_EFS_NOT_ALLOWED_IN_TRANSACTION,
  MD_ERROR_WIN_TRANSACTIONAL_OPEN_NOT_ALLOWED,
  MD_ERROR_WIN_LOG_GROWTH_FAILED,
  MD_ERROR_WIN_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE,
  MD_ERROR_WIN_TXF_METADATA_ALREADY_PRESENT,
  MD_ERROR_WIN_TRANSACTION_SCOPE_CALLBACKS_NOT_SET,
  MD_ERROR_WIN_TRANSACTION_REQUIRED_PROMOTION,
  MD_ERROR_WIN_CANNOT_EXECUTE_FILE_IN_TRANSACTION,
  MD_ERROR_WIN_TRANSACTIONS_NOT_FROZEN,
  MD_ERROR_WIN_TRANSACTION_FREEZE_IN_PROGRESS,
  MD_ERROR_WIN_NOT_SNAPSHOT_VOLUME,
  MD_ERROR_WIN_NO_SAVEPOINT_WITH_OPEN_FILES,
  MD_ERROR_WIN_DATA_LOST_REPAIR,
  MD_ERROR_WIN_SPARSE_NOT_ALLOWED_IN_TRANSACTION,
  MD_ERROR_WIN_TM_IDENTITY_MISMATCH,
  MD_ERROR_WIN_FLOATED_SECTION,
  MD_ERROR_WIN_CANNOT_ACCEPT_TRANSACTED_WORK,
  MD_ERROR_WIN_CANNOT_ABORT_TRANSACTIONS,
  MD_ERROR_WIN_BAD_CLUSTERS,
  MD_ERROR_WIN_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION,
  MD_ERROR_WIN_VOLUME_DIRTY,
  MD_ERROR_WIN_NO_LINK_TRACKING_IN_TRANSACTION,
  MD_ERROR_WIN_OPERATION_NOT_SUPPORTED_IN_TRANSACTION,
  MD_ERROR_WIN_EXPIRED_HANDLE,
  MD_ERROR_WIN_TRANSACTION_NOT_ENLISTED,
  MD_ERROR_WIN_CTX_WINSTATION_NAME_INVALID,
  MD_ERROR_WIN_CTX_INVALID_PD,
  MD_ERROR_WIN_CTX_PD_NOT_FOUND,
  MD_ERROR_WIN_CTX_WD_NOT_FOUND,
  MD_ERROR_WIN_CTX_CANNOT_MAKE_EVENTLOG_ENTRY,
  MD_ERROR_WIN_CTX_SERVICE_NAME_COLLISION,
  MD_ERROR_WIN_CTX_CLOSE_PENDING,
  MD_ERROR_WIN_CTX_NO_OUTBUF,
  MD_ERROR_WIN_CTX_MODEM_INF_NOT_FOUND,
  MD_ERROR_WIN_CTX_INVALID_MODEMNAME,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_ERROR,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_TIMEOUT,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_NO_CARRIER,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_NO_DIALTONE,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_BUSY,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_VOICE,
  MD_ERROR_WIN_CTX_TD_ERROR,
  MD_ERROR_WIN_CTX_WINSTATION_NOT_FOUND,
  MD_ERROR_WIN_CTX_WINSTATION_ALREADY_EXISTS,
  MD_ERROR_WIN_CTX_WINSTATION_BUSY,
  MD_ERROR_WIN_CTX_BAD_VIDEO_MODE,
  MD_ERROR_WIN_CTX_GRAPHICS_INVALID,
  MD_ERROR_WIN_CTX_LOGON_DISABLED,
  MD_ERROR_WIN_CTX_NOT_CONSOLE,
  MD_ERROR_WIN_CTX_CLIENT_QUERY_TIMEOUT,
  MD_ERROR_WIN_CTX_CONSOLE_DISCONNECT,
  MD_ERROR_WIN_CTX_CONSOLE_CONNECT,
  MD_ERROR_WIN_CTX_SHADOW_DENIED,
  MD_ERROR_WIN_CTX_WINSTATION_ACCESS_DENIED,
  MD_ERROR_WIN_CTX_INVALID_WD,
  MD_ERROR_WIN_CTX_SHADOW_INVALID,
  MD_ERROR_WIN_CTX_SHADOW_DISABLED,
  MD_ERROR_WIN_CTX_CLIENT_LICENSE_IN_USE,
  MD_ERROR_WIN_CTX_CLIENT_LICENSE_NOT_SET,
  MD_ERROR_WIN_CTX_LICENSE_NOT_AVAILABLE,
  MD_ERROR_WIN_CTX_LICENSE_CLIENT_INVALID,
  MD_ERROR_WIN_CTX_LICENSE_EXPIRED,
  MD_ERROR_WIN_CTX_SHADOW_NOT_RUNNING,
  MD_ERROR_WIN_CTX_SHADOW_ENDED_BY_MODE_CHANGE,
  MD_ERROR_WIN_ACTIVATION_COUNT_EXCEEDED,
  MD_ERROR_WIN_CTX_WINSTATIONS_DISABLED,
  MD_ERROR_WIN_CTX_ENCRYPTION_LEVEL_REQUIRED,
  MD_ERROR_WIN_CTX_SESSION_IN_USE,
  MD_ERROR_WIN_CTX_NO_FORCE_LOGOFF,
  MD_ERROR_WIN_CTX_ACCOUNT_RESTRICTION,
  MD_ERROR_WIN_RDP_PROTOCOL_ERROR,
  MD_ERROR_WIN_CTX_CDM_CONNECT,
  MD_ERROR_WIN_CTX_CDM_DISCONNECT,
  MD_ERROR_WIN_CTX_SECURITY_LAYER_ERROR,
  MD_ERROR_WIN_TS_INCOMPATIBLE_SESSIONS,
  MD_ERROR_WIN_TS_VIDEO_SUBSYSTEM_ERROR,
  MD_ERROR_WIN_DS_NOT_INSTALLED,
  MD_ERROR_WIN_DS_MEMBERSHIP_EVALUATED_LOCALLY,
  MD_ERROR_WIN_DS_NO_ATTRIBUTE_OR_VALUE,
  MD_ERROR_WIN_DS_INVALID_ATTRIBUTE_SYNTAX,
  MD_ERROR_WIN_DS_ATTRIBUTE_TYPE_UNDEFINED,
  MD_ERROR_WIN_DS_ATTRIBUTE_OR_VALUE_EXISTS,
  MD_ERROR_WIN_DS_BUSY,
  MD_ERROR_WIN_DS_UNAVAILABLE,
  MD_ERROR_WIN_DS_NO_RIDS_ALLOCATED,
  MD_ERROR_WIN_DS_NO_MORE_RIDS,
  MD_ERROR_WIN_DS_INCORRECT_ROLE_OWNER,
  MD_ERROR_WIN_DS_RIDMGR_INIT_ERROR,
  MD_ERROR_WIN_DS_OBJ_CLASS_VIOLATION,
  MD_ERROR_WIN_DS_CANT_ON_NON_LEAF,
  MD_ERROR_WIN_DS_CANT_ON_RDN,
  MD_ERROR_WIN_DS_CANT_MOD_OBJ_CLASS,
  MD_ERROR_WIN_DS_CROSS_DOM_MOVE_ERROR,
  MD_ERROR_WIN_DS_GC_NOT_AVAILABLE,
  MD_ERROR_WIN_SHARED_POLICY,
  MD_ERROR_WIN_POLICY_OBJECT_NOT_FOUND,
  MD_ERROR_WIN_POLICY_ONLY_IN_DS,
  MD_ERROR_WIN_PROMOTION_ACTIVE,
  MD_ERROR_WIN_NO_PROMOTION_ACTIVE,
  MD_ERROR_WIN_DS_OPERATIONS_ERROR,
  MD_ERROR_WIN_DS_PROTOCOL_ERROR,
  MD_ERROR_WIN_DS_TIMELIMIT_EXCEEDED,
  MD_ERROR_WIN_DS_SIZELIMIT_EXCEEDED,
  MD_ERROR_WIN_DS_ADMIN_LIMIT_EXCEEDED,
  MD_ERROR_WIN_DS_COMPARE_FALSE,
  MD_ERROR_WIN_DS_COMPARE_TRUE,
  MD_ERROR_WIN_DS_AUTH_METHOD_NOT_SUPPORTED,
  MD_ERROR_WIN_DS_STRONG_AUTH_REQUIRED,
  MD_ERROR_WIN_DS_INAPPROPRIATE_AUTH,
  MD_ERROR_WIN_DS_AUTH_UNKNOWN,
  MD_ERROR_WIN_DS_REFERRAL,
  MD_ERROR_WIN_DS_UNAVAILABLE_CRIT_EXTENSION,
  MD_ERROR_WIN_DS_CONFIDENTIALITY_REQUIRED,
  MD_ERROR_WIN_DS_INAPPROPRIATE_MATCHING,
  MD_ERROR_WIN_DS_CONSTRAINT_VIOLATION,
  MD_ERROR_WIN_DS_NO_SUCH_OBJECT,
  MD_ERROR_WIN_DS_ALIAS_PROBLEM,
  MD_ERROR_WIN_DS_INVALID_DN_SYNTAX,
  MD_ERROR_WIN_DS_IS_LEAF,
  MD_ERROR_WIN_DS_ALIAS_DEREF_PROBLEM,
  MD_ERROR_WIN_DS_UNWILLING_TO_PERFORM,
  MD_ERROR_WIN_DS_LOOP_DETECT,
  MD_ERROR_WIN_DS_NAMING_VIOLATION,
  MD_ERROR_WIN_DS_OBJECT_RESULTS_TOO_LARGE,
  MD_ERROR_WIN_DS_AFFECTS_MULTIPLE_DSAS,
  MD_ERROR_WIN_DS_SERVER_DOWN,
  MD_ERROR_WIN_DS_LOCAL_ERROR,
  MD_ERROR_WIN_DS_ENCODING_ERROR,
  MD_ERROR_WIN_DS_DECODING_ERROR,
  MD_ERROR_WIN_DS_FILTER_UNKNOWN,
  MD_ERROR_WIN_DS_PARAM_ERROR,
  MD_ERROR_WIN_DS_NOT_SUPPORTED,
  MD_ERROR_WIN_DS_NO_RESULTS_RETURNED,
  MD_ERROR_WIN_DS_CONTROL_NOT_FOUND,
  MD_ERROR_WIN_DS_CLIENT_LOOP,
  MD_ERROR_WIN_DS_REFERRAL_LIMIT_EXCEEDED,
  MD_ERROR_WIN_DS_SORT_CONTROL_MISSING,
  MD_ERROR_WIN_DS_OFFSET_RANGE_ERROR,
  MD_ERROR_WIN_DS_RIDMGR_DISABLED,
  MD_ERROR_WIN_DS_ROOT_MUST_BE_NC,
  MD_ERROR_WIN_DS_ADD_REPLICA_INHIBITED,
  MD_ERROR_WIN_DS_ATT_NOT_DEF_IN_SCHEMA,
  MD_ERROR_WIN_DS_MAX_OBJ_SIZE_EXCEEDED,
  MD_ERROR_WIN_DS_OBJ_STRING_NAME_EXISTS,
  MD_ERROR_WIN_DS_NO_RDN_DEFINED_IN_SCHEMA,
  MD_ERROR_WIN_DS_RDN_DOESNT_MATCH_SCHEMA,
  MD_ERROR_WIN_DS_NO_REQUESTED_ATTS_FOUND,
  MD_ERROR_WIN_DS_USER_BUFFER_TO_SMALL,
  MD_ERROR_WIN_DS_ATT_IS_NOT_ON_OBJ,
  MD_ERROR_WIN_DS_ILLEGAL_MOD_OPERATION,
  MD_ERROR_WIN_DS_OBJ_TOO_LARGE,
  MD_ERROR_WIN_DS_BAD_INSTANCE_TYPE,
  MD_ERROR_WIN_DS_MASTERDSA_REQUIRED,
  MD_ERROR_WIN_DS_OBJECT_CLASS_REQUIRED,
  MD_ERROR_WIN_DS_MISSING_REQUIRED_ATT,
  MD_ERROR_WIN_DS_ATT_NOT_DEF_FOR_CLASS,
  MD_ERROR_WIN_DS_ATT_ALREADY_EXISTS,
  MD_ERROR_WIN_DS_CANT_ADD_ATT_VALUES,
  MD_ERROR_WIN_DS_SINGLE_VALUE_CONSTRAINT,
  MD_ERROR_WIN_DS_RANGE_CONSTRAINT,
  MD_ERROR_WIN_DS_ATT_VAL_ALREADY_EXISTS,
  MD_ERROR_WIN_DS_CANT_REM_MISSING_ATT,
  MD_ERROR_WIN_DS_CANT_REM_MISSING_ATT_VAL,
  MD_ERROR_WIN_DS_ROOT_CANT_BE_SUBREF,
  MD_ERROR_WIN_DS_NO_CHAINING,
  MD_ERROR_WIN_DS_NO_CHAINED_EVAL,
  MD_ERROR_WIN_DS_NO_PARENT_OBJECT,
  MD_ERROR_WIN_DS_PARENT_IS_AN_ALIAS,
  MD_ERROR_WIN_DS_CANT_MIX_MASTER_AND_REPS,
  MD_ERROR_WIN_DS_CHILDREN_EXIST,
  MD_ERROR_WIN_DS_OBJ_NOT_FOUND,
  MD_ERROR_WIN_DS_ALIASED_OBJ_MISSING,
  MD_ERROR_WIN_DS_BAD_NAME_SYNTAX,
  MD_ERROR_WIN_DS_ALIAS_POINTS_TO_ALIAS,
  MD_ERROR_WIN_DS_CANT_DEREF_ALIAS,
  MD_ERROR_WIN_DS_OUT_OF_SCOPE,
  MD_ERROR_WIN_DS_OBJECT_BEING_REMOVED,
  MD_ERROR_WIN_DS_CANT_DELETE_DSA_OBJ,
  MD_ERROR_WIN_DS_GENERIC_ERROR,
  MD_ERROR_WIN_DS_DSA_MUST_BE_INT_MASTER,
  MD_ERROR_WIN_DS_CLASS_NOT_DSA,
  MD_ERROR_WIN_DS_INSUFF_ACCESS_RIGHTS,
  MD_ERROR_WIN_DS_ILLEGAL_SUPERIOR,
  MD_ERROR_WIN_DS_ATTRIBUTE_OWNED_BY_SAM,
  MD_ERROR_WIN_DS_NAME_TOO_MANY_PARTS,
  MD_ERROR_WIN_DS_NAME_TOO_LONG,
  MD_ERROR_WIN_DS_NAME_VALUE_TOO_LONG,
  MD_ERROR_WIN_DS_NAME_UNPARSEABLE,
  MD_ERROR_WIN_DS_NAME_TYPE_UNKNOWN,
  MD_ERROR_WIN_DS_NOT_AN_OBJECT,
  MD_ERROR_WIN_DS_SEC_DESC_TOO_SHORT,
  MD_ERROR_WIN_DS_SEC_DESC_INVALID,
  MD_ERROR_WIN_DS_NO_DELETED_NAME,
  MD_ERROR_WIN_DS_SUBREF_MUST_HAVE_PARENT,
  MD_ERROR_WIN_DS_NCNAME_MUST_BE_NC,
  MD_ERROR_WIN_DS_CANT_ADD_SYSTEM_ONLY,
  MD_ERROR_WIN_DS_CLASS_MUST_BE_CONCRETE,
  MD_ERROR_WIN_DS_INVALID_DMD,
  MD_ERROR_WIN_DS_OBJ_GUID_EXISTS,
  MD_ERROR_WIN_DS_NOT_ON_BACKLINK,
  MD_ERROR_WIN_DS_NO_CROSSREF_FOR_NC,
  MD_ERROR_WIN_DS_SHUTTING_DOWN,
  MD_ERROR_WIN_DS_UNKNOWN_OPERATION,
  MD_ERROR_WIN_DS_INVALID_ROLE_OWNER,
  MD_ERROR_WIN_DS_COULDNT_CONTACT_FSMO,
  MD_ERROR_WIN_DS_CROSS_NC_DN_RENAME,
  MD_ERROR_WIN_DS_CANT_MOD_SYSTEM_ONLY,
  MD_ERROR_WIN_DS_REPLICATOR_ONLY,
  MD_ERROR_WIN_DS_OBJ_CLASS_NOT_DEFINED,
  MD_ERROR_WIN_DS_OBJ_CLASS_NOT_SUBCLASS,
  MD_ERROR_WIN_DS_NAME_REFERENCE_INVALID,
  MD_ERROR_WIN_DS_CROSS_REF_EXISTS,
  MD_ERROR_WIN_DS_CANT_DEL_MASTER_CROSSREF,
  MD_ERROR_WIN_DS_SUBTREE_NOTIFY_NOT_NC_HEAD,
  MD_ERROR_WIN_DS_NOTIFY_FILTER_TOO_COMPLEX,
  MD_ERROR_WIN_DS_DUP_RDN,
  MD_ERROR_WIN_DS_DUP_OID,
  MD_ERROR_WIN_DS_DUP_MAPI_ID,
  MD_ERROR_WIN_DS_DUP_SCHEMA_ID_GUID,
  MD_ERROR_WIN_DS_DUP_LDAP_DISPLAY_NAME,
  MD_ERROR_WIN_DS_SEMANTIC_ATT_TEST,
  MD_ERROR_WIN_DS_SYNTAX_MISMATCH,
  MD_ERROR_WIN_DS_EXISTS_IN_MUST_HAVE,
  MD_ERROR_WIN_DS_EXISTS_IN_MAY_HAVE,
  MD_ERROR_WIN_DS_NONEXISTENT_MAY_HAVE,
  MD_ERROR_WIN_DS_NONEXISTENT_MUST_HAVE,
  MD_ERROR_WIN_DS_AUX_CLS_TEST_FAIL,
  MD_ERROR_WIN_DS_NONEXISTENT_POSS_SUP,
  MD_ERROR_WIN_DS_SUB_CLS_TEST_FAIL,
  MD_ERROR_WIN_DS_BAD_RDN_ATT_ID_SYNTAX,
  MD_ERROR_WIN_DS_EXISTS_IN_AUX_CLS,
  MD_ERROR_WIN_DS_EXISTS_IN_SUB_CLS,
  MD_ERROR_WIN_DS_EXISTS_IN_POSS_SUP,
  MD_ERROR_WIN_DS_RECALCSCHEMA_FAILED,
  MD_ERROR_WIN_DS_TREE_DELETE_NOT_FINISHED,
  MD_ERROR_WIN_DS_CANT_DELETE,
  MD_ERROR_WIN_DS_ATT_SCHEMA_REQ_ID,
  MD_ERROR_WIN_DS_BAD_ATT_SCHEMA_SYNTAX,
  MD_ERROR_WIN_DS_CANT_CACHE_ATT,
  MD_ERROR_WIN_DS_CANT_CACHE_CLASS,
  MD_ERROR_WIN_DS_CANT_REMOVE_ATT_CACHE,
  MD_ERROR_WIN_DS_CANT_REMOVE_CLASS_CACHE,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_DN,
  MD_ERROR_WIN_DS_MISSING_SUPREF,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_INSTANCE,
  MD_ERROR_WIN_DS_CODE_INCONSISTENCY,
  MD_ERROR_WIN_DS_DATABASE_ERROR,
  MD_ERROR_WIN_DS_GOVERNSID_MISSING,
  MD_ERROR_WIN_DS_MISSING_EXPECTED_ATT,
  MD_ERROR_WIN_DS_NCNAME_MISSING_CR_REF,
  MD_ERROR_WIN_DS_SECURITY_CHECKING_ERROR,
  MD_ERROR_WIN_DS_SCHEMA_NOT_LOADED,
  MD_ERROR_WIN_DS_SCHEMA_ALLOC_FAILED,
  MD_ERROR_WIN_DS_ATT_SCHEMA_REQ_SYNTAX,
  MD_ERROR_WIN_DS_GCVERIFY_ERROR,
  MD_ERROR_WIN_DS_DRA_SCHEMA_MISMATCH,
  MD_ERROR_WIN_DS_CANT_FIND_DSA_OBJ,
  MD_ERROR_WIN_DS_CANT_FIND_EXPECTED_NC,
  MD_ERROR_WIN_DS_CANT_FIND_NC_IN_CACHE,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_CHILD,
  MD_ERROR_WIN_DS_SECURITY_ILLEGAL_MODIFY,
  MD_ERROR_WIN_DS_CANT_REPLACE_HIDDEN_REC,
  MD_ERROR_WIN_DS_BAD_HIERARCHY_FILE,
  MD_ERROR_WIN_DS_BUILD_HIERARCHY_TABLE_FAILED,
  MD_ERROR_WIN_DS_CONFIG_PARAM_MISSING,
  MD_ERROR_WIN_DS_COUNTING_AB_INDICES_FAILED,
  MD_ERROR_WIN_DS_HIERARCHY_TABLE_MALLOC_FAILED,
  MD_ERROR_WIN_DS_INTERNAL_FAILURE,
  MD_ERROR_WIN_DS_UNKNOWN_ERROR,
  MD_ERROR_WIN_DS_ROOT_REQUIRES_CLASS_TOP,
  MD_ERROR_WIN_DS_REFUSING_FSMO_ROLES,
  MD_ERROR_WIN_DS_MISSING_FSMO_SETTINGS,
  MD_ERROR_WIN_DS_UNABLE_TO_SURRENDER_ROLES,
  MD_ERROR_WIN_DS_DRA_GENERIC,
  MD_ERROR_WIN_DS_DRA_INVALID_PARAMETER,
  MD_ERROR_WIN_DS_DRA_BUSY,
  MD_ERROR_WIN_DS_DRA_BAD_DN,
  MD_ERROR_WIN_DS_DRA_BAD_NC,
  MD_ERROR_WIN_DS_DRA_DN_EXISTS,
  MD_ERROR_WIN_DS_DRA_INTERNAL_ERROR,
  MD_ERROR_WIN_DS_DRA_INCONSISTENT_DIT,
  MD_ERROR_WIN_DS_DRA_CONNECTION_FAILED,
  MD_ERROR_WIN_DS_DRA_BAD_INSTANCE_TYPE,
  MD_ERROR_WIN_DS_DRA_OUT_OF_MEM,
  MD_ERROR_WIN_DS_DRA_MAIL_PROBLEM,
  MD_ERROR_WIN_DS_DRA_REF_ALREADY_EXISTS,
  MD_ERROR_WIN_DS_DRA_REF_NOT_FOUND,
  MD_ERROR_WIN_DS_DRA_OBJ_IS_REP_SOURCE,
  MD_ERROR_WIN_DS_DRA_DB_ERROR,
  MD_ERROR_WIN_DS_DRA_NO_REPLICA,
  MD_ERROR_WIN_DS_DRA_ACCESS_DENIED,
  MD_ERROR_WIN_DS_DRA_NOT_SUPPORTED,
  MD_ERROR_WIN_DS_DRA_RPC_CANCELLED,
  MD_ERROR_WIN_DS_DRA_SOURCE_DISABLED,
  MD_ERROR_WIN_DS_DRA_SINK_DISABLED,
  MD_ERROR_WIN_DS_DRA_NAME_COLLISION,
  MD_ERROR_WIN_DS_DRA_SOURCE_REINSTALLED,
  MD_ERROR_WIN_DS_DRA_MISSING_PARENT,
  MD_ERROR_WIN_DS_DRA_PREEMPTED,
  MD_ERROR_WIN_DS_DRA_ABANDON_SYNC,
  MD_ERROR_WIN_DS_DRA_SHUTDOWN,
  MD_ERROR_WIN_DS_DRA_INCOMPATIBLE_PARTIAL_SET,
  MD_ERROR_WIN_DS_DRA_SOURCE_IS_PARTIAL_REPLICA,
  MD_ERROR_WIN_DS_DRA_EXTN_CONNECTION_FAILED,
  MD_ERROR_WIN_DS_INSTALL_SCHEMA_MISMATCH,
  MD_ERROR_WIN_DS_DUP_LINK_ID,
  MD_ERROR_WIN_DS_NAME_ERROR_RESOLVING,
  MD_ERROR_WIN_DS_NAME_ERROR_NOT_FOUND,
  MD_ERROR_WIN_DS_NAME_ERROR_NOT_UNIQUE,
  MD_ERROR_WIN_DS_NAME_ERROR_NO_MAPPING,
  MD_ERROR_WIN_DS_NAME_ERROR_DOMAIN_ONLY,
  MD_ERROR_WIN_DS_NAME_ERROR_NO_SYNTACTICAL_MAPPING,
  MD_ERROR_WIN_DS_CONSTRUCTED_ATT_MOD,
  MD_ERROR_WIN_DS_WRONG_OM_OBJ_CLASS,
  MD_ERROR_WIN_DS_DRA_REPL_PENDING,
  MD_ERROR_WIN_DS_DS_REQUIRED,
  MD_ERROR_WIN_DS_INVALID_LDAP_DISPLAY_NAME,
  MD_ERROR_WIN_DS_NON_BASE_SEARCH,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_ATTS,
  MD_ERROR_WIN_DS_BACKLINK_WITHOUT_LINK,
  MD_ERROR_WIN_DS_EPOCH_MISMATCH,
  MD_ERROR_WIN_DS_SRC_NAME_MISMATCH,
  MD_ERROR_WIN_DS_SRC_AND_DST_NC_IDENTICAL,
  MD_ERROR_WIN_DS_DST_NC_MISMATCH,
  MD_ERROR_WIN_DS_NOT_AUTHORITIVE_FOR_DST_NC,
  MD_ERROR_WIN_DS_SRC_GUID_MISMATCH,
  MD_ERROR_WIN_DS_CANT_MOVE_DELETED_OBJECT,
  MD_ERROR_WIN_DS_PDC_OPERATION_IN_PROGRESS,
  MD_ERROR_WIN_DS_CROSS_DOMAIN_CLEANUP_REQD,
  MD_ERROR_WIN_DS_ILLEGAL_XDOM_MOVE_OPERATION,
  MD_ERROR_WIN_DS_CANT_WITH_ACCT_GROUP_MEMBERSHPS,
  MD_ERROR_WIN_DS_NC_MUST_HAVE_NC_PARENT,
  MD_ERROR_WIN_DS_CR_IMPOSSIBLE_TO_VALIDATE,
  MD_ERROR_WIN_DS_DST_DOMAIN_NOT_NATIVE,
  MD_ERROR_WIN_DS_MISSING_INFRASTRUCTURE_CONTAINER,
  MD_ERROR_WIN_DS_CANT_MOVE_ACCOUNT_GROUP,
  MD_ERROR_WIN_DS_CANT_MOVE_RESOURCE_GROUP,
  MD_ERROR_WIN_DS_INVALID_SEARCH_FLAG,
  MD_ERROR_WIN_DS_NO_TREE_DELETE_ABOVE_NC,
  MD_ERROR_WIN_DS_COULDNT_LOCK_TREE_FOR_DELETE,
  MD_ERROR_WIN_DS_COULDNT_IDENTIFY_OBJECTS_FOR_TREE_DELETE,
  MD_ERROR_WIN_DS_SAM_INIT_FAILURE,
  MD_ERROR_WIN_DS_SENSITIVE_GROUP_VIOLATION,
  MD_ERROR_WIN_DS_CANT_MOD_PRIMARYGROUPID,
  MD_ERROR_WIN_DS_ILLEGAL_BASE_SCHEMA_MOD,
  MD_ERROR_WIN_DS_NONSAFE_SCHEMA_CHANGE,
  MD_ERROR_WIN_DS_SCHEMA_UPDATE_DISALLOWED,
  MD_ERROR_WIN_DS_CANT_CREATE_UNDER_SCHEMA,
  MD_ERROR_WIN_DS_INSTALL_NO_SRC_SCH_VERSION,
  MD_ERROR_WIN_DS_INSTALL_NO_SCH_VERSION_IN_INIFILE,
  MD_ERROR_WIN_DS_INVALID_GROUP_TYPE,
  MD_ERROR_WIN_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN,
  MD_ERROR_WIN_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN,
  MD_ERROR_WIN_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER,
  MD_ERROR_WIN_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER,
  MD_ERROR_WIN_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER,
  MD_ERROR_WIN_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER,
  MD_ERROR_WIN_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER,
  MD_ERROR_WIN_DS_HAVE_PRIMARY_MEMBERS,
  MD_ERROR_WIN_DS_STRING_SD_CONVERSION_FAILED,
  MD_ERROR_WIN_DS_NAMING_MASTER_GC,
  MD_ERROR_WIN_DS_DNS_LOOKUP_FAILURE,
  MD_ERROR_WIN_DS_COULDNT_UPDATE_SPNS,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_SD,
  MD_ERROR_WIN_DS_KEY_NOT_UNIQUE,
  MD_ERROR_WIN_DS_WRONG_LINKED_ATT_SYNTAX,
  MD_ERROR_WIN_DS_SAM_NEED_BOOTKEY_PASSWORD,
  MD_ERROR_WIN_DS_SAM_NEED_BOOTKEY_FLOPPY,
  MD_ERROR_WIN_DS_CANT_START,
  MD_ERROR_WIN_DS_INIT_FAILURE,
  MD_ERROR_WIN_DS_NO_PKT_PRIVACY_ON_CONNECTION,
  MD_ERROR_WIN_DS_SOURCE_DOMAIN_IN_FOREST,
  MD_ERROR_WIN_DS_DESTINATION_DOMAIN_NOT_IN_FOREST,
  MD_ERROR_WIN_DS_DESTINATION_AUDITING_NOT_ENABLED,
  MD_ERROR_WIN_DS_CANT_FIND_DC_FOR_SRC_DOMAIN,
  MD_ERROR_WIN_DS_SRC_OBJ_NOT_GROUP_OR_USER,
  MD_ERROR_WIN_DS_SRC_SID_EXISTS_IN_FOREST,
  MD_ERROR_WIN_DS_SRC_AND_DST_OBJECT_CLASS_MISMATCH,
  MD_ERROR_WIN_SAM_INIT_FAILURE,
  MD_ERROR_WIN_DS_DRA_SCHEMA_INFO_SHIP,
  MD_ERROR_WIN_DS_DRA_SCHEMA_CONFLICT,
  MD_ERROR_WIN_DS_DRA_EARLIER_SCHEMA_CONFLICT,
  MD_ERROR_WIN_DS_DRA_OBJ_NC_MISMATCH,
  MD_ERROR_WIN_DS_NC_STILL_HAS_DSAS,
  MD_ERROR_WIN_DS_GC_REQUIRED,
  MD_ERROR_WIN_DS_LOCAL_MEMBER_OF_LOCAL_ONLY,
  MD_ERROR_WIN_DS_NO_FPO_IN_UNIVERSAL_GROUPS,
  MD_ERROR_WIN_DS_CANT_ADD_TO_GC,
  MD_ERROR_WIN_DS_NO_CHECKPOINT_WITH_PDC,
  MD_ERROR_WIN_DS_SOURCE_AUDITING_NOT_ENABLED,
  MD_ERROR_WIN_DS_CANT_CREATE_IN_NONDOMAIN_NC,
  MD_ERROR_WIN_DS_INVALID_NAME_FOR_SPN,
  MD_ERROR_WIN_DS_FILTER_USES_CONTRUCTED_ATTRS,
  MD_ERROR_WIN_DS_UNICODEPWD_NOT_IN_QUOTES,
  MD_ERROR_WIN_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED,
  MD_ERROR_WIN_DS_MUST_BE_RUN_ON_DST_DC,
  MD_ERROR_WIN_DS_SRC_DC_MUST_BE_SP4_OR_GREATER,
  MD_ERROR_WIN_DS_CANT_TREE_DELETE_CRITICAL_OBJ,
  MD_ERROR_WIN_DS_INIT_FAILURE_CONSOLE,
  MD_ERROR_WIN_DS_SAM_INIT_FAILURE_CONSOLE,
  MD_ERROR_WIN_DS_FOREST_VERSION_TOO_HIGH,
  MD_ERROR_WIN_DS_DOMAIN_VERSION_TOO_HIGH,
  MD_ERROR_WIN_DS_FOREST_VERSION_TOO_LOW,
  MD_ERROR_WIN_DS_DOMAIN_VERSION_TOO_LOW,
  MD_ERROR_WIN_DS_INCOMPATIBLE_VERSION,
  MD_ERROR_WIN_DS_LOW_DSA_VERSION,
  MD_ERROR_WIN_DS_NO_BEHAVIOR_VERSION_IN_MIXEDDOMAIN,
  MD_ERROR_WIN_DS_NOT_SUPPORTED_SORT_ORDER,
  MD_ERROR_WIN_DS_NAME_NOT_UNIQUE,
  MD_ERROR_WIN_DS_MACHINE_ACCOUNT_CREATED_PRENT4,
  MD_ERROR_WIN_DS_OUT_OF_VERSION_STORE,
  MD_ERROR_WIN_DS_INCOMPATIBLE_CONTROLS_USED,
  MD_ERROR_WIN_DS_NO_REF_DOMAIN,
  MD_ERROR_WIN_DS_RESERVED_LINK_ID,
  MD_ERROR_WIN_DS_LINK_ID_NOT_AVAILABLE,
  MD_ERROR_WIN_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER,
  MD_ERROR_WIN_DS_MODIFYDN_DISALLOWED_BY_INSTANCE_TYPE,
  MD_ERROR_WIN_DS_NO_OBJECT_MOVE_IN_SCHEMA_NC,
  MD_ERROR_WIN_DS_MODIFYDN_DISALLOWED_BY_FLAG,
  MD_ERROR_WIN_DS_MODIFYDN_WRONG_GRANDPARENT,
  MD_ERROR_WIN_DS_NAME_ERROR_TRUST_REFERRAL,
  MD_ERROR_WIN_NOT_SUPPORTED_ON_STANDARD_SERVER,
  MD_ERROR_WIN_DS_CANT_ACCESS_REMOTE_PART_OF_AD,
  MD_ERROR_WIN_DS_CR_IMPOSSIBLE_TO_VALIDATE_V2,
  MD_ERROR_WIN_DS_THREAD_LIMIT_EXCEEDED,
  MD_ERROR_WIN_DS_NOT_CLOSEST,
  MD_ERROR_WIN_DS_CANT_DERIVE_SPN_WITHOUT_SERVER_REF,
  MD_ERROR_WIN_DS_SINGLE_USER_MODE_FAILED,
  MD_ERROR_WIN_DS_NTDSCRIPT_SYNTAX_ERROR,
  MD_ERROR_WIN_DS_NTDSCRIPT_PROCESS_ERROR,
  MD_ERROR_WIN_DS_DIFFERENT_REPL_EPOCHS,
  MD_ERROR_WIN_DS_DRS_EXTENSIONS_CHANGED,
  MD_ERROR_WIN_DS_REPLICA_SET_CHANGE_NOT_ALLOWED_ON_DISABLED_CR,
  MD_ERROR_WIN_DS_NO_MSDS_INTID,
  MD_ERROR_WIN_DS_DUP_MSDS_INTID,
  MD_ERROR_WIN_DS_EXISTS_IN_RDNATTID,
  MD_ERROR_WIN_DS_AUTHORIZATION_FAILED,
  MD_ERROR_WIN_DS_INVALID_SCRIPT,
  MD_ERROR_WIN_DS_REMOTE_CROSSREF_OP_FAILED,
  MD_ERROR_WIN_DS_CROSS_REF_BUSY,
  MD_ERROR_WIN_DS_CANT_DERIVE_SPN_FOR_DELETED_DOMAIN,
  MD_ERROR_WIN_DS_CANT_DEMOTE_WITH_WRITEABLE_NC,
  MD_ERROR_WIN_DS_DUPLICATE_ID_FOUND,
  MD_ERROR_WIN_DS_INSUFFICIENT_ATTR_TO_CREATE_OBJECT,
  MD_ERROR_WIN_DS_GROUP_CONVERSION_ERROR,
  MD_ERROR_WIN_DS_CANT_MOVE_APP_BASIC_GROUP,
  MD_ERROR_WIN_DS_CANT_MOVE_APP_QUERY_GROUP,
  MD_ERROR_WIN_DS_ROLE_NOT_VERIFIED,
  MD_ERROR_WIN_DS_WKO_CONTAINER_CANNOT_BE_SPECIAL,
  MD_ERROR_WIN_DS_DOMAIN_RENAME_IN_PROGRESS,
  MD_ERROR_WIN_DS_EXISTING_AD_CHILD_NC,
  MD_ERROR_WIN_DS_REPL_LIFETIME_EXCEEDED,
  MD_ERROR_WIN_DS_DISALLOWED_IN_SYSTEM_CONTAINER,
  MD_ERROR_WIN_DS_LDAP_SEND_QUEUE_FULL,
  MD_ERROR_WIN_DS_DRA_OUT_SCHEDULE_WINDOW,
  MD_ERROR_WIN_DS_POLICY_NOT_KNOWN,
  MD_ERROR_WIN_NO_SITE_SETTINGS_OBJECT,
  MD_ERROR_WIN_NO_SECRETS,
  MD_ERROR_WIN_NO_WRITABLE_DC_FOUND,
  MD_ERROR_WIN_DS_NO_SERVER_OBJECT,
  MD_ERROR_WIN_DS_NO_NTDSA_OBJECT,
  MD_ERROR_WIN_DS_NON_ASQ_SEARCH,
  MD_ERROR_WIN_DS_AUDIT_FAILURE,
  MD_ERROR_WIN_DS_INVALID_SEARCH_FLAG_SUBTREE,
  MD_ERROR_WIN_DS_INVALID_SEARCH_FLAG_TUPLE,
  MD_ERROR_WIN_DS_HIERARCHY_TABLE_TOO_DEEP,
  MD_ERROR_WIN_DS_DRA_CORRUPT_UTD_VECTOR,
  MD_ERROR_WIN_DS_DRA_SECRETS_DENIED,
  MD_ERROR_WIN_DS_RESERVED_MAPI_ID,
  MD_ERROR_WIN_DS_MAPI_ID_NOT_AVAILABLE,
  MD_ERROR_WIN_DS_DRA_MISSING_KRBTGT_SECRET,
  MD_ERROR_WIN_DS_DOMAIN_NAME_EXISTS_IN_FOREST,
  MD_ERROR_WIN_DS_FLAT_NAME_EXISTS_IN_FOREST,
  MD_ERROR_WIN_INVALID_USER_PRINCIPAL_NAME,
  MD_ERROR_WIN_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS,
  MD_ERROR_WIN_DS_OID_NOT_FOUND,
  MD_ERROR_WIN_DS_DRA_RECYCLED_TARGET,
  MD_ERROR_WIN_DS_DISALLOWED_NC_REDIRECT,
  MD_ERROR_WIN_DS_HIGH_ADLDS_FFL,
  MD_ERROR_WIN_DS_HIGH_DSA_VERSION,
  MD_ERROR_WIN_DS_LOW_ADLDS_FFL,
  MD_ERROR_WIN_DOMAIN_SID_SAME_AS_LOCAL_WORKSTATION,
  MD_ERROR_WIN_DS_UNDELETE_SAM_VALIDATION_FAILED,
  MD_ERROR_WIN_INCORRECT_ACCOUNT_TYPE,
  MD_ERROR_WIN_DS_SPN_VALUE_NOT_UNIQUE_IN_FOREST,
  MD_ERROR_WIN_DS_UPN_VALUE_NOT_UNIQUE_IN_FOREST,
  MD_ERROR_WIN_DS_MISSING_FOREST_TRUST,
  MD_ERROR_WIN_DS_VALUE_KEY_NOT_UNIQUE,
  MD_ERROR_WIN_IPSEC_QM_POLICY_EXISTS,
  MD_ERROR_WIN_IPSEC_QM_POLICY_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_QM_POLICY_IN_USE,
  MD_ERROR_WIN_IPSEC_MM_POLICY_EXISTS,
  MD_ERROR_WIN_IPSEC_MM_POLICY_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_MM_POLICY_IN_USE,
  MD_ERROR_WIN_IPSEC_MM_FILTER_EXISTS,
  MD_ERROR_WIN_IPSEC_MM_FILTER_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_TRANSPORT_FILTER_EXISTS,
  MD_ERROR_WIN_IPSEC_TRANSPORT_FILTER_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_MM_AUTH_EXISTS,
  MD_ERROR_WIN_IPSEC_MM_AUTH_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_MM_AUTH_IN_USE,
  MD_ERROR_WIN_IPSEC_DEFAULT_MM_POLICY_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_DEFAULT_MM_AUTH_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_DEFAULT_QM_POLICY_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_TUNNEL_FILTER_EXISTS,
  MD_ERROR_WIN_IPSEC_TUNNEL_FILTER_NOT_FOUND,
  MD_ERROR_WIN_IPSEC_MM_FILTER_PENDING_DELETION,
  MD_ERROR_WIN_IPSEC_TRANSPORT_FILTER_PENDING_DELETION,
  MD_ERROR_WIN_IPSEC_TUNNEL_FILTER_PENDING_DELETION,
  MD_ERROR_WIN_IPSEC_MM_POLICY_PENDING_DELETION,
  MD_ERROR_WIN_IPSEC_MM_AUTH_PENDING_DELETION,
  MD_ERROR_WIN_IPSEC_QM_POLICY_PENDING_DELETION,
  MD_ERROR_WIN_IPSEC_IKE_NEG_STATUS_BEGIN,
  MD_ERROR_WIN_IPSEC_IKE_AUTH_FAIL,
  MD_ERROR_WIN_IPSEC_IKE_ATTRIB_FAIL,
  MD_ERROR_WIN_IPSEC_IKE_NEGOTIATION_PENDING,
  MD_ERROR_WIN_IPSEC_IKE_GENERAL_PROCESSING_ERROR,
  MD_ERROR_WIN_IPSEC_IKE_TIMED_OUT,
  MD_ERROR_WIN_IPSEC_IKE_NO_CERT,
  MD_ERROR_WIN_IPSEC_IKE_SA_DELETED,
  MD_ERROR_WIN_IPSEC_IKE_SA_REAPED,
  MD_ERROR_WIN_IPSEC_IKE_MM_ACQUIRE_DROP,
  MD_ERROR_WIN_IPSEC_IKE_QM_ACQUIRE_DROP,
  MD_ERROR_WIN_IPSEC_IKE_QUEUE_DROP_MM,
  MD_ERROR_WIN_IPSEC_IKE_QUEUE_DROP_NO_MM,
  MD_ERROR_WIN_IPSEC_IKE_DROP_NO_RESPONSE,
  MD_ERROR_WIN_IPSEC_IKE_MM_DELAY_DROP,
  MD_ERROR_WIN_IPSEC_IKE_QM_DELAY_DROP,
  MD_ERROR_WIN_IPSEC_IKE_ERROR,
  MD_ERROR_WIN_IPSEC_IKE_CRL_FAILED,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_KEY_USAGE,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_CERT_TYPE,
  MD_ERROR_WIN_IPSEC_IKE_NO_PRIVATE_KEY,
  MD_ERROR_WIN_IPSEC_IKE_SIMULTANEOUS_REKEY,
  MD_ERROR_WIN_IPSEC_IKE_DH_FAIL,
  MD_ERROR_WIN_IPSEC_IKE_CRITICAL_PAYLOAD_NOT_RECOGNIZED,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HEADER,
  MD_ERROR_WIN_IPSEC_IKE_NO_POLICY,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_SIGNATURE,
  MD_ERROR_WIN_IPSEC_IKE_KERBEROS_ERROR,
  MD_ERROR_WIN_IPSEC_IKE_NO_PUBLIC_KEY,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_SA,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_PROP,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_TRANS,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_KE,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_ID,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_CERT,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_CERT_REQ,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_HASH,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_SIG,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_NONCE,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_NOTIFY,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_DELETE,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_VENDOR,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_PAYLOAD,
  MD_ERROR_WIN_IPSEC_IKE_LOAD_SOFT_SA,
  MD_ERROR_WIN_IPSEC_IKE_SOFT_SA_TORN_DOWN,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_COOKIE,
  MD_ERROR_WIN_IPSEC_IKE_NO_PEER_CERT,
  MD_ERROR_WIN_IPSEC_IKE_PEER_CRL_FAILED,
  MD_ERROR_WIN_IPSEC_IKE_POLICY_CHANGE,
  MD_ERROR_WIN_IPSEC_IKE_NO_MM_POLICY,
  MD_ERROR_WIN_IPSEC_IKE_NOTCBPRIV,
  MD_ERROR_WIN_IPSEC_IKE_SECLOADFAIL,
  MD_ERROR_WIN_IPSEC_IKE_FAILSSPINIT,
  MD_ERROR_WIN_IPSEC_IKE_FAILQUERYSSP,
  MD_ERROR_WIN_IPSEC_IKE_SRVACQFAIL,
  MD_ERROR_WIN_IPSEC_IKE_SRVQUERYCRED,
  MD_ERROR_WIN_IPSEC_IKE_GETSPIFAIL,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_FILTER,
  MD_ERROR_WIN_IPSEC_IKE_OUT_OF_MEMORY,
  MD_ERROR_WIN_IPSEC_IKE_ADD_UPDATE_KEY_FAILED,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_POLICY,
  MD_ERROR_WIN_IPSEC_IKE_UNKNOWN_DOI,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_SITUATION,
  MD_ERROR_WIN_IPSEC_IKE_DH_FAILURE,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_GROUP,
  MD_ERROR_WIN_IPSEC_IKE_ENCRYPT,
  MD_ERROR_WIN_IPSEC_IKE_DECRYPT,
  MD_ERROR_WIN_IPSEC_IKE_POLICY_MATCH,
  MD_ERROR_WIN_IPSEC_IKE_UNSUPPORTED_ID,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HASH,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HASH_ALG,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HASH_SIZE,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_ENCRYPT_ALG,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_AUTH_ALG,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_SIG,
  MD_ERROR_WIN_IPSEC_IKE_LOAD_FAILED,
  MD_ERROR_WIN_IPSEC_IKE_RPC_DELETE,
  MD_ERROR_WIN_IPSEC_IKE_BENIGN_REINIT,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_RESPONDER_LIFETIME_NOTIFY,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_MAJOR_VERSION,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_CERT_KEYLEN,
  MD_ERROR_WIN_IPSEC_IKE_MM_LIMIT,
  MD_ERROR_WIN_IPSEC_IKE_NEGOTIATION_DISABLED,
  MD_ERROR_WIN_IPSEC_IKE_QM_LIMIT,
  MD_ERROR_WIN_IPSEC_IKE_MM_EXPIRED,
  MD_ERROR_WIN_IPSEC_IKE_PEER_MM_ASSUMED_INVALID,
  MD_ERROR_WIN_IPSEC_IKE_CERT_CHAIN_POLICY_MISMATCH,
  MD_ERROR_WIN_IPSEC_IKE_UNEXPECTED_MESSAGE_ID,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_AUTH_PAYLOAD,
  MD_ERROR_WIN_IPSEC_IKE_DOS_COOKIE_SENT,
  MD_ERROR_WIN_IPSEC_IKE_SHUTTING_DOWN,
  MD_ERROR_WIN_IPSEC_IKE_CGA_AUTH_FAILED,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_NATOA,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_MM_FOR_QM,
  MD_ERROR_WIN_IPSEC_IKE_QM_EXPIRED,
  MD_ERROR_WIN_IPSEC_IKE_TOO_MANY_FILTERS,
  MD_ERROR_WIN_IPSEC_IKE_NEG_STATUS_END,
  MD_ERROR_WIN_IPSEC_IKE_KILL_DUMMY_NAP_TUNNEL,
  MD_ERROR_WIN_IPSEC_IKE_INNER_IP_ASSIGNMENT_FAILURE,
  MD_ERROR_WIN_IPSEC_IKE_REQUIRE_CP_PAYLOAD_MISSING,
  MD_ERROR_WIN_IPSEC_KEY_MODULE_IMPERSONATION_NEGOTIATION_PENDING,
  MD_ERROR_WIN_IPSEC_IKE_COEXISTENCE_SUPPRESS,
  MD_ERROR_WIN_IPSEC_IKE_RATELIMIT_DROP,
  MD_ERROR_WIN_IPSEC_IKE_PEER_DOESNT_SUPPORT_MOBIKE,
  MD_ERROR_WIN_IPSEC_IKE_AUTHORIZATION_FAILURE,
  MD_ERROR_WIN_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_FAILURE,
  MD_ERROR_WIN_IPSEC_IKE_AUTHORIZATION_FAILURE_WITH_OPTIONAL_RETRY,
  MD_ERROR_WIN_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_AND_CERTMAP_FAILURE,
  MD_ERROR_WIN_IPSEC_IKE_NEG_STATUS_EXTENDED_END,
  MD_ERROR_WIN_IPSEC_BAD_SPI,
  MD_ERROR_WIN_IPSEC_SA_LIFETIME_EXPIRED,
  MD_ERROR_WIN_IPSEC_WRONG_SA,
  MD_ERROR_WIN_IPSEC_REPLAY_CHECK_FAILED,
  MD_ERROR_WIN_IPSEC_INVALID_PACKET,
  MD_ERROR_WIN_IPSEC_INTEGRITY_CHECK_FAILED,
  MD_ERROR_WIN_IPSEC_CLEAR_TEXT_DROP,
  MD_ERROR_WIN_IPSEC_AUTH_FIREWALL_DROP,
  MD_ERROR_WIN_IPSEC_THROTTLE_DROP,
  MD_ERROR_WIN_IPSEC_DOSP_BLOCK,
  MD_ERROR_WIN_IPSEC_DOSP_RECEIVED_MULTICAST,
  MD_ERROR_WIN_IPSEC_DOSP_INVALID_PACKET,
  MD_ERROR_WIN_IPSEC_DOSP_STATE_LOOKUP_FAILED,
  MD_ERROR_WIN_IPSEC_DOSP_MAX_ENTRIES,
  MD_ERROR_WIN_IPSEC_DOSP_KEYMOD_NOT_ALLOWED,
  MD_ERROR_WIN_IPSEC_DOSP_NOT_INSTALLED,
  MD_ERROR_WIN_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES,
  MD_ERROR_WIN_SXS_SECTION_NOT_FOUND,
  MD_ERROR_WIN_SXS_CANT_GEN_ACTCTX,
  MD_ERROR_WIN_SXS_INVALID_ACTCTXDATA_FORMAT,
  MD_ERROR_WIN_SXS_ASSEMBLY_NOT_FOUND,
  MD_ERROR_WIN_SXS_MANIFEST_FORMAT_ERROR,
  MD_ERROR_WIN_SXS_MANIFEST_PARSE_ERROR,
  MD_ERROR_WIN_SXS_ACTIVATION_CONTEXT_DISABLED,
  MD_ERROR_WIN_SXS_KEY_NOT_FOUND,
  MD_ERROR_WIN_SXS_VERSION_CONFLICT,
  MD_ERROR_WIN_SXS_WRONG_SECTION_TYPE,
  MD_ERROR_WIN_SXS_THREAD_QUERIES_DISABLED,
  MD_ERROR_WIN_SXS_PROCESS_DEFAULT_ALREADY_SET,
  MD_ERROR_WIN_SXS_UNKNOWN_ENCODING_GROUP,
  MD_ERROR_WIN_SXS_UNKNOWN_ENCODING,
  MD_ERROR_WIN_SXS_INVALID_XML_NAMESPACE_URI,
  MD_ERROR_WIN_SXS_ROOT_MANIFEST_DEPENDENCY_NOT_INSTALLED,
  MD_ERROR_WIN_SXS_LEAF_MANIFEST_DEPENDENCY_NOT_INSTALLED,
  MD_ERROR_WIN_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE,
  MD_ERROR_WIN_SXS_MANIFEST_MISSING_REQUIRED_DEFAULT_NAMESPACE,
  MD_ERROR_WIN_SXS_MANIFEST_INVALID_REQUIRED_DEFAULT_NAMESPACE,
  MD_ERROR_WIN_SXS_PRIVATE_MANIFEST_CROSS_PATH_WITH_REPARSE_POINT,
  MD_ERROR_WIN_SXS_DUPLICATE_DLL_NAME,
  MD_ERROR_WIN_SXS_DUPLICATE_WINDOWCLASS_NAME,
  MD_ERROR_WIN_SXS_DUPLICATE_CLSID,
  MD_ERROR_WIN_SXS_DUPLICATE_IID,
  MD_ERROR_WIN_SXS_DUPLICATE_TLBID,
  MD_ERROR_WIN_SXS_DUPLICATE_PROGID,
  MD_ERROR_WIN_SXS_DUPLICATE_ASSEMBLY_NAME,
  MD_ERROR_WIN_SXS_FILE_HASH_MISMATCH,
  MD_ERROR_WIN_SXS_POLICY_PARSE_ERROR,
  MD_ERROR_WIN_SXS_XML_E_MISSINGQUOTE,
  MD_ERROR_WIN_SXS_XML_E_COMMENTSYNTAX,
  MD_ERROR_WIN_SXS_XML_E_BADSTARTNAMECHAR,
  MD_ERROR_WIN_SXS_XML_E_BADNAMECHAR,
  MD_ERROR_WIN_SXS_XML_E_BADCHARINSTRING,
  MD_ERROR_WIN_SXS_XML_E_XMLDECLSYNTAX,
  MD_ERROR_WIN_SXS_XML_E_BADCHARDATA,
  MD_ERROR_WIN_SXS_XML_E_MISSINGWHITESPACE,
  MD_ERROR_WIN_SXS_XML_E_EXPECTINGTAGEND,
  MD_ERROR_WIN_SXS_XML_E_MISSINGSEMICOLON,
  MD_ERROR_WIN_SXS_XML_E_UNBALANCEDPAREN,
  MD_ERROR_WIN_SXS_XML_E_INTERNALERROR,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTED_WHITESPACE,
  MD_ERROR_WIN_SXS_XML_E_INCOMPLETE_ENCODING,
  MD_ERROR_WIN_SXS_XML_E_MISSING_PAREN,
  MD_ERROR_WIN_SXS_XML_E_EXPECTINGCLOSEQUOTE,
  MD_ERROR_WIN_SXS_XML_E_MULTIPLE_COLONS,
  MD_ERROR_WIN_SXS_XML_E_INVALID_DECIMAL,
  MD_ERROR_WIN_SXS_XML_E_INVALID_HEXIDECIMAL,
  MD_ERROR_WIN_SXS_XML_E_INVALID_UNICODE,
  MD_ERROR_WIN_SXS_XML_E_WHITESPACEORQUESTIONMARK,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTEDENDTAG,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDTAG,
  MD_ERROR_WIN_SXS_XML_E_DUPLICATEATTRIBUTE,
  MD_ERROR_WIN_SXS_XML_E_MULTIPLEROOTS,
  MD_ERROR_WIN_SXS_XML_E_INVALIDATROOTLEVEL,
  MD_ERROR_WIN_SXS_XML_E_BADXMLDECL,
  MD_ERROR_WIN_SXS_XML_E_MISSINGROOT,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTEDEOF,
  MD_ERROR_WIN_SXS_XML_E_BADPEREFINSUBSET,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDSTARTTAG,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDENDTAG,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDSTRING,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDCOMMENT,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDDECL,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDCDATA,
  MD_ERROR_WIN_SXS_XML_E_RESERVEDNAMESPACE,
  MD_ERROR_WIN_SXS_XML_E_INVALIDENCODING,
  MD_ERROR_WIN_SXS_XML_E_INVALIDSWITCH,
  MD_ERROR_WIN_SXS_XML_E_BADXMLCASE,
  MD_ERROR_WIN_SXS_XML_E_INVALID_STANDALONE,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTED_STANDALONE,
  MD_ERROR_WIN_SXS_XML_E_INVALID_VERSION,
  MD_ERROR_WIN_SXS_XML_E_MISSINGEQUALS,
  MD_ERROR_WIN_SXS_PROTECTION_RECOVERY_FAILED,
  MD_ERROR_WIN_SXS_PROTECTION_PUBLIC_KEY_TOO_SHORT,
  MD_ERROR_WIN_SXS_PROTECTION_CATALOG_NOT_VALID,
  MD_ERROR_WIN_SXS_UNTRANSLATABLE_HRESULT,
  MD_ERROR_WIN_SXS_PROTECTION_CATALOG_FILE_MISSING,
  MD_ERROR_WIN_SXS_MISSING_ASSEMBLY_IDENTITY_ATTRIBUTE,
  MD_ERROR_WIN_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE_NAME,
  MD_ERROR_WIN_SXS_ASSEMBLY_MISSING,
  MD_ERROR_WIN_SXS_CORRUPT_ACTIVATION_STACK,
  MD_ERROR_WIN_SXS_CORRUPTION,
  MD_ERROR_WIN_SXS_EARLY_DEACTIVATION,
  MD_ERROR_WIN_SXS_INVALID_DEACTIVATION,
  MD_ERROR_WIN_SXS_MULTIPLE_DEACTIVATION,
  MD_ERROR_WIN_SXS_PROCESS_TERMINATION_REQUESTED,
  MD_ERROR_WIN_SXS_RELEASE_ACTIVATION_CONTEXT,
  MD_ERROR_WIN_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY,
  MD_ERROR_WIN_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE,
  MD_ERROR_WIN_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME,
  MD_ERROR_WIN_SXS_IDENTITY_DUPLICATE_ATTRIBUTE,
  MD_ERROR_WIN_SXS_IDENTITY_PARSE_ERROR,
  MD_ERROR_WIN_MALFORMED_SUBSTITUTION_STRING,
  MD_ERROR_WIN_SXS_INCORRECT_PUBLIC_KEY_TOKEN,
  MD_ERROR_WIN_UNMAPPED_SUBSTITUTION_STRING,
  MD_ERROR_WIN_SXS_ASSEMBLY_NOT_LOCKED,
  MD_ERROR_WIN_SXS_COMPONENT_STORE_CORRUPT,
  MD_ERROR_WIN_ADVANCED_INSTALLER_FAILED,
  MD_ERROR_WIN_XML_ENCODING_MISMATCH,
  MD_ERROR_WIN_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT,
  MD_ERROR_WIN_SXS_IDENTITIES_DIFFERENT,
  MD_ERROR_WIN_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT,
  MD_ERROR_WIN_SXS_FILE_NOT_PART_OF_ASSEMBLY,
  MD_ERROR_WIN_SXS_MANIFEST_TOO_BIG,
  MD_ERROR_WIN_SXS_SETTING_NOT_REGISTERED,
  MD_ERROR_WIN_SXS_TRANSACTION_CLOSURE_INCOMPLETE,
  MD_ERROR_WIN_SMI_PRIMITIVE_INSTALLER_FAILED,
  MD_ERROR_WIN_GENERIC_COMMAND_FAILED,
  MD_ERROR_WIN_SXS_FILE_HASH_MISSING,
  MD_ERROR_WIN_SXS_DUPLICATE_ACTIVATABLE_CLASS,
  MD_ERROR_WIN_EVT_INVALID_CHANNEL_PATH,
  MD_ERROR_WIN_EVT_INVALID_QUERY,
  MD_ERROR_WIN_EVT_PUBLISHER_METADATA_NOT_FOUND,
  MD_ERROR_WIN_EVT_EVENT_TEMPLATE_NOT_FOUND,
  MD_ERROR_WIN_EVT_INVALID_PUBLISHER_NAME,
  MD_ERROR_WIN_EVT_INVALID_EVENT_DATA,
  MD_ERROR_WIN_EVT_CHANNEL_NOT_FOUND,
  MD_ERROR_WIN_EVT_MALFORMED_XML_TEXT,
  MD_ERROR_WIN_EVT_SUBSCRIPTION_TO_DIRECT_CHANNEL,
  MD_ERROR_WIN_EVT_CONFIGURATION_ERROR,
  MD_ERROR_WIN_EVT_QUERY_RESULT_STALE,
  MD_ERROR_WIN_EVT_QUERY_RESULT_INVALID_POSITION,
  MD_ERROR_WIN_EVT_NON_VALIDATING_MSXML,
  MD_ERROR_WIN_EVT_FILTER_ALREADYSCOPED,
  MD_ERROR_WIN_EVT_FILTER_NOTELTSET,
  MD_ERROR_WIN_EVT_FILTER_INVARG,
  MD_ERROR_WIN_EVT_FILTER_INVTEST,
  MD_ERROR_WIN_EVT_FILTER_INVTYPE,
  MD_ERROR_WIN_EVT_FILTER_PARSEERR,
  MD_ERROR_WIN_EVT_FILTER_UNSUPPORTEDOP,
  MD_ERROR_WIN_EVT_FILTER_UNEXPECTEDTOKEN,
  MD_ERROR_WIN_EVT_INVALID_OPERATION_OVER_ENABLED_DIRECT_CHANNEL,
  MD_ERROR_WIN_EVT_INVALID_CHANNEL_PROPERTY_VALUE,
  MD_ERROR_WIN_EVT_INVALID_PUBLISHER_PROPERTY_VALUE,
  MD_ERROR_WIN_EVT_CHANNEL_CANNOT_ACTIVATE,
  MD_ERROR_WIN_EVT_FILTER_TOO_COMPLEX,
  MD_ERROR_WIN_EVT_MESSAGE_NOT_FOUND,
  MD_ERROR_WIN_EVT_MESSAGE_ID_NOT_FOUND,
  MD_ERROR_WIN_EVT_UNRESOLVED_VALUE_INSERT,
  MD_ERROR_WIN_EVT_UNRESOLVED_PARAMETER_INSERT,
  MD_ERROR_WIN_EVT_MAX_INSERTS_REACHED,
  MD_ERROR_WIN_EVT_EVENT_DEFINITION_NOT_FOUND,
  MD_ERROR_WIN_EVT_MESSAGE_LOCALE_NOT_FOUND,
  MD_ERROR_WIN_EVT_VERSION_TOO_OLD,
  MD_ERROR_WIN_EVT_VERSION_TOO_NEW,
  MD_ERROR_WIN_EVT_CANNOT_OPEN_CHANNEL_OF_QUERY,
  MD_ERROR_WIN_EVT_PUBLISHER_DISABLED,
  MD_ERROR_WIN_EVT_FILTER_OUT_OF_RANGE,
  MD_ERROR_WIN_EC_SUBSCRIPTION_CANNOT_ACTIVATE,
  MD_ERROR_WIN_EC_LOG_DISABLED,
  MD_ERROR_WIN_EC_CIRCULAR_FORWARDING,
  MD_ERROR_WIN_EC_CREDSTORE_FULL,
  MD_ERROR_WIN_EC_CRED_NOT_FOUND,
  MD_ERROR_WIN_EC_NO_ACTIVE_CHANNEL,
  MD_ERROR_WIN_MUI_FILE_NOT_FOUND,
  MD_ERROR_WIN_MUI_INVALID_FILE,
  MD_ERROR_WIN_MUI_INVALID_RC_CONFIG,
  MD_ERROR_WIN_MUI_INVALID_LOCALE_NAME,
  MD_ERROR_WIN_MUI_INVALID_ULTIMATEFALLBACK_NAME,
  MD_ERROR_WIN_MUI_FILE_NOT_LOADED,
  MD_ERROR_WIN_RESOURCE_ENUM_USER_STOP,
  MD_ERROR_WIN_MUI_INTLSETTINGS_UILANG_NOT_INSTALLED,
  MD_ERROR_WIN_MUI_INTLSETTINGS_INVALID_LOCALE_NAME,
  MD_ERROR_WIN_MRM_RUNTIME_NO_DEFAULT_OR_NEUTRAL_RESOURCE,
  MD_ERROR_WIN_MRM_INVALID_PRICONFIG,
  MD_ERROR_WIN_MRM_INVALID_FILE_TYPE,
  MD_ERROR_WIN_MRM_UNKNOWN_QUALIFIER,
  MD_ERROR_WIN_MRM_INVALID_QUALIFIER_VALUE,
  MD_ERROR_WIN_MRM_NO_CANDIDATE,
  MD_ERROR_WIN_MRM_NO_MATCH_OR_DEFAULT_CANDIDATE,
  MD_ERROR_WIN_MRM_RESOURCE_TYPE_MISMATCH,
  MD_ERROR_WIN_MRM_DUPLICATE_MAP_NAME,
  MD_ERROR_WIN_MRM_DUPLICATE_ENTRY,
  MD_ERROR_WIN_MRM_INVALID_RESOURCE_IDENTIFIER,
  MD_ERROR_WIN_MRM_FILEPATH_TOO_LONG,
  MD_ERROR_WIN_MRM_UNSUPPORTED_DIRECTORY_TYPE,
  MD_ERROR_WIN_MRM_INVALID_PRI_FILE,
  MD_ERROR_WIN_MRM_NAMED_RESOURCE_NOT_FOUND,
  MD_ERROR_WIN_MRM_MAP_NOT_FOUND,
  MD_ERROR_WIN_MRM_UNSUPPORTED_PROFILE_TYPE,
  MD_ERROR_WIN_MRM_INVALID_QUALIFIER_OPERATOR,
  MD_ERROR_WIN_MRM_INDETERMINATE_QUALIFIER_VALUE,
  MD_ERROR_WIN_MRM_AUTOMERGE_ENABLED,
  MD_ERROR_WIN_MRM_TOO_MANY_RESOURCES,
  MD_ERROR_WIN_MRM_UNSUPPORTED_FILE_TYPE_FOR_MERGE,
  MD_ERROR_WIN_MRM_UNSUPPORTED_FILE_TYPE_FOR_LOAD_UNLOAD_PRI_FILE,
  MD_ERROR_WIN_MRM_NO_CURRENT_VIEW_ON_THREAD,
  MD_ERROR_WIN_DIFFERENT_PROFILE_RESOURCE_MANAGER_EXIST,
  MD_ERROR_WIN_OPERATION_NOT_ALLOWED_FROM_SYSTEM_COMPONENT,
  MD_ERROR_WIN_MRM_DIRECT_REF_TO_NON_DEFAULT_RESOURCE,
  MD_ERROR_WIN_MRM_GENERATION_COUNT_MISMATCH,
  MD_ERROR_WIN_PRI_MERGE_VERSION_MISMATCH,
  MD_ERROR_WIN_PRI_MERGE_MISSING_SCHEMA,
  MD_ERROR_WIN_PRI_MERGE_LOAD_FILE_FAILED,
  MD_ERROR_WIN_PRI_MERGE_ADD_FILE_FAILED,
  MD_ERROR_WIN_PRI_MERGE_WRITE_FILE_FAILED,
  MD_ERROR_WIN_PRI_MERGE_MULTIPLE_PACKAGE_FAMILIES_NOT_ALLOWED,
  MD_ERROR_WIN_PRI_MERGE_MULTIPLE_MAIN_PACKAGES_NOT_ALLOWED,
  MD_ERROR_WIN_PRI_MERGE_BUNDLE_PACKAGES_NOT_ALLOWED,
  MD_ERROR_WIN_PRI_MERGE_MAIN_PACKAGE_REQUIRED,
  MD_ERROR_WIN_PRI_MERGE_RESOURCE_PACKAGE_REQUIRED,
  MD_ERROR_WIN_PRI_MERGE_INVALID_FILE_NAME,
  MD_ERROR_WIN_MRM_PACKAGE_NOT_FOUND,
  MD_ERROR_WIN_MRM_MISSING_DEFAULT_LANGUAGE,
  MD_ERROR_WIN_MCA_INVALID_CAPABILITIES_STRING,
  MD_ERROR_WIN_MCA_INVALID_VCP_VERSION,
  MD_ERROR_WIN_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION,
  MD_ERROR_WIN_MCA_MCCS_VERSION_MISMATCH,
  MD_ERROR_WIN_MCA_UNSUPPORTED_MCCS_VERSION,
  MD_ERROR_WIN_MCA_INTERNAL_ERROR,
  MD_ERROR_WIN_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED,
  MD_ERROR_WIN_MCA_UNSUPPORTED_COLOR_TEMPERATURE,
  MD_ERROR_WIN_AMBIGUOUS_SYSTEM_DEVICE,
  MD_ERROR_WIN_SYSTEM_DEVICE_NOT_FOUND,
  MD_ERROR_WIN_HASH_NOT_SUPPORTED,
  MD_ERROR_WIN_HASH_NOT_PRESENT,
  MD_ERROR_WIN_SECONDARY_IC_PROVIDER_NOT_REGISTERED,
  MD_ERROR_WIN_GPIO_CLIENT_INFORMATION_INVALID,
  MD_ERROR_WIN_GPIO_VERSION_NOT_SUPPORTED,
  MD_ERROR_WIN_GPIO_INVALID_REGISTRATION_PACKET,
  MD_ERROR_WIN_GPIO_OPERATION_DENIED,
  MD_ERROR_WIN_GPIO_INCOMPATIBLE_CONNECT_MODE,
  MD_ERROR_WIN_GPIO_INTERRUPT_ALREADY_UNMASKED,
  MD_ERROR_WIN_CANNOT_SWITCH_RUNLEVEL,
  MD_ERROR_WIN_INVALID_RUNLEVEL_SETTING,
  MD_ERROR_WIN_RUNLEVEL_SWITCH_TIMEOUT,
  MD_ERROR_WIN_RUNLEVEL_SWITCH_AGENT_TIMEOUT,
  MD_ERROR_WIN_RUNLEVEL_SWITCH_IN_PROGRESS,
  MD_ERROR_WIN_SERVICES_FAILED_AUTOSTART,
  MD_ERROR_WIN_COM_TASK_STOP_PENDING,
  MD_ERROR_WIN_INSTALL_OPEN_PACKAGE_FAILED,
  MD_ERROR_WIN_INSTALL_PACKAGE_NOT_FOUND,
  MD_ERROR_WIN_INSTALL_INVALID_PACKAGE,
  MD_ERROR_WIN_INSTALL_RESOLVE_DEPENDENCY_FAILED,
  MD_ERROR_WIN_INSTALL_OUT_OF_DISK_SPACE,
  MD_ERROR_WIN_INSTALL_NETWORK_FAILURE,
  MD_ERROR_WIN_INSTALL_REGISTRATION_FAILURE,
  MD_ERROR_WIN_INSTALL_DEREGISTRATION_FAILURE,
  MD_ERROR_WIN_INSTALL_CANCEL,
  MD_ERROR_WIN_INSTALL_FAILED,
  MD_ERROR_WIN_REMOVE_FAILED,
  MD_ERROR_WIN_PACKAGE_ALREADY_EXISTS,
  MD_ERROR_WIN_NEEDS_REMEDIATION,
  MD_ERROR_WIN_INSTALL_PREREQUISITE_FAILED,
  MD_ERROR_WIN_PACKAGE_REPOSITORY_CORRUPTED,
  MD_ERROR_WIN_INSTALL_POLICY_FAILURE,
  MD_ERROR_WIN_PACKAGE_UPDATING,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_POLICY,
  MD_ERROR_WIN_PACKAGES_IN_USE,
  MD_ERROR_WIN_RECOVERY_FILE_CORRUPT,
  MD_ERROR_WIN_INVALID_STAGED_SIGNATURE,
  MD_ERROR_WIN_DELETING_EXISTING_APPLICATIONDATA_STORE_FAILED,
  MD_ERROR_WIN_INSTALL_PACKAGE_DOWNGRADE,
  MD_ERROR_WIN_SYSTEM_NEEDS_REMEDIATION,
  MD_ERROR_WIN_APPX_INTEGRITY_FAILURE_CLR_NGEN,
  MD_ERROR_WIN_RESILIENCY_FILE_CORRUPT,
  MD_ERROR_WIN_INSTALL_FIREWALL_SERVICE_NOT_RUNNING,
  MD_ERROR_WIN_PACKAGE_MOVE_FAILED,
  MD_ERROR_WIN_INSTALL_VOLUME_NOT_EMPTY,
  MD_ERROR_WIN_INSTALL_VOLUME_OFFLINE,
  MD_ERROR_WIN_INSTALL_VOLUME_CORRUPT,
  MD_ERROR_WIN_NEEDS_REGISTRATION,
  MD_ERROR_WIN_INSTALL_WRONG_PROCESSOR_ARCHITECTURE,
  MD_ERROR_WIN_DEV_SIDELOAD_LIMIT_EXCEEDED,
  MD_ERROR_WIN_INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE,
  MD_ERROR_WIN_PACKAGE_NOT_SUPPORTED_ON_FILESYSTEM,
  MD_ERROR_WIN_PACKAGE_MOVE_BLOCKED_BY_STREAMING,
  MD_ERROR_WIN_INSTALL_OPTIONAL_PACKAGE_APPLICATIONID_NOT_UNIQUE,
  MD_ERROR_WIN_PACKAGE_STAGING_ONHOLD,
  MD_ERROR_WIN_INSTALL_INVALID_RELATED_SET_UPDATE,
  MD_ERROR_WIN_INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_USER_LOG_OFF,
  MD_ERROR_WIN_PROVISION_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_PROVISIONED,
  MD_ERROR_WIN_PACKAGES_REPUTATION_CHECK_FAILED,
  MD_ERROR_WIN_PACKAGES_REPUTATION_CHECK_TIMEDOUT,
  MD_ERROR_WIN_DEPLOYMENT_OPTION_NOT_SUPPORTED,
  MD_ERROR_WIN_APPINSTALLER_ACTIVATION_BLOCKED,
  MD_ERROR_WIN_REGISTRATION_FROM_REMOTE_DRIVE_NOT_SUPPORTED,
  MD_ERROR_WIN_APPX_RAW_DATA_WRITE_FAILED,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_PACKAGE,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_MACHINE,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_PROFILE_POLICY,
  MD_ERROR_WIN_DEPLOYMENT_FAILED_CONFLICTING_MUTABLE_PACKAGE_DIRECTORY,
  MD_ERROR_WIN_SINGLETON_RESOURCE_INSTALLED_IN_ACTIVE_USER,
  MD_ERROR_WIN_DIFFERENT_VERSION_OF_PACKAGED_SERVICE_INSTALLED,
  MD_ERROR_WIN_SERVICE_EXISTS_AS_NON_PACKAGED_SERVICE,
  MD_ERROR_WIN_PACKAGED_SERVICE_REQUIRES_ADMIN_PRIVILEGES,
  MD_ERROR_WIN_REDIRECTION_TO_DEFAULT_ACCOUNT_NOT_ALLOWED,
  MD_ERROR_WIN_PACKAGE_LACKS_CAPABILITY_TO_DEPLOY_ON_HOST,
  MD_ERROR_WIN_UNSIGNED_PACKAGE_INVALID_CONTENT,
  MD_ERROR_WIN_UNSIGNED_PACKAGE_INVALID_PUBLISHER_NAMESPACE,
  MD_ERROR_WIN_SIGNED_PACKAGE_INVALID_PUBLISHER_NAMESPACE,
  MD_ERROR_WIN_PACKAGE_EXTERNAL_LOCATION_NOT_ALLOWED,
  MD_ERROR_WIN_INSTALL_FULLTRUST_HOSTRUNTIME_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY,
  MD_ERROR_WIN_STATE_LOAD_STORE_FAILED,
  MD_ERROR_WIN_STATE_GET_VERSION_FAILED,
  MD_ERROR_WIN_STATE_SET_VERSION_FAILED,
  MD_ERROR_WIN_STATE_STRUCTURED_RESET_FAILED,
  MD_ERROR_WIN_STATE_OPEN_CONTAINER_FAILED,
  MD_ERROR_WIN_STATE_CREATE_CONTAINER_FAILED,
  MD_ERROR_WIN_STATE_DELETE_CONTAINER_FAILED,
  MD_ERROR_WIN_STATE_READ_SETTING_FAILED,
  MD_ERROR_WIN_STATE_WRITE_SETTING_FAILED,
  MD_ERROR_WIN_STATE_DELETE_SETTING_FAILED,
  MD_ERROR_WIN_STATE_QUERY_SETTING_FAILED,
  MD_ERROR_WIN_STATE_READ_COMPOSITE_SETTING_FAILED,
  MD_ERROR_WIN_STATE_WRITE_COMPOSITE_SETTING_FAILED,
  MD_ERROR_WIN_STATE_ENUMERATE_CONTAINER_FAILED,
  MD_ERROR_WIN_STATE_ENUMERATE_SETTINGS_FAILED,
  MD_ERROR_WIN_STATE_COMPOSITE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED,
  MD_ERROR_WIN_STATE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED,
  MD_ERROR_WIN_STATE_SETTING_NAME_SIZE_LIMIT_EXCEEDED,
  MD_ERROR_WIN_STATE_CONTAINER_NAME_SIZE_LIMIT_EXCEEDED,
  MD_ERROR_WIN_API_UNAVAILABLE,
};

// The content of this array was created from winerror.h in the 10 SDK
// (version 10.0.19041.0) with the following script:
//
// egrep -o '#define ERROR_[A-Z_0-9]+\s+[0-9]+L' winerror.h
// | tr -d '\r'
// | sed -r 's@#define (ERROR_[A-Z_0-9]+)\s+([0-9]+)L@\2 \1@'
// | sort -n | sed -r 's@([0-9]+) ([A-Z_0-9]+)@  "\2",@'
static const char* kWinErrorStrings[] = {
  "ERROR_SUCCESS",
  "ERROR_INVALID_FUNCTION",
  "ERROR_FILE_NOT_FOUND",
  "ERROR_PATH_NOT_FOUND",
  "ERROR_TOO_MANY_OPEN_FILES",
  "ERROR_ACCESS_DENIED",
  "ERROR_INVALID_HANDLE",
  "ERROR_ARENA_TRASHED",
  "ERROR_NOT_ENOUGH_MEMORY",
  "ERROR_INVALID_BLOCK",
  "ERROR_BAD_ENVIRONMENT",
  "ERROR_BAD_FORMAT",
  "ERROR_INVALID_ACCESS",
  "ERROR_INVALID_DATA",
  "ERROR_OUTOFMEMORY",
  "ERROR_INVALID_DRIVE",
  "ERROR_CURRENT_DIRECTORY",
  "ERROR_NOT_SAME_DEVICE",
  "ERROR_NO_MORE_FILES",
  "ERROR_WRITE_PROTECT",
  "ERROR_BAD_UNIT",
  "ERROR_NOT_READY",
  "ERROR_BAD_COMMAND",
  "ERROR_CRC",
  "ERROR_BAD_LENGTH",
  "ERROR_SEEK",
  "ERROR_NOT_DOS_DISK",
  "ERROR_SECTOR_NOT_FOUND",
  "ERROR_OUT_OF_PAPER",
  "ERROR_WRITE_FAULT",
  "ERROR_READ_FAULT",
  "ERROR_GEN_FAILURE",
  "ERROR_SHARING_VIOLATION",
  "ERROR_LOCK_VIOLATION",
  "ERROR_WRONG_DISK",
  "ERROR_SHARING_BUFFER_EXCEEDED",
  "ERROR_HANDLE_EOF",
  "ERROR_HANDLE_DISK_FULL",
  "ERROR_NOT_SUPPORTED",
  "ERROR_REM_NOT_LIST",
  "ERROR_DUP_NAME",
  "ERROR_BAD_NETPATH",
  "ERROR_NETWORK_BUSY",
  "ERROR_DEV_NOT_EXIST",
  "ERROR_TOO_MANY_CMDS",
  "ERROR_ADAP_HDW_ERR",
  "ERROR_BAD_NET_RESP",
  "ERROR_UNEXP_NET_ERR",
  "ERROR_BAD_REM_ADAP",
  "ERROR_PRINTQ_FULL",
  "ERROR_NO_SPOOL_SPACE",
  "ERROR_PRINT_CANCELLED",
  "ERROR_NETNAME_DELETED",
  "ERROR_NETWORK_ACCESS_DENIED",
  "ERROR_BAD_DEV_TYPE",
  "ERROR_BAD_NET_NAME",
  "ERROR_TOO_MANY_NAMES",
  "ERROR_TOO_MANY_SESS",
  "ERROR_SHARING_PAUSED",
  "ERROR_REQ_NOT_ACCEP",
  "ERROR_REDIR_PAUSED",
  "ERROR_FILE_EXISTS",
  "ERROR_CANNOT_MAKE",
  "ERROR_FAIL_I24",
  "ERROR_OUT_OF_STRUCTURES",
  "ERROR_ALREADY_ASSIGNED",
  "ERROR_INVALID_PASSWORD",
  "ERROR_INVALID_PARAMETER",
  "ERROR_NET_WRITE_FAULT",
  "ERROR_NO_PROC_SLOTS",
  "ERROR_TOO_MANY_SEMAPHORES",
  "ERROR_EXCL_SEM_ALREADY_OWNED",
  "ERROR_SEM_IS_SET",
  "ERROR_TOO_MANY_SEM_REQUESTS",
  "ERROR_INVALID_AT_INTERRUPT_TIME",
  "ERROR_SEM_OWNER_DIED",
  "ERROR_SEM_USER_LIMIT",
  "ERROR_DISK_CHANGE",
  "ERROR_DRIVE_LOCKED",
  "ERROR_BROKEN_PIPE",
  "ERROR_OPEN_FAILED",
  "ERROR_BUFFER_OVERFLOW",
  "ERROR_DISK_FULL",
  "ERROR_NO_MORE_SEARCH_HANDLES",
  "ERROR_INVALID_TARGET_HANDLE",
  "ERROR_INVALID_CATEGORY",
  "ERROR_INVALID_VERIFY_SWITCH",
  "ERROR_BAD_DRIVER_LEVEL",
  "ERROR_CALL_NOT_IMPLEMENTED",
  "ERROR_SEM_TIMEOUT",
  "ERROR_INSUFFICIENT_BUFFER",
  "ERROR_INVALID_NAME",
  "ERROR_INVALID_LEVEL",
  "ERROR_NO_VOLUME_LABEL",
  "ERROR_MOD_NOT_FOUND",
  "ERROR_PROC_NOT_FOUND",
  "ERROR_WAIT_NO_CHILDREN",
  "ERROR_CHILD_NOT_COMPLETE",
  "ERROR_DIRECT_ACCESS_HANDLE",
  "ERROR_NEGATIVE_SEEK",
  "ERROR_SEEK_ON_DEVICE",
  "ERROR_IS_JOIN_TARGET",
  "ERROR_IS_JOINED",
  "ERROR_IS_SUBSTED",
  "ERROR_NOT_JOINED",
  "ERROR_NOT_SUBSTED",
  "ERROR_JOIN_TO_JOIN",
  "ERROR_SUBST_TO_SUBST",
  "ERROR_JOIN_TO_SUBST",
  "ERROR_SUBST_TO_JOIN",
  "ERROR_BUSY_DRIVE",
  "ERROR_SAME_DRIVE",
  "ERROR_DIR_NOT_ROOT",
  "ERROR_DIR_NOT_EMPTY",
  "ERROR_IS_SUBST_PATH",
  "ERROR_IS_JOIN_PATH",
  "ERROR_PATH_BUSY",
  "ERROR_IS_SUBST_TARGET",
  "ERROR_SYSTEM_TRACE",
  "ERROR_INVALID_EVENT_COUNT",
  "ERROR_TOO_MANY_MUXWAITERS",
  "ERROR_INVALID_LIST_FORMAT",
  "ERROR_LABEL_TOO_LONG",
  "ERROR_TOO_MANY_TCBS",
  "ERROR_SIGNAL_REFUSED",
  "ERROR_DISCARDED",
  "ERROR_NOT_LOCKED",
  "ERROR_BAD_THREADID_ADDR",
  "ERROR_BAD_ARGUMENTS",
  "ERROR_BAD_PATHNAME",
  "ERROR_SIGNAL_PENDING",
  "ERROR_MAX_THRDS_REACHED",
  "ERROR_LOCK_FAILED",
  "ERROR_BUSY",
  "ERROR_DEVICE_SUPPORT_IN_PROGRESS",
  "ERROR_CANCEL_VIOLATION",
  "ERROR_ATOMIC_LOCKS_NOT_SUPPORTED",
  "ERROR_INVALID_SEGMENT_NUMBER",
  "ERROR_INVALID_ORDINAL",
  "ERROR_ALREADY_EXISTS",
  "ERROR_INVALID_FLAG_NUMBER",
  "ERROR_SEM_NOT_FOUND",
  "ERROR_INVALID_STARTING_CODESEG",
  "ERROR_INVALID_STACKSEG",
  "ERROR_INVALID_MODULETYPE",
  "ERROR_INVALID_EXE_SIGNATURE",
  "ERROR_EXE_MARKED_INVALID",
  "ERROR_BAD_EXE_FORMAT",
  "ERROR_INVALID_MINALLOCSIZE",
  "ERROR_DYNLINK_FROM_INVALID_RING",
  "ERROR_IOPL_NOT_ENABLED",
  "ERROR_INVALID_SEGDPL",
  "ERROR_RING2SEG_MUST_BE_MOVABLE",
  "ERROR_RELOC_CHAIN_XEEDS_SEGLIM",
  "ERROR_INFLOOP_IN_RELOC_CHAIN",
  "ERROR_ENVVAR_NOT_FOUND",
  "ERROR_NO_SIGNAL_SENT",
  "ERROR_FILENAME_EXCED_RANGE",
  "ERROR_RING2_STACK_IN_USE",
  "ERROR_META_EXPANSION_TOO_LONG",
  "ERROR_INVALID_SIGNAL_NUMBER",
  "ERROR_THREAD_1_INACTIVE",
  "ERROR_LOCKED",
  "ERROR_TOO_MANY_MODULES",
  "ERROR_NESTING_NOT_ALLOWED",
  "ERROR_EXE_MACHINE_TYPE_MISMATCH",
  "ERROR_EXE_CANNOT_MODIFY_SIGNED_BINARY",
  "ERROR_EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY",
  "ERROR_FILE_CHECKED_OUT",
  "ERROR_CHECKOUT_REQUIRED",
  "ERROR_BAD_FILE_TYPE",
  "ERROR_FILE_TOO_LARGE",
  "ERROR_FORMS_AUTH_REQUIRED",
  "ERROR_VIRUS_INFECTED",
  "ERROR_VIRUS_DELETED",
  "ERROR_PIPE_LOCAL",
  "ERROR_BAD_PIPE",
  "ERROR_PIPE_BUSY",
  "ERROR_NO_DATA",
  "ERROR_PIPE_NOT_CONNECTED",
  "ERROR_MORE_DATA",
  "ERROR_NO_WORK_DONE",
  "ERROR_VC_DISCONNECTED",
  "ERROR_INVALID_EA_NAME",
  "ERROR_EA_LIST_INCONSISTENT",
  "ERROR_NO_MORE_ITEMS",
  "ERROR_CANNOT_COPY",
  "ERROR_DIRECTORY",
  "ERROR_EAS_DIDNT_FIT",
  "ERROR_EA_FILE_CORRUPT",
  "ERROR_EA_TABLE_FULL",
  "ERROR_INVALID_EA_HANDLE",
  "ERROR_EAS_NOT_SUPPORTED",
  "ERROR_NOT_OWNER",
  "ERROR_TOO_MANY_POSTS",
  "ERROR_PARTIAL_COPY",
  "ERROR_OPLOCK_NOT_GRANTED",
  "ERROR_INVALID_OPLOCK_PROTOCOL",
  "ERROR_DISK_TOO_FRAGMENTED",
  "ERROR_DELETE_PENDING",
  "ERROR_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING",
  "ERROR_SHORT_NAMES_NOT_ENABLED_ON_VOLUME",
  "ERROR_SECURITY_STREAM_IS_INCONSISTENT",
  "ERROR_INVALID_LOCK_RANGE",
  "ERROR_IMAGE_SUBSYSTEM_NOT_PRESENT",
  "ERROR_NOTIFICATION_GUID_ALREADY_DEFINED",
  "ERROR_INVALID_EXCEPTION_HANDLER",
  "ERROR_DUPLICATE_PRIVILEGES",
  "ERROR_NO_RANGES_PROCESSED",
  "ERROR_NOT_ALLOWED_ON_SYSTEM_FILE",
  "ERROR_DISK_RESOURCES_EXHAUSTED",
  "ERROR_INVALID_TOKEN",
  "ERROR_DEVICE_FEATURE_NOT_SUPPORTED",
  "ERROR_MR_MID_NOT_FOUND",
  "ERROR_SCOPE_NOT_FOUND",
  "ERROR_UNDEFINED_SCOPE",
  "ERROR_INVALID_CAP",
  "ERROR_DEVICE_UNREACHABLE",
  "ERROR_DEVICE_NO_RESOURCES",
  "ERROR_DATA_CHECKSUM_ERROR",
  "ERROR_INTERMIXED_KERNEL_EA_OPERATION",
  "ERROR_FILE_LEVEL_TRIM_NOT_SUPPORTED",
  "ERROR_OFFSET_ALIGNMENT_VIOLATION",
  "ERROR_INVALID_FIELD_IN_PARAMETER_LIST",
  "ERROR_OPERATION_IN_PROGRESS",
  "ERROR_BAD_DEVICE_PATH",
  "ERROR_TOO_MANY_DESCRIPTORS",
  "ERROR_SCRUB_DATA_DISABLED",
  "ERROR_NOT_REDUNDANT_STORAGE",
  "ERROR_RESIDENT_FILE_NOT_SUPPORTED",
  "ERROR_COMPRESSED_FILE_NOT_SUPPORTED",
  "ERROR_DIRECTORY_NOT_SUPPORTED",
  "ERROR_NOT_READ_FROM_COPY",
  "ERROR_FT_WRITE_FAILURE",
  "ERROR_FT_DI_SCAN_REQUIRED",
  "ERROR_INVALID_KERNEL_INFO_VERSION",
  "ERROR_INVALID_PEP_INFO_VERSION",
  "ERROR_OBJECT_NOT_EXTERNALLY_BACKED",
  "ERROR_EXTERNAL_BACKING_PROVIDER_UNKNOWN",
  "ERROR_COMPRESSION_NOT_BENEFICIAL",
  "ERROR_STORAGE_TOPOLOGY_ID_MISMATCH",
  "ERROR_BLOCKED_BY_PARENTAL_CONTROLS",
  "ERROR_BLOCK_TOO_MANY_REFERENCES",
  "ERROR_MARKED_TO_DISALLOW_WRITES",
  "ERROR_ENCLAVE_FAILURE",
  "ERROR_FAIL_NOACTION_REBOOT",
  "ERROR_FAIL_SHUTDOWN",
  "ERROR_FAIL_RESTART",
  "ERROR_MAX_SESSIONS_REACHED",
  "ERROR_NETWORK_ACCESS_DENIED_EDP",
  "ERROR_DEVICE_HINT_NAME_BUFFER_TOO_SMALL",
  "ERROR_EDP_POLICY_DENIES_OPERATION",
  "ERROR_EDP_DPL_POLICY_CANT_BE_SATISFIED",
  "ERROR_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT",
  "ERROR_DEVICE_IN_MAINTENANCE",
  "ERROR_NOT_SUPPORTED_ON_DAX",
  "ERROR_DAX_MAPPING_EXISTS",
  "ERROR_CLOUD_FILE_PROVIDER_NOT_RUNNING",
  "ERROR_CLOUD_FILE_METADATA_CORRUPT",
  "ERROR_CLOUD_FILE_METADATA_TOO_LARGE",
  "ERROR_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE",
  "ERROR_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH",
  "ERROR_CHILD_PROCESS_BLOCKED",
  "ERROR_STORAGE_LOST_DATA_PERSISTENCE",
  "ERROR_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE",
  "ERROR_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT",
  "ERROR_FILE_SYSTEM_VIRTUALIZATION_BUSY",
  "ERROR_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN",
  "ERROR_GDI_HANDLE_LEAK",
  "ERROR_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS",
  "ERROR_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED",
  "ERROR_NOT_A_CLOUD_FILE",
  "ERROR_CLOUD_FILE_NOT_IN_SYNC",
  "ERROR_CLOUD_FILE_ALREADY_CONNECTED",
  "ERROR_CLOUD_FILE_NOT_SUPPORTED",
  "ERROR_CLOUD_FILE_INVALID_REQUEST",
  "ERROR_CLOUD_FILE_READ_ONLY_VOLUME",
  "ERROR_CLOUD_FILE_CONNECTED_PROVIDER_ONLY",
  "ERROR_CLOUD_FILE_VALIDATION_FAILED",
  "ERROR_SMB1_NOT_AVAILABLE",
  "ERROR_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION",
  "ERROR_CLOUD_FILE_AUTHENTICATION_FAILED",
  "ERROR_CLOUD_FILE_INSUFFICIENT_RESOURCES",
  "ERROR_CLOUD_FILE_NETWORK_UNAVAILABLE",
  "ERROR_CLOUD_FILE_UNSUCCESSFUL",
  "ERROR_CLOUD_FILE_NOT_UNDER_SYNC_ROOT",
  "ERROR_CLOUD_FILE_IN_USE",
  "ERROR_CLOUD_FILE_PINNED",
  "ERROR_CLOUD_FILE_REQUEST_ABORTED",
  "ERROR_CLOUD_FILE_PROPERTY_CORRUPT",
  "ERROR_CLOUD_FILE_ACCESS_DENIED",
  "ERROR_CLOUD_FILE_INCOMPATIBLE_HARDLINKS",
  "ERROR_CLOUD_FILE_PROPERTY_LOCK_CONFLICT",
  "ERROR_CLOUD_FILE_REQUEST_CANCELED",
  "ERROR_EXTERNAL_SYSKEY_NOT_SUPPORTED",
  "ERROR_THREAD_MODE_ALREADY_BACKGROUND",
  "ERROR_THREAD_MODE_NOT_BACKGROUND",
  "ERROR_PROCESS_MODE_ALREADY_BACKGROUND",
  "ERROR_PROCESS_MODE_NOT_BACKGROUND",
  "ERROR_CLOUD_FILE_PROVIDER_TERMINATED",
  "ERROR_NOT_A_CLOUD_SYNC_ROOT",
  "ERROR_FILE_PROTECTED_UNDER_DPL",
  "ERROR_VOLUME_NOT_CLUSTER_ALIGNED",
  "ERROR_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND",
  "ERROR_APPX_FILE_NOT_ENCRYPTED",
  "ERROR_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED",
  "ERROR_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET",
  "ERROR_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE",
  "ERROR_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER",
  "ERROR_LINUX_SUBSYSTEM_NOT_PRESENT",
  "ERROR_FT_READ_FAILURE",
  "ERROR_STORAGE_RESERVE_ID_INVALID",
  "ERROR_STORAGE_RESERVE_DOES_NOT_EXIST",
  "ERROR_STORAGE_RESERVE_ALREADY_EXISTS",
  "ERROR_STORAGE_RESERVE_NOT_EMPTY",
  "ERROR_NOT_A_DAX_VOLUME",
  "ERROR_NOT_DAX_MAPPABLE",
  "ERROR_TIME_SENSITIVE_THREAD",
  "ERROR_DPL_NOT_SUPPORTED_FOR_USER",
  "ERROR_CASE_DIFFERING_NAMES_IN_DIR",
  "ERROR_FILE_NOT_SUPPORTED",
  "ERROR_CLOUD_FILE_REQUEST_TIMEOUT",
  "ERROR_NO_TASK_QUEUE",
  "ERROR_SRC_SRV_DLL_LOAD_FAILED",
  "ERROR_NOT_SUPPORTED_WITH_BTT",
  "ERROR_ENCRYPTION_DISABLED",
  "ERROR_ENCRYPTING_METADATA_DISALLOWED",
  "ERROR_CANT_CLEAR_ENCRYPTION_FLAG",
  "ERROR_NO_SUCH_DEVICE",
  "ERROR_CLOUD_FILE_DEHYDRATION_DISALLOWED",
  "ERROR_FILE_SNAP_IN_PROGRESS",
  "ERROR_FILE_SNAP_USER_SECTION_NOT_SUPPORTED",
  "ERROR_FILE_SNAP_MODIFY_NOT_SUPPORTED",
  "ERROR_FILE_SNAP_IO_NOT_COORDINATED",
  "ERROR_FILE_SNAP_UNEXPECTED_ERROR",
  "ERROR_FILE_SNAP_INVALID_PARAMETER",
  "ERROR_UNSATISFIED_DEPENDENCIES",
  "ERROR_CASE_SENSITIVE_PATH",
  "ERROR_UNEXPECTED_NTCACHEMANAGER_ERROR",
  "ERROR_LINUX_SUBSYSTEM_UPDATE_REQUIRED",
  "ERROR_DLP_POLICY_WARNS_AGAINST_OPERATION",
  "ERROR_DLP_POLICY_DENIES_OPERATION",
  "ERROR_DLP_POLICY_SILENTLY_FAIL",
  "ERROR_CAPAUTHZ_NOT_DEVUNLOCKED",
  "ERROR_CAPAUTHZ_CHANGE_TYPE",
  "ERROR_CAPAUTHZ_NOT_PROVISIONED",
  "ERROR_CAPAUTHZ_NOT_AUTHORIZED",
  "ERROR_CAPAUTHZ_NO_POLICY",
  "ERROR_CAPAUTHZ_DB_CORRUPTED",
  "ERROR_CAPAUTHZ_SCCD_INVALID_CATALOG",
  "ERROR_CAPAUTHZ_SCCD_NO_AUTH_ENTITY",
  "ERROR_CAPAUTHZ_SCCD_PARSE_ERROR",
  "ERROR_CAPAUTHZ_SCCD_DEV_MODE_REQUIRED",
  "ERROR_CAPAUTHZ_SCCD_NO_CAPABILITY_MATCH",
  "ERROR_CIMFS_IMAGE_CORRUPT",
  "ERROR_PNP_QUERY_REMOVE_DEVICE_TIMEOUT",
  "ERROR_PNP_QUERY_REMOVE_RELATED_DEVICE_TIMEOUT",
  "ERROR_PNP_QUERY_REMOVE_UNRELATED_DEVICE_TIMEOUT",
  "ERROR_DEVICE_HARDWARE_ERROR",
  "ERROR_INVALID_ADDRESS",
  "ERROR_HAS_SYSTEM_CRITICAL_FILES",
  "ERROR_USER_PROFILE_LOAD",
  "ERROR_ARITHMETIC_OVERFLOW",
  "ERROR_PIPE_CONNECTED",
  "ERROR_PIPE_LISTENING",
  "ERROR_VERIFIER_STOP",
  "ERROR_ABIOS_ERROR",
  "ERROR_WX86_WARNING",
  "ERROR_WX86_ERROR",
  "ERROR_TIMER_NOT_CANCELED",
  "ERROR_UNWIND",
  "ERROR_BAD_STACK",
  "ERROR_INVALID_UNWIND_TARGET",
  "ERROR_INVALID_PORT_ATTRIBUTES",
  "ERROR_PORT_MESSAGE_TOO_LONG",
  "ERROR_INVALID_QUOTA_LOWER",
  "ERROR_DEVICE_ALREADY_ATTACHED",
  "ERROR_INSTRUCTION_MISALIGNMENT",
  "ERROR_PROFILING_NOT_STARTED",
  "ERROR_PROFILING_NOT_STOPPED",
  "ERROR_COULD_NOT_INTERPRET",
  "ERROR_PROFILING_AT_LIMIT",
  "ERROR_CANT_WAIT",
  "ERROR_CANT_TERMINATE_SELF",
  "ERROR_UNEXPECTED_MM_CREATE_ERR",
  "ERROR_UNEXPECTED_MM_MAP_ERROR",
  "ERROR_UNEXPECTED_MM_EXTEND_ERR",
  "ERROR_BAD_FUNCTION_TABLE",
  "ERROR_NO_GUID_TRANSLATION",
  "ERROR_INVALID_LDT_SIZE",
  "ERROR_INVALID_LDT_OFFSET",
  "ERROR_INVALID_LDT_DESCRIPTOR",
  "ERROR_TOO_MANY_THREADS",
  "ERROR_THREAD_NOT_IN_PROCESS",
  "ERROR_PAGEFILE_QUOTA_EXCEEDED",
  "ERROR_LOGON_SERVER_CONFLICT",
  "ERROR_SYNCHRONIZATION_REQUIRED",
  "ERROR_NET_OPEN_FAILED",
  "ERROR_IO_PRIVILEGE_FAILED",
  "ERROR_CONTROL_C_EXIT",
  "ERROR_MISSING_SYSTEMFILE",
  "ERROR_UNHANDLED_EXCEPTION",
  "ERROR_APP_INIT_FAILURE",
  "ERROR_PAGEFILE_CREATE_FAILED",
  "ERROR_INVALID_IMAGE_HASH",
  "ERROR_NO_PAGEFILE",
  "ERROR_ILLEGAL_FLOAT_CONTEXT",
  "ERROR_NO_EVENT_PAIR",
  "ERROR_DOMAIN_CTRLR_CONFIG_ERROR",
  "ERROR_ILLEGAL_CHARACTER",
  "ERROR_UNDEFINED_CHARACTER",
  "ERROR_FLOPPY_VOLUME",
  "ERROR_BIOS_FAILED_TO_CONNECT_INTERRUPT",
  "ERROR_BACKUP_CONTROLLER",
  "ERROR_MUTANT_LIMIT_EXCEEDED",
  "ERROR_FS_DRIVER_REQUIRED",
  "ERROR_CANNOT_LOAD_REGISTRY_FILE",
  "ERROR_DEBUG_ATTACH_FAILED",
  "ERROR_SYSTEM_PROCESS_TERMINATED",
  "ERROR_DATA_NOT_ACCEPTED",
  "ERROR_VDM_HARD_ERROR",
  "ERROR_DRIVER_CANCEL_TIMEOUT",
  "ERROR_REPLY_MESSAGE_MISMATCH",
  "ERROR_LOST_WRITEBEHIND_DATA",
  "ERROR_CLIENT_SERVER_PARAMETERS_INVALID",
  "ERROR_NOT_TINY_STREAM",
  "ERROR_STACK_OVERFLOW_READ",
  "ERROR_CONVERT_TO_LARGE",
  "ERROR_FOUND_OUT_OF_SCOPE",
  "ERROR_ALLOCATE_BUCKET",
  "ERROR_MARSHALL_OVERFLOW",
  "ERROR_INVALID_VARIANT",
  "ERROR_BAD_COMPRESSION_BUFFER",
  "ERROR_AUDIT_FAILED",
  "ERROR_TIMER_RESOLUTION_NOT_SET",
  "ERROR_INSUFFICIENT_LOGON_INFO",
  "ERROR_BAD_DLL_ENTRYPOINT",
  "ERROR_BAD_SERVICE_ENTRYPOINT",
  "ERROR_IP_ADDRESS_CONFLICT1",
  "ERROR_IP_ADDRESS_CONFLICT2",
  "ERROR_REGISTRY_QUOTA_LIMIT",
  "ERROR_NO_CALLBACK_ACTIVE",
  "ERROR_PWD_TOO_SHORT",
  "ERROR_PWD_TOO_RECENT",
  "ERROR_PWD_HISTORY_CONFLICT",
  "ERROR_UNSUPPORTED_COMPRESSION",
  "ERROR_INVALID_HW_PROFILE",
  "ERROR_INVALID_PLUGPLAY_DEVICE_PATH",
  "ERROR_QUOTA_LIST_INCONSISTENT",
  "ERROR_EVALUATION_EXPIRATION",
  "ERROR_ILLEGAL_DLL_RELOCATION",
  "ERROR_DLL_INIT_FAILED_LOGOFF",
  "ERROR_VALIDATE_CONTINUE",
  "ERROR_NO_MORE_MATCHES",
  "ERROR_RANGE_LIST_CONFLICT",
  "ERROR_SERVER_SID_MISMATCH",
  "ERROR_CANT_ENABLE_DENY_ONLY",
  "ERROR_FLOAT_MULTIPLE_FAULTS",
  "ERROR_FLOAT_MULTIPLE_TRAPS",
  "ERROR_NOINTERFACE",
  "ERROR_DRIVER_FAILED_SLEEP",
  "ERROR_CORRUPT_SYSTEM_FILE",
  "ERROR_COMMITMENT_MINIMUM",
  "ERROR_PNP_RESTART_ENUMERATION",
  "ERROR_SYSTEM_IMAGE_BAD_SIGNATURE",
  "ERROR_PNP_REBOOT_REQUIRED",
  "ERROR_INSUFFICIENT_POWER",
  "ERROR_MULTIPLE_FAULT_VIOLATION",
  "ERROR_SYSTEM_SHUTDOWN",
  "ERROR_PORT_NOT_SET",
  "ERROR_DS_VERSION_CHECK_FAILURE",
  "ERROR_RANGE_NOT_FOUND",
  "ERROR_NOT_SAFE_MODE_DRIVER",
  "ERROR_FAILED_DRIVER_ENTRY",
  "ERROR_DEVICE_ENUMERATION_ERROR",
  "ERROR_MOUNT_POINT_NOT_RESOLVED",
  "ERROR_INVALID_DEVICE_OBJECT_PARAMETER",
  "ERROR_MCA_OCCURED",
  "ERROR_DRIVER_DATABASE_ERROR",
  "ERROR_SYSTEM_HIVE_TOO_LARGE",
  "ERROR_DRIVER_FAILED_PRIOR_UNLOAD",
  "ERROR_VOLSNAP_PREPARE_HIBERNATE",
  "ERROR_HIBERNATION_FAILURE",
  "ERROR_PWD_TOO_LONG",
  "ERROR_FILE_SYSTEM_LIMITATION",
  "ERROR_ASSERTION_FAILURE",
  "ERROR_ACPI_ERROR",
  "ERROR_WOW_ASSERTION",
  "ERROR_PNP_BAD_MPS_TABLE",
  "ERROR_PNP_TRANSLATION_FAILED",
  "ERROR_PNP_IRQ_TRANSLATION_FAILED",
  "ERROR_PNP_INVALID_ID",
  "ERROR_WAKE_SYSTEM_DEBUGGER",
  "ERROR_HANDLES_CLOSED",
  "ERROR_EXTRANEOUS_INFORMATION",
  "ERROR_RXACT_COMMIT_NECESSARY",
  "ERROR_MEDIA_CHECK",
  "ERROR_GUID_SUBSTITUTION_MADE",
  "ERROR_STOPPED_ON_SYMLINK",
  "ERROR_LONGJUMP",
  "ERROR_PLUGPLAY_QUERY_VETOED",
  "ERROR_UNWIND_CONSOLIDATE",
  "ERROR_REGISTRY_HIVE_RECOVERED",
  "ERROR_DLL_MIGHT_BE_INSECURE",
  "ERROR_DLL_MIGHT_BE_INCOMPATIBLE",
  "ERROR_DBG_EXCEPTION_NOT_HANDLED",
  "ERROR_DBG_REPLY_LATER",
  "ERROR_DBG_UNABLE_TO_PROVIDE_HANDLE",
  "ERROR_DBG_TERMINATE_THREAD",
  "ERROR_DBG_TERMINATE_PROCESS",
  "ERROR_DBG_CONTROL_C",
  "ERROR_DBG_PRINTEXCEPTION_C",
  "ERROR_DBG_RIPEXCEPTION",
  "ERROR_DBG_CONTROL_BREAK",
  "ERROR_DBG_COMMAND_EXCEPTION",
  "ERROR_OBJECT_NAME_EXISTS",
  "ERROR_THREAD_WAS_SUSPENDED",
  "ERROR_IMAGE_NOT_AT_BASE",
  "ERROR_RXACT_STATE_CREATED",
  "ERROR_SEGMENT_NOTIFICATION",
  "ERROR_BAD_CURRENT_DIRECTORY",
  "ERROR_FT_READ_RECOVERY_FROM_BACKUP",
  "ERROR_FT_WRITE_RECOVERY",
  "ERROR_IMAGE_MACHINE_TYPE_MISMATCH",
  "ERROR_RECEIVE_PARTIAL",
  "ERROR_RECEIVE_EXPEDITED",
  "ERROR_RECEIVE_PARTIAL_EXPEDITED",
  "ERROR_EVENT_DONE",
  "ERROR_EVENT_PENDING",
  "ERROR_CHECKING_FILE_SYSTEM",
  "ERROR_FATAL_APP_EXIT",
  "ERROR_PREDEFINED_HANDLE",
  "ERROR_WAS_UNLOCKED",
  "ERROR_SERVICE_NOTIFICATION",
  "ERROR_WAS_LOCKED",
  "ERROR_LOG_HARD_ERROR",
  "ERROR_ALREADY_WIN32",
  "ERROR_IMAGE_MACHINE_TYPE_MISMATCH_EXE",
  "ERROR_NO_YIELD_PERFORMED",
  "ERROR_TIMER_RESUME_IGNORED",
  "ERROR_ARBITRATION_UNHANDLED",
  "ERROR_CARDBUS_NOT_SUPPORTED",
  "ERROR_MP_PROCESSOR_MISMATCH",
  "ERROR_HIBERNATED",
  "ERROR_RESUME_HIBERNATION",
  "ERROR_FIRMWARE_UPDATED",
  "ERROR_DRIVERS_LEAKING_LOCKED_PAGES",
  "ERROR_WAKE_SYSTEM",
  "ERROR_WAIT_1",
  "ERROR_WAIT_2",
  "ERROR_WAIT_3",
  "ERROR_WAIT_63",
  "ERROR_ABANDONED_WAIT_0",
  "ERROR_ABANDONED_WAIT_63",
  "ERROR_USER_APC",
  "ERROR_KERNEL_APC",
  "ERROR_ALERTED",
  "ERROR_ELEVATION_REQUIRED",
  "ERROR_REPARSE",
  "ERROR_OPLOCK_BREAK_IN_PROGRESS",
  "ERROR_VOLUME_MOUNTED",
  "ERROR_RXACT_COMMITTED",
  "ERROR_NOTIFY_CLEANUP",
  "ERROR_PRIMARY_TRANSPORT_CONNECT_FAILED",
  "ERROR_PAGE_FAULT_TRANSITION",
  "ERROR_PAGE_FAULT_DEMAND_ZERO",
  "ERROR_PAGE_FAULT_COPY_ON_WRITE",
  "ERROR_PAGE_FAULT_GUARD_PAGE",
  "ERROR_PAGE_FAULT_PAGING_FILE",
  "ERROR_CACHE_PAGE_LOCKED",
  "ERROR_CRASH_DUMP",
  "ERROR_BUFFER_ALL_ZEROS",
  "ERROR_REPARSE_OBJECT",
  "ERROR_RESOURCE_REQUIREMENTS_CHANGED",
  "ERROR_TRANSLATION_COMPLETE",
  "ERROR_NOTHING_TO_TERMINATE",
  "ERROR_PROCESS_NOT_IN_JOB",
  "ERROR_PROCESS_IN_JOB",
  "ERROR_VOLSNAP_HIBERNATE_READY",
  "ERROR_FSFILTER_OP_COMPLETED_SUCCESSFULLY",
  "ERROR_INTERRUPT_VECTOR_ALREADY_CONNECTED",
  "ERROR_INTERRUPT_STILL_CONNECTED",
  "ERROR_WAIT_FOR_OPLOCK",
  "ERROR_DBG_EXCEPTION_HANDLED",
  "ERROR_DBG_CONTINUE",
  "ERROR_CALLBACK_POP_STACK",
  "ERROR_COMPRESSION_DISABLED",
  "ERROR_CANTFETCHBACKWARDS",
  "ERROR_CANTSCROLLBACKWARDS",
  "ERROR_ROWSNOTRELEASED",
  "ERROR_BAD_ACCESSOR_FLAGS",
  "ERROR_ERRORS_ENCOUNTERED",
  "ERROR_NOT_CAPABLE",
  "ERROR_REQUEST_OUT_OF_SEQUENCE",
  "ERROR_VERSION_PARSE_ERROR",
  "ERROR_BADSTARTPOSITION",
  "ERROR_MEMORY_HARDWARE",
  "ERROR_DISK_REPAIR_DISABLED",
  "ERROR_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE",
  "ERROR_SYSTEM_POWERSTATE_TRANSITION",
  "ERROR_SYSTEM_POWERSTATE_COMPLEX_TRANSITION",
  "ERROR_MCA_EXCEPTION",
  "ERROR_ACCESS_AUDIT_BY_POLICY",
  "ERROR_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY",
  "ERROR_ABANDON_HIBERFILE",
  "ERROR_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED",
  "ERROR_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR",
  "ERROR_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR",
  "ERROR_BAD_MCFG_TABLE",
  "ERROR_DISK_REPAIR_REDIRECTED",
  "ERROR_DISK_REPAIR_UNSUCCESSFUL",
  "ERROR_CORRUPT_LOG_OVERFULL",
  "ERROR_CORRUPT_LOG_CORRUPTED",
  "ERROR_CORRUPT_LOG_UNAVAILABLE",
  "ERROR_CORRUPT_LOG_DELETED_FULL",
  "ERROR_CORRUPT_LOG_CLEARED",
  "ERROR_ORPHAN_NAME_EXHAUSTED",
  "ERROR_OPLOCK_SWITCHED_TO_NEW_HANDLE",
  "ERROR_CANNOT_GRANT_REQUESTED_OPLOCK",
  "ERROR_CANNOT_BREAK_OPLOCK",
  "ERROR_OPLOCK_HANDLE_CLOSED",
  "ERROR_NO_ACE_CONDITION",
  "ERROR_INVALID_ACE_CONDITION",
  "ERROR_FILE_HANDLE_REVOKED",
  "ERROR_IMAGE_AT_DIFFERENT_BASE",
  "ERROR_ENCRYPTED_IO_NOT_POSSIBLE",
  "ERROR_FILE_METADATA_OPTIMIZATION_IN_PROGRESS",
  "ERROR_QUOTA_ACTIVITY",
  "ERROR_HANDLE_REVOKED",
  "ERROR_CALLBACK_INVOKE_INLINE",
  "ERROR_CPU_SET_INVALID",
  "ERROR_ENCLAVE_NOT_TERMINATED",
  "ERROR_ENCLAVE_VIOLATION",
  "ERROR_EA_ACCESS_DENIED",
  "ERROR_OPERATION_ABORTED",
  "ERROR_IO_INCOMPLETE",
  "ERROR_IO_PENDING",
  "ERROR_NOACCESS",
  "ERROR_SWAPERROR",
  "ERROR_STACK_OVERFLOW",
  "ERROR_INVALID_MESSAGE",
  "ERROR_CAN_NOT_COMPLETE",
  "ERROR_INVALID_FLAGS",
  "ERROR_UNRECOGNIZED_VOLUME",
  "ERROR_FILE_INVALID",
  "ERROR_FULLSCREEN_MODE",
  "ERROR_NO_TOKEN",
  "ERROR_BADDB",
  "ERROR_BADKEY",
  "ERROR_CANTOPEN",
  "ERROR_CANTREAD",
  "ERROR_CANTWRITE",
  "ERROR_REGISTRY_RECOVERED",
  "ERROR_REGISTRY_CORRUPT",
  "ERROR_REGISTRY_IO_FAILED",
  "ERROR_NOT_REGISTRY_FILE",
  "ERROR_KEY_DELETED",
  "ERROR_NO_LOG_SPACE",
  "ERROR_KEY_HAS_CHILDREN",
  "ERROR_CHILD_MUST_BE_VOLATILE",
  "ERROR_NOTIFY_ENUM_DIR",
  "ERROR_DEPENDENT_SERVICES_RUNNING",
  "ERROR_INVALID_SERVICE_CONTROL",
  "ERROR_SERVICE_REQUEST_TIMEOUT",
  "ERROR_SERVICE_NO_THREAD",
  "ERROR_SERVICE_DATABASE_LOCKED",
  "ERROR_SERVICE_ALREADY_RUNNING",
  "ERROR_INVALID_SERVICE_ACCOUNT",
  "ERROR_SERVICE_DISABLED",
  "ERROR_CIRCULAR_DEPENDENCY",
  "ERROR_SERVICE_DOES_NOT_EXIST",
  "ERROR_SERVICE_CANNOT_ACCEPT_CTRL",
  "ERROR_SERVICE_NOT_ACTIVE",
  "ERROR_FAILED_SERVICE_CONTROLLER_CONNECT",
  "ERROR_EXCEPTION_IN_SERVICE",
  "ERROR_DATABASE_DOES_NOT_EXIST",
  "ERROR_SERVICE_SPECIFIC_ERROR",
  "ERROR_PROCESS_ABORTED",
  "ERROR_SERVICE_DEPENDENCY_FAIL",
  "ERROR_SERVICE_LOGON_FAILED",
  "ERROR_SERVICE_START_HANG",
  "ERROR_INVALID_SERVICE_LOCK",
  "ERROR_SERVICE_MARKED_FOR_DELETE",
  "ERROR_SERVICE_EXISTS",
  "ERROR_ALREADY_RUNNING_LKG",
  "ERROR_SERVICE_DEPENDENCY_DELETED",
  "ERROR_BOOT_ALREADY_ACCEPTED",
  "ERROR_SERVICE_NEVER_STARTED",
  "ERROR_DUPLICATE_SERVICE_NAME",
  "ERROR_DIFFERENT_SERVICE_ACCOUNT",
  "ERROR_CANNOT_DETECT_DRIVER_FAILURE",
  "ERROR_CANNOT_DETECT_PROCESS_ABORT",
  "ERROR_NO_RECOVERY_PROGRAM",
  "ERROR_SERVICE_NOT_IN_EXE",
  "ERROR_NOT_SAFEBOOT_SERVICE",
  "ERROR_END_OF_MEDIA",
  "ERROR_FILEMARK_DETECTED",
  "ERROR_BEGINNING_OF_MEDIA",
  "ERROR_SETMARK_DETECTED",
  "ERROR_NO_DATA_DETECTED",
  "ERROR_PARTITION_FAILURE",
  "ERROR_INVALID_BLOCK_LENGTH",
  "ERROR_DEVICE_NOT_PARTITIONED",
  "ERROR_UNABLE_TO_LOCK_MEDIA",
  "ERROR_UNABLE_TO_UNLOAD_MEDIA",
  "ERROR_MEDIA_CHANGED",
  "ERROR_BUS_RESET",
  "ERROR_NO_MEDIA_IN_DRIVE",
  "ERROR_NO_UNICODE_TRANSLATION",
  "ERROR_DLL_INIT_FAILED",
  "ERROR_SHUTDOWN_IN_PROGRESS",
  "ERROR_NO_SHUTDOWN_IN_PROGRESS",
  "ERROR_IO_DEVICE",
  "ERROR_SERIAL_NO_DEVICE",
  "ERROR_IRQ_BUSY",
  "ERROR_MORE_WRITES",
  "ERROR_COUNTER_TIMEOUT",
  "ERROR_FLOPPY_ID_MARK_NOT_FOUND",
  "ERROR_FLOPPY_WRONG_CYLINDER",
  "ERROR_FLOPPY_UNKNOWN_ERROR",
  "ERROR_FLOPPY_BAD_REGISTERS",
  "ERROR_DISK_RECALIBRATE_FAILED",
  "ERROR_DISK_OPERATION_FAILED",
  "ERROR_DISK_RESET_FAILED",
  "ERROR_EOM_OVERFLOW",
  "ERROR_NOT_ENOUGH_SERVER_MEMORY",
  "ERROR_POSSIBLE_DEADLOCK",
  "ERROR_MAPPED_ALIGNMENT",
  "ERROR_SET_POWER_STATE_VETOED",
  "ERROR_SET_POWER_STATE_FAILED",
  "ERROR_TOO_MANY_LINKS",
  "ERROR_OLD_WIN_VERSION",
  "ERROR_APP_WRONG_OS",
  "ERROR_SINGLE_INSTANCE_APP",
  "ERROR_RMODE_APP",
  "ERROR_INVALID_DLL",
  "ERROR_NO_ASSOCIATION",
  "ERROR_DDE_FAIL",
  "ERROR_DLL_NOT_FOUND",
  "ERROR_NO_MORE_USER_HANDLES",
  "ERROR_MESSAGE_SYNC_ONLY",
  "ERROR_SOURCE_ELEMENT_EMPTY",
  "ERROR_DESTINATION_ELEMENT_FULL",
  "ERROR_ILLEGAL_ELEMENT_ADDRESS",
  "ERROR_MAGAZINE_NOT_PRESENT",
  "ERROR_DEVICE_REINITIALIZATION_NEEDED",
  "ERROR_DEVICE_REQUIRES_CLEANING",
  "ERROR_DEVICE_DOOR_OPEN",
  "ERROR_DEVICE_NOT_CONNECTED",
  "ERROR_NOT_FOUND",
  "ERROR_NO_MATCH",
  "ERROR_SET_NOT_FOUND",
  "ERROR_POINT_NOT_FOUND",
  "ERROR_NO_TRACKING_SERVICE",
  "ERROR_NO_VOLUME_ID",
  "ERROR_UNABLE_TO_REMOVE_REPLACED",
  "ERROR_UNABLE_TO_MOVE_REPLACEMENT",
  "ERROR_UNABLE_TO_MOVE_REPLACEMENT_2",
  "ERROR_JOURNAL_DELETE_IN_PROGRESS",
  "ERROR_JOURNAL_NOT_ACTIVE",
  "ERROR_POTENTIAL_FILE_FOUND",
  "ERROR_JOURNAL_ENTRY_DELETED",
  "ERROR_VRF_CFG_AND_IO_ENABLED",
  "ERROR_PARTITION_TERMINATING",
  "ERROR_SHUTDOWN_IS_SCHEDULED",
  "ERROR_SHUTDOWN_USERS_LOGGED_ON",
  "ERROR_BAD_DEVICE",
  "ERROR_CONNECTION_UNAVAIL",
  "ERROR_DEVICE_ALREADY_REMEMBERED",
  "ERROR_NO_NET_OR_BAD_PATH",
  "ERROR_BAD_PROVIDER",
  "ERROR_CANNOT_OPEN_PROFILE",
  "ERROR_BAD_PROFILE",
  "ERROR_NOT_CONTAINER",
  "ERROR_EXTENDED_ERROR",
  "ERROR_INVALID_GROUPNAME",
  "ERROR_INVALID_COMPUTERNAME",
  "ERROR_INVALID_EVENTNAME",
  "ERROR_INVALID_DOMAINNAME",
  "ERROR_INVALID_SERVICENAME",
  "ERROR_INVALID_NETNAME",
  "ERROR_INVALID_SHARENAME",
  "ERROR_INVALID_PASSWORDNAME",
  "ERROR_INVALID_MESSAGENAME",
  "ERROR_INVALID_MESSAGEDEST",
  "ERROR_SESSION_CREDENTIAL_CONFLICT",
  "ERROR_REMOTE_SESSION_LIMIT_EXCEEDED",
  "ERROR_DUP_DOMAINNAME",
  "ERROR_NO_NETWORK",
  "ERROR_CANCELLED",
  "ERROR_USER_MAPPED_FILE",
  "ERROR_CONNECTION_REFUSED",
  "ERROR_GRACEFUL_DISCONNECT",
  "ERROR_ADDRESS_ALREADY_ASSOCIATED",
  "ERROR_ADDRESS_NOT_ASSOCIATED",
  "ERROR_CONNECTION_INVALID",
  "ERROR_CONNECTION_ACTIVE",
  "ERROR_NETWORK_UNREACHABLE",
  "ERROR_HOST_UNREACHABLE",
  "ERROR_PROTOCOL_UNREACHABLE",
  "ERROR_PORT_UNREACHABLE",
  "ERROR_REQUEST_ABORTED",
  "ERROR_CONNECTION_ABORTED",
  "ERROR_RETRY",
  "ERROR_CONNECTION_COUNT_LIMIT",
  "ERROR_LOGIN_TIME_RESTRICTION",
  "ERROR_LOGIN_WKSTA_RESTRICTION",
  "ERROR_INCORRECT_ADDRESS",
  "ERROR_ALREADY_REGISTERED",
  "ERROR_SERVICE_NOT_FOUND",
  "ERROR_NOT_AUTHENTICATED",
  "ERROR_NOT_LOGGED_ON",
  "ERROR_CONTINUE",
  "ERROR_ALREADY_INITIALIZED",
  "ERROR_NO_MORE_DEVICES",
  "ERROR_NO_SUCH_SITE",
  "ERROR_DOMAIN_CONTROLLER_EXISTS",
  "ERROR_ONLY_IF_CONNECTED",
  "ERROR_OVERRIDE_NOCHANGES",
  "ERROR_BAD_USER_PROFILE",
  "ERROR_NOT_SUPPORTED_ON_SBS",
  "ERROR_SERVER_SHUTDOWN_IN_PROGRESS",
  "ERROR_HOST_DOWN",
  "ERROR_NON_ACCOUNT_SID",
  "ERROR_NON_DOMAIN_SID",
  "ERROR_APPHELP_BLOCK",
  "ERROR_ACCESS_DISABLED_BY_POLICY",
  "ERROR_REG_NAT_CONSUMPTION",
  "ERROR_CSCSHARE_OFFLINE",
  "ERROR_PKINIT_FAILURE",
  "ERROR_SMARTCARD_SUBSYSTEM_FAILURE",
  "ERROR_DOWNGRADE_DETECTED",
  "ERROR_MACHINE_LOCKED",
  "ERROR_SMB_GUEST_LOGON_BLOCKED",
  "ERROR_CALLBACK_SUPPLIED_INVALID_DATA",
  "ERROR_SYNC_FOREGROUND_REFRESH_REQUIRED",
  "ERROR_DRIVER_BLOCKED",
  "ERROR_INVALID_IMPORT_OF_NON_DLL",
  "ERROR_ACCESS_DISABLED_WEBBLADE",
  "ERROR_ACCESS_DISABLED_WEBBLADE_TAMPER",
  "ERROR_RECOVERY_FAILURE",
  "ERROR_ALREADY_FIBER",
  "ERROR_ALREADY_THREAD",
  "ERROR_STACK_BUFFER_OVERRUN",
  "ERROR_PARAMETER_QUOTA_EXCEEDED",
  "ERROR_DEBUGGER_INACTIVE",
  "ERROR_DELAY_LOAD_FAILED",
  "ERROR_VDM_DISALLOWED",
  "ERROR_UNIDENTIFIED_ERROR",
  "ERROR_INVALID_CRUNTIME_PARAMETER",
  "ERROR_BEYOND_VDL",
  "ERROR_INCOMPATIBLE_SERVICE_SID_TYPE",
  "ERROR_DRIVER_PROCESS_TERMINATED",
  "ERROR_IMPLEMENTATION_LIMIT",
  "ERROR_PROCESS_IS_PROTECTED",
  "ERROR_SERVICE_NOTIFY_CLIENT_LAGGING",
  "ERROR_DISK_QUOTA_EXCEEDED",
  "ERROR_CONTENT_BLOCKED",
  "ERROR_INCOMPATIBLE_SERVICE_PRIVILEGE",
  "ERROR_APP_HANG",
  "ERROR_INVALID_LABEL",
  "ERROR_NOT_ALL_ASSIGNED",
  "ERROR_SOME_NOT_MAPPED",
  "ERROR_NO_QUOTAS_FOR_ACCOUNT",
  "ERROR_LOCAL_USER_SESSION_KEY",
  "ERROR_NULL_LM_PASSWORD",
  "ERROR_UNKNOWN_REVISION",
  "ERROR_REVISION_MISMATCH",
  "ERROR_INVALID_OWNER",
  "ERROR_INVALID_PRIMARY_GROUP",
  "ERROR_NO_IMPERSONATION_TOKEN",
  "ERROR_CANT_DISABLE_MANDATORY",
  "ERROR_NO_LOGON_SERVERS",
  "ERROR_NO_SUCH_LOGON_SESSION",
  "ERROR_NO_SUCH_PRIVILEGE",
  "ERROR_PRIVILEGE_NOT_HELD",
  "ERROR_INVALID_ACCOUNT_NAME",
  "ERROR_USER_EXISTS",
  "ERROR_NO_SUCH_USER",
  "ERROR_GROUP_EXISTS",
  "ERROR_NO_SUCH_GROUP",
  "ERROR_MEMBER_IN_GROUP",
  "ERROR_MEMBER_NOT_IN_GROUP",
  "ERROR_LAST_ADMIN",
  "ERROR_WRONG_PASSWORD",
  "ERROR_ILL_FORMED_PASSWORD",
  "ERROR_PASSWORD_RESTRICTION",
  "ERROR_LOGON_FAILURE",
  "ERROR_ACCOUNT_RESTRICTION",
  "ERROR_INVALID_LOGON_HOURS",
  "ERROR_INVALID_WORKSTATION",
  "ERROR_PASSWORD_EXPIRED",
  "ERROR_ACCOUNT_DISABLED",
  "ERROR_NONE_MAPPED",
  "ERROR_TOO_MANY_LUIDS_REQUESTED",
  "ERROR_LUIDS_EXHAUSTED",
  "ERROR_INVALID_SUB_AUTHORITY",
  "ERROR_INVALID_ACL",
  "ERROR_INVALID_SID",
  "ERROR_INVALID_SECURITY_DESCR",
  "ERROR_BAD_INHERITANCE_ACL",
  "ERROR_SERVER_DISABLED",
  "ERROR_SERVER_NOT_DISABLED",
  "ERROR_INVALID_ID_AUTHORITY",
  "ERROR_ALLOTTED_SPACE_EXCEEDED",
  "ERROR_INVALID_GROUP_ATTRIBUTES",
  "ERROR_BAD_IMPERSONATION_LEVEL",
  "ERROR_CANT_OPEN_ANONYMOUS",
  "ERROR_BAD_VALIDATION_CLASS",
  "ERROR_BAD_TOKEN_TYPE",
  "ERROR_NO_SECURITY_ON_OBJECT",
  "ERROR_CANT_ACCESS_DOMAIN_INFO",
  "ERROR_INVALID_SERVER_STATE",
  "ERROR_INVALID_DOMAIN_STATE",
  "ERROR_INVALID_DOMAIN_ROLE",
  "ERROR_NO_SUCH_DOMAIN",
  "ERROR_DOMAIN_EXISTS",
  "ERROR_DOMAIN_LIMIT_EXCEEDED",
  "ERROR_INTERNAL_DB_CORRUPTION",
  "ERROR_INTERNAL_ERROR",
  "ERROR_GENERIC_NOT_MAPPED",
  "ERROR_BAD_DESCRIPTOR_FORMAT",
  "ERROR_NOT_LOGON_PROCESS",
  "ERROR_LOGON_SESSION_EXISTS",
  "ERROR_NO_SUCH_PACKAGE",
  "ERROR_BAD_LOGON_SESSION_STATE",
  "ERROR_LOGON_SESSION_COLLISION",
  "ERROR_INVALID_LOGON_TYPE",
  "ERROR_CANNOT_IMPERSONATE",
  "ERROR_RXACT_INVALID_STATE",
  "ERROR_RXACT_COMMIT_FAILURE",
  "ERROR_SPECIAL_ACCOUNT",
  "ERROR_SPECIAL_GROUP",
  "ERROR_SPECIAL_USER",
  "ERROR_MEMBERS_PRIMARY_GROUP",
  "ERROR_TOKEN_ALREADY_IN_USE",
  "ERROR_NO_SUCH_ALIAS",
  "ERROR_MEMBER_NOT_IN_ALIAS",
  "ERROR_MEMBER_IN_ALIAS",
  "ERROR_ALIAS_EXISTS",
  "ERROR_LOGON_NOT_GRANTED",
  "ERROR_TOO_MANY_SECRETS",
  "ERROR_SECRET_TOO_LONG",
  "ERROR_INTERNAL_DB_ERROR",
  "ERROR_TOO_MANY_CONTEXT_IDS",
  "ERROR_LOGON_TYPE_NOT_GRANTED",
  "ERROR_NT_CROSS_ENCRYPTION_REQUIRED",
  "ERROR_NO_SUCH_MEMBER",
  "ERROR_INVALID_MEMBER",
  "ERROR_TOO_MANY_SIDS",
  "ERROR_LM_CROSS_ENCRYPTION_REQUIRED",
  "ERROR_NO_INHERITANCE",
  "ERROR_FILE_CORRUPT",
  "ERROR_DISK_CORRUPT",
  "ERROR_NO_USER_SESSION_KEY",
  "ERROR_LICENSE_QUOTA_EXCEEDED",
  "ERROR_WRONG_TARGET_NAME",
  "ERROR_MUTUAL_AUTH_FAILED",
  "ERROR_TIME_SKEW",
  "ERROR_CURRENT_DOMAIN_NOT_ALLOWED",
  "ERROR_INVALID_WINDOW_HANDLE",
  "ERROR_INVALID_MENU_HANDLE",
  "ERROR_INVALID_CURSOR_HANDLE",
  "ERROR_INVALID_ACCEL_HANDLE",
  "ERROR_INVALID_HOOK_HANDLE",
  "ERROR_INVALID_DWP_HANDLE",
  "ERROR_TLW_WITH_WSCHILD",
  "ERROR_CANNOT_FIND_WND_CLASS",
  "ERROR_WINDOW_OF_OTHER_THREAD",
  "ERROR_HOTKEY_ALREADY_REGISTERED",
  "ERROR_CLASS_ALREADY_EXISTS",
  "ERROR_CLASS_DOES_NOT_EXIST",
  "ERROR_CLASS_HAS_WINDOWS",
  "ERROR_INVALID_INDEX",
  "ERROR_INVALID_ICON_HANDLE",
  "ERROR_PRIVATE_DIALOG_INDEX",
  "ERROR_LISTBOX_ID_NOT_FOUND",
  "ERROR_NO_WILDCARD_CHARACTERS",
  "ERROR_CLIPBOARD_NOT_OPEN",
  "ERROR_HOTKEY_NOT_REGISTERED",
  "ERROR_WINDOW_NOT_DIALOG",
  "ERROR_CONTROL_ID_NOT_FOUND",
  "ERROR_INVALID_COMBOBOX_MESSAGE",
  "ERROR_WINDOW_NOT_COMBOBOX",
  "ERROR_INVALID_EDIT_HEIGHT",
  "ERROR_DC_NOT_FOUND",
  "ERROR_INVALID_HOOK_FILTER",
  "ERROR_INVALID_FILTER_PROC",
  "ERROR_HOOK_NEEDS_HMOD",
  "ERROR_GLOBAL_ONLY_HOOK",
  "ERROR_JOURNAL_HOOK_SET",
  "ERROR_HOOK_NOT_INSTALLED",
  "ERROR_INVALID_LB_MESSAGE",
  "ERROR_SETCOUNT_ON_BAD_LB",
  "ERROR_LB_WITHOUT_TABSTOPS",
  "ERROR_DESTROY_OBJECT_OF_OTHER_THREAD",
  "ERROR_CHILD_WINDOW_MENU",
  "ERROR_NO_SYSTEM_MENU",
  "ERROR_INVALID_MSGBOX_STYLE",
  "ERROR_INVALID_SPI_VALUE",
  "ERROR_SCREEN_ALREADY_LOCKED",
  "ERROR_HWNDS_HAVE_DIFF_PARENT",
  "ERROR_NOT_CHILD_WINDOW",
  "ERROR_INVALID_GW_COMMAND",
  "ERROR_INVALID_THREAD_ID",
  "ERROR_NON_MDICHILD_WINDOW",
  "ERROR_POPUP_ALREADY_ACTIVE",
  "ERROR_NO_SCROLLBARS",
  "ERROR_INVALID_SCROLLBAR_RANGE",
  "ERROR_INVALID_SHOWWIN_COMMAND",
  "ERROR_NO_SYSTEM_RESOURCES",
  "ERROR_NONPAGED_SYSTEM_RESOURCES",
  "ERROR_PAGED_SYSTEM_RESOURCES",
  "ERROR_WORKING_SET_QUOTA",
  "ERROR_PAGEFILE_QUOTA",
  "ERROR_COMMITMENT_LIMIT",
  "ERROR_MENU_ITEM_NOT_FOUND",
  "ERROR_INVALID_KEYBOARD_HANDLE",
  "ERROR_HOOK_TYPE_NOT_ALLOWED",
  "ERROR_REQUIRES_INTERACTIVE_WINDOWSTATION",
  "ERROR_TIMEOUT",
  "ERROR_INVALID_MONITOR_HANDLE",
  "ERROR_INCORRECT_SIZE",
  "ERROR_SYMLINK_CLASS_DISABLED",
  "ERROR_SYMLINK_NOT_SUPPORTED",
  "ERROR_XML_PARSE_ERROR",
  "ERROR_XMLDSIG_ERROR",
  "ERROR_RESTART_APPLICATION",
  "ERROR_WRONG_COMPARTMENT",
  "ERROR_AUTHIP_FAILURE",
  "ERROR_NO_NVRAM_RESOURCES",
  "ERROR_NOT_GUI_PROCESS",
  "ERROR_EVENTLOG_FILE_CORRUPT",
  "ERROR_EVENTLOG_CANT_START",
  "ERROR_LOG_FILE_FULL",
  "ERROR_EVENTLOG_FILE_CHANGED",
  "ERROR_CONTAINER_ASSIGNED",
  "ERROR_JOB_NO_CONTAINER",
  "ERROR_INVALID_TASK_NAME",
  "ERROR_INVALID_TASK_INDEX",
  "ERROR_THREAD_ALREADY_IN_TASK",
  "ERROR_INSTALL_SERVICE_FAILURE",
  "ERROR_INSTALL_USEREXIT",
  "ERROR_INSTALL_FAILURE",
  "ERROR_INSTALL_SUSPEND",
  "ERROR_UNKNOWN_PRODUCT",
  "ERROR_UNKNOWN_FEATURE",
  "ERROR_UNKNOWN_COMPONENT",
  "ERROR_UNKNOWN_PROPERTY",
  "ERROR_INVALID_HANDLE_STATE",
  "ERROR_BAD_CONFIGURATION",
  "ERROR_INDEX_ABSENT",
  "ERROR_INSTALL_SOURCE_ABSENT",
  "ERROR_INSTALL_PACKAGE_VERSION",
  "ERROR_PRODUCT_UNINSTALLED",
  "ERROR_BAD_QUERY_SYNTAX",
  "ERROR_INVALID_FIELD",
  "ERROR_DEVICE_REMOVED",
  "ERROR_INSTALL_ALREADY_RUNNING",
  "ERROR_INSTALL_PACKAGE_OPEN_FAILED",
  "ERROR_INSTALL_PACKAGE_INVALID",
  "ERROR_INSTALL_UI_FAILURE",
  "ERROR_INSTALL_LOG_FAILURE",
  "ERROR_INSTALL_LANGUAGE_UNSUPPORTED",
  "ERROR_INSTALL_TRANSFORM_FAILURE",
  "ERROR_INSTALL_PACKAGE_REJECTED",
  "ERROR_FUNCTION_NOT_CALLED",
  "ERROR_FUNCTION_FAILED",
  "ERROR_INVALID_TABLE",
  "ERROR_DATATYPE_MISMATCH",
  "ERROR_UNSUPPORTED_TYPE",
  "ERROR_CREATE_FAILED",
  "ERROR_INSTALL_TEMP_UNWRITABLE",
  "ERROR_INSTALL_PLATFORM_UNSUPPORTED",
  "ERROR_INSTALL_NOTUSED",
  "ERROR_PATCH_PACKAGE_OPEN_FAILED",
  "ERROR_PATCH_PACKAGE_INVALID",
  "ERROR_PATCH_PACKAGE_UNSUPPORTED",
  "ERROR_PRODUCT_VERSION",
  "ERROR_INVALID_COMMAND_LINE",
  "ERROR_INSTALL_REMOTE_DISALLOWED",
  "ERROR_SUCCESS_REBOOT_INITIATED",
  "ERROR_PATCH_TARGET_NOT_FOUND",
  "ERROR_PATCH_PACKAGE_REJECTED",
  "ERROR_INSTALL_TRANSFORM_REJECTED",
  "ERROR_INSTALL_REMOTE_PROHIBITED",
  "ERROR_PATCH_REMOVAL_UNSUPPORTED",
  "ERROR_UNKNOWN_PATCH",
  "ERROR_PATCH_NO_SEQUENCE",
  "ERROR_PATCH_REMOVAL_DISALLOWED",
  "ERROR_INVALID_PATCH_XML",
  "ERROR_PATCH_MANAGED_ADVERTISED_PRODUCT",
  "ERROR_INSTALL_SERVICE_SAFEBOOT",
  "ERROR_FAIL_FAST_EXCEPTION",
  "ERROR_INSTALL_REJECTED",
  "ERROR_DYNAMIC_CODE_BLOCKED",
  "ERROR_NOT_SAME_OBJECT",
  "ERROR_STRICT_CFG_VIOLATION",
  "ERROR_SET_CONTEXT_DENIED",
  "ERROR_CROSS_PARTITION_VIOLATION",
  "ERROR_RETURN_ADDRESS_HIJACK_ATTEMPT",
  "ERROR_INVALID_USER_BUFFER",
  "ERROR_UNRECOGNIZED_MEDIA",
  "ERROR_NO_TRUST_LSA_SECRET",
  "ERROR_NO_TRUST_SAM_ACCOUNT",
  "ERROR_TRUSTED_DOMAIN_FAILURE",
  "ERROR_TRUSTED_RELATIONSHIP_FAILURE",
  "ERROR_TRUST_FAILURE",
  "ERROR_NETLOGON_NOT_STARTED",
  "ERROR_ACCOUNT_EXPIRED",
  "ERROR_REDIRECTOR_HAS_OPEN_HANDLES",
  "ERROR_PRINTER_DRIVER_ALREADY_INSTALLED",
  "ERROR_UNKNOWN_PORT",
  "ERROR_UNKNOWN_PRINTER_DRIVER",
  "ERROR_UNKNOWN_PRINTPROCESSOR",
  "ERROR_INVALID_SEPARATOR_FILE",
  "ERROR_INVALID_PRIORITY",
  "ERROR_INVALID_PRINTER_NAME",
  "ERROR_PRINTER_ALREADY_EXISTS",
  "ERROR_INVALID_PRINTER_COMMAND",
  "ERROR_INVALID_DATATYPE",
  "ERROR_INVALID_ENVIRONMENT",
  "ERROR_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT",
  "ERROR_NOLOGON_WORKSTATION_TRUST_ACCOUNT",
  "ERROR_NOLOGON_SERVER_TRUST_ACCOUNT",
  "ERROR_DOMAIN_TRUST_INCONSISTENT",
  "ERROR_SERVER_HAS_OPEN_HANDLES",
  "ERROR_RESOURCE_DATA_NOT_FOUND",
  "ERROR_RESOURCE_TYPE_NOT_FOUND",
  "ERROR_RESOURCE_NAME_NOT_FOUND",
  "ERROR_RESOURCE_LANG_NOT_FOUND",
  "ERROR_NOT_ENOUGH_QUOTA",
  "ERROR_INVALID_TIME",
  "ERROR_INVALID_FORM_NAME",
  "ERROR_INVALID_FORM_SIZE",
  "ERROR_ALREADY_WAITING",
  "ERROR_PRINTER_DELETED",
  "ERROR_INVALID_PRINTER_STATE",
  "ERROR_PASSWORD_MUST_CHANGE",
  "ERROR_DOMAIN_CONTROLLER_NOT_FOUND",
  "ERROR_ACCOUNT_LOCKED_OUT",
  "ERROR_NO_SITENAME",
  "ERROR_CANT_ACCESS_FILE",
  "ERROR_CANT_RESOLVE_FILENAME",
  "ERROR_KM_DRIVER_BLOCKED",
  "ERROR_CONTEXT_EXPIRED",
  "ERROR_PER_USER_TRUST_QUOTA_EXCEEDED",
  "ERROR_ALL_USER_TRUST_QUOTA_EXCEEDED",
  "ERROR_USER_DELETE_TRUST_QUOTA_EXCEEDED",
  "ERROR_AUTHENTICATION_FIREWALL_FAILED",
  "ERROR_REMOTE_PRINT_CONNECTIONS_BLOCKED",
  "ERROR_NTLM_BLOCKED",
  "ERROR_PASSWORD_CHANGE_REQUIRED",
  "ERROR_LOST_MODE_LOGON_RESTRICTION",
  "ERROR_INVALID_PIXEL_FORMAT",
  "ERROR_BAD_DRIVER",
  "ERROR_INVALID_WINDOW_STYLE",
  "ERROR_METAFILE_NOT_SUPPORTED",
  "ERROR_TRANSFORM_NOT_SUPPORTED",
  "ERROR_CLIPPING_NOT_SUPPORTED",
  "ERROR_INVALID_CMM",
  "ERROR_INVALID_PROFILE",
  "ERROR_TAG_NOT_FOUND",
  "ERROR_TAG_NOT_PRESENT",
  "ERROR_DUPLICATE_TAG",
  "ERROR_PROFILE_NOT_ASSOCIATED_WITH_DEVICE",
  "ERROR_PROFILE_NOT_FOUND",
  "ERROR_INVALID_COLORSPACE",
  "ERROR_ICM_NOT_ENABLED",
  "ERROR_DELETING_ICM_XFORM",
  "ERROR_INVALID_TRANSFORM",
  "ERROR_COLORSPACE_MISMATCH",
  "ERROR_INVALID_COLORINDEX",
  "ERROR_PROFILE_DOES_NOT_MATCH_DEVICE",
  "ERROR_CONNECTED_OTHER_PASSWORD",
  "ERROR_CONNECTED_OTHER_PASSWORD_DEFAULT",
  "ERROR_BAD_USERNAME",
  "ERROR_NOT_CONNECTED",
  "ERROR_OPEN_FILES",
  "ERROR_ACTIVE_CONNECTIONS",
  "ERROR_DEVICE_IN_USE",
  "ERROR_UNKNOWN_PRINT_MONITOR",
  "ERROR_PRINTER_DRIVER_IN_USE",
  "ERROR_SPOOL_FILE_NOT_FOUND",
  "ERROR_SPL_NO_STARTDOC",
  "ERROR_SPL_NO_ADDJOB",
  "ERROR_PRINT_PROCESSOR_ALREADY_INSTALLED",
  "ERROR_PRINT_MONITOR_ALREADY_INSTALLED",
  "ERROR_INVALID_PRINT_MONITOR",
  "ERROR_PRINT_MONITOR_IN_USE",
  "ERROR_PRINTER_HAS_JOBS_QUEUED",
  "ERROR_SUCCESS_REBOOT_REQUIRED",
  "ERROR_SUCCESS_RESTART_REQUIRED",
  "ERROR_PRINTER_NOT_FOUND",
  "ERROR_PRINTER_DRIVER_WARNED",
  "ERROR_PRINTER_DRIVER_BLOCKED",
  "ERROR_PRINTER_DRIVER_PACKAGE_IN_USE",
  "ERROR_CORE_DRIVER_PACKAGE_NOT_FOUND",
  "ERROR_FAIL_REBOOT_REQUIRED",
  "ERROR_FAIL_REBOOT_INITIATED",
  "ERROR_PRINTER_DRIVER_DOWNLOAD_NEEDED",
  "ERROR_PRINT_JOB_RESTART_REQUIRED",
  "ERROR_INVALID_PRINTER_DRIVER_MANIFEST",
  "ERROR_PRINTER_NOT_SHAREABLE",
  "ERROR_REQUEST_PAUSED",
  "ERROR_APPEXEC_CONDITION_NOT_SATISFIED",
  "ERROR_APPEXEC_HANDLE_INVALIDATED",
  "ERROR_APPEXEC_INVALID_HOST_GENERATION",
  "ERROR_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION",
  "ERROR_APPEXEC_INVALID_HOST_STATE",
  "ERROR_APPEXEC_NO_DONOR",
  "ERROR_APPEXEC_HOST_ID_MISMATCH",
  "ERROR_APPEXEC_UNKNOWN_USER",
  "ERROR_IO_REISSUE_AS_CACHED",
  "ERROR_WINS_INTERNAL",
  "ERROR_CAN_NOT_DEL_LOCAL_WINS",
  "ERROR_STATIC_INIT",
  "ERROR_INC_BACKUP",
  "ERROR_FULL_BACKUP",
  "ERROR_REC_NON_EXISTENT",
  "ERROR_RPL_NOT_ALLOWED",
  "ERROR_DHCP_ADDRESS_CONFLICT",
  "ERROR_WMI_GUID_NOT_FOUND",
  "ERROR_WMI_INSTANCE_NOT_FOUND",
  "ERROR_WMI_ITEMID_NOT_FOUND",
  "ERROR_WMI_TRY_AGAIN",
  "ERROR_WMI_DP_NOT_FOUND",
  "ERROR_WMI_UNRESOLVED_INSTANCE_REF",
  "ERROR_WMI_ALREADY_ENABLED",
  "ERROR_WMI_GUID_DISCONNECTED",
  "ERROR_WMI_SERVER_UNAVAILABLE",
  "ERROR_WMI_DP_FAILED",
  "ERROR_WMI_INVALID_MOF",
  "ERROR_WMI_INVALID_REGINFO",
  "ERROR_WMI_ALREADY_DISABLED",
  "ERROR_WMI_READ_ONLY",
  "ERROR_WMI_SET_FAILURE",
  "ERROR_NOT_APPCONTAINER",
  "ERROR_APPCONTAINER_REQUIRED",
  "ERROR_NOT_SUPPORTED_IN_APPCONTAINER",
  "ERROR_INVALID_PACKAGE_SID_LENGTH",
  "ERROR_INVALID_MEDIA",
  "ERROR_INVALID_LIBRARY",
  "ERROR_INVALID_MEDIA_POOL",
  "ERROR_DRIVE_MEDIA_MISMATCH",
  "ERROR_MEDIA_OFFLINE",
  "ERROR_LIBRARY_OFFLINE",
  "ERROR_EMPTY",
  "ERROR_NOT_EMPTY",
  "ERROR_MEDIA_UNAVAILABLE",
  "ERROR_RESOURCE_DISABLED",
  "ERROR_INVALID_CLEANER",
  "ERROR_UNABLE_TO_CLEAN",
  "ERROR_OBJECT_NOT_FOUND",
  "ERROR_DATABASE_FAILURE",
  "ERROR_DATABASE_FULL",
  "ERROR_MEDIA_INCOMPATIBLE",
  "ERROR_RESOURCE_NOT_PRESENT",
  "ERROR_INVALID_OPERATION",
  "ERROR_MEDIA_NOT_AVAILABLE",
  "ERROR_DEVICE_NOT_AVAILABLE",
  "ERROR_REQUEST_REFUSED",
  "ERROR_INVALID_DRIVE_OBJECT",
  "ERROR_LIBRARY_FULL",
  "ERROR_MEDIUM_NOT_ACCESSIBLE",
  "ERROR_UNABLE_TO_LOAD_MEDIUM",
  "ERROR_UNABLE_TO_INVENTORY_DRIVE",
  "ERROR_UNABLE_TO_INVENTORY_SLOT",
  "ERROR_UNABLE_TO_INVENTORY_TRANSPORT",
  "ERROR_TRANSPORT_FULL",
  "ERROR_CONTROLLING_IEPORT",
  "ERROR_UNABLE_TO_EJECT_MOUNTED_MEDIA",
  "ERROR_CLEANER_SLOT_SET",
  "ERROR_CLEANER_SLOT_NOT_SET",
  "ERROR_CLEANER_CARTRIDGE_SPENT",
  "ERROR_UNEXPECTED_OMID",
  "ERROR_CANT_DELETE_LAST_ITEM",
  "ERROR_MESSAGE_EXCEEDS_MAX_SIZE",
  "ERROR_VOLUME_CONTAINS_SYS_FILES",
  "ERROR_INDIGENOUS_TYPE",
  "ERROR_NO_SUPPORTING_DRIVES",
  "ERROR_CLEANER_CARTRIDGE_INSTALLED",
  "ERROR_IEPORT_FULL",
  "ERROR_FILE_OFFLINE",
  "ERROR_REMOTE_STORAGE_NOT_ACTIVE",
  "ERROR_REMOTE_STORAGE_MEDIA_ERROR",
  "ERROR_NOT_A_REPARSE_POINT",
  "ERROR_REPARSE_ATTRIBUTE_CONFLICT",
  "ERROR_INVALID_REPARSE_DATA",
  "ERROR_REPARSE_TAG_INVALID",
  "ERROR_REPARSE_TAG_MISMATCH",
  "ERROR_REPARSE_POINT_ENCOUNTERED",
  "ERROR_APP_DATA_NOT_FOUND",
  "ERROR_APP_DATA_EXPIRED",
  "ERROR_APP_DATA_CORRUPT",
  "ERROR_APP_DATA_LIMIT_EXCEEDED",
  "ERROR_APP_DATA_REBOOT_REQUIRED",
  "ERROR_SECUREBOOT_ROLLBACK_DETECTED",
  "ERROR_SECUREBOOT_POLICY_VIOLATION",
  "ERROR_SECUREBOOT_INVALID_POLICY",
  "ERROR_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND",
  "ERROR_SECUREBOOT_POLICY_NOT_SIGNED",
  "ERROR_SECUREBOOT_NOT_ENABLED",
  "ERROR_SECUREBOOT_FILE_REPLACED",
  "ERROR_SECUREBOOT_POLICY_NOT_AUTHORIZED",
  "ERROR_SECUREBOOT_POLICY_UNKNOWN",
  "ERROR_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION",
  "ERROR_SECUREBOOT_PLATFORM_ID_MISMATCH",
  "ERROR_SECUREBOOT_POLICY_ROLLBACK_DETECTED",
  "ERROR_SECUREBOOT_POLICY_UPGRADE_MISMATCH",
  "ERROR_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING",
  "ERROR_SECUREBOOT_NOT_BASE_POLICY",
  "ERROR_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY",
  "ERROR_OFFLOAD_READ_FLT_NOT_SUPPORTED",
  "ERROR_OFFLOAD_WRITE_FLT_NOT_SUPPORTED",
  "ERROR_OFFLOAD_READ_FILE_NOT_SUPPORTED",
  "ERROR_OFFLOAD_WRITE_FILE_NOT_SUPPORTED",
  "ERROR_ALREADY_HAS_STREAM_ID",
  "ERROR_SMR_GARBAGE_COLLECTION_REQUIRED",
  "ERROR_WOF_WIM_HEADER_CORRUPT",
  "ERROR_WOF_WIM_RESOURCE_TABLE_CORRUPT",
  "ERROR_WOF_FILE_RESOURCE_TABLE_CORRUPT",
  "ERROR_VOLUME_NOT_SIS_ENABLED",
  "ERROR_SYSTEM_INTEGRITY_ROLLBACK_DETECTED",
  "ERROR_SYSTEM_INTEGRITY_POLICY_VIOLATION",
  "ERROR_SYSTEM_INTEGRITY_INVALID_POLICY",
  "ERROR_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED",
  "ERROR_SYSTEM_INTEGRITY_TOO_MANY_POLICIES",
  "ERROR_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED",
  "ERROR_VSM_NOT_INITIALIZED",
  "ERROR_VSM_DMA_PROTECTION_NOT_IN_USE",
  "ERROR_PLATFORM_MANIFEST_NOT_AUTHORIZED",
  "ERROR_PLATFORM_MANIFEST_INVALID",
  "ERROR_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED",
  "ERROR_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED",
  "ERROR_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND",
  "ERROR_PLATFORM_MANIFEST_NOT_ACTIVE",
  "ERROR_PLATFORM_MANIFEST_NOT_SIGNED",
  "ERROR_DEPENDENT_RESOURCE_EXISTS",
  "ERROR_DEPENDENCY_NOT_FOUND",
  "ERROR_DEPENDENCY_ALREADY_EXISTS",
  "ERROR_RESOURCE_NOT_ONLINE",
  "ERROR_HOST_NODE_NOT_AVAILABLE",
  "ERROR_RESOURCE_NOT_AVAILABLE",
  "ERROR_RESOURCE_NOT_FOUND",
  "ERROR_SHUTDOWN_CLUSTER",
  "ERROR_CANT_EVICT_ACTIVE_NODE",
  "ERROR_OBJECT_ALREADY_EXISTS",
  "ERROR_OBJECT_IN_LIST",
  "ERROR_GROUP_NOT_AVAILABLE",
  "ERROR_GROUP_NOT_FOUND",
  "ERROR_GROUP_NOT_ONLINE",
  "ERROR_HOST_NODE_NOT_RESOURCE_OWNER",
  "ERROR_HOST_NODE_NOT_GROUP_OWNER",
  "ERROR_RESMON_CREATE_FAILED",
  "ERROR_RESMON_ONLINE_FAILED",
  "ERROR_RESOURCE_ONLINE",
  "ERROR_QUORUM_RESOURCE",
  "ERROR_NOT_QUORUM_CAPABLE",
  "ERROR_CLUSTER_SHUTTING_DOWN",
  "ERROR_INVALID_STATE",
  "ERROR_RESOURCE_PROPERTIES_STORED",
  "ERROR_NOT_QUORUM_CLASS",
  "ERROR_CORE_RESOURCE",
  "ERROR_QUORUM_RESOURCE_ONLINE_FAILED",
  "ERROR_QUORUMLOG_OPEN_FAILED",
  "ERROR_CLUSTERLOG_CORRUPT",
  "ERROR_CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE",
  "ERROR_CLUSTERLOG_EXCEEDS_MAXSIZE",
  "ERROR_CLUSTERLOG_CHKPOINT_NOT_FOUND",
  "ERROR_CLUSTERLOG_NOT_ENOUGH_SPACE",
  "ERROR_QUORUM_OWNER_ALIVE",
  "ERROR_NETWORK_NOT_AVAILABLE",
  "ERROR_NODE_NOT_AVAILABLE",
  "ERROR_ALL_NODES_NOT_AVAILABLE",
  "ERROR_RESOURCE_FAILED",
  "ERROR_CLUSTER_INVALID_NODE",
  "ERROR_CLUSTER_NODE_EXISTS",
  "ERROR_CLUSTER_JOIN_IN_PROGRESS",
  "ERROR_CLUSTER_NODE_NOT_FOUND",
  "ERROR_CLUSTER_LOCAL_NODE_NOT_FOUND",
  "ERROR_CLUSTER_NETWORK_EXISTS",
  "ERROR_CLUSTER_NETWORK_NOT_FOUND",
  "ERROR_CLUSTER_NETINTERFACE_EXISTS",
  "ERROR_CLUSTER_NETINTERFACE_NOT_FOUND",
  "ERROR_CLUSTER_INVALID_REQUEST",
  "ERROR_CLUSTER_INVALID_NETWORK_PROVIDER",
  "ERROR_CLUSTER_NODE_DOWN",
  "ERROR_CLUSTER_NODE_UNREACHABLE",
  "ERROR_CLUSTER_NODE_NOT_MEMBER",
  "ERROR_CLUSTER_JOIN_NOT_IN_PROGRESS",
  "ERROR_CLUSTER_INVALID_NETWORK",
  "ERROR_CLUSTER_NODE_UP",
  "ERROR_CLUSTER_IPADDR_IN_USE",
  "ERROR_CLUSTER_NODE_NOT_PAUSED",
  "ERROR_CLUSTER_NO_SECURITY_CONTEXT",
  "ERROR_CLUSTER_NETWORK_NOT_INTERNAL",
  "ERROR_CLUSTER_NODE_ALREADY_UP",
  "ERROR_CLUSTER_NODE_ALREADY_DOWN",
  "ERROR_CLUSTER_NETWORK_ALREADY_ONLINE",
  "ERROR_CLUSTER_NETWORK_ALREADY_OFFLINE",
  "ERROR_CLUSTER_NODE_ALREADY_MEMBER",
  "ERROR_CLUSTER_LAST_INTERNAL_NETWORK",
  "ERROR_CLUSTER_NETWORK_HAS_DEPENDENTS",
  "ERROR_INVALID_OPERATION_ON_QUORUM",
  "ERROR_DEPENDENCY_NOT_ALLOWED",
  "ERROR_CLUSTER_NODE_PAUSED",
  "ERROR_NODE_CANT_HOST_RESOURCE",
  "ERROR_CLUSTER_NODE_NOT_READY",
  "ERROR_CLUSTER_NODE_SHUTTING_DOWN",
  "ERROR_CLUSTER_JOIN_ABORTED",
  "ERROR_CLUSTER_INCOMPATIBLE_VERSIONS",
  "ERROR_CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED",
  "ERROR_CLUSTER_SYSTEM_CONFIG_CHANGED",
  "ERROR_CLUSTER_RESOURCE_TYPE_NOT_FOUND",
  "ERROR_CLUSTER_RESTYPE_NOT_SUPPORTED",
  "ERROR_CLUSTER_RESNAME_NOT_FOUND",
  "ERROR_CLUSTER_NO_RPC_PACKAGES_REGISTERED",
  "ERROR_CLUSTER_OWNER_NOT_IN_PREFLIST",
  "ERROR_CLUSTER_DATABASE_SEQMISMATCH",
  "ERROR_RESMON_INVALID_STATE",
  "ERROR_CLUSTER_GUM_NOT_LOCKER",
  "ERROR_QUORUM_DISK_NOT_FOUND",
  "ERROR_DATABASE_BACKUP_CORRUPT",
  "ERROR_CLUSTER_NODE_ALREADY_HAS_DFS_ROOT",
  "ERROR_RESOURCE_PROPERTY_UNCHANGEABLE",
  "ERROR_NO_ADMIN_ACCESS_POINT",
  "ERROR_CLUSTER_MEMBERSHIP_INVALID_STATE",
  "ERROR_CLUSTER_QUORUMLOG_NOT_FOUND",
  "ERROR_CLUSTER_MEMBERSHIP_HALT",
  "ERROR_CLUSTER_INSTANCE_ID_MISMATCH",
  "ERROR_CLUSTER_NETWORK_NOT_FOUND_FOR_IP",
  "ERROR_CLUSTER_PROPERTY_DATA_TYPE_MISMATCH",
  "ERROR_CLUSTER_EVICT_WITHOUT_CLEANUP",
  "ERROR_CLUSTER_PARAMETER_MISMATCH",
  "ERROR_NODE_CANNOT_BE_CLUSTERED",
  "ERROR_CLUSTER_WRONG_OS_VERSION",
  "ERROR_CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME",
  "ERROR_CLUSCFG_ALREADY_COMMITTED",
  "ERROR_CLUSCFG_ROLLBACK_FAILED",
  "ERROR_CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT",
  "ERROR_CLUSTER_OLD_VERSION",
  "ERROR_CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME",
  "ERROR_CLUSTER_NO_NET_ADAPTERS",
  "ERROR_CLUSTER_POISONED",
  "ERROR_CLUSTER_GROUP_MOVING",
  "ERROR_CLUSTER_RESOURCE_TYPE_BUSY",
  "ERROR_RESOURCE_CALL_TIMED_OUT",
  "ERROR_INVALID_CLUSTER_IPV6_ADDRESS",
  "ERROR_CLUSTER_INTERNAL_INVALID_FUNCTION",
  "ERROR_CLUSTER_PARAMETER_OUT_OF_BOUNDS",
  "ERROR_CLUSTER_PARTIAL_SEND",
  "ERROR_CLUSTER_REGISTRY_INVALID_FUNCTION",
  "ERROR_CLUSTER_INVALID_STRING_TERMINATION",
  "ERROR_CLUSTER_INVALID_STRING_FORMAT",
  "ERROR_CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS",
  "ERROR_CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS",
  "ERROR_CLUSTER_NULL_DATA",
  "ERROR_CLUSTER_PARTIAL_READ",
  "ERROR_CLUSTER_PARTIAL_WRITE",
  "ERROR_CLUSTER_CANT_DESERIALIZE_DATA",
  "ERROR_DEPENDENT_RESOURCE_PROPERTY_CONFLICT",
  "ERROR_CLUSTER_NO_QUORUM",
  "ERROR_CLUSTER_INVALID_IPV6_NETWORK",
  "ERROR_CLUSTER_INVALID_IPV6_TUNNEL_NETWORK",
  "ERROR_QUORUM_NOT_ALLOWED_IN_THIS_GROUP",
  "ERROR_DEPENDENCY_TREE_TOO_COMPLEX",
  "ERROR_EXCEPTION_IN_RESOURCE_CALL",
  "ERROR_CLUSTER_RHS_FAILED_INITIALIZATION",
  "ERROR_CLUSTER_NOT_INSTALLED",
  "ERROR_CLUSTER_RESOURCES_MUST_BE_ONLINE_ON_THE_SAME_NODE",
  "ERROR_CLUSTER_MAX_NODES_IN_CLUSTER",
  "ERROR_CLUSTER_TOO_MANY_NODES",
  "ERROR_CLUSTER_OBJECT_ALREADY_USED",
  "ERROR_NONCORE_GROUPS_FOUND",
  "ERROR_FILE_SHARE_RESOURCE_CONFLICT",
  "ERROR_CLUSTER_EVICT_INVALID_REQUEST",
  "ERROR_CLUSTER_SINGLETON_RESOURCE",
  "ERROR_CLUSTER_GROUP_SINGLETON_RESOURCE",
  "ERROR_CLUSTER_RESOURCE_PROVIDER_FAILED",
  "ERROR_CLUSTER_RESOURCE_CONFIGURATION_ERROR",
  "ERROR_CLUSTER_GROUP_BUSY",
  "ERROR_CLUSTER_NOT_SHARED_VOLUME",
  "ERROR_CLUSTER_INVALID_SECURITY_DESCRIPTOR",
  "ERROR_CLUSTER_SHARED_VOLUMES_IN_USE",
  "ERROR_CLUSTER_USE_SHARED_VOLUMES_API",
  "ERROR_CLUSTER_BACKUP_IN_PROGRESS",
  "ERROR_NON_CSV_PATH",
  "ERROR_CSV_VOLUME_NOT_LOCAL",
  "ERROR_CLUSTER_WATCHDOG_TERMINATING",
  "ERROR_CLUSTER_RESOURCE_VETOED_MOVE_INCOMPATIBLE_NODES",
  "ERROR_CLUSTER_INVALID_NODE_WEIGHT",
  "ERROR_CLUSTER_RESOURCE_VETOED_CALL",
  "ERROR_RESMON_SYSTEM_RESOURCES_LACKING",
  "ERROR_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_DESTINATION",
  "ERROR_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_SOURCE",
  "ERROR_CLUSTER_GROUP_QUEUED",
  "ERROR_CLUSTER_RESOURCE_LOCKED_STATUS",
  "ERROR_CLUSTER_SHARED_VOLUME_FAILOVER_NOT_ALLOWED",
  "ERROR_CLUSTER_NODE_DRAIN_IN_PROGRESS",
  "ERROR_CLUSTER_DISK_NOT_CONNECTED",
  "ERROR_DISK_NOT_CSV_CAPABLE",
  "ERROR_RESOURCE_NOT_IN_AVAILABLE_STORAGE",
  "ERROR_CLUSTER_SHARED_VOLUME_REDIRECTED",
  "ERROR_CLUSTER_SHARED_VOLUME_NOT_REDIRECTED",
  "ERROR_CLUSTER_CANNOT_RETURN_PROPERTIES",
  "ERROR_CLUSTER_RESOURCE_CONTAINS_UNSUPPORTED_DIFF_AREA_FOR_SHARED_VOLUMES",
  "ERROR_CLUSTER_RESOURCE_IS_IN_MAINTENANCE_MODE",
  "ERROR_CLUSTER_AFFINITY_CONFLICT",
  "ERROR_CLUSTER_RESOURCE_IS_REPLICA_VIRTUAL_MACHINE",
  "ERROR_CLUSTER_UPGRADE_INCOMPATIBLE_VERSIONS",
  "ERROR_CLUSTER_UPGRADE_FIX_QUORUM_NOT_SUPPORTED",
  "ERROR_CLUSTER_UPGRADE_RESTART_REQUIRED",
  "ERROR_CLUSTER_UPGRADE_IN_PROGRESS",
  "ERROR_CLUSTER_UPGRADE_INCOMPLETE",
  "ERROR_CLUSTER_NODE_IN_GRACE_PERIOD",
  "ERROR_CLUSTER_CSV_IO_PAUSE_TIMEOUT",
  "ERROR_NODE_NOT_ACTIVE_CLUSTER_MEMBER",
  "ERROR_CLUSTER_RESOURCE_NOT_MONITORED",
  "ERROR_CLUSTER_RESOURCE_DOES_NOT_SUPPORT_UNMONITORED",
  "ERROR_CLUSTER_RESOURCE_IS_REPLICATED",
  "ERROR_CLUSTER_NODE_ISOLATED",
  "ERROR_CLUSTER_NODE_QUARANTINED",
  "ERROR_CLUSTER_DATABASE_UPDATE_CONDITION_FAILED",
  "ERROR_CLUSTER_SPACE_DEGRADED",
  "ERROR_CLUSTER_TOKEN_DELEGATION_NOT_SUPPORTED",
  "ERROR_CLUSTER_CSV_INVALID_HANDLE",
  "ERROR_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR",
  "ERROR_GROUPSET_NOT_AVAILABLE",
  "ERROR_GROUPSET_NOT_FOUND",
  "ERROR_GROUPSET_CANT_PROVIDE",
  "ERROR_CLUSTER_FAULT_DOMAIN_PARENT_NOT_FOUND",
  "ERROR_CLUSTER_FAULT_DOMAIN_INVALID_HIERARCHY",
  "ERROR_CLUSTER_FAULT_DOMAIN_FAILED_S2D_VALIDATION",
  "ERROR_CLUSTER_FAULT_DOMAIN_S2D_CONNECTIVITY_LOSS",
  "ERROR_CLUSTER_INVALID_INFRASTRUCTURE_FILESERVER_NAME",
  "ERROR_CLUSTERSET_MANAGEMENT_CLUSTER_UNREACHABLE",
  "ERROR_ENCRYPTION_FAILED",
  "ERROR_DECRYPTION_FAILED",
  "ERROR_FILE_ENCRYPTED",
  "ERROR_NO_RECOVERY_POLICY",
  "ERROR_NO_EFS",
  "ERROR_WRONG_EFS",
  "ERROR_NO_USER_KEYS",
  "ERROR_FILE_NOT_ENCRYPTED",
  "ERROR_NOT_EXPORT_FORMAT",
  "ERROR_FILE_READ_ONLY",
  "ERROR_DIR_EFS_DISALLOWED",
  "ERROR_EFS_SERVER_NOT_TRUSTED",
  "ERROR_BAD_RECOVERY_POLICY",
  "ERROR_EFS_ALG_BLOB_TOO_BIG",
  "ERROR_VOLUME_NOT_SUPPORT_EFS",
  "ERROR_EFS_DISABLED",
  "ERROR_EFS_VERSION_NOT_SUPPORT",
  "ERROR_CS_ENCRYPTION_INVALID_SERVER_RESPONSE",
  "ERROR_CS_ENCRYPTION_UNSUPPORTED_SERVER",
  "ERROR_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE",
  "ERROR_CS_ENCRYPTION_NEW_ENCRYPTED_FILE",
  "ERROR_CS_ENCRYPTION_FILE_NOT_CSE",
  "ERROR_ENCRYPTION_POLICY_DENIES_OPERATION",
  "ERROR_WIP_ENCRYPTION_FAILED",
  "ERROR_NO_BROWSER_SERVERS_FOUND",
  "ERROR_CLUSTER_OBJECT_IS_CLUSTER_SET_VM",
  "ERROR_LOG_SECTOR_INVALID",
  "ERROR_LOG_SECTOR_PARITY_INVALID",
  "ERROR_LOG_SECTOR_REMAPPED",
  "ERROR_LOG_BLOCK_INCOMPLETE",
  "ERROR_LOG_INVALID_RANGE",
  "ERROR_LOG_BLOCKS_EXHAUSTED",
  "ERROR_LOG_READ_CONTEXT_INVALID",
  "ERROR_LOG_RESTART_INVALID",
  "ERROR_LOG_BLOCK_VERSION",
  "ERROR_LOG_BLOCK_INVALID",
  "ERROR_LOG_READ_MODE_INVALID",
  "ERROR_LOG_NO_RESTART",
  "ERROR_LOG_METADATA_CORRUPT",
  "ERROR_LOG_METADATA_INVALID",
  "ERROR_LOG_METADATA_INCONSISTENT",
  "ERROR_LOG_RESERVATION_INVALID",
  "ERROR_LOG_CANT_DELETE",
  "ERROR_LOG_CONTAINER_LIMIT_EXCEEDED",
  "ERROR_LOG_START_OF_LOG",
  "ERROR_LOG_POLICY_ALREADY_INSTALLED",
  "ERROR_LOG_POLICY_NOT_INSTALLED",
  "ERROR_LOG_POLICY_INVALID",
  "ERROR_LOG_POLICY_CONFLICT",
  "ERROR_LOG_PINNED_ARCHIVE_TAIL",
  "ERROR_LOG_RECORD_NONEXISTENT",
  "ERROR_LOG_RECORDS_RESERVED_INVALID",
  "ERROR_LOG_SPACE_RESERVED_INVALID",
  "ERROR_LOG_TAIL_INVALID",
  "ERROR_LOG_FULL",
  "ERROR_COULD_NOT_RESIZE_LOG",
  "ERROR_LOG_MULTIPLEXED",
  "ERROR_LOG_DEDICATED",
  "ERROR_LOG_ARCHIVE_NOT_IN_PROGRESS",
  "ERROR_LOG_ARCHIVE_IN_PROGRESS",
  "ERROR_LOG_EPHEMERAL",
  "ERROR_LOG_NOT_ENOUGH_CONTAINERS",
  "ERROR_LOG_CLIENT_ALREADY_REGISTERED",
  "ERROR_LOG_CLIENT_NOT_REGISTERED",
  "ERROR_LOG_FULL_HANDLER_IN_PROGRESS",
  "ERROR_LOG_CONTAINER_READ_FAILED",
  "ERROR_LOG_CONTAINER_WRITE_FAILED",
  "ERROR_LOG_CONTAINER_OPEN_FAILED",
  "ERROR_LOG_CONTAINER_STATE_INVALID",
  "ERROR_LOG_STATE_INVALID",
  "ERROR_LOG_PINNED",
  "ERROR_LOG_METADATA_FLUSH_FAILED",
  "ERROR_LOG_INCONSISTENT_SECURITY",
  "ERROR_LOG_APPENDED_FLUSH_FAILED",
  "ERROR_LOG_PINNED_RESERVATION",
  "ERROR_INVALID_TRANSACTION",
  "ERROR_TRANSACTION_NOT_ACTIVE",
  "ERROR_TRANSACTION_REQUEST_NOT_VALID",
  "ERROR_TRANSACTION_NOT_REQUESTED",
  "ERROR_TRANSACTION_ALREADY_ABORTED",
  "ERROR_TRANSACTION_ALREADY_COMMITTED",
  "ERROR_TM_INITIALIZATION_FAILED",
  "ERROR_RESOURCEMANAGER_READ_ONLY",
  "ERROR_TRANSACTION_NOT_JOINED",
  "ERROR_TRANSACTION_SUPERIOR_EXISTS",
  "ERROR_CRM_PROTOCOL_ALREADY_EXISTS",
  "ERROR_TRANSACTION_PROPAGATION_FAILED",
  "ERROR_CRM_PROTOCOL_NOT_FOUND",
  "ERROR_TRANSACTION_INVALID_MARSHALL_BUFFER",
  "ERROR_CURRENT_TRANSACTION_NOT_VALID",
  "ERROR_TRANSACTION_NOT_FOUND",
  "ERROR_RESOURCEMANAGER_NOT_FOUND",
  "ERROR_ENLISTMENT_NOT_FOUND",
  "ERROR_TRANSACTIONMANAGER_NOT_FOUND",
  "ERROR_TRANSACTIONMANAGER_NOT_ONLINE",
  "ERROR_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION",
  "ERROR_TRANSACTION_NOT_ROOT",
  "ERROR_TRANSACTION_OBJECT_EXPIRED",
  "ERROR_TRANSACTION_RESPONSE_NOT_ENLISTED",
  "ERROR_TRANSACTION_RECORD_TOO_LONG",
  "ERROR_IMPLICIT_TRANSACTION_NOT_SUPPORTED",
  "ERROR_TRANSACTION_INTEGRITY_VIOLATED",
  "ERROR_TRANSACTIONMANAGER_IDENTITY_MISMATCH",
  "ERROR_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT",
  "ERROR_TRANSACTION_MUST_WRITETHROUGH",
  "ERROR_TRANSACTION_NO_SUPERIOR",
  "ERROR_HEURISTIC_DAMAGE_POSSIBLE",
  "ERROR_TRANSACTIONAL_CONFLICT",
  "ERROR_RM_NOT_ACTIVE",
  "ERROR_RM_METADATA_CORRUPT",
  "ERROR_DIRECTORY_NOT_RM",
  "ERROR_TRANSACTIONS_UNSUPPORTED_REMOTE",
  "ERROR_LOG_RESIZE_INVALID_SIZE",
  "ERROR_OBJECT_NO_LONGER_EXISTS",
  "ERROR_STREAM_MINIVERSION_NOT_FOUND",
  "ERROR_STREAM_MINIVERSION_NOT_VALID",
  "ERROR_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION",
  "ERROR_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT",
  "ERROR_CANT_CREATE_MORE_STREAM_MINIVERSIONS",
  "ERROR_REMOTE_FILE_VERSION_MISMATCH",
  "ERROR_HANDLE_NO_LONGER_VALID",
  "ERROR_NO_TXF_METADATA",
  "ERROR_LOG_CORRUPTION_DETECTED",
  "ERROR_CANT_RECOVER_WITH_HANDLE_OPEN",
  "ERROR_RM_DISCONNECTED",
  "ERROR_ENLISTMENT_NOT_SUPERIOR",
  "ERROR_RECOVERY_NOT_NEEDED",
  "ERROR_RM_ALREADY_STARTED",
  "ERROR_FILE_IDENTITY_NOT_PERSISTENT",
  "ERROR_CANT_BREAK_TRANSACTIONAL_DEPENDENCY",
  "ERROR_CANT_CROSS_RM_BOUNDARY",
  "ERROR_TXF_DIR_NOT_EMPTY",
  "ERROR_INDOUBT_TRANSACTIONS_EXIST",
  "ERROR_TM_VOLATILE",
  "ERROR_ROLLBACK_TIMER_EXPIRED",
  "ERROR_TXF_ATTRIBUTE_CORRUPT",
  "ERROR_EFS_NOT_ALLOWED_IN_TRANSACTION",
  "ERROR_TRANSACTIONAL_OPEN_NOT_ALLOWED",
  "ERROR_LOG_GROWTH_FAILED",
  "ERROR_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE",
  "ERROR_TXF_METADATA_ALREADY_PRESENT",
  "ERROR_TRANSACTION_SCOPE_CALLBACKS_NOT_SET",
  "ERROR_TRANSACTION_REQUIRED_PROMOTION",
  "ERROR_CANNOT_EXECUTE_FILE_IN_TRANSACTION",
  "ERROR_TRANSACTIONS_NOT_FROZEN",
  "ERROR_TRANSACTION_FREEZE_IN_PROGRESS",
  "ERROR_NOT_SNAPSHOT_VOLUME",
  "ERROR_NO_SAVEPOINT_WITH_OPEN_FILES",
  "ERROR_DATA_LOST_REPAIR",
  "ERROR_SPARSE_NOT_ALLOWED_IN_TRANSACTION",
  "ERROR_TM_IDENTITY_MISMATCH",
  "ERROR_FLOATED_SECTION",
  "ERROR_CANNOT_ACCEPT_TRANSACTED_WORK",
  "ERROR_CANNOT_ABORT_TRANSACTIONS",
  "ERROR_BAD_CLUSTERS",
  "ERROR_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION",
  "ERROR_VOLUME_DIRTY",
  "ERROR_NO_LINK_TRACKING_IN_TRANSACTION",
  "ERROR_OPERATION_NOT_SUPPORTED_IN_TRANSACTION",
  "ERROR_EXPIRED_HANDLE",
  "ERROR_TRANSACTION_NOT_ENLISTED",
  "ERROR_CTX_WINSTATION_NAME_INVALID",
  "ERROR_CTX_INVALID_PD",
  "ERROR_CTX_PD_NOT_FOUND",
  "ERROR_CTX_WD_NOT_FOUND",
  "ERROR_CTX_CANNOT_MAKE_EVENTLOG_ENTRY",
  "ERROR_CTX_SERVICE_NAME_COLLISION",
  "ERROR_CTX_CLOSE_PENDING",
  "ERROR_CTX_NO_OUTBUF",
  "ERROR_CTX_MODEM_INF_NOT_FOUND",
  "ERROR_CTX_INVALID_MODEMNAME",
  "ERROR_CTX_MODEM_RESPONSE_ERROR",
  "ERROR_CTX_MODEM_RESPONSE_TIMEOUT",
  "ERROR_CTX_MODEM_RESPONSE_NO_CARRIER",
  "ERROR_CTX_MODEM_RESPONSE_NO_DIALTONE",
  "ERROR_CTX_MODEM_RESPONSE_BUSY",
  "ERROR_CTX_MODEM_RESPONSE_VOICE",
  "ERROR_CTX_TD_ERROR",
  "ERROR_CTX_WINSTATION_NOT_FOUND",
  "ERROR_CTX_WINSTATION_ALREADY_EXISTS",
  "ERROR_CTX_WINSTATION_BUSY",
  "ERROR_CTX_BAD_VIDEO_MODE",
  "ERROR_CTX_GRAPHICS_INVALID",
  "ERROR_CTX_LOGON_DISABLED",
  "ERROR_CTX_NOT_CONSOLE",
  "ERROR_CTX_CLIENT_QUERY_TIMEOUT",
  "ERROR_CTX_CONSOLE_DISCONNECT",
  "ERROR_CTX_CONSOLE_CONNECT",
  "ERROR_CTX_SHADOW_DENIED",
  "ERROR_CTX_WINSTATION_ACCESS_DENIED",
  "ERROR_CTX_INVALID_WD",
  "ERROR_CTX_SHADOW_INVALID",
  "ERROR_CTX_SHADOW_DISABLED",
  "ERROR_CTX_CLIENT_LICENSE_IN_USE",
  "ERROR_CTX_CLIENT_LICENSE_NOT_SET",
  "ERROR_CTX_LICENSE_NOT_AVAILABLE",
  "ERROR_CTX_LICENSE_CLIENT_INVALID",
  "ERROR_CTX_LICENSE_EXPIRED",
  "ERROR_CTX_SHADOW_NOT_RUNNING",
  "ERROR_CTX_SHADOW_ENDED_BY_MODE_CHANGE",
  "ERROR_ACTIVATION_COUNT_EXCEEDED",
  "ERROR_CTX_WINSTATIONS_DISABLED",
  "ERROR_CTX_ENCRYPTION_LEVEL_REQUIRED",
  "ERROR_CTX_SESSION_IN_USE",
  "ERROR_CTX_NO_FORCE_LOGOFF",
  "ERROR_CTX_ACCOUNT_RESTRICTION",
  "ERROR_RDP_PROTOCOL_ERROR",
  "ERROR_CTX_CDM_CONNECT",
  "ERROR_CTX_CDM_DISCONNECT",
  "ERROR_CTX_SECURITY_LAYER_ERROR",
  "ERROR_TS_INCOMPATIBLE_SESSIONS",
  "ERROR_TS_VIDEO_SUBSYSTEM_ERROR",
  "ERROR_DS_NOT_INSTALLED",
  "ERROR_DS_MEMBERSHIP_EVALUATED_LOCALLY",
  "ERROR_DS_NO_ATTRIBUTE_OR_VALUE",
  "ERROR_DS_INVALID_ATTRIBUTE_SYNTAX",
  "ERROR_DS_ATTRIBUTE_TYPE_UNDEFINED",
  "ERROR_DS_ATTRIBUTE_OR_VALUE_EXISTS",
  "ERROR_DS_BUSY",
  "ERROR_DS_UNAVAILABLE",
  "ERROR_DS_NO_RIDS_ALLOCATED",
  "ERROR_DS_NO_MORE_RIDS",
  "ERROR_DS_INCORRECT_ROLE_OWNER",
  "ERROR_DS_RIDMGR_INIT_ERROR",
  "ERROR_DS_OBJ_CLASS_VIOLATION",
  "ERROR_DS_CANT_ON_NON_LEAF",
  "ERROR_DS_CANT_ON_RDN",
  "ERROR_DS_CANT_MOD_OBJ_CLASS",
  "ERROR_DS_CROSS_DOM_MOVE_ERROR",
  "ERROR_DS_GC_NOT_AVAILABLE",
  "ERROR_SHARED_POLICY",
  "ERROR_POLICY_OBJECT_NOT_FOUND",
  "ERROR_POLICY_ONLY_IN_DS",
  "ERROR_PROMOTION_ACTIVE",
  "ERROR_NO_PROMOTION_ACTIVE",
  "ERROR_DS_OPERATIONS_ERROR",
  "ERROR_DS_PROTOCOL_ERROR",
  "ERROR_DS_TIMELIMIT_EXCEEDED",
  "ERROR_DS_SIZELIMIT_EXCEEDED",
  "ERROR_DS_ADMIN_LIMIT_EXCEEDED",
  "ERROR_DS_COMPARE_FALSE",
  "ERROR_DS_COMPARE_TRUE",
  "ERROR_DS_AUTH_METHOD_NOT_SUPPORTED",
  "ERROR_DS_STRONG_AUTH_REQUIRED",
  "ERROR_DS_INAPPROPRIATE_AUTH",
  "ERROR_DS_AUTH_UNKNOWN",
  "ERROR_DS_REFERRAL",
  "ERROR_DS_UNAVAILABLE_CRIT_EXTENSION",
  "ERROR_DS_CONFIDENTIALITY_REQUIRED",
  "ERROR_DS_INAPPROPRIATE_MATCHING",
  "ERROR_DS_CONSTRAINT_VIOLATION",
  "ERROR_DS_NO_SUCH_OBJECT",
  "ERROR_DS_ALIAS_PROBLEM",
  "ERROR_DS_INVALID_DN_SYNTAX",
  "ERROR_DS_IS_LEAF",
  "ERROR_DS_ALIAS_DEREF_PROBLEM",
  "ERROR_DS_UNWILLING_TO_PERFORM",
  "ERROR_DS_LOOP_DETECT",
  "ERROR_DS_NAMING_VIOLATION",
  "ERROR_DS_OBJECT_RESULTS_TOO_LARGE",
  "ERROR_DS_AFFECTS_MULTIPLE_DSAS",
  "ERROR_DS_SERVER_DOWN",
  "ERROR_DS_LOCAL_ERROR",
  "ERROR_DS_ENCODING_ERROR",
  "ERROR_DS_DECODING_ERROR",
  "ERROR_DS_FILTER_UNKNOWN",
  "ERROR_DS_PARAM_ERROR",
  "ERROR_DS_NOT_SUPPORTED",
  "ERROR_DS_NO_RESULTS_RETURNED",
  "ERROR_DS_CONTROL_NOT_FOUND",
  "ERROR_DS_CLIENT_LOOP",
  "ERROR_DS_REFERRAL_LIMIT_EXCEEDED",
  "ERROR_DS_SORT_CONTROL_MISSING",
  "ERROR_DS_OFFSET_RANGE_ERROR",
  "ERROR_DS_RIDMGR_DISABLED",
  "ERROR_DS_ROOT_MUST_BE_NC",
  "ERROR_DS_ADD_REPLICA_INHIBITED",
  "ERROR_DS_ATT_NOT_DEF_IN_SCHEMA",
  "ERROR_DS_MAX_OBJ_SIZE_EXCEEDED",
  "ERROR_DS_OBJ_STRING_NAME_EXISTS",
  "ERROR_DS_NO_RDN_DEFINED_IN_SCHEMA",
  "ERROR_DS_RDN_DOESNT_MATCH_SCHEMA",
  "ERROR_DS_NO_REQUESTED_ATTS_FOUND",
  "ERROR_DS_USER_BUFFER_TO_SMALL",
  "ERROR_DS_ATT_IS_NOT_ON_OBJ",
  "ERROR_DS_ILLEGAL_MOD_OPERATION",
  "ERROR_DS_OBJ_TOO_LARGE",
  "ERROR_DS_BAD_INSTANCE_TYPE",
  "ERROR_DS_MASTERDSA_REQUIRED",
  "ERROR_DS_OBJECT_CLASS_REQUIRED",
  "ERROR_DS_MISSING_REQUIRED_ATT",
  "ERROR_DS_ATT_NOT_DEF_FOR_CLASS",
  "ERROR_DS_ATT_ALREADY_EXISTS",
  "ERROR_DS_CANT_ADD_ATT_VALUES",
  "ERROR_DS_SINGLE_VALUE_CONSTRAINT",
  "ERROR_DS_RANGE_CONSTRAINT",
  "ERROR_DS_ATT_VAL_ALREADY_EXISTS",
  "ERROR_DS_CANT_REM_MISSING_ATT",
  "ERROR_DS_CANT_REM_MISSING_ATT_VAL",
  "ERROR_DS_ROOT_CANT_BE_SUBREF",
  "ERROR_DS_NO_CHAINING",
  "ERROR_DS_NO_CHAINED_EVAL",
  "ERROR_DS_NO_PARENT_OBJECT",
  "ERROR_DS_PARENT_IS_AN_ALIAS",
  "ERROR_DS_CANT_MIX_MASTER_AND_REPS",
  "ERROR_DS_CHILDREN_EXIST",
  "ERROR_DS_OBJ_NOT_FOUND",
  "ERROR_DS_ALIASED_OBJ_MISSING",
  "ERROR_DS_BAD_NAME_SYNTAX",
  "ERROR_DS_ALIAS_POINTS_TO_ALIAS",
  "ERROR_DS_CANT_DEREF_ALIAS",
  "ERROR_DS_OUT_OF_SCOPE",
  "ERROR_DS_OBJECT_BEING_REMOVED",
  "ERROR_DS_CANT_DELETE_DSA_OBJ",
  "ERROR_DS_GENERIC_ERROR",
  "ERROR_DS_DSA_MUST_BE_INT_MASTER",
  "ERROR_DS_CLASS_NOT_DSA",
  "ERROR_DS_INSUFF_ACCESS_RIGHTS",
  "ERROR_DS_ILLEGAL_SUPERIOR",
  "ERROR_DS_ATTRIBUTE_OWNED_BY_SAM",
  "ERROR_DS_NAME_TOO_MANY_PARTS",
  "ERROR_DS_NAME_TOO_LONG",
  "ERROR_DS_NAME_VALUE_TOO_LONG",
  "ERROR_DS_NAME_UNPARSEABLE",
  "ERROR_DS_NAME_TYPE_UNKNOWN",
  "ERROR_DS_NOT_AN_OBJECT",
  "ERROR_DS_SEC_DESC_TOO_SHORT",
  "ERROR_DS_SEC_DESC_INVALID",
  "ERROR_DS_NO_DELETED_NAME",
  "ERROR_DS_SUBREF_MUST_HAVE_PARENT",
  "ERROR_DS_NCNAME_MUST_BE_NC",
  "ERROR_DS_CANT_ADD_SYSTEM_ONLY",
  "ERROR_DS_CLASS_MUST_BE_CONCRETE",
  "ERROR_DS_INVALID_DMD",
  "ERROR_DS_OBJ_GUID_EXISTS",
  "ERROR_DS_NOT_ON_BACKLINK",
  "ERROR_DS_NO_CROSSREF_FOR_NC",
  "ERROR_DS_SHUTTING_DOWN",
  "ERROR_DS_UNKNOWN_OPERATION",
  "ERROR_DS_INVALID_ROLE_OWNER",
  "ERROR_DS_COULDNT_CONTACT_FSMO",
  "ERROR_DS_CROSS_NC_DN_RENAME",
  "ERROR_DS_CANT_MOD_SYSTEM_ONLY",
  "ERROR_DS_REPLICATOR_ONLY",
  "ERROR_DS_OBJ_CLASS_NOT_DEFINED",
  "ERROR_DS_OBJ_CLASS_NOT_SUBCLASS",
  "ERROR_DS_NAME_REFERENCE_INVALID",
  "ERROR_DS_CROSS_REF_EXISTS",
  "ERROR_DS_CANT_DEL_MASTER_CROSSREF",
  "ERROR_DS_SUBTREE_NOTIFY_NOT_NC_HEAD",
  "ERROR_DS_NOTIFY_FILTER_TOO_COMPLEX",
  "ERROR_DS_DUP_RDN",
  "ERROR_DS_DUP_OID",
  "ERROR_DS_DUP_MAPI_ID",
  "ERROR_DS_DUP_SCHEMA_ID_GUID",
  "ERROR_DS_DUP_LDAP_DISPLAY_NAME",
  "ERROR_DS_SEMANTIC_ATT_TEST",
  "ERROR_DS_SYNTAX_MISMATCH",
  "ERROR_DS_EXISTS_IN_MUST_HAVE",
  "ERROR_DS_EXISTS_IN_MAY_HAVE",
  "ERROR_DS_NONEXISTENT_MAY_HAVE",
  "ERROR_DS_NONEXISTENT_MUST_HAVE",
  "ERROR_DS_AUX_CLS_TEST_FAIL",
  "ERROR_DS_NONEXISTENT_POSS_SUP",
  "ERROR_DS_SUB_CLS_TEST_FAIL",
  "ERROR_DS_BAD_RDN_ATT_ID_SYNTAX",
  "ERROR_DS_EXISTS_IN_AUX_CLS",
  "ERROR_DS_EXISTS_IN_SUB_CLS",
  "ERROR_DS_EXISTS_IN_POSS_SUP",
  "ERROR_DS_RECALCSCHEMA_FAILED",
  "ERROR_DS_TREE_DELETE_NOT_FINISHED",
  "ERROR_DS_CANT_DELETE",
  "ERROR_DS_ATT_SCHEMA_REQ_ID",
  "ERROR_DS_BAD_ATT_SCHEMA_SYNTAX",
  "ERROR_DS_CANT_CACHE_ATT",
  "ERROR_DS_CANT_CACHE_CLASS",
  "ERROR_DS_CANT_REMOVE_ATT_CACHE",
  "ERROR_DS_CANT_REMOVE_CLASS_CACHE",
  "ERROR_DS_CANT_RETRIEVE_DN",
  "ERROR_DS_MISSING_SUPREF",
  "ERROR_DS_CANT_RETRIEVE_INSTANCE",
  "ERROR_DS_CODE_INCONSISTENCY",
  "ERROR_DS_DATABASE_ERROR",
  "ERROR_DS_GOVERNSID_MISSING",
  "ERROR_DS_MISSING_EXPECTED_ATT",
  "ERROR_DS_NCNAME_MISSING_CR_REF",
  "ERROR_DS_SECURITY_CHECKING_ERROR",
  "ERROR_DS_SCHEMA_NOT_LOADED",
  "ERROR_DS_SCHEMA_ALLOC_FAILED",
  "ERROR_DS_ATT_SCHEMA_REQ_SYNTAX",
  "ERROR_DS_GCVERIFY_ERROR",
  "ERROR_DS_DRA_SCHEMA_MISMATCH",
  "ERROR_DS_CANT_FIND_DSA_OBJ",
  "ERROR_DS_CANT_FIND_EXPECTED_NC",
  "ERROR_DS_CANT_FIND_NC_IN_CACHE",
  "ERROR_DS_CANT_RETRIEVE_CHILD",
  "ERROR_DS_SECURITY_ILLEGAL_MODIFY",
  "ERROR_DS_CANT_REPLACE_HIDDEN_REC",
  "ERROR_DS_BAD_HIERARCHY_FILE",
  "ERROR_DS_BUILD_HIERARCHY_TABLE_FAILED",
  "ERROR_DS_CONFIG_PARAM_MISSING",
  "ERROR_DS_COUNTING_AB_INDICES_FAILED",
  "ERROR_DS_HIERARCHY_TABLE_MALLOC_FAILED",
  "ERROR_DS_INTERNAL_FAILURE",
  "ERROR_DS_UNKNOWN_ERROR",
  "ERROR_DS_ROOT_REQUIRES_CLASS_TOP",
  "ERROR_DS_REFUSING_FSMO_ROLES",
  "ERROR_DS_MISSING_FSMO_SETTINGS",
  "ERROR_DS_UNABLE_TO_SURRENDER_ROLES",
  "ERROR_DS_DRA_GENERIC",
  "ERROR_DS_DRA_INVALID_PARAMETER",
  "ERROR_DS_DRA_BUSY",
  "ERROR_DS_DRA_BAD_DN",
  "ERROR_DS_DRA_BAD_NC",
  "ERROR_DS_DRA_DN_EXISTS",
  "ERROR_DS_DRA_INTERNAL_ERROR",
  "ERROR_DS_DRA_INCONSISTENT_DIT",
  "ERROR_DS_DRA_CONNECTION_FAILED",
  "ERROR_DS_DRA_BAD_INSTANCE_TYPE",
  "ERROR_DS_DRA_OUT_OF_MEM",
  "ERROR_DS_DRA_MAIL_PROBLEM",
  "ERROR_DS_DRA_REF_ALREADY_EXISTS",
  "ERROR_DS_DRA_REF_NOT_FOUND",
  "ERROR_DS_DRA_OBJ_IS_REP_SOURCE",
  "ERROR_DS_DRA_DB_ERROR",
  "ERROR_DS_DRA_NO_REPLICA",
  "ERROR_DS_DRA_ACCESS_DENIED",
  "ERROR_DS_DRA_NOT_SUPPORTED",
  "ERROR_DS_DRA_RPC_CANCELLED",
  "ERROR_DS_DRA_SOURCE_DISABLED",
  "ERROR_DS_DRA_SINK_DISABLED",
  "ERROR_DS_DRA_NAME_COLLISION",
  "ERROR_DS_DRA_SOURCE_REINSTALLED",
  "ERROR_DS_DRA_MISSING_PARENT",
  "ERROR_DS_DRA_PREEMPTED",
  "ERROR_DS_DRA_ABANDON_SYNC",
  "ERROR_DS_DRA_SHUTDOWN",
  "ERROR_DS_DRA_INCOMPATIBLE_PARTIAL_SET",
  "ERROR_DS_DRA_SOURCE_IS_PARTIAL_REPLICA",
  "ERROR_DS_DRA_EXTN_CONNECTION_FAILED",
  "ERROR_DS_INSTALL_SCHEMA_MISMATCH",
  "ERROR_DS_DUP_LINK_ID",
  "ERROR_DS_NAME_ERROR_RESOLVING",
  "ERROR_DS_NAME_ERROR_NOT_FOUND",
  "ERROR_DS_NAME_ERROR_NOT_UNIQUE",
  "ERROR_DS_NAME_ERROR_NO_MAPPING",
  "ERROR_DS_NAME_ERROR_DOMAIN_ONLY",
  "ERROR_DS_NAME_ERROR_NO_SYNTACTICAL_MAPPING",
  "ERROR_DS_CONSTRUCTED_ATT_MOD",
  "ERROR_DS_WRONG_OM_OBJ_CLASS",
  "ERROR_DS_DRA_REPL_PENDING",
  "ERROR_DS_DS_REQUIRED",
  "ERROR_DS_INVALID_LDAP_DISPLAY_NAME",
  "ERROR_DS_NON_BASE_SEARCH",
  "ERROR_DS_CANT_RETRIEVE_ATTS",
  "ERROR_DS_BACKLINK_WITHOUT_LINK",
  "ERROR_DS_EPOCH_MISMATCH",
  "ERROR_DS_SRC_NAME_MISMATCH",
  "ERROR_DS_SRC_AND_DST_NC_IDENTICAL",
  "ERROR_DS_DST_NC_MISMATCH",
  "ERROR_DS_NOT_AUTHORITIVE_FOR_DST_NC",
  "ERROR_DS_SRC_GUID_MISMATCH",
  "ERROR_DS_CANT_MOVE_DELETED_OBJECT",
  "ERROR_DS_PDC_OPERATION_IN_PROGRESS",
  "ERROR_DS_CROSS_DOMAIN_CLEANUP_REQD",
  "ERROR_DS_ILLEGAL_XDOM_MOVE_OPERATION",
  "ERROR_DS_CANT_WITH_ACCT_GROUP_MEMBERSHPS",
  "ERROR_DS_NC_MUST_HAVE_NC_PARENT",
  "ERROR_DS_CR_IMPOSSIBLE_TO_VALIDATE",
  "ERROR_DS_DST_DOMAIN_NOT_NATIVE",
  "ERROR_DS_MISSING_INFRASTRUCTURE_CONTAINER",
  "ERROR_DS_CANT_MOVE_ACCOUNT_GROUP",
  "ERROR_DS_CANT_MOVE_RESOURCE_GROUP",
  "ERROR_DS_INVALID_SEARCH_FLAG",
  "ERROR_DS_NO_TREE_DELETE_ABOVE_NC",
  "ERROR_DS_COULDNT_LOCK_TREE_FOR_DELETE",
  "ERROR_DS_COULDNT_IDENTIFY_OBJECTS_FOR_TREE_DELETE",
  "ERROR_DS_SAM_INIT_FAILURE",
  "ERROR_DS_SENSITIVE_GROUP_VIOLATION",
  "ERROR_DS_CANT_MOD_PRIMARYGROUPID",
  "ERROR_DS_ILLEGAL_BASE_SCHEMA_MOD",
  "ERROR_DS_NONSAFE_SCHEMA_CHANGE",
  "ERROR_DS_SCHEMA_UPDATE_DISALLOWED",
  "ERROR_DS_CANT_CREATE_UNDER_SCHEMA",
  "ERROR_DS_INSTALL_NO_SRC_SCH_VERSION",
  "ERROR_DS_INSTALL_NO_SCH_VERSION_IN_INIFILE",
  "ERROR_DS_INVALID_GROUP_TYPE",
  "ERROR_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN",
  "ERROR_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN",
  "ERROR_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER",
  "ERROR_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER",
  "ERROR_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER",
  "ERROR_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER",
  "ERROR_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER",
  "ERROR_DS_HAVE_PRIMARY_MEMBERS",
  "ERROR_DS_STRING_SD_CONVERSION_FAILED",
  "ERROR_DS_NAMING_MASTER_GC",
  "ERROR_DS_DNS_LOOKUP_FAILURE",
  "ERROR_DS_COULDNT_UPDATE_SPNS",
  "ERROR_DS_CANT_RETRIEVE_SD",
  "ERROR_DS_KEY_NOT_UNIQUE",
  "ERROR_DS_WRONG_LINKED_ATT_SYNTAX",
  "ERROR_DS_SAM_NEED_BOOTKEY_PASSWORD",
  "ERROR_DS_SAM_NEED_BOOTKEY_FLOPPY",
  "ERROR_DS_CANT_START",
  "ERROR_DS_INIT_FAILURE",
  "ERROR_DS_NO_PKT_PRIVACY_ON_CONNECTION",
  "ERROR_DS_SOURCE_DOMAIN_IN_FOREST",
  "ERROR_DS_DESTINATION_DOMAIN_NOT_IN_FOREST",
  "ERROR_DS_DESTINATION_AUDITING_NOT_ENABLED",
  "ERROR_DS_CANT_FIND_DC_FOR_SRC_DOMAIN",
  "ERROR_DS_SRC_OBJ_NOT_GROUP_OR_USER",
  "ERROR_DS_SRC_SID_EXISTS_IN_FOREST",
  "ERROR_DS_SRC_AND_DST_OBJECT_CLASS_MISMATCH",
  "ERROR_SAM_INIT_FAILURE",
  "ERROR_DS_DRA_SCHEMA_INFO_SHIP",
  "ERROR_DS_DRA_SCHEMA_CONFLICT",
  "ERROR_DS_DRA_EARLIER_SCHEMA_CONFLICT",
  "ERROR_DS_DRA_OBJ_NC_MISMATCH",
  "ERROR_DS_NC_STILL_HAS_DSAS",
  "ERROR_DS_GC_REQUIRED",
  "ERROR_DS_LOCAL_MEMBER_OF_LOCAL_ONLY",
  "ERROR_DS_NO_FPO_IN_UNIVERSAL_GROUPS",
  "ERROR_DS_CANT_ADD_TO_GC",
  "ERROR_DS_NO_CHECKPOINT_WITH_PDC",
  "ERROR_DS_SOURCE_AUDITING_NOT_ENABLED",
  "ERROR_DS_CANT_CREATE_IN_NONDOMAIN_NC",
  "ERROR_DS_INVALID_NAME_FOR_SPN",
  "ERROR_DS_FILTER_USES_CONTRUCTED_ATTRS",
  "ERROR_DS_UNICODEPWD_NOT_IN_QUOTES",
  "ERROR_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED",
  "ERROR_DS_MUST_BE_RUN_ON_DST_DC",
  "ERROR_DS_SRC_DC_MUST_BE_SP4_OR_GREATER",
  "ERROR_DS_CANT_TREE_DELETE_CRITICAL_OBJ",
  "ERROR_DS_INIT_FAILURE_CONSOLE",
  "ERROR_DS_SAM_INIT_FAILURE_CONSOLE",
  "ERROR_DS_FOREST_VERSION_TOO_HIGH",
  "ERROR_DS_DOMAIN_VERSION_TOO_HIGH",
  "ERROR_DS_FOREST_VERSION_TOO_LOW",
  "ERROR_DS_DOMAIN_VERSION_TOO_LOW",
  "ERROR_DS_INCOMPATIBLE_VERSION",
  "ERROR_DS_LOW_DSA_VERSION",
  "ERROR_DS_NO_BEHAVIOR_VERSION_IN_MIXEDDOMAIN",
  "ERROR_DS_NOT_SUPPORTED_SORT_ORDER",
  "ERROR_DS_NAME_NOT_UNIQUE",
  "ERROR_DS_MACHINE_ACCOUNT_CREATED_PRENT4",
  "ERROR_DS_OUT_OF_VERSION_STORE",
  "ERROR_DS_INCOMPATIBLE_CONTROLS_USED",
  "ERROR_DS_NO_REF_DOMAIN",
  "ERROR_DS_RESERVED_LINK_ID",
  "ERROR_DS_LINK_ID_NOT_AVAILABLE",
  "ERROR_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER",
  "ERROR_DS_MODIFYDN_DISALLOWED_BY_INSTANCE_TYPE",
  "ERROR_DS_NO_OBJECT_MOVE_IN_SCHEMA_NC",
  "ERROR_DS_MODIFYDN_DISALLOWED_BY_FLAG",
  "ERROR_DS_MODIFYDN_WRONG_GRANDPARENT",
  "ERROR_DS_NAME_ERROR_TRUST_REFERRAL",
  "ERROR_NOT_SUPPORTED_ON_STANDARD_SERVER",
  "ERROR_DS_CANT_ACCESS_REMOTE_PART_OF_AD",
  "ERROR_DS_CR_IMPOSSIBLE_TO_VALIDATE_V2",
  "ERROR_DS_THREAD_LIMIT_EXCEEDED",
  "ERROR_DS_NOT_CLOSEST",
  "ERROR_DS_CANT_DERIVE_SPN_WITHOUT_SERVER_REF",
  "ERROR_DS_SINGLE_USER_MODE_FAILED",
  "ERROR_DS_NTDSCRIPT_SYNTAX_ERROR",
  "ERROR_DS_NTDSCRIPT_PROCESS_ERROR",
  "ERROR_DS_DIFFERENT_REPL_EPOCHS",
  "ERROR_DS_DRS_EXTENSIONS_CHANGED",
  "ERROR_DS_REPLICA_SET_CHANGE_NOT_ALLOWED_ON_DISABLED_CR",
  "ERROR_DS_NO_MSDS_INTID",
  "ERROR_DS_DUP_MSDS_INTID",
  "ERROR_DS_EXISTS_IN_RDNATTID",
  "ERROR_DS_AUTHORIZATION_FAILED",
  "ERROR_DS_INVALID_SCRIPT",
  "ERROR_DS_REMOTE_CROSSREF_OP_FAILED",
  "ERROR_DS_CROSS_REF_BUSY",
  "ERROR_DS_CANT_DERIVE_SPN_FOR_DELETED_DOMAIN",
  "ERROR_DS_CANT_DEMOTE_WITH_WRITEABLE_NC",
  "ERROR_DS_DUPLICATE_ID_FOUND",
  "ERROR_DS_INSUFFICIENT_ATTR_TO_CREATE_OBJECT",
  "ERROR_DS_GROUP_CONVERSION_ERROR",
  "ERROR_DS_CANT_MOVE_APP_BASIC_GROUP",
  "ERROR_DS_CANT_MOVE_APP_QUERY_GROUP",
  "ERROR_DS_ROLE_NOT_VERIFIED",
  "ERROR_DS_WKO_CONTAINER_CANNOT_BE_SPECIAL",
  "ERROR_DS_DOMAIN_RENAME_IN_PROGRESS",
  "ERROR_DS_EXISTING_AD_CHILD_NC",
  "ERROR_DS_REPL_LIFETIME_EXCEEDED",
  "ERROR_DS_DISALLOWED_IN_SYSTEM_CONTAINER",
  "ERROR_DS_LDAP_SEND_QUEUE_FULL",
  "ERROR_DS_DRA_OUT_SCHEDULE_WINDOW",
  "ERROR_DS_POLICY_NOT_KNOWN",
  "ERROR_NO_SITE_SETTINGS_OBJECT",
  "ERROR_NO_SECRETS",
  "ERROR_NO_WRITABLE_DC_FOUND",
  "ERROR_DS_NO_SERVER_OBJECT",
  "ERROR_DS_NO_NTDSA_OBJECT",
  "ERROR_DS_NON_ASQ_SEARCH",
  "ERROR_DS_AUDIT_FAILURE",
  "ERROR_DS_INVALID_SEARCH_FLAG_SUBTREE",
  "ERROR_DS_INVALID_SEARCH_FLAG_TUPLE",
  "ERROR_DS_HIERARCHY_TABLE_TOO_DEEP",
  "ERROR_DS_DRA_CORRUPT_UTD_VECTOR",
  "ERROR_DS_DRA_SECRETS_DENIED",
  "ERROR_DS_RESERVED_MAPI_ID",
  "ERROR_DS_MAPI_ID_NOT_AVAILABLE",
  "ERROR_DS_DRA_MISSING_KRBTGT_SECRET",
  "ERROR_DS_DOMAIN_NAME_EXISTS_IN_FOREST",
  "ERROR_DS_FLAT_NAME_EXISTS_IN_FOREST",
  "ERROR_INVALID_USER_PRINCIPAL_NAME",
  "ERROR_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS",
  "ERROR_DS_OID_NOT_FOUND",
  "ERROR_DS_DRA_RECYCLED_TARGET",
  "ERROR_DS_DISALLOWED_NC_REDIRECT",
  "ERROR_DS_HIGH_ADLDS_FFL",
  "ERROR_DS_HIGH_DSA_VERSION",
  "ERROR_DS_LOW_ADLDS_FFL",
  "ERROR_DOMAIN_SID_SAME_AS_LOCAL_WORKSTATION",
  "ERROR_DS_UNDELETE_SAM_VALIDATION_FAILED",
  "ERROR_INCORRECT_ACCOUNT_TYPE",
  "ERROR_DS_SPN_VALUE_NOT_UNIQUE_IN_FOREST",
  "ERROR_DS_UPN_VALUE_NOT_UNIQUE_IN_FOREST",
  "ERROR_DS_MISSING_FOREST_TRUST",
  "ERROR_DS_VALUE_KEY_NOT_UNIQUE",
  "ERROR_IPSEC_QM_POLICY_EXISTS",
  "ERROR_IPSEC_QM_POLICY_NOT_FOUND",
  "ERROR_IPSEC_QM_POLICY_IN_USE",
  "ERROR_IPSEC_MM_POLICY_EXISTS",
  "ERROR_IPSEC_MM_POLICY_NOT_FOUND",
  "ERROR_IPSEC_MM_POLICY_IN_USE",
  "ERROR_IPSEC_MM_FILTER_EXISTS",
  "ERROR_IPSEC_MM_FILTER_NOT_FOUND",
  "ERROR_IPSEC_TRANSPORT_FILTER_EXISTS",
  "ERROR_IPSEC_TRANSPORT_FILTER_NOT_FOUND",
  "ERROR_IPSEC_MM_AUTH_EXISTS",
  "ERROR_IPSEC_MM_AUTH_NOT_FOUND",
  "ERROR_IPSEC_MM_AUTH_IN_USE",
  "ERROR_IPSEC_DEFAULT_MM_POLICY_NOT_FOUND",
  "ERROR_IPSEC_DEFAULT_MM_AUTH_NOT_FOUND",
  "ERROR_IPSEC_DEFAULT_QM_POLICY_NOT_FOUND",
  "ERROR_IPSEC_TUNNEL_FILTER_EXISTS",
  "ERROR_IPSEC_TUNNEL_FILTER_NOT_FOUND",
  "ERROR_IPSEC_MM_FILTER_PENDING_DELETION",
  "ERROR_IPSEC_TRANSPORT_FILTER_PENDING_DELETION",
  "ERROR_IPSEC_TUNNEL_FILTER_PENDING_DELETION",
  "ERROR_IPSEC_MM_POLICY_PENDING_DELETION",
  "ERROR_IPSEC_MM_AUTH_PENDING_DELETION",
  "ERROR_IPSEC_QM_POLICY_PENDING_DELETION",
  "ERROR_IPSEC_IKE_NEG_STATUS_BEGIN",
  "ERROR_IPSEC_IKE_AUTH_FAIL",
  "ERROR_IPSEC_IKE_ATTRIB_FAIL",
  "ERROR_IPSEC_IKE_NEGOTIATION_PENDING",
  "ERROR_IPSEC_IKE_GENERAL_PROCESSING_ERROR",
  "ERROR_IPSEC_IKE_TIMED_OUT",
  "ERROR_IPSEC_IKE_NO_CERT",
  "ERROR_IPSEC_IKE_SA_DELETED",
  "ERROR_IPSEC_IKE_SA_REAPED",
  "ERROR_IPSEC_IKE_MM_ACQUIRE_DROP",
  "ERROR_IPSEC_IKE_QM_ACQUIRE_DROP",
  "ERROR_IPSEC_IKE_QUEUE_DROP_MM",
  "ERROR_IPSEC_IKE_QUEUE_DROP_NO_MM",
  "ERROR_IPSEC_IKE_DROP_NO_RESPONSE",
  "ERROR_IPSEC_IKE_MM_DELAY_DROP",
  "ERROR_IPSEC_IKE_QM_DELAY_DROP",
  "ERROR_IPSEC_IKE_ERROR",
  "ERROR_IPSEC_IKE_CRL_FAILED",
  "ERROR_IPSEC_IKE_INVALID_KEY_USAGE",
  "ERROR_IPSEC_IKE_INVALID_CERT_TYPE",
  "ERROR_IPSEC_IKE_NO_PRIVATE_KEY",
  "ERROR_IPSEC_IKE_SIMULTANEOUS_REKEY",
  "ERROR_IPSEC_IKE_DH_FAIL",
  "ERROR_IPSEC_IKE_CRITICAL_PAYLOAD_NOT_RECOGNIZED",
  "ERROR_IPSEC_IKE_INVALID_HEADER",
  "ERROR_IPSEC_IKE_NO_POLICY",
  "ERROR_IPSEC_IKE_INVALID_SIGNATURE",
  "ERROR_IPSEC_IKE_KERBEROS_ERROR",
  "ERROR_IPSEC_IKE_NO_PUBLIC_KEY",
  "ERROR_IPSEC_IKE_PROCESS_ERR",
  "ERROR_IPSEC_IKE_PROCESS_ERR_SA",
  "ERROR_IPSEC_IKE_PROCESS_ERR_PROP",
  "ERROR_IPSEC_IKE_PROCESS_ERR_TRANS",
  "ERROR_IPSEC_IKE_PROCESS_ERR_KE",
  "ERROR_IPSEC_IKE_PROCESS_ERR_ID",
  "ERROR_IPSEC_IKE_PROCESS_ERR_CERT",
  "ERROR_IPSEC_IKE_PROCESS_ERR_CERT_REQ",
  "ERROR_IPSEC_IKE_PROCESS_ERR_HASH",
  "ERROR_IPSEC_IKE_PROCESS_ERR_SIG",
  "ERROR_IPSEC_IKE_PROCESS_ERR_NONCE",
  "ERROR_IPSEC_IKE_PROCESS_ERR_NOTIFY",
  "ERROR_IPSEC_IKE_PROCESS_ERR_DELETE",
  "ERROR_IPSEC_IKE_PROCESS_ERR_VENDOR",
  "ERROR_IPSEC_IKE_INVALID_PAYLOAD",
  "ERROR_IPSEC_IKE_LOAD_SOFT_SA",
  "ERROR_IPSEC_IKE_SOFT_SA_TORN_DOWN",
  "ERROR_IPSEC_IKE_INVALID_COOKIE",
  "ERROR_IPSEC_IKE_NO_PEER_CERT",
  "ERROR_IPSEC_IKE_PEER_CRL_FAILED",
  "ERROR_IPSEC_IKE_POLICY_CHANGE",
  "ERROR_IPSEC_IKE_NO_MM_POLICY",
  "ERROR_IPSEC_IKE_NOTCBPRIV",
  "ERROR_IPSEC_IKE_SECLOADFAIL",
  "ERROR_IPSEC_IKE_FAILSSPINIT",
  "ERROR_IPSEC_IKE_FAILQUERYSSP",
  "ERROR_IPSEC_IKE_SRVACQFAIL",
  "ERROR_IPSEC_IKE_SRVQUERYCRED",
  "ERROR_IPSEC_IKE_GETSPIFAIL",
  "ERROR_IPSEC_IKE_INVALID_FILTER",
  "ERROR_IPSEC_IKE_OUT_OF_MEMORY",
  "ERROR_IPSEC_IKE_ADD_UPDATE_KEY_FAILED",
  "ERROR_IPSEC_IKE_INVALID_POLICY",
  "ERROR_IPSEC_IKE_UNKNOWN_DOI",
  "ERROR_IPSEC_IKE_INVALID_SITUATION",
  "ERROR_IPSEC_IKE_DH_FAILURE",
  "ERROR_IPSEC_IKE_INVALID_GROUP",
  "ERROR_IPSEC_IKE_ENCRYPT",
  "ERROR_IPSEC_IKE_DECRYPT",
  "ERROR_IPSEC_IKE_POLICY_MATCH",
  "ERROR_IPSEC_IKE_UNSUPPORTED_ID",
  "ERROR_IPSEC_IKE_INVALID_HASH",
  "ERROR_IPSEC_IKE_INVALID_HASH_ALG",
  "ERROR_IPSEC_IKE_INVALID_HASH_SIZE",
  "ERROR_IPSEC_IKE_INVALID_ENCRYPT_ALG",
  "ERROR_IPSEC_IKE_INVALID_AUTH_ALG",
  "ERROR_IPSEC_IKE_INVALID_SIG",
  "ERROR_IPSEC_IKE_LOAD_FAILED",
  "ERROR_IPSEC_IKE_RPC_DELETE",
  "ERROR_IPSEC_IKE_BENIGN_REINIT",
  "ERROR_IPSEC_IKE_INVALID_RESPONDER_LIFETIME_NOTIFY",
  "ERROR_IPSEC_IKE_INVALID_MAJOR_VERSION",
  "ERROR_IPSEC_IKE_INVALID_CERT_KEYLEN",
  "ERROR_IPSEC_IKE_MM_LIMIT",
  "ERROR_IPSEC_IKE_NEGOTIATION_DISABLED",
  "ERROR_IPSEC_IKE_QM_LIMIT",
  "ERROR_IPSEC_IKE_MM_EXPIRED",
  "ERROR_IPSEC_IKE_PEER_MM_ASSUMED_INVALID",
  "ERROR_IPSEC_IKE_CERT_CHAIN_POLICY_MISMATCH",
  "ERROR_IPSEC_IKE_UNEXPECTED_MESSAGE_ID",
  "ERROR_IPSEC_IKE_INVALID_AUTH_PAYLOAD",
  "ERROR_IPSEC_IKE_DOS_COOKIE_SENT",
  "ERROR_IPSEC_IKE_SHUTTING_DOWN",
  "ERROR_IPSEC_IKE_CGA_AUTH_FAILED",
  "ERROR_IPSEC_IKE_PROCESS_ERR_NATOA",
  "ERROR_IPSEC_IKE_INVALID_MM_FOR_QM",
  "ERROR_IPSEC_IKE_QM_EXPIRED",
  "ERROR_IPSEC_IKE_TOO_MANY_FILTERS",
  "ERROR_IPSEC_IKE_NEG_STATUS_END",
  "ERROR_IPSEC_IKE_KILL_DUMMY_NAP_TUNNEL",
  "ERROR_IPSEC_IKE_INNER_IP_ASSIGNMENT_FAILURE",
  "ERROR_IPSEC_IKE_REQUIRE_CP_PAYLOAD_MISSING",
  "ERROR_IPSEC_KEY_MODULE_IMPERSONATION_NEGOTIATION_PENDING",
  "ERROR_IPSEC_IKE_COEXISTENCE_SUPPRESS",
  "ERROR_IPSEC_IKE_RATELIMIT_DROP",
  "ERROR_IPSEC_IKE_PEER_DOESNT_SUPPORT_MOBIKE",
  "ERROR_IPSEC_IKE_AUTHORIZATION_FAILURE",
  "ERROR_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_FAILURE",
  "ERROR_IPSEC_IKE_AUTHORIZATION_FAILURE_WITH_OPTIONAL_RETRY",
  "ERROR_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_AND_CERTMAP_FAILURE",
  "ERROR_IPSEC_IKE_NEG_STATUS_EXTENDED_END",
  "ERROR_IPSEC_BAD_SPI",
  "ERROR_IPSEC_SA_LIFETIME_EXPIRED",
  "ERROR_IPSEC_WRONG_SA",
  "ERROR_IPSEC_REPLAY_CHECK_FAILED",
  "ERROR_IPSEC_INVALID_PACKET",
  "ERROR_IPSEC_INTEGRITY_CHECK_FAILED",
  "ERROR_IPSEC_CLEAR_TEXT_DROP",
  "ERROR_IPSEC_AUTH_FIREWALL_DROP",
  "ERROR_IPSEC_THROTTLE_DROP",
  "ERROR_IPSEC_DOSP_BLOCK",
  "ERROR_IPSEC_DOSP_RECEIVED_MULTICAST",
  "ERROR_IPSEC_DOSP_INVALID_PACKET",
  "ERROR_IPSEC_DOSP_STATE_LOOKUP_FAILED",
  "ERROR_IPSEC_DOSP_MAX_ENTRIES",
  "ERROR_IPSEC_DOSP_KEYMOD_NOT_ALLOWED",
  "ERROR_IPSEC_DOSP_NOT_INSTALLED",
  "ERROR_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES",
  "ERROR_SXS_SECTION_NOT_FOUND",
  "ERROR_SXS_CANT_GEN_ACTCTX",
  "ERROR_SXS_INVALID_ACTCTXDATA_FORMAT",
  "ERROR_SXS_ASSEMBLY_NOT_FOUND",
  "ERROR_SXS_MANIFEST_FORMAT_ERROR",
  "ERROR_SXS_MANIFEST_PARSE_ERROR",
  "ERROR_SXS_ACTIVATION_CONTEXT_DISABLED",
  "ERROR_SXS_KEY_NOT_FOUND",
  "ERROR_SXS_VERSION_CONFLICT",
  "ERROR_SXS_WRONG_SECTION_TYPE",
  "ERROR_SXS_THREAD_QUERIES_DISABLED",
  "ERROR_SXS_PROCESS_DEFAULT_ALREADY_SET",
  "ERROR_SXS_UNKNOWN_ENCODING_GROUP",
  "ERROR_SXS_UNKNOWN_ENCODING",
  "ERROR_SXS_INVALID_XML_NAMESPACE_URI",
  "ERROR_SXS_ROOT_MANIFEST_DEPENDENCY_NOT_INSTALLED",
  "ERROR_SXS_LEAF_MANIFEST_DEPENDENCY_NOT_INSTALLED",
  "ERROR_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE",
  "ERROR_SXS_MANIFEST_MISSING_REQUIRED_DEFAULT_NAMESPACE",
  "ERROR_SXS_MANIFEST_INVALID_REQUIRED_DEFAULT_NAMESPACE",
  "ERROR_SXS_PRIVATE_MANIFEST_CROSS_PATH_WITH_REPARSE_POINT",
  "ERROR_SXS_DUPLICATE_DLL_NAME",
  "ERROR_SXS_DUPLICATE_WINDOWCLASS_NAME",
  "ERROR_SXS_DUPLICATE_CLSID",
  "ERROR_SXS_DUPLICATE_IID",
  "ERROR_SXS_DUPLICATE_TLBID",
  "ERROR_SXS_DUPLICATE_PROGID",
  "ERROR_SXS_DUPLICATE_ASSEMBLY_NAME",
  "ERROR_SXS_FILE_HASH_MISMATCH",
  "ERROR_SXS_POLICY_PARSE_ERROR",
  "ERROR_SXS_XML_E_MISSINGQUOTE",
  "ERROR_SXS_XML_E_COMMENTSYNTAX",
  "ERROR_SXS_XML_E_BADSTARTNAMECHAR",
  "ERROR_SXS_XML_E_BADNAMECHAR",
  "ERROR_SXS_XML_E_BADCHARINSTRING",
  "ERROR_SXS_XML_E_XMLDECLSYNTAX",
  "ERROR_SXS_XML_E_BADCHARDATA",
  "ERROR_SXS_XML_E_MISSINGWHITESPACE",
  "ERROR_SXS_XML_E_EXPECTINGTAGEND",
  "ERROR_SXS_XML_E_MISSINGSEMICOLON",
  "ERROR_SXS_XML_E_UNBALANCEDPAREN",
  "ERROR_SXS_XML_E_INTERNALERROR",
  "ERROR_SXS_XML_E_UNEXPECTED_WHITESPACE",
  "ERROR_SXS_XML_E_INCOMPLETE_ENCODING",
  "ERROR_SXS_XML_E_MISSING_PAREN",
  "ERROR_SXS_XML_E_EXPECTINGCLOSEQUOTE",
  "ERROR_SXS_XML_E_MULTIPLE_COLONS",
  "ERROR_SXS_XML_E_INVALID_DECIMAL",
  "ERROR_SXS_XML_E_INVALID_HEXIDECIMAL",
  "ERROR_SXS_XML_E_INVALID_UNICODE",
  "ERROR_SXS_XML_E_WHITESPACEORQUESTIONMARK",
  "ERROR_SXS_XML_E_UNEXPECTEDENDTAG",
  "ERROR_SXS_XML_E_UNCLOSEDTAG",
  "ERROR_SXS_XML_E_DUPLICATEATTRIBUTE",
  "ERROR_SXS_XML_E_MULTIPLEROOTS",
  "ERROR_SXS_XML_E_INVALIDATROOTLEVEL",
  "ERROR_SXS_XML_E_BADXMLDECL",
  "ERROR_SXS_XML_E_MISSINGROOT",
  "ERROR_SXS_XML_E_UNEXPECTEDEOF",
  "ERROR_SXS_XML_E_BADPEREFINSUBSET",
  "ERROR_SXS_XML_E_UNCLOSEDSTARTTAG",
  "ERROR_SXS_XML_E_UNCLOSEDENDTAG",
  "ERROR_SXS_XML_E_UNCLOSEDSTRING",
  "ERROR_SXS_XML_E_UNCLOSEDCOMMENT",
  "ERROR_SXS_XML_E_UNCLOSEDDECL",
  "ERROR_SXS_XML_E_UNCLOSEDCDATA",
  "ERROR_SXS_XML_E_RESERVEDNAMESPACE",
  "ERROR_SXS_XML_E_INVALIDENCODING",
  "ERROR_SXS_XML_E_INVALIDSWITCH",
  "ERROR_SXS_XML_E_BADXMLCASE",
  "ERROR_SXS_XML_E_INVALID_STANDALONE",
  "ERROR_SXS_XML_E_UNEXPECTED_STANDALONE",
  "ERROR_SXS_XML_E_INVALID_VERSION",
  "ERROR_SXS_XML_E_MISSINGEQUALS",
  "ERROR_SXS_PROTECTION_RECOVERY_FAILED",
  "ERROR_SXS_PROTECTION_PUBLIC_KEY_TOO_SHORT",
  "ERROR_SXS_PROTECTION_CATALOG_NOT_VALID",
  "ERROR_SXS_UNTRANSLATABLE_HRESULT",
  "ERROR_SXS_PROTECTION_CATALOG_FILE_MISSING",
  "ERROR_SXS_MISSING_ASSEMBLY_IDENTITY_ATTRIBUTE",
  "ERROR_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE_NAME",
  "ERROR_SXS_ASSEMBLY_MISSING",
  "ERROR_SXS_CORRUPT_ACTIVATION_STACK",
  "ERROR_SXS_CORRUPTION",
  "ERROR_SXS_EARLY_DEACTIVATION",
  "ERROR_SXS_INVALID_DEACTIVATION",
  "ERROR_SXS_MULTIPLE_DEACTIVATION",
  "ERROR_SXS_PROCESS_TERMINATION_REQUESTED",
  "ERROR_SXS_RELEASE_ACTIVATION_CONTEXT",
  "ERROR_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY",
  "ERROR_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE",
  "ERROR_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME",
  "ERROR_SXS_IDENTITY_DUPLICATE_ATTRIBUTE",
  "ERROR_SXS_IDENTITY_PARSE_ERROR",
  "ERROR_MALFORMED_SUBSTITUTION_STRING",
  "ERROR_SXS_INCORRECT_PUBLIC_KEY_TOKEN",
  "ERROR_UNMAPPED_SUBSTITUTION_STRING",
  "ERROR_SXS_ASSEMBLY_NOT_LOCKED",
  "ERROR_SXS_COMPONENT_STORE_CORRUPT",
  "ERROR_ADVANCED_INSTALLER_FAILED",
  "ERROR_XML_ENCODING_MISMATCH",
  "ERROR_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT",
  "ERROR_SXS_IDENTITIES_DIFFERENT",
  "ERROR_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT",
  "ERROR_SXS_FILE_NOT_PART_OF_ASSEMBLY",
  "ERROR_SXS_MANIFEST_TOO_BIG",
  "ERROR_SXS_SETTING_NOT_REGISTERED",
  "ERROR_SXS_TRANSACTION_CLOSURE_INCOMPLETE",
  "ERROR_SMI_PRIMITIVE_INSTALLER_FAILED",
  "ERROR_GENERIC_COMMAND_FAILED",
  "ERROR_SXS_FILE_HASH_MISSING",
  "ERROR_SXS_DUPLICATE_ACTIVATABLE_CLASS",
  "ERROR_EVT_INVALID_CHANNEL_PATH",
  "ERROR_EVT_INVALID_QUERY",
  "ERROR_EVT_PUBLISHER_METADATA_NOT_FOUND",
  "ERROR_EVT_EVENT_TEMPLATE_NOT_FOUND",
  "ERROR_EVT_INVALID_PUBLISHER_NAME",
  "ERROR_EVT_INVALID_EVENT_DATA",
  "ERROR_EVT_CHANNEL_NOT_FOUND",
  "ERROR_EVT_MALFORMED_XML_TEXT",
  "ERROR_EVT_SUBSCRIPTION_TO_DIRECT_CHANNEL",
  "ERROR_EVT_CONFIGURATION_ERROR",
  "ERROR_EVT_QUERY_RESULT_STALE",
  "ERROR_EVT_QUERY_RESULT_INVALID_POSITION",
  "ERROR_EVT_NON_VALIDATING_MSXML",
  "ERROR_EVT_FILTER_ALREADYSCOPED",
  "ERROR_EVT_FILTER_NOTELTSET",
  "ERROR_EVT_FILTER_INVARG",
  "ERROR_EVT_FILTER_INVTEST",
  "ERROR_EVT_FILTER_INVTYPE",
  "ERROR_EVT_FILTER_PARSEERR",
  "ERROR_EVT_FILTER_UNSUPPORTEDOP",
  "ERROR_EVT_FILTER_UNEXPECTEDTOKEN",
  "ERROR_EVT_INVALID_OPERATION_OVER_ENABLED_DIRECT_CHANNEL",
  "ERROR_EVT_INVALID_CHANNEL_PROPERTY_VALUE",
  "ERROR_EVT_INVALID_PUBLISHER_PROPERTY_VALUE",
  "ERROR_EVT_CHANNEL_CANNOT_ACTIVATE",
  "ERROR_EVT_FILTER_TOO_COMPLEX",
  "ERROR_EVT_MESSAGE_NOT_FOUND",
  "ERROR_EVT_MESSAGE_ID_NOT_FOUND",
  "ERROR_EVT_UNRESOLVED_VALUE_INSERT",
  "ERROR_EVT_UNRESOLVED_PARAMETER_INSERT",
  "ERROR_EVT_MAX_INSERTS_REACHED",
  "ERROR_EVT_EVENT_DEFINITION_NOT_FOUND",
  "ERROR_EVT_MESSAGE_LOCALE_NOT_FOUND",
  "ERROR_EVT_VERSION_TOO_OLD",
  "ERROR_EVT_VERSION_TOO_NEW",
  "ERROR_EVT_CANNOT_OPEN_CHANNEL_OF_QUERY",
  "ERROR_EVT_PUBLISHER_DISABLED",
  "ERROR_EVT_FILTER_OUT_OF_RANGE",
  "ERROR_EC_SUBSCRIPTION_CANNOT_ACTIVATE",
  "ERROR_EC_LOG_DISABLED",
  "ERROR_EC_CIRCULAR_FORWARDING",
  "ERROR_EC_CREDSTORE_FULL",
  "ERROR_EC_CRED_NOT_FOUND",
  "ERROR_EC_NO_ACTIVE_CHANNEL",
  "ERROR_MUI_FILE_NOT_FOUND",
  "ERROR_MUI_INVALID_FILE",
  "ERROR_MUI_INVALID_RC_CONFIG",
  "ERROR_MUI_INVALID_LOCALE_NAME",
  "ERROR_MUI_INVALID_ULTIMATEFALLBACK_NAME",
  "ERROR_MUI_FILE_NOT_LOADED",
  "ERROR_RESOURCE_ENUM_USER_STOP",
  "ERROR_MUI_INTLSETTINGS_UILANG_NOT_INSTALLED",
  "ERROR_MUI_INTLSETTINGS_INVALID_LOCALE_NAME",
  "ERROR_MRM_RUNTIME_NO_DEFAULT_OR_NEUTRAL_RESOURCE",
  "ERROR_MRM_INVALID_PRICONFIG",
  "ERROR_MRM_INVALID_FILE_TYPE",
  "ERROR_MRM_UNKNOWN_QUALIFIER",
  "ERROR_MRM_INVALID_QUALIFIER_VALUE",
  "ERROR_MRM_NO_CANDIDATE",
  "ERROR_MRM_NO_MATCH_OR_DEFAULT_CANDIDATE",
  "ERROR_MRM_RESOURCE_TYPE_MISMATCH",
  "ERROR_MRM_DUPLICATE_MAP_NAME",
  "ERROR_MRM_DUPLICATE_ENTRY",
  "ERROR_MRM_INVALID_RESOURCE_IDENTIFIER",
  "ERROR_MRM_FILEPATH_TOO_LONG",
  "ERROR_MRM_UNSUPPORTED_DIRECTORY_TYPE",
  "ERROR_MRM_INVALID_PRI_FILE",
  "ERROR_MRM_NAMED_RESOURCE_NOT_FOUND",
  "ERROR_MRM_MAP_NOT_FOUND",
  "ERROR_MRM_UNSUPPORTED_PROFILE_TYPE",
  "ERROR_MRM_INVALID_QUALIFIER_OPERATOR",
  "ERROR_MRM_INDETERMINATE_QUALIFIER_VALUE",
  "ERROR_MRM_AUTOMERGE_ENABLED",
  "ERROR_MRM_TOO_MANY_RESOURCES",
  "ERROR_MRM_UNSUPPORTED_FILE_TYPE_FOR_MERGE",
  "ERROR_MRM_UNSUPPORTED_FILE_TYPE_FOR_LOAD_UNLOAD_PRI_FILE",
  "ERROR_MRM_NO_CURRENT_VIEW_ON_THREAD",
  "ERROR_DIFFERENT_PROFILE_RESOURCE_MANAGER_EXIST",
  "ERROR_OPERATION_NOT_ALLOWED_FROM_SYSTEM_COMPONENT",
  "ERROR_MRM_DIRECT_REF_TO_NON_DEFAULT_RESOURCE",
  "ERROR_MRM_GENERATION_COUNT_MISMATCH",
  "ERROR_PRI_MERGE_VERSION_MISMATCH",
  "ERROR_PRI_MERGE_MISSING_SCHEMA",
  "ERROR_PRI_MERGE_LOAD_FILE_FAILED",
  "ERROR_PRI_MERGE_ADD_FILE_FAILED",
  "ERROR_PRI_MERGE_WRITE_FILE_FAILED",
  "ERROR_PRI_MERGE_MULTIPLE_PACKAGE_FAMILIES_NOT_ALLOWED",
  "ERROR_PRI_MERGE_MULTIPLE_MAIN_PACKAGES_NOT_ALLOWED",
  "ERROR_PRI_MERGE_BUNDLE_PACKAGES_NOT_ALLOWED",
  "ERROR_PRI_MERGE_MAIN_PACKAGE_REQUIRED",
  "ERROR_PRI_MERGE_RESOURCE_PACKAGE_REQUIRED",
  "ERROR_PRI_MERGE_INVALID_FILE_NAME",
  "ERROR_MRM_PACKAGE_NOT_FOUND",
  "ERROR_MRM_MISSING_DEFAULT_LANGUAGE",
  "ERROR_MCA_INVALID_CAPABILITIES_STRING",
  "ERROR_MCA_INVALID_VCP_VERSION",
  "ERROR_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION",
  "ERROR_MCA_MCCS_VERSION_MISMATCH",
  "ERROR_MCA_UNSUPPORTED_MCCS_VERSION",
  "ERROR_MCA_INTERNAL_ERROR",
  "ERROR_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED",
  "ERROR_MCA_UNSUPPORTED_COLOR_TEMPERATURE",
  "ERROR_AMBIGUOUS_SYSTEM_DEVICE",
  "ERROR_SYSTEM_DEVICE_NOT_FOUND",
  "ERROR_HASH_NOT_SUPPORTED",
  "ERROR_HASH_NOT_PRESENT",
  "ERROR_SECONDARY_IC_PROVIDER_NOT_REGISTERED",
  "ERROR_GPIO_CLIENT_INFORMATION_INVALID",
  "ERROR_GPIO_VERSION_NOT_SUPPORTED",
  "ERROR_GPIO_INVALID_REGISTRATION_PACKET",
  "ERROR_GPIO_OPERATION_DENIED",
  "ERROR_GPIO_INCOMPATIBLE_CONNECT_MODE",
  "ERROR_GPIO_INTERRUPT_ALREADY_UNMASKED",
  "ERROR_CANNOT_SWITCH_RUNLEVEL",
  "ERROR_INVALID_RUNLEVEL_SETTING",
  "ERROR_RUNLEVEL_SWITCH_TIMEOUT",
  "ERROR_RUNLEVEL_SWITCH_AGENT_TIMEOUT",
  "ERROR_RUNLEVEL_SWITCH_IN_PROGRESS",
  "ERROR_SERVICES_FAILED_AUTOSTART",
  "ERROR_COM_TASK_STOP_PENDING",
  "ERROR_INSTALL_OPEN_PACKAGE_FAILED",
  "ERROR_INSTALL_PACKAGE_NOT_FOUND",
  "ERROR_INSTALL_INVALID_PACKAGE",
  "ERROR_INSTALL_RESOLVE_DEPENDENCY_FAILED",
  "ERROR_INSTALL_OUT_OF_DISK_SPACE",
  "ERROR_INSTALL_NETWORK_FAILURE",
  "ERROR_INSTALL_REGISTRATION_FAILURE",
  "ERROR_INSTALL_DEREGISTRATION_FAILURE",
  "ERROR_INSTALL_CANCEL",
  "ERROR_INSTALL_FAILED",
  "ERROR_REMOVE_FAILED",
  "ERROR_PACKAGE_ALREADY_EXISTS",
  "ERROR_NEEDS_REMEDIATION",
  "ERROR_INSTALL_PREREQUISITE_FAILED",
  "ERROR_PACKAGE_REPOSITORY_CORRUPTED",
  "ERROR_INSTALL_POLICY_FAILURE",
  "ERROR_PACKAGE_UPDATING",
  "ERROR_DEPLOYMENT_BLOCKED_BY_POLICY",
  "ERROR_PACKAGES_IN_USE",
  "ERROR_RECOVERY_FILE_CORRUPT",
  "ERROR_INVALID_STAGED_SIGNATURE",
  "ERROR_DELETING_EXISTING_APPLICATIONDATA_STORE_FAILED",
  "ERROR_INSTALL_PACKAGE_DOWNGRADE",
  "ERROR_SYSTEM_NEEDS_REMEDIATION",
  "ERROR_APPX_INTEGRITY_FAILURE_CLR_NGEN",
  "ERROR_RESILIENCY_FILE_CORRUPT",
  "ERROR_INSTALL_FIREWALL_SERVICE_NOT_RUNNING",
  "ERROR_PACKAGE_MOVE_FAILED",
  "ERROR_INSTALL_VOLUME_NOT_EMPTY",
  "ERROR_INSTALL_VOLUME_OFFLINE",
  "ERROR_INSTALL_VOLUME_CORRUPT",
  "ERROR_NEEDS_REGISTRATION",
  "ERROR_INSTALL_WRONG_PROCESSOR_ARCHITECTURE",
  "ERROR_DEV_SIDELOAD_LIMIT_EXCEEDED",
  "ERROR_INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE",
  "ERROR_PACKAGE_NOT_SUPPORTED_ON_FILESYSTEM",
  "ERROR_PACKAGE_MOVE_BLOCKED_BY_STREAMING",
  "ERROR_INSTALL_OPTIONAL_PACKAGE_APPLICATIONID_NOT_UNIQUE",
  "ERROR_PACKAGE_STAGING_ONHOLD",
  "ERROR_INSTALL_INVALID_RELATED_SET_UPDATE",
  "ERROR_INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY",
  "ERROR_DEPLOYMENT_BLOCKED_BY_USER_LOG_OFF",
  "ERROR_PROVISION_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_PROVISIONED",
  "ERROR_PACKAGES_REPUTATION_CHECK_FAILED",
  "ERROR_PACKAGES_REPUTATION_CHECK_TIMEDOUT",
  "ERROR_DEPLOYMENT_OPTION_NOT_SUPPORTED",
  "ERROR_APPINSTALLER_ACTIVATION_BLOCKED",
  "ERROR_REGISTRATION_FROM_REMOTE_DRIVE_NOT_SUPPORTED",
  "ERROR_APPX_RAW_DATA_WRITE_FAILED",
  "ERROR_DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_PACKAGE",
  "ERROR_DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_MACHINE",
  "ERROR_DEPLOYMENT_BLOCKED_BY_PROFILE_POLICY",
  "ERROR_DEPLOYMENT_FAILED_CONFLICTING_MUTABLE_PACKAGE_DIRECTORY",
  "ERROR_SINGLETON_RESOURCE_INSTALLED_IN_ACTIVE_USER",
  "ERROR_DIFFERENT_VERSION_OF_PACKAGED_SERVICE_INSTALLED",
  "ERROR_SERVICE_EXISTS_AS_NON_PACKAGED_SERVICE",
  "ERROR_PACKAGED_SERVICE_REQUIRES_ADMIN_PRIVILEGES",
  "ERROR_REDIRECTION_TO_DEFAULT_ACCOUNT_NOT_ALLOWED",
  "ERROR_PACKAGE_LACKS_CAPABILITY_TO_DEPLOY_ON_HOST",
  "ERROR_UNSIGNED_PACKAGE_INVALID_CONTENT",
  "ERROR_UNSIGNED_PACKAGE_INVALID_PUBLISHER_NAMESPACE",
  "ERROR_SIGNED_PACKAGE_INVALID_PUBLISHER_NAMESPACE",
  "ERROR_PACKAGE_EXTERNAL_LOCATION_NOT_ALLOWED",
  "ERROR_INSTALL_FULLTRUST_HOSTRUNTIME_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY",
  "ERROR_STATE_LOAD_STORE_FAILED",
  "ERROR_STATE_GET_VERSION_FAILED",
  "ERROR_STATE_SET_VERSION_FAILED",
  "ERROR_STATE_STRUCTURED_RESET_FAILED",
  "ERROR_STATE_OPEN_CONTAINER_FAILED",
  "ERROR_STATE_CREATE_CONTAINER_FAILED",
  "ERROR_STATE_DELETE_CONTAINER_FAILED",
  "ERROR_STATE_READ_SETTING_FAILED",
  "ERROR_STATE_WRITE_SETTING_FAILED",
  "ERROR_STATE_DELETE_SETTING_FAILED",
  "ERROR_STATE_QUERY_SETTING_FAILED",
  "ERROR_STATE_READ_COMPOSITE_SETTING_FAILED",
  "ERROR_STATE_WRITE_COMPOSITE_SETTING_FAILED",
  "ERROR_STATE_ENUMERATE_CONTAINER_FAILED",
  "ERROR_STATE_ENUMERATE_SETTINGS_FAILED",
  "ERROR_STATE_COMPOSITE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED",
  "ERROR_STATE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED",
  "ERROR_STATE_SETTING_NAME_SIZE_LIMIT_EXCEEDED",
  "ERROR_STATE_CONTAINER_NAME_SIZE_LIMIT_EXCEEDED",
  "ERROR_API_UNAVAILABLE",
};

const char* WinErrorToString(uint16_t error) {
  auto itr = std::find(std::begin(kWinErrorValues),
                       std::end(kWinErrorValues),
                       error);
  if (itr == std::end(kWinErrorValues)) {
    return nullptr;
  }

  return kWinErrorStrings[itr - std::begin(kWinErrorValues)];
}

}  // namespace google_breakpad
