/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 et cindent: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "AudioEventTimeline.h"
#include "AudioNodeTrack.h"

#include "mozilla/ErrorResult.h"

static float LinearInterpolate(double t0, float v0, double t1, float v1,
                               double t) {
  return v0 + (v1 - v0) * ((t - t0) / (t1 - t0));
}

static float ExponentialInterpolate(double t0, float v0, double t1, float v1,
                                    double t) {
  return v0 * fdlibm_powf(v1 / v0, (t - t0) / (t1 - t0));
}

static float ExponentialApproach(double t0, double v0, float v1,
                                 double timeConstant, double t) {
  if (!mozilla::dom::WebAudioUtils::FuzzyEqual(timeConstant, 0.0)) {
    return v1 + (v0 - v1) * fdlibm_expf(-(t - t0) / timeConstant);
  } else {
    return v1;
  }
}

static float ExtractValueFromCurve(double startTime, float* aCurve,
                                   uint32_t aCurveLength, double duration,
                                   double t) {
  if (t >= startTime + duration) {
    // After the duration, return the last curve value
    return aCurve[aCurveLength - 1];
  }
  double ratio = std::max((t - startTime) / duration, 0.0);
  if (ratio >= 1.0) {
    return aCurve[aCurveLength - 1];
  }
  uint32_t current = uint32_t(floor((aCurveLength - 1) * ratio));
  uint32_t next = current + 1;
  double step = duration / double(aCurveLength - 1);
  if (next < aCurveLength) {
    double t0 = current * step;
    double t1 = next * step;
    return LinearInterpolate(t0, aCurve[current], t1, aCurve[next],
                             t - startTime);
  } else {
    return aCurve[current];
  }
}

namespace mozilla::dom {

AudioTimelineEvent::AudioTimelineEvent(Type aType, double aTime, float aValue,
                                       double aTimeConstant, double aDuration,
                                       const float* aCurve,
                                       uint32_t aCurveLength)
    : mType(aType),
      mCurve(nullptr),
      mTimeConstant(aTimeConstant),
      mDuration(aDuration),
      mTime(aTime) {
  if (aType == AudioTimelineEvent::SetValueCurve) {
    SetCurveParams(aCurve, aCurveLength);
  } else {
    mValue = aValue;
  }
}

AudioTimelineEvent::AudioTimelineEvent(AudioNodeTrack* aTrack)
    : mType(Track),
      mCurve(nullptr),
      mTrack(aTrack),
      mTimeConstant(0.0),
      mDuration(0.0),
      mTime(0.0) {}

AudioTimelineEvent::AudioTimelineEvent(const AudioTimelineEvent& rhs) {
  PodCopy(this, &rhs, 1);

  if (rhs.mType == AudioTimelineEvent::SetValueCurve) {
    SetCurveParams(rhs.mCurve, rhs.mCurveLength);
  } else if (rhs.mType == AudioTimelineEvent::Track) {
    new (&mTrack) decltype(mTrack)(rhs.mTrack);
  }
}

AudioTimelineEvent::~AudioTimelineEvent() {
  if (mType == AudioTimelineEvent::SetValueCurve) {
    delete[] mCurve;
  }
}

template <class TimeType>
double AudioTimelineEvent::EndTime() const {
  MOZ_ASSERT(mType != AudioTimelineEvent::SetTarget);
  if (mType == AudioTimelineEvent::SetValueCurve) {
    return Time<TimeType>() + mDuration;
  }
  return Time<TimeType>();
};

float AudioTimelineEvent::EndValue() const {
  if (mType == AudioTimelineEvent::SetValueCurve) {
    return mCurve[mCurveLength - 1];
  }
  return mValue;
};

template <class TimeType>
float AudioEventTimeline::ComputeSetTargetStartValue(
    const AudioTimelineEvent* aPreviousEvent, TimeType aTime) {
  mSetTargetStartTime = aTime;
  mSetTargetStartValue =
      GetValuesAtTimeHelperInternal(aTime, aPreviousEvent, nullptr);
  return mSetTargetStartValue;
}

template void AudioEventTimeline::CleanupEventsOlderThan(double);
template void AudioEventTimeline::CleanupEventsOlderThan(int64_t);
template <class TimeType>
void AudioEventTimeline::CleanupEventsOlderThan(TimeType aTime) {
  auto TimeOf =
      [](const decltype(mEvents)::const_iterator& aEvent) -> TimeType {
    return aEvent->Time<TimeType>();
  };

  if (mSimpleValue.isSome()) {
    return;  // already only a single event
  }

  // Find first event to keep.  Keep one event prior to aTime.
  auto begin = mEvents.cbegin();
  auto end = mEvents.cend();
  auto event = begin + 1;
  for (; event < end && aTime > TimeOf(event); ++event) {
    MOZ_ASSERT(!(event - 1)->mTrack,
               "AudioParam tracks should never be destroyed on the real-time "
               "thread.");
  }
  auto firstToKeep = event - 1;

  if (firstToKeep->mType != AudioTimelineEvent::SetTarget) {
    // The value is constant if there is a single remaining non-SetTarget event
    // that has already passed.
    if (end - firstToKeep == 1 && aTime >= firstToKeep->EndTime<TimeType>()) {
      mSimpleValue.emplace(firstToKeep->EndValue());
    }
  } else {
    // The firstToKeep event is a SetTarget.  Set its initial value if
    // not already set.  First find the most recent event where the value at
    // the end time of the event is known, either from the event or for
    // SetTarget events because it has already been calculated.  This may not
    // have been calculated if GetValuesAtTime() was not called for the start
    // time of the SetTarget event.
    for (event = firstToKeep;
         event > begin && event->mType == AudioTimelineEvent::SetTarget &&
         TimeOf(event) > mSetTargetStartTime.Get<TimeType>();
         --event) {
    }
    // Compute SetTarget start times.
    for (; event < firstToKeep; ++event) {
      MOZ_ASSERT((event + 1)->mType == AudioTimelineEvent::SetTarget);
      ComputeSetTargetStartValue(&*event, TimeOf(event + 1));
    }
  }
  if (firstToKeep == begin) {
    return;
  }

  JS::AutoSuppressGCAnalysis suppress;  // for null mTrack
  mEvents.RemoveElementsRange(begin, firstToKeep);
}

// This method computes the AudioParam value at a given time based on the event
// timeline
template <class TimeType>
void AudioEventTimeline::GetValuesAtTimeHelper(TimeType aTime, float* aBuffer,
                                               const size_t aSize) {
  MOZ_ASSERT(aBuffer);
  MOZ_ASSERT(aSize);

  auto TimeOf = [](const AudioTimelineEvent& aEvent) -> TimeType {
    return aEvent.Time<TimeType>();
  };

  size_t eventIndex = 0;
  const AudioTimelineEvent* previous = nullptr;

  // Let's remove old events except the last one: we need it to calculate some
  // curves.
  CleanupEventsOlderThan(aTime);

  for (size_t bufferIndex = 0; bufferIndex < aSize; ++bufferIndex, ++aTime) {
    bool timeMatchesEventIndex = false;
    const AudioTimelineEvent* next;
    for (;; ++eventIndex) {
      if (eventIndex >= mEvents.Length()) {
        next = nullptr;
        break;
      }

      next = &mEvents[eventIndex];
      if (aTime < TimeOf(*next)) {
        break;
      }

#ifdef DEBUG
      MOZ_ASSERT(next->mType == AudioTimelineEvent::SetValueAtTime ||
                 next->mType == AudioTimelineEvent::SetTarget ||
                 next->mType == AudioTimelineEvent::LinearRamp ||
                 next->mType == AudioTimelineEvent::ExponentialRamp ||
                 next->mType == AudioTimelineEvent::SetValueCurve);
#endif

      if (TimesEqual(aTime, TimeOf(*next))) {
        timeMatchesEventIndex = true;
        aBuffer[bufferIndex] = GetValueAtTimeOfEvent<TimeType>(next, previous);
        // Advance to next event, which may or may not have the same time.
      }
      previous = next;
    }

    if (timeMatchesEventIndex) {
      // The time matches one of the events exactly.
      MOZ_ASSERT(TimesEqual(aTime, TimeOf(mEvents[eventIndex - 1])));
    } else {
      aBuffer[bufferIndex] =
          GetValuesAtTimeHelperInternal(aTime, previous, next);
    }
  }
}
template void AudioEventTimeline::GetValuesAtTimeHelper(double aTime,
                                                        float* aBuffer,
                                                        const size_t aSize);
template void AudioEventTimeline::GetValuesAtTimeHelper(int64_t aTime,
                                                        float* aBuffer,
                                                        const size_t aSize);

template <class TimeType>
float AudioEventTimeline::GetValueAtTimeOfEvent(
    const AudioTimelineEvent* aEvent, const AudioTimelineEvent* aPrevious) {
  TimeType time = aEvent->Time<TimeType>();
  switch (aEvent->mType) {
    case AudioTimelineEvent::SetTarget:
      // Start the curve, from the last value of the previous event.
      return ComputeSetTargetStartValue(aPrevious, time);
    case AudioTimelineEvent::SetValueCurve:
      // SetValueCurve events can be handled no matter what their event
      // node is (if they have one)
      return ExtractValueFromCurve(time, aEvent->mCurve, aEvent->mCurveLength,
                                   aEvent->mDuration, time);
      break;
    default:
      // For other event types
      return aEvent->mValue;
  }
}

template <class TimeType>
float AudioEventTimeline::GetValuesAtTimeHelperInternal(
    TimeType aTime, const AudioTimelineEvent* aPrevious,
    const AudioTimelineEvent* aNext) {
  // If the requested time is before all of the existing events
  if (!aPrevious) {
    return mDefaultValue;
  }

  auto TimeOf = [](const AudioTimelineEvent* aEvent) -> TimeType {
    return aEvent->Time<TimeType>();
  };
  auto EndTimeOf = [](const AudioTimelineEvent* aEvent) -> double {
    return aEvent->EndTime<TimeType>();
  };

  // SetTarget nodes can be handled no matter what their next node is (if
  // they have one)
  if (aPrevious->mType == AudioTimelineEvent::SetTarget) {
    return ExponentialApproach(TimeOf(aPrevious), mSetTargetStartValue,
                               aPrevious->mValue, aPrevious->mTimeConstant,
                               aTime);
  }

  // SetValueCurve events can be handled no matter what their next node is
  // (if they have one), when aTime is in the curve region.
  if (aPrevious->mType == AudioTimelineEvent::SetValueCurve &&
      aTime <= TimeOf(aPrevious) + aPrevious->mDuration) {
    return ExtractValueFromCurve(TimeOf(aPrevious), aPrevious->mCurve,
                                 aPrevious->mCurveLength, aPrevious->mDuration,
                                 aTime);
  }

  // Handle the cases where our range ends up in a ramp event
  if (aNext) {
    switch (aNext->mType) {
      case AudioTimelineEvent::LinearRamp:
        return LinearInterpolate(EndTimeOf(aPrevious), aPrevious->EndValue(),
                                 TimeOf(aNext), aNext->mValue, aTime);

      case AudioTimelineEvent::ExponentialRamp:
        return ExponentialInterpolate(EndTimeOf(aPrevious),
                                      aPrevious->EndValue(), TimeOf(aNext),
                                      aNext->mValue, aTime);

      case AudioTimelineEvent::SetValueAtTime:
      case AudioTimelineEvent::SetTarget:
      case AudioTimelineEvent::SetValueCurve:
        break;
      case AudioTimelineEvent::SetValue:
      case AudioTimelineEvent::Cancel:
      case AudioTimelineEvent::Track:
        MOZ_ASSERT(false, "Should have been handled earlier.");
    }
  }

  // Now handle all other cases
  switch (aPrevious->mType) {
    case AudioTimelineEvent::SetValueAtTime:
    case AudioTimelineEvent::LinearRamp:
    case AudioTimelineEvent::ExponentialRamp:
      // If the next event type is neither linear or exponential ramp, the
      // value is constant.
      return aPrevious->mValue;
    case AudioTimelineEvent::SetValueCurve:
      return ExtractValueFromCurve(aPrevious->Time<TimeType>(),
                                   aPrevious->mCurve, aPrevious->mCurveLength,
                                   aPrevious->mDuration, aTime);
    case AudioTimelineEvent::SetTarget:
      MOZ_FALLTHROUGH_ASSERT("AudioTimelineEvent::SetTarget");
    case AudioTimelineEvent::SetValue:
    case AudioTimelineEvent::Cancel:
    case AudioTimelineEvent::Track:
      MOZ_ASSERT(false, "Should have been handled earlier.");
  }

  MOZ_ASSERT(false, "unreached");
  return 0.0f;
}
template float AudioEventTimeline::GetValuesAtTimeHelperInternal(
    double aTime, const AudioTimelineEvent* aPrevious,
    const AudioTimelineEvent* aNext);
template float AudioEventTimeline::GetValuesAtTimeHelperInternal(
    int64_t aTime, const AudioTimelineEvent* aPrevious,
    const AudioTimelineEvent* aNext);

const AudioTimelineEvent* AudioEventTimeline::GetPreviousEvent(
    double aTime) const {
  const AudioTimelineEvent* previous = nullptr;
  const AudioTimelineEvent* next = nullptr;

  auto TimeOf = [](const AudioTimelineEvent& aEvent) -> double {
    return aEvent.Time<double>();
  };

  bool bailOut = false;
  for (unsigned i = 0; !bailOut && i < mEvents.Length(); ++i) {
    switch (mEvents[i].mType) {
      case AudioTimelineEvent::SetValueAtTime:
      case AudioTimelineEvent::SetTarget:
      case AudioTimelineEvent::LinearRamp:
      case AudioTimelineEvent::ExponentialRamp:
      case AudioTimelineEvent::SetValueCurve:
        if (aTime == TimeOf(mEvents[i])) {
          // Find the last event with the same time
          do {
            ++i;
          } while (i < mEvents.Length() && aTime == TimeOf(mEvents[i]));
          return &mEvents[i - 1];
        }
        previous = next;
        next = &mEvents[i];
        if (aTime < TimeOf(mEvents[i])) {
          bailOut = true;
        }
        break;
      default:
        MOZ_ASSERT(false, "unreached");
    }
  }
  // Handle the case where the time is past all of the events
  if (!bailOut) {
    previous = next;
  }

  return previous;
}

}  // namespace mozilla::dom
