/* -*- Mode: indent-tabs-mode: nil; js-indent-level: 2 -*- */
/* vim: set sts=2 sw=2 et tw=80: */
"use strict";

add_task(async function testPopupBorderRadius() {
  let extension = ExtensionTestUtils.loadExtension({
    background() {
      browser.tabs.query({ active: true, currentWindow: true }, tabs => {
        browser.pageAction.show(tabs[0].id);
      });
    },

    manifest: {
      browser_action: {
        default_popup: "popup.html",
        default_area: "navbar",
        browser_style: false,
      },

      page_action: {
        default_popup: "popup.html",
        browser_style: false,
      },
    },

    files: {
      "popup.html": `<!DOCTYPE html>
        <html>
          <head><meta charset="utf-8"></head>
          <body style="width: 100px; height: 100px;"></body>
        </html>`,
    },
  });

  await extension.startup();

  let widget = getBrowserActionWidget(extension);
  // If the panel doesn't allows embedding in subview then
  // radius will be 0, otherwise 8.  In practice we always
  // disallow subview.
  let expectedRadius = widget.disallowSubView ? "8px" : "0px";

  async function testPanel(browser, standAlone = true) {
    let panel = getPanelForNode(browser);
    let arrowContent = panel.panelContent;

    let panelStyle = getComputedStyle(arrowContent);
    is(panelStyle.overflow, "clip", "overflow is clipped");

    let stack = browser.parentNode;
    let viewNode = stack.parentNode === panel ? browser : stack.parentNode;
    let viewStyle = getComputedStyle(viewNode);

    let props = [
      "borderTopLeftRadius",
      "borderTopRightRadius",
      "borderBottomRightRadius",
      "borderBottomLeftRadius",
    ];

    let bodyStyle = await SpecialPowers.spawn(
      browser,
      [props],
      async function (props) {
        let bodyStyle = content.getComputedStyle(content.document.body);

        return new Map(props.map(prop => [prop, bodyStyle[prop]]));
      }
    );

    for (let prop of props) {
      if (standAlone) {
        is(
          viewStyle[prop],
          panelStyle[prop],
          `Panel and view ${prop} should be the same`
        );
        is(
          bodyStyle.get(prop),
          panelStyle[prop],
          `Panel and body ${prop} should be the same`
        );
      } else {
        is(viewStyle[prop], expectedRadius, `View node ${prop} should be 0px`);
        is(
          bodyStyle.get(prop),
          expectedRadius,
          `Body node ${prop} should be 0px`
        );
      }
    }
  }

  {
    info("Test stand-alone browserAction popup");

    clickBrowserAction(extension);
    let browser = await awaitExtensionPanel(extension);
    await testPanel(browser);
    await closeBrowserAction(extension);
  }

  {
    info("Test menu panel browserAction popup");

    CustomizableUI.addWidgetToArea(widget.id, getCustomizableUIPanelID());

    clickBrowserAction(extension);
    let browser = await awaitExtensionPanel(extension);
    await testPanel(browser, false);
    await closeBrowserAction(extension);
  }

  {
    info("Test pageAction popup");

    clickPageAction(extension);
    let browser = await awaitExtensionPanel(extension);
    await testPanel(browser);
    await closePageAction(extension);
  }

  await extension.unload();
});
