/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package mozilla.components.lib.state

import androidx.annotation.CheckResult
import androidx.annotation.VisibleForTesting
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import java.lang.ref.WeakReference
import java.util.Collections
import java.util.concurrent.ConcurrentHashMap

/**
 * A generic store holding an immutable [State].
 *
 * The [State] can only be modified by dispatching [Action]s which will create a new state and notify all registered
 * [Observer]s.
 *
 * @param initialState The initial state until a dispatched [Action] creates a new state.
 * @param reducer A function that gets the current [State] and [Action] passed in and will return a new [State].
 * @param middleware Optional list of [Middleware] sitting between the [Store] and the [Reducer].
 */
open class Store<S : State, A : Action>(
    initialState: S,
    private val reducer: Reducer<S, A>,
    private val middleware: List<Middleware<S, A>> = emptyList(),
) {
    private var reducerChain: ((A) -> Unit)? = null

    @VisibleForTesting
    internal val subscriptions = Collections.newSetFromMap(ConcurrentHashMap<Subscription<S, A>, Boolean>())

    private val mutableStateFlow = MutableStateFlow(initialState)

    /**
     * The current [State].
     */
    val state: S
        get() = mutableStateFlow.value

    /**
     * An observable flow which will emit the store state as it updates.
     *
     * @return the current state as a [StateFlow]
     */
    val stateFlow: StateFlow<S> = mutableStateFlow.asStateFlow()

    /**
     * Registers an [Observer] function that will be invoked whenever the [State] changes.
     *
     * It's the responsibility of the caller to keep track of the returned [Subscription] and call
     * [Subscription.unsubscribe] to stop observing and avoid potentially leaking memory by keeping an unused [Observer]
     * registered. It's is recommend to use one of the `observe` extension methods that unsubscribe automatically.
     *
     * The created [Subscription] is in paused state until explicitly resumed by calling [Subscription.resume].
     * While paused the [Subscription] will not receive any state updates. Once resumed the [observer]
     * will get invoked immediately with the latest state.
     *
     * @return A [Subscription] object that can be used to unsubscribe from further state changes.
     */
    @CheckResult(suggest = "observe")
    @Synchronized
    fun observeManually(observer: Observer<S>): Subscription<S, A> {
        val subscription = Subscription(observer, store = this)
        subscriptions.add(subscription)

        return subscription
    }

    /**
     * Dispatch an [Action] to the store in order to trigger a [State] change.
     * This function may be invoked on any thread.
     * Invocations are serialized by synchronizing on `this@Store`,
     * preventing concurrent modification of the underlying store.
     * Long running reducers and/or middlewares can and will impact all consumers.
     *
     * @return Unit. Previously this returned a new Job that was launched here, but this no longer happens.
     */
    fun dispatch(action: A) {
        synchronized(this) {
            if (reducerChain == null) {
                var chain: (A) -> Unit = { action ->
                    val newState = reducer(state, action)
                    if (newState != mutableStateFlow.value) {
                        mutableStateFlow.value = newState
                        subscriptions.forEach { subscription -> subscription.dispatch(newState) }
                    }
                }
                middleware.reversed().forEach { middleware ->
                    val next = chain
                    chain = { action -> middleware(this, next, action) }
                }
                reducerChain = chain
            }
            reducerChain?.invoke(action)
        }
    }

    /**
     * A [Subscription] is returned whenever an observer is registered via the [observeManually] method. Calling
     * [unsubscribe] on the [Subscription] will unregister the observer.
     */
    class Subscription<S : State, A : Action> internal constructor(
        internal val observer: Observer<S>,
        store: Store<S, A>,
    ) {
        private val storeReference = WeakReference(store)
        internal var binding: Binding? = null
        private var active = false

        /**
         * Resumes the [Subscription]. The [Observer] will get notified for every state change.
         * Additionally it will get invoked immediately with the latest state.
         */
        @Synchronized
        fun resume() {
            active = true

            storeReference.get()?.state?.let(observer)
        }

        /**
         * Pauses the [Subscription]. The [Observer] will not get notified when the state changes
         * until [resume] is called.
         */
        @Synchronized
        fun pause() {
            active = false
        }

        /**
         * Notifies this subscription's observer of a state change.
         *
         * @param state the updated state.
         */
        @Synchronized
        internal fun dispatch(state: S) {
            if (active) {
                observer.invoke(state)
            }
        }

        /**
         * Unsubscribe from the [Store].
         *
         * Calling this method will clear all references and the subscription will not longer be
         * active.
         */
        @Synchronized
        fun unsubscribe() {
            active = false

            storeReference.get()?.subscriptions?.remove(this)
            storeReference.clear()

            binding?.unbind()
        }

        interface Binding {
            fun unbind()
        }
    }
}
