/*
 * Event emitter component.
 *
 * Copyright (c) 2014 Component contributors <dev@component.io>
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of the library source tree.
 *
 * https://github.com/component/emitter
 */

"use strict";

/**
 * Initialize a new `Emitter`.
 *
 * @public
 */

function Emitter(obj) {
  if (obj) {
    return mixin(obj);
  }
}

/**
 * Mixin the emitter properties.
 *
 * @param {object} obj
 * @return {object}
 * @private
 */

function mixin(obj) {
  for (const key in Emitter.prototype) {
    obj[key] = Emitter.prototype[key];
  }
  return obj;
}

/**
 * Listen on the given `event` with `fn`.
 *
 * @param {string} event
 * @param {Function} fn
 * @return {Emitter}
 * @public
 */

Emitter.prototype.on = function (event, fn) {
  this._callbacks = this._callbacks || {};
  (this._callbacks["$" + event] = this._callbacks["$" + event] || []).push(fn);
  return this;
};

/**
 * Emit `event` with the given args.
 *
 * @param {string} event
 * @param {Mixed} ...
 * @return {Emitter}
 */

Emitter.prototype.emit = function (event) {
  this._callbacks = this._callbacks || {};

  const args = new Array(arguments.length - 1);
  let callbacks = this._callbacks["$" + event];

  for (let i = 1; i < arguments.length; i++) {
    args[i - 1] = arguments[i];
  }

  if (callbacks) {
    callbacks = callbacks.slice(0);
    for (let i = 0, len = callbacks.length; i < len; ++i) {
      callbacks[i].apply(this, args);
    }
  }

  return this;
};

module.exports = Emitter;
