/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.fenix.ui

import android.os.Build
import android.os.Build.VERSION.SDK_INT
import androidx.compose.ui.test.junit4.AndroidComposeTestRule
import androidx.core.net.toUri
import androidx.test.espresso.intent.rule.IntentsRule
import org.junit.Ignore
import org.junit.Rule
import org.junit.Test
import org.mozilla.fenix.customannotations.SkipLeaks
import org.mozilla.fenix.customannotations.SmokeTest
import org.mozilla.fenix.helpers.AppAndSystemHelper.assertExternalAppOpens
import org.mozilla.fenix.helpers.AppAndSystemHelper.assertNativeAppOpens
import org.mozilla.fenix.helpers.AppAndSystemHelper.deleteDownloadedFileOnStorage
import org.mozilla.fenix.helpers.AppAndSystemHelper.setNetworkEnabled
import org.mozilla.fenix.helpers.Constants.PackageName.GMAIL_APP
import org.mozilla.fenix.helpers.Constants.PackageName.GOOGLE_APPS_PHOTOS
import org.mozilla.fenix.helpers.Constants.PackageName.GOOGLE_DOCS
import org.mozilla.fenix.helpers.HomeActivityTestRule
import org.mozilla.fenix.helpers.MatcherHelper.itemWithResIdAndText
import org.mozilla.fenix.helpers.MatcherHelper.itemWithText
import org.mozilla.fenix.helpers.TestAssetHelper.getGenericAsset
import org.mozilla.fenix.helpers.TestHelper.clickSnackbarButton
import org.mozilla.fenix.helpers.TestHelper.mDevice
import org.mozilla.fenix.helpers.TestHelper.verifySnackBarText
import org.mozilla.fenix.helpers.TestSetup
import org.mozilla.fenix.helpers.perf.DetectMemoryLeaksRule
import org.mozilla.fenix.ui.robots.browserScreen
import org.mozilla.fenix.ui.robots.clickPageObject
import org.mozilla.fenix.ui.robots.downloadRobot
import org.mozilla.fenix.ui.robots.homeScreen
import org.mozilla.fenix.ui.robots.navigationToolbar
import org.mozilla.fenix.ui.robots.notificationShade

/**
 *  Tests for verifying basic functionality of download
 *
 *  - Initiates a download
 *  - Verifies download prompt
 *  - Verifies download notification and actions
 *  - Verifies managing downloads inside the Downloads listing.
 **/
class DownloadTest : TestSetup() {
    // Remote test page managed by Mozilla Mobile QA team at https://github.com/mozilla-mobile/testapp
    private val downloadTestPage =
        "https://storage.googleapis.com/mobile_test_assets/test_app/downloads.html"
    private var downloadFile: String = ""

    @get:Rule
    val composeTestRule =
        AndroidComposeTestRule(
            HomeActivityTestRule.withDefaultSettingsOverrides(),
        ) { it.activity }

    @get:Rule
    val intentsTestRule = IntentsRule()

    @get:Rule
    val memoryLeaksRule = DetectMemoryLeaksRule()

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/243844
    @Test
    fun verifyTheDownloadPromptsTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "web_icon.png")
            verifyDownloadCompleteSnackbar(fileName = "web_icon.png")
            clickSnackbarButton(composeTestRule = this@DownloadTest.composeTestRule, "OPEN")
            verifyPhotosAppOpens()
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2299405
    @Ignore("Failing, see https://bugzilla.mozilla.org/show_bug.cgi?id=1987355")
    @Test
    fun verifyTheDownloadFailedNotificationsTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "1GB.zip")
            setNetworkEnabled(enabled = false)
            verifyDownloadFailedSnackbar(fileName = "1GB.zip")
            clickSnackbarButton(composeTestRule, "DETAILS")
        }.openNotificationShade {
            verifySystemNotificationExists("Download failed")
        }.closeNotificationTray(composeTestRule) {
        }
        downloadRobot(composeTestRule) {
            verifyDownloadFileFailedMessage("1GB.zip")
            setNetworkEnabled(enabled = true)
            clickTryAgainDownloadMenuButton()
            verifyPauseDownloadMenuButtonButton()
        }
        downloadRobot(composeTestRule) {
        }.openNotificationShade {
            expandNotificationMessage("1GB.zip")
            clickDownloadNotificationControlButton("CANCEL")
            verifySystemNotificationDoesNotExist("1GB.zip")
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2298616
    @SkipLeaks(reasons = ["https://bugzilla.mozilla.org/show_bug.cgi?id=2006672"])
    @Test
    fun verifyDownloadCompleteNotificationTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "web_icon.png")
            verifyDownloadCompleteSnackbar(fileName = "web_icon.png")
            waitUntilDownloadSnackbarGone()
        }
        mDevice.openNotification()
        notificationShade {
            verifySystemNotificationExists("Download completed")
            clickNotification("Download completed")
            assertExternalAppOpens(GOOGLE_APPS_PHOTOS)
            mDevice.pressBack()
            mDevice.openNotification()
            verifySystemNotificationExists("Download completed")
            swipeDownloadNotification(
                composeTestRule,
                direction = "Left",
                shouldDismissNotification = true,
                canExpandNotification = false,
                notificationItem = "web_icon.png",
            )
            verifySystemNotificationDoesNotExist("Firefox Fenix")
        }.closeNotificationTray(composeTestRule) {
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/451563
    @SmokeTest
    @Test
    fun pauseResumeCancelDownloadTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "3GB.zip")
            verifySnackBarText("Download in progress")
            waitUntilDownloadSnackbarGone()
        }
        mDevice.openNotification()
        notificationShade {
            expandNotificationMessage("3GB.zip")
            clickDownloadNotificationControlButton("PAUSE")
            verifySystemNotificationExists("Download paused")
            clickDownloadNotificationControlButton("RESUME")
            clickDownloadNotificationControlButton("CANCEL")
            verifySystemNotificationDoesNotExist("3GB.zip")
            mDevice.pressBack()
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyEmptyDownloadsList()
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2301474
    @Test
    fun openDownloadedFileFromDownloadsMenuTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "web_icon.png")
            verifyDownloadCompleteSnackbar(fileName = "web_icon.png")
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyDownloadedFileExistsInDownloadsList("web_icon.png")
            clickDownloadedItem("web_icon.png")
            verifyPhotosAppOpens()
            mDevice.pressBack()
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/1114970
    @Test
    @SkipLeaks(reasons = ["https://bugzilla.mozilla.org/show_bug.cgi?id=2004099"])
    fun deleteDownloadedFileTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "smallZip.zip")
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyDownloadedFileExistsInDownloadsList("smallZip.zip")
            clickDownloadItemMenuIcon("smallZip.zip")
            deleteDownloadedItem("smallZip.zip")
            clickSnackbarButton(composeTestRule, "Undo")
            verifyDownloadedFileExistsInDownloadsList("smallZip.zip")
            clickDownloadItemMenuIcon("smallZip.zip")
            deleteDownloadedItem("smallZip.zip")
            verifyEmptyDownloadsList()
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2302662
    @Test
    fun deleteMultipleDownloadedFilesTest() {
        val firstDownloadedFile = "smallZip.zip"
        val secondDownloadedFile = "textfile.txt"

        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = firstDownloadedFile)
            verifyDownloadCompleteSnackbar(fileName = firstDownloadedFile)
        }
        browserScreen(composeTestRule) {
        }.clickDownloadLink(secondDownloadedFile) {
        }.clickDownload {
            verifyDownloadCompleteSnackbar(fileName = secondDownloadedFile)
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyDownloadedFileExistsInDownloadsList(firstDownloadedFile)
            verifyDownloadedFileExistsInDownloadsList(secondDownloadedFile)
            longClickDownloadedItem(firstDownloadedFile)
            clickDownloadedItem(secondDownloadedFile)
            openMultiSelectMoreOptionsMenu()
            clickMultiSelectRemoveButton()
            clickMultiSelectDeleteDialogButton()
            clickSnackbarButton(composeTestRule, "Undo")
            verifyDownloadedFileExistsInDownloadsList(firstDownloadedFile)
            verifyDownloadedFileExistsInDownloadsList(secondDownloadedFile)
            longClickDownloadedItem(firstDownloadedFile)
            clickDownloadedItem(secondDownloadedFile)
            openMultiSelectMoreOptionsMenu()
            clickMultiSelectRemoveButton()
            clickMultiSelectDeleteDialogButton()
            verifyEmptyDownloadsList()
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2301537
    @Test
    fun fileDeletedFromStorageIsDeletedEverywhereTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "smallZip.zip")
            verifyDownloadCompleteSnackbar(fileName = "smallZip.zip")
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyDownloadedFileExistsInDownloadsList("smallZip.zip")
            deleteDownloadedFileOnStorage("smallZip.zip")
        }.exitDownloadsManagerToBrowser {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyEmptyDownloadsList()
        }.exitDownloadsManagerToBrowser {
        }

        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "smallZip.zip")
            verifyDownloadCompleteSnackbar(fileName = "smallZip.zip")
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyDownloadedFileExistsInDownloadsList("smallZip.zip")
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2466505
    @Test
    fun systemNotificationCantBeDismissedWhileInProgressTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "3GB.zip")
        }
        browserScreen(composeTestRule) {
        }.openNotificationShade {
            swipeDownloadNotification(composeTestRule, direction = "Left", shouldDismissNotification = false, notificationItem = "3GB.zip")
            expandNotificationMessage("3GB.zip")
            clickDownloadNotificationControlButton("PAUSE")
            notificationShade {
            }.closeNotificationTray(composeTestRule) {
            }
            browserScreen(composeTestRule) {
            }.openNotificationShade {
                swipeDownloadNotification(composeTestRule, direction = "Right", shouldDismissNotification = true, notificationItem = "3GB.zip")
                verifySystemNotificationDoesNotExist("3GB.zip")
            }
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2299297
    @Ignore("Failing, see https://bugzilla.mozilla.org/show_bug.cgi?id=1987355")
    @Test
    fun notificationCanBeDismissedIfDownloadIsInterruptedTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "1GB.zip")
            setNetworkEnabled(enabled = false)
            verifyDownloadFailedSnackbar(fileName = "1GB.zip")
        }
        browserScreen(composeTestRule) {
        }.openNotificationShade {
            verifySystemNotificationExists("Download failed")
            swipeDownloadNotification(
                composeTestRule,
                direction = "Left",
                shouldDismissNotification = true,
                canExpandNotification = true,
                notificationItem = "1GB.zip",
            )
            verifySystemNotificationDoesNotExist("Firefox Fenix")
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/1632384
    @Test
    fun warningWhenClosingPrivateTabsWhileDownloadingTest() {
        homeScreen(composeTestRule) {
        }.togglePrivateBrowsingMode()
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "3GB.zip")
        }
        browserScreen(composeTestRule) {
        }.openTabDrawer(composeTestRule) {
            closeTab()
        }
        browserScreen(composeTestRule) {
            verifyCancelPrivateDownloadsPrompt("1")
            clickStayInPrivateBrowsingPromptButton()
        }.openNotificationShade {
            if (SDK_INT == Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
                // On API 34 we first need to expand the system notification before verifying that the app name is displayed
                expandNotificationMessage("3GB.zip")
                verifySystemNotificationExists("Firefox Fenix")
            } else {
                verifySystemNotificationExists("Firefox Fenix")
            }
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2302663
    @Test
    fun cancelActivePrivateBrowsingDownloadsTest() {
        homeScreen(composeTestRule) {
        }.togglePrivateBrowsingMode()
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "3GB.zip")
        }
        browserScreen(composeTestRule) {
        }.openTabDrawer(composeTestRule) {
            closeTab()
        }
        browserScreen(composeTestRule) {
            verifyCancelPrivateDownloadsPrompt("1")
            clickCancelPrivateDownloadsPromptButton()
        }.openNotificationShade {
            verifySystemNotificationDoesNotExist("Firefox Fenix")
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2048448
    // Save edited PDF file from the share overlay
    @SmokeTest
    @Test
    fun saveAsPdfFunctionalityTest() {
        val genericURL = mockWebServer.getGenericAsset(3)
        downloadFile = "pdfForm.pdf"

        navigationToolbar(composeTestRule) {
        }.enterURLAndEnterToBrowser(genericURL.url) {
            clickPageObject(composeTestRule, itemWithText("PDF form file"))
            waitForPageToLoad()
            clickPageObject(composeTestRule, itemWithResIdAndText("android:id/button2", "Cancel"))
            fillPdfForm("Firefox")
        }.openThreeDotMenu {
        }.clickShareButton {
        }.clickSaveAsPDF(composeTestRule) {
           verifyDownloadPrompt(downloadFile)
        }.clickDownload {
            verifyDownloadCompleteSnackbar(fileName = downloadFile)
            clickSnackbarButton(composeTestRule = composeTestRule, "OPEN")
            assertExternalAppOpens(GOOGLE_DOCS)
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/244125
    @Ignore("Failing, see https://bugzilla.mozilla.org/show_bug.cgi?id=1983769")
    @Test
    fun restartDownloadFromAppNotificationAfterConnectionIsInterruptedTest() {
        downloadFile = "3GB.zip"

        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "3GB.zip")
            setNetworkEnabled(false)
            verifyDownloadFailedSnackbar(fileName = "3GB.zip")
            setNetworkEnabled(true)
            clickSnackbarButton(composeTestRule, "DETAILS")
            verifyDownloadFileFailedMessage("3GB.zip")
            setNetworkEnabled(enabled = true)
            clickTryAgainDownloadMenuButton()
            verifyPauseDownloadMenuButtonButton()
        }
        downloadRobot(composeTestRule) {
        }.openNotificationShade {
            expandNotificationMessage("3GB.zip")
            clickDownloadNotificationControlButton("CANCEL")
            verifySystemNotificationDoesNotExist("3GB.zip")
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2981843
    @Test
    fun verifyTheDownloadFiltersTest() {
        val firstDownloadedFile = "smallZip.zip"
        val secondDownloadedFile = "web_icon.png"

        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = firstDownloadedFile)
            verifyDownloadCompleteSnackbar(fileName = firstDownloadedFile)
        }
        browserScreen(composeTestRule) {
        }.clickDownloadLink(secondDownloadedFile) {
        }.clickDownload {
            verifyDownloadCompleteSnackbar(fileName = secondDownloadedFile)
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            clickDownloadsFilter("Images")
            verifyDownloadedFileExistsInDownloadsList(secondDownloadedFile)
            verifyDownloadFileIsNotDisplayed(firstDownloadedFile)
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/2987000
    @Test
    fun shareDownloadedFileTest() {
        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "web_icon.png")
            verifyDownloadCompleteSnackbar(fileName = "web_icon.png")
        }
        browserScreen(composeTestRule) {
        }.openThreeDotMenu {
        }.clickDownloadsButton {
            verifyDownloadedFileExistsInDownloadsList("web_icon.png")
            clickDownloadItemMenuIcon("web_icon.png")
        }.shareDownloadedItem("web_icon.png") {
            expandAndroidShareLayout("Gmail")
            clickSharingApp("Gmail", GMAIL_APP)
            assertNativeAppOpens(composeTestRule, GMAIL_APP)
        }
    }

    // TestRail link: https://mozilla.testrail.io/index.php?/cases/view/457111
    @Ignore("Failing, see https://bugzilla.mozilla.org/show_bug.cgi?id=1983769")
    @Test
    fun downloadRestartAfterConnectionIsReestablishedTest() {
        downloadFile = "3GB.zip"

        downloadRobot(composeTestRule) {
            openPageAndDownloadFile(url = downloadTestPage.toUri(), downloadFile = "3GB.zip")
            downloadRobot(composeTestRule) {
            }.openNotificationShade {
                expandNotificationMessage("3GB.zip")
                clickDownloadNotificationControlButton("PAUSE")
                setNetworkEnabled(false)
                verifySystemNotificationExists("Download paused")
                clickDownloadNotificationControlButton("RESUME")
                verifySystemNotificationExists("Download failed")
                setNetworkEnabled(enabled = true)
                clickDownloadNotificationControlButton("TRY AGAIN")
                expandNotificationMessage("3GB.zip")
                clickDownloadNotificationControlButton("CANCEL")
                verifySystemNotificationDoesNotExist("3GB.zip")
            }
        }
    }
}
