/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

let Cu = Components.utils;
let Ci = Components.interfaces;

Cu.import("resource:///modules/tabview/utils.jsm");

// Bug 671101 - directly using webProgress in this context
// causes docShells to leak
this.__defineGetter__("webProgress", function () {
  let ifaceReq = docShell.QueryInterface(Ci.nsIInterfaceRequestor);
  return ifaceReq.getInterface(Ci.nsIWebProgress);
});

// ----------
// WindowEventHandler
//
// Handles events dispatched by the content window.
let WindowEventHandler = {
  // ----------
  // Function: onDOMWillOpenModalDialog
  // Sends a synchronous message when the "onDOMWillOpenModalDialog" event
  // is fired right before a modal dialog will be opened by the current page.
  onDOMWillOpenModalDialog: function WEH_onDOMWillOpenModalDialog(event) {
    // (event.isTrusted == true) when the event is generated by a user action
    // and does not originate from a script.
    if (!event.isTrusted)
      return;

    // we're intentionally sending a synchronous message to handle this event
    // as quick as possible, switch the selected tab and hide the tabview
    // before the modal dialog is shown
    sendSyncMessage("Panorama:DOMWillOpenModalDialog");
  },

  // ----------
  // Function: onMozAfterPaint
  // Sends an asynchronous message when the "onMozAfterPaint" event
  // is fired.
  onMozAfterPaint: function WEH_onMozAfterPaint(event) {
    if (event.clientRects.length > 0) {
      sendAsyncMessage("Panorama:MozAfterPaint");
    }
  }
};

// add event listeners
addEventListener("DOMWillOpenModalDialog", WindowEventHandler.onDOMWillOpenModalDialog, false);
addEventListener("MozAfterPaint", WindowEventHandler.onMozAfterPaint, false);

// ----------
// WindowMessageHandler
//
// Handles messages sent by the chrome process.
let WindowMessageHandler = {
  // ----------
  // Function: isDocumentLoaded
  // Checks if the currently active document is loaded.
  isDocumentLoaded: function WMH_isDocumentLoaded(cx) {
    let isLoaded = (content.document.readyState != "uninitialized" &&
                    !webProgress.isLoadingDocument);

    sendAsyncMessage(cx.name, {isLoaded: isLoaded});
  },

  // ----------
  // Function: isImageDocument
  // Checks if the currently active document is an image document or not.
  isImageDocument: function WMH_isImageDocument(cx) {
    let isImageDocument = (content.document instanceof Ci.nsIImageDocument);

    sendAsyncMessage(cx.name, {isImageDocument: isImageDocument});
  }
};

// add message listeners
addMessageListener("Panorama:isDocumentLoaded", WindowMessageHandler.isDocumentLoaded);
addMessageListener("Panorama:isImageDocument", WindowMessageHandler.isImageDocument);

