/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/*
 * Form Autofill frame script.
 */

"use strict";

/* eslint-env mozilla/frame-script */

const {classes: Cc, interfaces: Ci, utils: Cu, results: Cr} = Components;

Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://formautofill/FormAutofillContent.jsm");
Cu.import("resource://formautofill/FormAutofillUtils.jsm");

/**
 * Handles content's interactions for the frame.
 *
 * NOTE: Declares it by "var" to make it accessible in unit tests.
 */
var FormAutofillFrameScript = {
  init() {
    addEventListener("focusin", this);
    addMessageListener("FormAutofill:PreviewProfile", this);
    addMessageListener("FormAutoComplete:PopupClosed", this);
    addMessageListener("FormAutoComplete:PopupOpened", this);
  },

  handleEvent(evt) {
    if (!evt.isTrusted) {
      return;
    }

    if (!Services.prefs.getBoolPref("extensions.formautofill.addresses.enabled")) {
      return;
    }

    switch (evt.type) {
      case "focusin": {
        let element = evt.target;
        let doc = element.ownerDocument;

        if (!FormAutofillUtils.isFieldEligibleForAutofill(element)) {
          return;
        }

        let doIdentifyAutofillFields =
          () => setTimeout(() => FormAutofillContent.identifyAutofillFields(element));

        if (doc.readyState === "loading") {
          doc.addEventListener("DOMContentLoaded", doIdentifyAutofillFields, {once: true});
        } else {
          doIdentifyAutofillFields();
        }
        break;
      }
    }
  },

  receiveMessage(message) {
    if (!Services.prefs.getBoolPref("extensions.formautofill.addresses.enabled")) {
      return;
    }

    const doc = content.document;
    const {chromeEventHandler} = doc.ownerGlobal.getInterface(Ci.nsIDocShell);

    switch (message.name) {
      case "FormAutofill:PreviewProfile": {
        FormAutofillContent._previewProfile(doc);
        break;
      }
      case "FormAutoComplete:PopupClosed": {
        FormAutofillContent._previewProfile(doc);
        chromeEventHandler.removeEventListener("keydown", FormAutofillContent._onKeyDown,
                                               {capturing: true});
        break;
      }
      case "FormAutoComplete:PopupOpened": {
        chromeEventHandler.addEventListener("keydown", FormAutofillContent._onKeyDown,
                                            {capturing: true});
      }
    }
  },
};

FormAutofillFrameScript.init();
