#!/bin/sh
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Script to update the mozilla in-tree copy of the PDFium library.
# Run this within the /modules/pdfium directory of the source tree.

set -e

COMMIT_SELECTOR=""
COMMIT=""
COMMIT_DATE=""

print_help() {
    echo "usage: ./update.sh [--commit COMMIT]"
    echo ""
    echo "Update Tool for PDFium Library"
    echo "optional arguments:"
    echo "  --commit COMMIT  specify a commit hash or a branch name (default: master)"
}

checkout_commit() {
    if [ -d "pdfium" ]; then
        echo "Removing pdfium"
        rm -rf pdfium
    fi

    echo "Downloading pdfium"
    git clone https://pdfium.googlesource.com/pdfium

    echo "Parsing $1"
    if [[ $1 =~ ^[0-9A-Fa-f]+$ ]]; then
        COMMIT=$(git -C pdfium rev-parse "$1")
    else
        COMMIT=$(git -C pdfium rev-parse remotes/origin/"$1")
    fi
    COMMIT_DATE=$(git -C pdfium show -s --format=%ci "$COMMIT")

    echo "Checking out $COMMIT"
    git -C pdfium checkout "$COMMIT"
}

cleanup_files() {
    # Remove git source control files.
    echo "Removing pdfium/.git and pdfium/.gitignore"
    rm -rf pdfium/.git
    rm pdfium/.gitignore

    # We remove PDFium's copy of FreeType and link directly to our in-tree copy
    # at /modules/freetype2.
    # (Our in-tree copy of FreeType is kept up-to-date and the FreeType API is
    # generally backwards compatible, so it is unlikely that our copy of the
    # PDFium source will be incompatible with our in-tree copy of FreeType.)
    echo "Removing pdfium/third_party/freetype"
    rm -rf pdfium/third_party/freetype

    # We link PDFium to our in-tree copy of libjpeg at /media/libjpeg
    echo "Removing pdfium/third_party/libjpeg"
    rm -rf pdfium/third_party/libjpeg

    # We link PDFium to our in-tree copy of zlib at /modules/zlib
    echo "Removing pdfium/third_party/zlib_v128"
    rm -rf pdfium/third_party/zlib_v128

    # We don't use libpng since the XFA features are disabled.
    echo "Removing pdfium/third_party/libpng16"
    rm -rf pdfium/third_party/libpng16

    # We don't use libtiff since the XFA features are disabled.
    echo "Removing pdfium/third_party/libtiff"
    rm -rf pdfium/third_party/libtiff

    # We don't use pymock since we don't run the pymock based tests.
    echo "Removing pdfium/third_party/pymock"
    rm -rf pdfium/third_party/pymock

    # We don't currently need PDfium to support JPEG 2000 since we are not
    # currently using PDFium as a general PDF viewer. Currently we only
    # process PDF files generated by SkiaPDF, and SkiaPDF doesn't output
    # JPEG 2000 images.
    echo "Removing pdfium/third_party/libopenjpeg20"
    rm -rf pdfium/third_party/libopenjpeg20

    # We don't currently use any testing data or any testing programs of PDFium.
    echo "Removing pdfium/test and pdfium/testing"
    rm -rf pdfium/test
    rm -rf pdfium/testing

    # We don't need XFA implementations since the XFA features are disabled.
    echo "Removing pdfium/xfa"
    rm -rf pdfium/xfa
}

apply_patches() {
    # Patch to fix build errors due to lacking GDI+ prerequisite headers
    echo "Applying patches/bug1368948_gdiplus_prerequisite.patch"
    patch -p3 < patches/bug1368948_gdiplus_prerequisite.patch

    # Patch to use freetype library within Gecko
    echo "Applying patches/bug1368948_use_gecko_freetype.patch"
    patch -p3 < patches/bug1368948_use_gecko_freetype.patch

    # Patch to use libjpeg library within Gecko
    echo "Applying patches/bug1368948_use_gecko_libjpeg.patch"
    patch -p3 < patches/bug1368948_use_gecko_libjpeg.patch

    # Patch to use zlib library within Gecko
    echo "Applying patches/bug1368948_use_gecko_zlib.patch"
    patch -p3 < patches/bug1368948_use_gecko_zlib.patch

    # Patch to remove openjpeg support
    echo "Applying patches/bug1368948_remove_openjpeg.patch"
    patch -p3 < patches/bug1368948_remove_openjpeg.patch

    # Patch to avoid using GDIPlusExt while rendering EMF
    echo "Applying patches/bug1367948_avoid_using_GDIPlusExt.patch"
    patch -p3 < patches/bug1367948_avoid_using_GDIPlusExt.patch
}

update_readme() {
    echo "Updating README_MOZILLA"
    PREFIX="The git commit ID last used to import was"
    perl -p -i -e "s/${PREFIX} [0-9A-Fa-f]+ \(.+\)/${PREFIX} ${COMMIT} (${COMMIT_DATE})/" README_MOZILLA
}

if [ "$#" == 0 ]; then
    COMMIT_SELECTOR="master"
elif [ "$#" == 2 -a "$1" == "--commit" ]; then
    COMMIT_SELECTOR="$2"
else
    print_help
    exit 1
fi

cd $(dirname $0)
checkout_commit "$COMMIT_SELECTOR"
cleanup_files
apply_patches
update_readme
