// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

//! Helper macros for implementing the FFI API for metric types.

/// Get a metric object by ID from the corresponding map, then
/// execute the provided closure with it.
///
/// # Arguments
///
/// * `$map` - The name of the hash map within `metrics::__glean_metric_maps`
///            as generated by glean_parser.
/// * `$id`  - The ID of the metric to get.
/// * `$f`   - The function or closure to execute with the retrieved metric.
macro_rules! with_metric {
    (BOOLEAN_MAP, $id:ident, $f:expr) => {
        maybe_labeled_with_metric!(BOOLEAN_MAP, $id, $f)
    };
    (COUNTER_MAP, $id:ident, $f:expr) => {
        maybe_labeled_with_metric!(COUNTER_MAP, $id, $f)
    };
    (STRING_MAP, $id:ident, $f:expr) => {
        maybe_labeled_with_metric!(STRING_MAP, $id, $f)
    };
    ($map:ident, $id:ident, $f:expr) => {
        just_with_metric!($map, $id, $f)
    };
}

/// Get a metric object by id from the corresponding map, then
/// execute the provided closure with it.
///
/// Ignores the possibility that the $id might be for a labeled submetric.
///
/// # Arguments
///
/// * `$map` - The name of the hash map within `metrics::__glean_metrics_maps`
///            as generated by glean_parser.
/// * `$id`  - The ID of the metric to get.
/// * `$f`   - The function or closure to execute with the retrieved metric.
macro_rules! just_with_metric {
    ($map:ident, $id:ident, $f:expr) => {
        match $crate::metrics::__glean_metric_maps::$map.get(&$id.into()) {
            Some(metric) => $f(metric),
            None => panic!("No metric for id {}", $id),
        }
    };
}

/// Get a metric object by id from the corresponding map, then
/// execute the provided closure with it.
///
/// Requires that the provided $map be of a type that can be labeled, since it
/// assumes the presence of a same-named map in
/// `metrics::_glean_metrics_map::submetric_maps`.
///
/// # Arguments
///
/// * `$map` - The name of the hash map within `metrics::__glean_metrics_maps`
///            and `metrics::__glean_metrics_maps::submetric_maps` as generated
///            by glean_parser.
/// * `$id`  - The ID of the metric to get.
/// * `$f`   - The function or closure to execute with the retrieved metric.
macro_rules! maybe_labeled_with_metric {
    ($map:ident, $id:ident, $f:expr) => {
        if $id >= $crate::metrics::__glean_metric_maps::submetric_maps::MIN_LABELED_SUBMETRIC_ID {
            let map = $crate::metrics::__glean_metric_maps::submetric_maps::$map
                .read()
                .expect("Read lock for labeled metric map was poisoned");
            match map.get(&$id.into()) {
                Some(metric) => $f(metric),
                None => panic!("No submetric for id {}", $id),
            }
        } else {
            just_with_metric!($map, $id, $f)
        }
    };
}

/// Test whether a value is stored for the given metric.
///
/// # Arguments
///
/// * `$metric`  - The metric to test.
/// * `$storage` - the storage name to look into.
macro_rules! test_has {
    ($metric:ident, $storage:ident) => {{
        let storage = if $storage.is_empty() {
            None
        } else {
            Some($storage.to_utf8())
        };
        $metric.test_get_value(storage.as_deref()).is_some()
    }};
}

/// Get the currently stored value for the given metric.
///
/// # Arguments
///
/// * `$metric`  - The metric to test.
/// * `$storage` - the storage name to look into.
macro_rules! test_get {
    ($metric:ident, $storage:ident) => {{
        let storage = if $storage.is_empty() {
            None
        } else {
            Some($storage.to_utf8())
        };
        $metric.test_get_value(storage.as_deref()).unwrap()
    }};
}

/// Get the submetric id for a given labeled metric and label.
///
/// # Arguments
///
/// * `$id`						- The id of the labeled metric.
/// * `$label`				- The label of the submetric.
/// * `$labeled_map`	- The name of the labeled metric's map for retrieval.
/// * `$submetric_map`- The name of the submetrics' map for storage.
/// * `$metric_type`  - The submetric's type (needed for an internal closure).
macro_rules! labeled_submetric_get {
    ($id:ident, $label:ident, $labeled_map:ident, $submetric_map:ident, $metric_type:ty) => {{
        let tuple = ($id, $label.to_utf8().into());
        {
            let map = $crate::metrics::__glean_metric_maps::submetric_maps::LABELED_METRICS_TO_IDS
                .read()
                .expect("read lock of submetric ids was poisoned");
            if let Some(submetric_id) = map.get(&tuple) {
                return *submetric_id;
            }
        }

        // Gotta actually create a new submetric with a new id.
        let submetric_id =
            $crate::metrics::__glean_metric_maps::submetric_maps::NEXT_LABELED_SUBMETRIC_ID
                .fetch_add(1, Ordering::SeqCst);
        {
            with_metric!(
                $labeled_map,
                $id,
                |metric: &$crate::private::LabeledMetric<$metric_type>| {
                    let submetric = metric.get(&tuple.1);
                    let mut map =
                        $crate::metrics::__glean_metric_maps::submetric_maps::$submetric_map
                            .write()
                            .expect("write lock of submetric map was poisoned");
                    map.insert(submetric_id.into(), submetric);
                }
            );
        }

        let mut map = $crate::metrics::__glean_metric_maps::submetric_maps::LABELED_METRICS_TO_IDS
            .write()
            .expect("write lock of submetric ids was poisoned");
        map.insert(tuple, submetric_id);
        submetric_id
    }};
}
