/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Roaming code.
 *
 * The Initial Developer of the Original Code is 
 * Ben Bucksch <http://www.bucksch.org> of
 * Beonex <http://www.beonex.com>
 * Portions created by the Initial Developer are Copyright (C) 2002-2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "mozSRoaming.h"
#include "nsIComponentManager.h"
#include "nsIProfile.h"
#include "nsDirectoryServiceUtils.h"
#include "nsAppDirectoryServiceDefs.h"
#include "mozSRoamingCopy.h"
#include "mozSRoamingStream.h"
#include "nsILocalFile.h"
#include "nsIObserverService.h"
#include "nsIIOService.h"
#include "nsNetUtil.h"
#include "nsCRT.h"

// UI
#include "nsIDialogParamBlock.h"
#include "nsIDOMWindow.h"
#include "nsIWindowWatcher.h"
#include "plstr.h"
static NS_DEFINE_CID(kIOServiceCID, NS_IOSERVICE_CID);

//added for password 
#define kRegTreeProfile (NS_LITERAL_STRING("Profiles")) 
#define kRegTreeRoaming (NS_LITERAL_STRING("Roaming")) 


#define kPrefEnabled "roaming.enabled"
#define kPrefMethod "roaming.method"
#define kPrefFiles "roaming.files"
#define kPrefSavePreference "roaming.savePreference"
#define kPrefMethodStream "stream"
#define kPrefMethodCopy "copy"

#define kConflDlg "chrome://sroaming/content/transfer/conflictResolve.xul"


NS_IMPL_ISUPPORTS1(mozSRoaming,
                   nsISessionRoaming)

mozSRoaming::mozSRoaming()
    : mFiles(10),mRegistry(0)
{
  mHavePrefs = PR_FALSE;
  mIsRoaming = PR_FALSE;
  mMethod = 0;
}

mozSRoaming::~mozSRoaming()
{
}



/*
 * mozISRoaming implementation
 */

NS_IMETHODIMP
mozSRoaming::BeginSession()
{
  //printf("\n\n\n!!!! beginsession\n\n\n\n");
  nsresult rv = NS_OK;

  if (!mHavePrefs)
    rv = ReadRoamingPrefs();
  if (NS_FAILED(rv))
    return rv;

  if (!mIsRoaming)
    return NS_OK;

  mozSRoamingProtocol* proto = CreateMethodHandler();
  if (!proto)
    return NS_ERROR_ABORT;

  rv = proto->Init(this);
  if (NS_FAILED(rv))
  {
    //printf("error 0x%x\n", rv);
    return rv;
  }

  PrefsDone();

  rv = proto->Download();
  if (NS_FAILED(rv))
  {
    //printf("error 0x%x\n", rv);
    return rv;
  }

  delete proto;
  return NS_OK;
}

nsresult mozSRoaming::GenerateLiPrefs()
{
  nsresult rv = NS_OK;
  nsCOMPtr<nsIPrefService> prefService = do_GetService(NS_PREFSERVICE_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIPrefBranch> prefBranch;
  rv = prefService->GetBranch(nsnull, getter_AddRefs(prefBranch));
  NS_ENSURE_SUCCESS(rv, rv);

  PRUint32 prefCount;
  char** prefArray;
  rv = prefBranch->GetChildList("", &prefCount, &prefArray);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIFile> profileFile = ProfileDir();
  NS_ENSURE_TRUE(profileFile, NS_ERROR_FAILURE);

  profileFile->AppendNative(nsCAutoString("liprefs.js"));
  
  nsCOMPtr<nsIOutputStream> fileStream;
  rv = NS_NewLocalFileOutputStream(getter_AddRefs(fileStream), profileFile);
  NS_ENSURE_SUCCESS(rv, rv);

  PRInt32 i;
  for (i = 0; i < prefCount; i++ ) {
    PRBool hasUserValue;
    nsCAutoString prefName(prefArray[i]);
    rv = prefBranch->PrefHasUserValue(prefName.get(), &hasUserValue);
    if (NS_FAILED(rv) || !hasUserValue)
      continue;
    if (prefName.Find("roaming.") != kNotFound ||
        prefName.Find("directory") != kNotFound ||
        prefName.Find(".dir") != kNotFound ||
        prefName.Find("mail.root.") != kNotFound)
      continue;
    nsCAutoString line("user_pref(\"");
    line.Append(prefName);
    line.Append("\", ");
    PRInt32 type;
    prefBranch->GetPrefType(prefName.get(), &type);
    switch (type) {
      case nsIPrefBranch::PREF_BOOL:
        PRBool boolValue;
        rv = prefBranch->GetBoolPref(prefName.get(), &boolValue);
        NS_ENSURE_SUCCESS(rv, rv);
        
        line.Append(boolValue ? "true" : "false");
        break;
      case nsIPrefBranch::PREF_INT:
        PRInt32 intValue;
        rv = prefBranch->GetIntPref(prefName.get(), &intValue);
        NS_ENSURE_SUCCESS(rv, rv);
        
        line.AppendInt(intValue);
        break;
      default:
        case nsIPrefBranch::PREF_STRING:
          nsXPIDLCString strValue;
          rv = prefBranch->GetCharPref(prefName.get(), getter_Copies(strValue));
          NS_ENSURE_SUCCESS(rv, rv);

          line.Append("\"");
          nsCAutoString value(strValue.get());
          int replaceIndex = 0;
          value.ReplaceSubstring("\\", "\\\\");
          value.ReplaceSubstring("\"", "\\\"");
          line.Append(value.get());
          line.Append("\"");
          break;
    }
    line.Append(");"NS_LINEBREAK);
    PRUint32 retval;
    fileStream->Write(line.get(), line.Length(), &retval);
  }
  NS_FREE_XPCOM_ALLOCATED_POINTER_ARRAY(prefCount, prefArray);
  
  rv = fileStream->Flush();
  fileStream->Close();

  return rv;
}

NS_IMETHODIMP
mozSRoaming::EndSession()
{
  //printf("\n\n\n!!!! endsession\n\n\n\n");
  nsresult rv = NS_OK;

  rv = GenerateLiPrefs();
  NS_ENSURE_SUCCESS(rv, rv);

  if (!mHavePrefs)
    rv = ReadRoamingPrefs();
  if (NS_FAILED(rv))
    return rv;

  if (!mIsRoaming || mSavePreference)
    return NS_OK;

  mozSRoamingProtocol* proto = CreateMethodHandler();
  if (!proto)
    return NS_ERROR_ABORT;

  rv = proto->Init(this);
  if (NS_FAILED(rv))
  {
    //printf("error 0x%x\n", rv);
    return rv;
  }

  PrefsDone();

  RestoreCloseNet(PR_TRUE);

  rv = proto->Upload();
  if (NS_FAILED(rv))
  {
    //printf("error 0x%x\n", rv);
    return rv;
  }

  RestoreCloseNet(PR_FALSE);

  delete proto;
  return NS_OK;
}

NS_IMETHODIMP
mozSRoaming::IsRoaming(PRBool *_retval)
{
  if (!mHavePrefs)
    ReadRoamingPrefs();

  *_retval = IsRoaming();
  return NS_OK;
}



/*
 * Public functions
 */

PRBool mozSRoaming::IsRoaming()
{
  return mIsRoaming;
}

nsCStringArray* mozSRoaming::FilesToRoam()
{
  nsresult rv = NS_OK;
  nsCOMPtr<nsIPrefService> prefService = 
    do_GetService(NS_PREFSERVICE_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, nsnull);

  nsCOMPtr<nsIPrefBranch> prefBranch;
  rv = prefService->GetBranch("roaming.files.", getter_AddRefs(prefBranch));
  NS_ENSURE_SUCCESS(rv, nsnull);

  PRUint32 prefCount;
  char** prefNames;
  rv = prefBranch->GetChildList("", &prefCount, &prefNames);
  NS_ENSURE_SUCCESS(rv, nsnull);

  mFiles.Clear();
  PRInt32 i;
  for (i = 0; i < prefCount; i++) {
    PRBool enabled;
    rv = prefBranch->GetBoolPref(prefNames[i], &enabled);
    if (enabled) {
      nsCString fileName(prefNames[i]);
      mFiles.InsertCStringAt(fileName, 0);
    }
  }
  NS_FREE_XPCOM_ALLOCATED_POINTER_ARRAY(prefCount, prefNames);

  return &mFiles;
}

nsCOMPtr<nsIFile> mozSRoaming::ProfileDir()
{
  nsCOMPtr<nsIFile> result;
  NS_GetSpecialDirectory(NS_APP_USER_PROFILE_50_DIR,
                         getter_AddRefs(result));
  return result;
}

PRInt32 mozSRoaming::Method()
{
  return mMethod;
}

nsresult
mozSRoaming::ConflictResolveUI(PRBool download, const nsCStringArray& files,
                               nsCStringArray* result)
{
  //printf("mozSRoaming::CheckConflicts\n");
  //nsCStringArray* result = new nsCStringArray();
  if (files.Count() < 1)
    return NS_OK;

  nsresult rv;
  nsCOMPtr<nsIWindowWatcher> windowWatcher
                             (do_GetService(NS_WINDOWWATCHER_CONTRACTID, &rv));
  if (NS_FAILED(rv))
    return rv;

  /* nsIDialogParamBlock is a method to pass ints and strings
     to and from XUL dialogs.
     To dialog (upon open)
       Int array
         Item 0: 1 = download, 2 = upload
         Item 1: Number of files (n below)
       String array
         Item 1..(n): Either filename or comma-separated (no spaces) string:
                      - filename
                      - last modified (unix time) server
                      - size (bytes) server
                      - last modified (unix time) local
                      - size (bytes) local
                      e.g. "bookmarks.html,100024563,325,100024535,245" or
                      "bookmarks.html"
     From dialog (upon close)
       Int array
         Item 0:      3 = OK, 4 = Cancel
         Item 1..(n): if OK:
                      1 = Use server version, 2 = Use local version.
                      For each file. Indices are the same as To/String
  */
  nsCOMPtr<nsIDialogParamBlock> ioParamBlock
             (do_CreateInstance("@mozilla.org/embedcomp/dialogparam;1", &rv));
  if (NS_FAILED(rv))
    return rv;

  // download/upload
  ioParamBlock->SetInt(0, download ? 1 : 2);

  // filenames
  ioParamBlock->SetInt(1, files.Count());
  PRInt32 i;
  for (i = files.Count() - 1; i >= 0; i--)
  {
    NS_ConvertASCIItoUCS2 filename(*files.CStringAt(i));
    ioParamBlock->SetString(i + 1, filename.get());
  }

  nsCOMPtr<nsIDOMWindow> window;
  rv = windowWatcher->OpenWindow(nsnull,
                                 kConflDlg,
                                 nsnull,
                                 "centerscreen,chrome,modal,titlebar",
                                 ioParamBlock,
                                 getter_AddRefs(window));
  if (NS_FAILED(rv))
    return rv;

  PRInt32 value = 0;
  ioParamBlock->GetInt(0, &value);
  //printf("got back: %s - %d\n", files.CStringAt(i)->get(), value);
  if (value != 3 && value != 4)
    return NS_ERROR_INVALID_ARG;
  if (value == 4) // cancel
  {
    //printf("Cancel clicked\n");
    return NS_ERROR_ABORT;
  }
  //printf("OK clicked\n");

  /* I am assuming that the sequence of iteration here is the same as in the
     last |for| statement. If that is not true, the indices gotten from
     param block will not match the array and we will interpret the result
     wrongly. */
  for (i = files.Count() - 1; i >= 0; i--)
  {
    ioParamBlock->GetInt(i + 1, &value);
    //printf("got back: %s - %d\n", files.CStringAt(i)->get(), value);
    if (value != 1 && value != 2)
      return NS_ERROR_INVALID_ARG;
    if (download
        ? value == 1
        : value == 2)
      result->AppendCString(*files.CStringAt(i));
  }

  //*result = files;
  //printf("CheckConflicts done\n");
  return NS_OK;
}


nsresult mozSRoaming::GetRegistry(nsCOMPtr<nsIRegistry>& result)
{
  if (mRegistry)
  {
    result = mRegistry;
    return NS_OK;
  }

  nsresult rv;
  nsCOMPtr<nsIRegistry> registry(do_CreateInstance(NS_REGISTRY_CONTRACTID,
                                                   &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  rv = registry->OpenWellKnownRegistry(nsIRegistry::ApplicationRegistry);
  NS_ENSURE_SUCCESS(rv, rv);

  mRegistry = registry;
  result = registry;
  return NS_OK;
}


nsresult mozSRoaming::GetRegistryTree(nsRegistryKey& result)
{
  nsRegistryKey regkey = 0;

  nsresult rv;
  nsCOMPtr<nsIProfile> profMan(do_GetService(NS_PROFILE_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  nsXPIDLString profile;
  rv = profMan->GetCurrentProfile(getter_Copies(profile));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIRegistry> registry;
  rv = GetRegistry(registry);
  NS_ENSURE_SUCCESS(rv, rv);

  rv = registry->GetKey(nsIRegistry::Common,
                        kRegTreeProfile.get(),
                        &regkey);
  NS_ENSURE_SUCCESS(rv, rv);

  rv = registry->GetKey(regkey,
                        profile.get(),
                        &regkey);
  NS_ENSURE_SUCCESS(rv, rv);

  rv = registry->GetKey(regkey,
                        kRegTreeRoaming.get(),
                        &regkey);
  NS_ENSURE_SUCCESS(rv, rv);

  result = regkey;
  return NS_OK;
}

/*
 * Internal functions
 */

nsresult mozSRoaming::ReadRoamingPrefs()
{
  //printf("mozSRoaming::ReadRoamingPrefs\n");
  nsresult rv = NS_OK;

  nsCOMPtr<nsIPrefBranch> pref(do_GetService(NS_PREFSERVICE_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  rv = pref->GetBoolPref(kPrefEnabled, &mIsRoaming);

  rv = pref->GetBoolPref(kPrefSavePreference, &mSavePreference);
  rv = pref->SetBoolPref(kPrefSavePreference, PR_FALSE);
  
  //printf("roaming enabled: %s\n", mIsRoaming?"Yes":"No");
  if (!mIsRoaming)
    return NS_OK;

  // Method
  nsXPIDLCString method;
  rv = pref->GetCharPref(kPrefMethod, getter_Copies(method));

  if (NS_FAILED(rv))
  {
    //printf("registry read of Method failed: error 0x%x\n", rv);
    return rv;
  }
  if (0 == strcmp(method.get(), kPrefMethodStream))
    mMethod = 1;
  else if (0 == strcmp(method.get(), kPrefMethodCopy))
    mMethod = 2;
  else
    mMethod = 1;
  //printf("method: %d\n", mMethod);
  //printf("leaving readprefs\n");
  return NS_OK;
}

void mozSRoaming::PrefsDone()
{
  //mPrefService->ResetPrefs();
}


mozSRoamingProtocol* mozSRoaming::CreateMethodHandler()
{
  //printf("mozSRoaming::CreateMethodHandler\n");
  if (mMethod == 1)
    return new mozSRoamingStream;
  else if (mMethod == 2)
    return new mozSRoamingCopy;
  else // 0=unknown, e.g. prefs not yet read, or invalid
    return 0;
}

/* A workaround for the fact that the network library is shut down during
   profile shutdown (for dynamic profile changing), *before* the
   app is told to write down profile changes and thus before we try to upload,
   so our upload will fail.
   So, I just power netlib up again and then power it down again by sending
   the corresponding profile change notification. Might not be the
   best solution, but has small impact on existing code and WFM. */
nsresult mozSRoaming::RestoreCloseNet(PRBool restore)
{
  const char* topic = restore ? "profile-change-net-restore"
                              : "profile-change-net-teardown";
  //printf("%s...\n", topic);
  nsresult rv;
  nsCOMPtr<nsIObserverService> observerService
                       (do_GetService("@mozilla.org/observer-service;1", &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsISupports> subject(do_GetService(NS_PROFILE_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  rv = observerService->NotifyObservers(subject, topic,
                                        NS_LITERAL_STRING("switch").get());
  //printf("%s done\n", topic);
  return rv;
}
