/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Roaming code.
 *
 * The Initial Developer of the Original Code is 
 * Ben Bucksch <http://www.bucksch.org> of
 * Beonex <http://www.beonex.com>
 * Portions created by the Initial Developer are Copyright (C) 2002-2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "mozSRoamingCopy.h"
#include "nsIComponentManager.h"
#include "nsILocalFile.h"
#include "nsXPIDLString.h"

#define kPrefRemote "roaming.copy.remoteDir"

// Internal helper functions unrelated to class

/* @param fileSubPath  works for subpaths or just filenames?
                       doesn't really matter for us. */
nsresult CopyFile(nsCOMPtr<nsIFile> fromDir,
                  nsCOMPtr<nsIFile> toDir,
                  nsAString& fileSubPath)
{
  nsresult rv;

  nsCOMPtr<nsIFile> fromFile;
  rv = fromDir->Clone(getter_AddRefs(fromFile));
  if (NS_FAILED(rv))
    return rv;
  rv = fromFile->Append(fileSubPath);
  if (NS_FAILED(rv))
    return rv;

  nsCOMPtr<nsIFile> toFileOld;
  rv = toDir->Clone(getter_AddRefs(toFileOld));
  if (NS_FAILED(rv))
    return rv;
  rv = toFileOld->Append(fileSubPath);
  if (NS_FAILED(rv))
    return rv;

  PRBool exists;
  rv = fromFile->Exists(&exists);
  if (NS_FAILED(rv))
    return rv;

  nsXPIDLCString path1, path2;
  fromFile->GetNativePath(path1);
  toFileOld->GetNativePath(path2);
  //printf("trying to copy from -%s- to -%s-\n", path1.get(), path2.get());

  if (exists)
  {
    rv = toFileOld->Remove(PR_FALSE);  // XXX needed?
    //if (NS_FAILED(rv))
    //    printf("couldn't remove\n");
    rv = fromFile->CopyTo(toDir, fileSubPath);
  }
  else
  {
    //printf("source file not found\n");
    rv = NS_ERROR_FILE_NOT_FOUND;
  }
  return rv;
}

void AppendElementsToStrArray(nsCStringArray& target, nsCStringArray& source)
{
  for (PRInt32 i = source.Count() - 1; i >= 0; i--)
    target.AppendCString(*source.CStringAt(i));
}


mozSRoamingCopy::mozSRoamingCopy()
{
  //printf("mozSRoamingCopy ctor\n");
}

mozSRoamingCopy::~mozSRoamingCopy()
{
  //printf("mozSRoamingCopy dtor\n");
}



/*
 * nsSRoamingProtocol implementation
 */

nsresult mozSRoamingCopy::Init(mozSRoaming* aController)
{
  //printf("mozSRoamingCopy::Init\n");
  nsresult rv;
  mController = aController;
  if (!mController)
    return NS_ERROR_INVALID_ARG;

  nsCOMPtr<nsIPrefBranch> pref(do_GetService(NS_PREFSERVICE_CONTRACTID, &rv));
  NS_ENSURE_SUCCESS(rv, rv);

  nsXPIDLCString remoteDirPref;
  rv = pref->GetCharPref(kPrefRemote, getter_Copies(remoteDirPref));
  if (NS_FAILED(rv))
  {
    //printf("registry read of remote dir failed: error 0x%x\n", rv);
    return rv;
  }

  nsCOMPtr<nsILocalFile> lf;  // getting around dumb getter
  rv = NS_NewNativeLocalFile(remoteDirPref, PR_FALSE,
                             getter_AddRefs(lf));
  if (NS_FAILED(rv))
    return rv;
  mRemoteDir = do_QueryInterface(lf, &rv);
  if (NS_FAILED(rv))
    return rv;

  nsXPIDLCString path;
  rv = lf->GetNativePath(path);
  if (NS_FAILED(rv))
    return rv;
  //printf("remote dir: -%s-\n", path.get());

  mProfileDir = mController->ProfileDir();
  if (!mProfileDir)
    return NS_ERROR_FILE_NOT_FOUND;

  mProfileDir->GetNativePath(path);
  //printf("profile dir: -%s-\n", path.get());

  return NS_OK;
}

nsresult mozSRoamingCopy::Download()
{
  //printf("mozSRoamingCopy::Download\n");
  return DownUpLoad(PR_TRUE);
}

nsresult mozSRoamingCopy::Upload()
{
  //printf("mozSRoamingCopy::Upload\n");
  return DownUpLoad(PR_FALSE);
}

nsresult mozSRoamingCopy::DownUpLoad(PRBool download)
{
  nsresult rv = NS_OK;

  nsCStringArray* files = mController->FilesToRoam();
  NS_ENSURE_TRUE(files, NS_ERROR_FAILURE);
  //printf("got %d files\n", files->Count());

  // Check for conflicts
  nsCStringArray conflicts(10);
  nsCStringArray copyfiles(10);
  PRInt32 i;
  for (i = files->Count() - 1; i >= 0; i--)
  {
    nsCString& file = *files->CStringAt(i);
    //printf("checking file -%s-\n", file.get());
    NS_ConvertASCIItoUCS2 fileL(file);

    //local part
    nsCOMPtr<nsIFile> tmp;
    rv = mProfileDir->Clone(getter_AddRefs(tmp));
    nsCOMPtr<nsILocalFile> profileFile(do_QueryInterface(tmp));
    if (NS_FAILED(rv))
      return rv;
    rv = profileFile->AppendRelativePath(fileL);
    if (NS_FAILED(rv))
      return rv;
    nsAutoString filename;
    rv = profileFile->GetLeafName(filename);
    if (NS_FAILED(rv))
      return rv;
    //remote part
    nsCOMPtr<nsIFile> tmp2;
    rv = mRemoteDir->Clone(getter_AddRefs(tmp2));
    nsCOMPtr<nsILocalFile> remoteFile(do_QueryInterface(tmp2));
    if (NS_FAILED(rv))
      return rv;
    rv = remoteFile->AppendRelativePath(filename);
    if (NS_FAILED(rv))
      return rv;
    // avoid conflicts for missing files
    PRBool remoteExists = PR_TRUE;
    PRBool profileExists = PR_TRUE;
    remoteFile->Exists(&remoteExists);
    profileFile->Exists(&profileExists);
    if (download)
    {
      if (!remoteExists)
        continue;
      else if (!remoteExists)
      {
        copyfiles.AppendCString(file);
        continue;
        /* actually, this code is not needed given how the last modified
           code below works, but for readability and just in case... */
      }
    }
    else
    {
      if (!profileExists)
        continue;
      else if (!remoteExists)
      {
        copyfiles.AppendCString(file);
        continue;
      }
    }

    PRInt64 profileTime = 0;
    PRInt64 remoteTime = 0;
    profileFile->GetLastModifiedTime(&profileTime);
    remoteFile->GetLastModifiedTime(&remoteTime);
    //printf("mod time profile: %qd, remote: %qd\n", profileTime, remoteTime);
    // do we have a conflict?
    if (download
        ? profileTime > remoteTime
        : profileTime < remoteTime )
    {
      //printf("conflict found!\n");
      PRInt64 profileSize = 0;
      PRInt64 remoteSize = 0;
      profileFile->GetFileSize(&profileSize);
      remoteFile->GetFileSize(&remoteSize); 
      char buffer[256];
      sprintf(buffer, ",%lld,%lld,%lld,%lld", 
              remoteTime, remoteSize,
              profileTime, profileSize);      
      file.Append(buffer);
      conflicts.AppendCString(file);
    }
    else
      copyfiles.AppendCString(file);
  }

  // Ask user about conflicts
  nsCStringArray copyfiles_conflicts(10);
  rv = mController->ConflictResolveUI(download, conflicts,
                                      &copyfiles_conflicts);
  if (NS_FAILED(rv))
    return rv;
  AppendElementsToStrArray(copyfiles, copyfiles_conflicts);

  // Copy
  //printf("copying %d files\n", copyfiles.Count());
  for (i = copyfiles.Count() - 1; i >= 0; i--)
  {
    nsCString& file = *copyfiles.CStringAt(i);
    PRInt32 offset = file.FindChar(',');
    nsCOMPtr<nsIFile> localDir;
    rv = mProfileDir->Clone(getter_AddRefs(localDir));
    if (strstr(file.get(), "ImapMail")) { //crack for msgFilterRules.dat
      nsCOMPtr<nsILocalFile> tmp(do_QueryInterface(localDir));
      nsAutoString filename(ToNewUnicode(file));
      rv = tmp->AppendRelativePath(filename);
      nsCOMPtr<nsIFile> tmp2(do_QueryInterface(tmp));
      rv = tmp2->GetParent(getter_AddRefs(localDir));
      file.Assign("msgFilterRules.dat");
      if (NS_FAILED(rv))
        return rv;
    }
    NS_ConvertASCIItoUCS2 fileL(Substring(file, 0, offset));
    if (download)
      rv = CopyFile(mRemoteDir, localDir, fileL);
    else
      rv = CopyFile(localDir, mRemoteDir, fileL);
    //if (NS_FAILED(rv))
    //    printf("Copy of file -%s- failed!\n", file.get());
  }

  return rv;
}
