/*
 * (C) Copyright The MITRE Corporation 1999  All rights reserved.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * The program provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * Please see release.txt distributed with this file for more information.
 *
 */

/**
 * XSL expression class definitions.
 * Much of this code was ported from XSL:P. <BR />
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * <BR/>
 * <PRE>
 * Modifications:
 * 19990806: Larry Fitzpatrick
 *   - changed constant short declarations in many of the classes
 *     with enumerations, commented with //--LF
 * </PRE>
 *
**/

#ifndef MITREXSL_EXPR_H
#define MITREXSL_EXPR_H

#include <math.h>
#include "String.h"
#include "ErrorObserver.h"
#include "NodeSet.h"
#include "List.h"
#include "Stack.h"
#include "dom.h"
#include "ExprResult.h"
#include "baseutils.h"
#include "MITREObject.h"
#include "primitives.h"

class ContextState : public ErrorObserver {
public:
     /**
      * Returns the value of a given variable binding within the current scope
      * @param the name to which the desired variable value has been bound
      * @return the ExprResult which has been bound to the variable with the given
      * name
     **/
    virtual ExprResult* getVariable(String& name) = 0;

    /**
     * Returns the Stack of context NodeSets
     * @return the Stack of context NodeSets
    **/
    virtual Stack* getNodeSetStack() = 0;

    virtual MBool isStripSpaceAllowed(Node* node) = 0;

}; //-- ContextState

/**
 * A Base Class for all XSL Expressions
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class Expr : public MITREObject {

public:

    /**
     * Virtual destructor, important for subclasses
    **/
    virtual ~Expr() {};

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs) = 0;

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str) = 0;

}; //-- Expr


/**
 * A base Pattern class
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class Pattern {

public:

    /**
     * Virtual destructor, important for subclasses
    **/
    virtual ~Pattern() {};

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs) = 0;

    /**
     * Determines whether this Pattern matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs) = 0;


    /**
     * Returns the String representation of this Pattern.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Pattern.
    **/
    virtual void toString(String& dest) = 0;

}; //-- Pattern


/**
 * A Base class for all Expressions and Patterns
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class PatternExpr :
    public Expr,
    public Pattern
{

public:

    /**
     * Virtual destructor, important for subclasses
    **/
    virtual ~PatternExpr() {};

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs) = 0;

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs) = 0;

    /**
     * Determines whether this PatternExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs) = 0;

    /**
     * Returns the String representation of this PatternExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PatternExpr.
    **/
    virtual void toString(String& dest) = 0;

}; //-- PatternExpr

/**
 * Represents an AttributeValueTemplate
**/
class AttributeValueTemplate: public Expr {

public:

    AttributeValueTemplate();

    virtual ~AttributeValueTemplate();

    /**
     * Adds the given Expr to this AttributeValueTemplate
    **/
    void addExpr(Expr* expr);

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);

private:
    List expressions;
};


/**
 * This class represents a NodeTestExpr as defined by the XSL
 * Working Draft
 * <PRE>
 * </PRE>
 * @author Keith Visco (kvisco@mitre.org)
 * This file was ported from XSL:P
**/
class NodeExpr : public PatternExpr {

public:

    //-- NodeExpr Types
    //-- LF - changed from const short to enum
    enum NodeExprType {
        ATTRIBUTE_EXPR =  1,
        ELEMENT_EXPR,
        TEXT_EXPR,
        COMMENT_EXPR,
        PI_EXPR,
        NODE_EXPR,
        WILD_CARD,
        ROOT_EXPR
    };

    virtual ~NodeExpr() {};

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs) = 0;

    /**
     * Returns the type of this NodeExpr
     * @return the type of this NodeExpr
    **/
    virtual short getType() = 0;

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs) = 0;

    /**
     * Returns the String representation of this Pattern.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Pattern.
    **/
    virtual void toString(String& dest) = 0;

}; //-- NodeExpr

/**
 * This class represents a AttributeExpr as defined by the XSL
 * Working Draft
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * This file was ported from XSL:P
**/
class AttributeExpr : public NodeExpr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    AttributeExpr();
    AttributeExpr(String& name);
    virtual ~AttributeExpr();

    void setWild(MBool isWild);

    /**
     * Returns the name of this AttributeExpr
     * @return the name of this AttributeExpr
    **/
    const String& getName();

    /**
     * Sets the name of this AttributeExpr
     * @param name the name of the element that this AttributeExpr matches
    **/
    void setName(const String& name);

    //-----------------------------------/
   //- Method signatures from NodeExpr -/
  //-----------------------------------/

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the type of this NodeExpr
     * @return the type of this NodeExpr
    **/
    short getType();

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

private:

    String name;
    MBool  isWild;

}; //-- AttributeExpr

/**
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * This file was ported from XSL:P
**/
class BasicNodeExpr : public NodeExpr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Creates a new BasicNodeExpr of type NodeExpr::NODE_EXPR, which matches
     * any node
    **/
    BasicNodeExpr();

    /**
     * Creates a new BasicNodeExpr of the given type
    **/
    BasicNodeExpr::BasicNodeExpr(NodeExpr::NodeExprType nodeExprType);

    /**
     * Destroys this BasicNodeExpr
    **/
    virtual ~BasicNodeExpr();

    //-----------------------------------/
   //- Method signatures from NodeExpr -/
  //-----------------------------------/

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the type of this NodeExpr
     * @return the type of this NodeExpr
    **/
    short getType();

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

private:
    NodeExpr::NodeExprType type;
}; //-- BasicNodeExpr

/**
 * This class represents a ElementExpr as defined by the XSL
 * Working Draft
 * <PRE>
 * </PRE>
 * @author Keith Visco (kvisco@mitre.org)
 * This file was ported from XSL:P
**/
class ElementExpr : public NodeExpr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    ElementExpr();
    ElementExpr(String& name);
    virtual ~ElementExpr();

    /**
     * Returns the name of this ElementExpr
     * @return the name of this ElementExpr
    **/
    const String& getName();

    /**
     * Sets the name of this ElementExpr
     * @param name the name of the element that this ElementExpr matches
    **/
    void setName(const String& name);

    //-----------------------------------/
   //- Method signatures from NodeExpr -/
  //-----------------------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the type of this NodeExpr
     * @return the type of this NodeExpr
    **/
    short getType();

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

private:

    String name;

}; //-- ElementExpr

/**
 * This class represents a IdentityExpr, which only matches a node
 * if it is equal to the context node
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * <BR>This class was ported from XSL:P, an open source Java based XSL processor
**/
class IdentityExpr : public Expr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

}; //-- IdentityExpr

/**
 * This class represents a ParentExpr, which only selects a node
 * if it is equal to the context node's parent
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * <BR>This class was ported from XSL:P, an open source Java based XSL processor
**/
class ParentExpr : public Expr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);


    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

}; //-- ParentExpr

/**
 * This class represents a TextExpr, which only matches any text node
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * <BR>This class was ported from XSL:P, an open source Java based XSL processor
**/
class TextExpr : public NodeExpr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the type of this NodeExpr
     * @return the type of this NodeExpr
    **/
    virtual short getType();

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);


    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

}; //-- TextExpr

/**
 * This class represents a WildCardExpr as defined by the XSL
 * Working Draft
 * <PRE>
 * </PRE>
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * <BR>This class was ported from XSL:P, an open source Java based XSL processor
**/
class WildCardExpr : public NodeExpr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the type of this NodeExpr
     * @return the type of this NodeExpr
    **/
    virtual short getType();

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);


    /**
     * Returns the String representation of this NodeExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this NodeExpr.
    **/
    virtual void toString(String& dest);

}; //-- WildCardExpr


/**
 * Represents an ordered list of Predicates,
 * for use with Step and Filter Expressions
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class PredicateList  {

public:

    /**
     * Creates a new PredicateList
    **/
    PredicateList();
    /**
     * Destructor, will delete all Expressions in the list, so remove
     * any you may need
    **/
    virtual ~PredicateList();

    /**
     * Adds the given Expr to the list
     * @param expr the Expr to add to the list
    **/
    void add(Expr* expr);


    void evaluatePredicates(NodeSet* nodes, ContextState* cs);

    /**
     * returns true if this predicate list is empty
    **/
    MBool isEmpty();

    /**
     * Removes the given Expr from the list
     * @param expr the Expr to remove from the list
    **/
    Expr* remove(Expr* expr);

    /**
     * Returns the String representation of this PredicateList.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PredicateList.
    **/
    virtual void toString(String& dest);

private:
    //-- list of predicates
    List predicates;
}; //-- PredicateList

class LocationStep : public PredicateList, public PatternExpr {

public:

    // Axis Identifier Types
    //-- LF changed from static const short to enum
    enum _LocationStepType {
        ANCESTOR_AXIS = 0,
        ANCESTOR_OR_SELF_AXIS,
        ATTRIBUTE_AXIS,
        CHILD_AXIS,
        DESCENDANT_AXIS,
        DESCENDANT_OR_SELF_AXIS,
        FOLLOWING_AXIS,
        FOLLOWING_SIBLING_AXIS,
        NAMESPACE_AXIS,
        PARENT_AXIS,
        PRECEDING_AXIS,
        PRECEDING_SIBLING_AXIS,
        SELF_AXIS
    };


    /**
     * Creates a new LocationStep using the default Axis Identifier and no
     * NodeExpr (which matches nothing)
    **/
    LocationStep();

    /**
     * Creates a new LocationStep using the default Axis Identifier and
     * the given NodeExpr
     * @param nodeExpr the NodeExpr to use when matching Nodes
    **/
    LocationStep(NodeExpr* nodeExpr);

    /**
     * Creates a new LocationStep using the given NodeExpr and Axis Identifier
     * @param nodeExpr the NodeExpr to use when matching Nodes
     * @param axisIdentifier the Axis Identifier in which to search for nodes
    **/
    LocationStep(NodeExpr* nodeExpr, short axisIdentifier);

    /**
     * Destructor, will delete all predicates and the given NodeExpr
    **/
    virtual ~LocationStep();

    /**
     * Sets the Axis Identifier for this LocationStep
     * @param axisIdentifier the Axis in which to search for nodes
    **/
    void setAxisIdentifier(short axisIdentifier);

    /**
     * Sets the NodeExpr of this LocationStep for use when matching nodes
     * @param nodeExpr the NodeExpr to use when matching nodes
    **/
    void setNodeExpr(NodeExpr* nodeExpr);

      //------------------------------------/
     //- Virtual methods from PatternExpr -/
    //------------------------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Determines whether this PatternExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this PatternExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PatternExpr.
    **/
    virtual void toString(String& dest);

private:

  NodeExpr* nodeExpr;
  short     axisIdentifier;

    void fromDescendants(Node* context, ContextState* cs, NodeSet* nodes);

}; //-- LocationStep


class FilterExpr : public PredicateList, public PatternExpr {

public:

    /**
     * Creates a new FilterExpr using the default no default Expr
     * (which evaluates to nothing)
    **/
    FilterExpr();

    /**
     * Creates a new FilterExpr using the given Expr
     * @param expr the Expr to use for evaluation
    **/
    FilterExpr(Expr* expr);

    /**
     * Destructor, will delete all predicates and the given Expr
    **/
    virtual ~FilterExpr();

    /**
     * Sets the Expr of this FilterExpr for evaluation
     * @param expr the Expr to use for evaluation
    **/
    void setExpr(Expr* expr);

      //------------------------------------/
     //- Virtual methods from PatternExpr -/
    //------------------------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Determines whether this PatternExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this PatternExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PatternExpr.
    **/
    virtual void toString(String& dest);

private:

  Expr*     expr;

}; //-- FilterExpr


class NumberExpr : public Expr {

public:

    NumberExpr();
    NumberExpr(double dbl);
    ~NumberExpr();

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);

private:

    NumberResult numberResult;
};

/**
 * Represents a String expression
**/
class StringExpr : public Expr {

public:

    StringExpr();
    StringExpr(String& value);
    StringExpr(const String& value);
    ~StringExpr();

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);

private:

    String value;
}; //-- StringExpr

/**
 * Represents an AdditiveExpr, a binary expression that
 * performs an additive operation between it's lvalue and rvalue:<BR/>
 *  +   : add
 *  -   : subtract
**/
class AdditiveExpr : public Expr {

public:

    //-- AdditiveExpr Types
    //-- LF, changed from static const short to enum
    enum _AdditiveExprType { ADDITION = 1, SUBTRACTION };

     AdditiveExpr();
     AdditiveExpr(Expr* leftExpr, Expr* rightExpr, short op);
     ~AdditiveExpr();

    /**
     * Sets the left side of this AdditiveExpr
    **/
    void setLeftExpr(Expr* leftExpr);

    /**
     * Sets the right side of this AdditiveExpr
    **/
    void setRightExpr(Expr* rightExpr);


    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);



private:
    short op;
    Expr* leftExpr;
    Expr* rightExpr;
}; //-- AdditiveExpr

/**
 * Represents a MultiplicativeExpr, a binary expression that
 * performs a multiplicative operation between it's lvalue and rvalue:<BR/>
 *  *   : multiply
 * mod  : modulus
 * div  : divide
 *
**/
class MultiplicativeExpr : public Expr {

public:

    //-- MultiplicativeExpr Types
    //-- LF, changed from static const short to enum
    enum _MultiplicativeExprType { DIVIDE = 1, MULTIPLY, MODULUS };

     MultiplicativeExpr();
     MultiplicativeExpr(Expr* leftExpr, Expr* rightExpr, short op);
     ~MultiplicativeExpr();

    /**
     * Sets the left side of this MultiplicativeExpr
    **/
    void setLeftExpr(Expr* leftExpr);

    /**
     * Sets the right side of this MultiplicativeExpr
    **/
    void setRightExpr(Expr* rightExpr);


    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);



private:
    short op;
    Expr* leftExpr;
    Expr* rightExpr;
}; //-- MultiplicativeExpr

/**
 * Represents a RelationalExpr, an expression that compares it's lvalue
 * to it's rvalue using:<BR/>
 * =  : equal to
 * <  : less than
 * >  : greater than
 * <= : less than or equal to
 * >= : greater than or equal to
 *
**/
class RelationalExpr : public Expr {

public:

    //-- RelationalExpr Types
    //-- LF, changed from static const short to enum
    enum _RelationalExprType {
        EQUAL = 1,
        NOT_EQUAL,
        LESS_THAN,
        GREATER_THAN,
        LESS_OR_EQUAL,
        GREATER_OR_EQUAL
    };

     RelationalExpr();
     RelationalExpr(Expr* leftExpr, Expr* rightExpr, short op);
     ~RelationalExpr();

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);



private:
    short op;
    Expr* leftExpr;
    Expr* rightExpr;
}; //-- RelationalExpr

/**
 * VariableRefExpr<BR>
 * Represents a variable reference ($refname)
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class VariableRefExpr : public Expr {

public:

    VariableRefExpr();
    VariableRefExpr(const String& name);
    VariableRefExpr(String& name);
    ~VariableRefExpr();

    /**
     * Sets the name of the variable of reference
    **/
    void setName(const String& name);

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the String representation of this Expr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this Expr.
    **/
    virtual void toString(String& str);

private:
    String name;

}; //-- VariableRefExpr

/**
 *  Represents a PathExpr
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class PathExpr : public PatternExpr {

public:

    //-- Path Operators
    //-- RELATIVE_OP is the default
    //-- LF, changed from static const short to enum
    enum _PathOperator { ANCESTOR_OP=1, PARENT_OP, RELATIVE_OP} ;

    /**
     * Creates a new PathExpr
    **/
    PathExpr();

    /**
     * Destructor, will delete all Pattern Expressions
    **/
    virtual ~PathExpr();

    /**
     * Adds the PatternExpr to this PathExpr
     * @param expr the Expr to add to this PathExpr
     * @param index the index at which to add the given Expr
    **/
    void PathExpr::addPatternExpr(int index, PatternExpr* expr, short ancestryOp);

    /**
     * Adds the PatternExpr to this PathExpr
     * @param expr the Expr to add to this PathExpr
    **/
    void addPatternExpr(PatternExpr* expr, short ancestryOp);

    virtual MBool isAbsolute();

      //------------------------------------/
     //- Virtual methods from PatternExpr -/
    //------------------------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Determines whether this PatternExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this PatternExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PatternExpr.
    **/
    virtual void toString(String& dest);

private:

    struct PathExprItem {
        PatternExpr* pExpr;
        short ancestryOp;
    };

   List expressions;

   /**
    * Selects from the descendants of the context node
    * all nodes that match the PatternExpr
    * -- this will be moving to a Utility class
   **/
   void fromDescendants(PatternExpr* pExpr,
                        Node* context,
                        ContextState* cs,
                        NodeSet* nodes);

}; //-- PathExpr

/**
 * This class represents a RootExpr, which only matches the Document node
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
 * <BR>This class was ported from XSL:P, an open source Java based XSL processor
**/
class RootExpr : public PathExpr {

public:

      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    virtual MBool isAbsolute();

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Determines whether this NodeExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this PatternExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PatternExpr.
    **/
    virtual void toString(String& dest);


}; //-- RootExpr

/**
 *  Represents a UnionExpr
 * @author <a href="mailto:kvisco@mitre.org">Keith Visco</a>
**/
class UnionExpr : public PatternExpr {

public:

    /**
     * Creates a new UnionExpr
    **/
    UnionExpr();

    /**
     * Destructor, will delete all Path Expressions
    **/
    virtual ~UnionExpr();

    /**
     * Adds the PathExpr to this UnionExpr
     * @param expr the Expr to add to this UnionExpr
    **/
    void addPathExpr(PathExpr* expr);

    /**
     * Adds the PathExpr to this UnionExpr at the specified index
     * @param expr the Expr to add to this UnionExpr
    **/
    void addPathExpr(int index, PathExpr* expr);

      //------------------------------------/
     //- Virtual methods from PatternExpr -/
    //------------------------------------/

    /**
     * Evaluates this Expr based on the given context node and processor state
     * @param context the context node for evaluation of this Expr
     * @param ps the ContextState containing the stack information needed
     * for evaluation
     * @return the result of the evaluation
    **/
    virtual ExprResult* evaluate(Node* context, ContextState* cs);

    /**
     * Returns the default priority of this Pattern based on the given Node,
     * context Node, and ContextState.
     * If this pattern does not match the given Node under the current context Node and
     * ContextState then Negative Infinity is returned.
    **/
    virtual double getDefaultPriority(Node* node, Node* context, ContextState* cs);

    /**
     * Determines whether this PatternExpr matches the given node within
     * the given context
    **/
    virtual MBool matches(Node* node, Node* context, ContextState* cs);

    /**
     * Returns the String representation of this PatternExpr.
     * @param dest the String to use when creating the String
     * representation. The String representation will be appended to
     * any data in the destination String, to allow cascading calls to
     * other #toString() methods for Expressions.
     * @return the String representation of this PatternExpr.
    **/
    virtual void toString(String& dest);

private:

   List expressions;

}; //-- UnionExpr

/* */
#endif


