/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *
 * The contents of this file are subject to the Netscape Public License
 * Version 1.0 (the "NPL"); you may not use this file except in
 * compliance with the NPL.  You may obtain a copy of the NPL at
 * http://www.mozilla.org/NPL/
 *
 * Software distributed under the NPL is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the NPL
 * for the specific language governing rights and limitations under the
 * NPL.
 *
 * The Initial Developer of this code under the NPL is Netscape
 * Communications Corporation.  Portions created by Netscape are
 * Copyright (C) 1998 Netscape Communications Corporation.  All Rights
 * Reserved.
 */
#ifndef nsUnitConversion_h__
#define nsUnitConversion_h__

#include "nscore.h"
#include <math.h>

/// handy constants
#define TWIPS_PER_POINT_INT           20
#define TWIPS_PER_POINT_FLOAT         20.0f
#define ROUND_EXCLUSIVE_CONST_FLOAT   0.4999999999999999f   // XXX this should be derived from platform FLOAT_MIN
#define ROUND_CONST_FLOAT             0.5f
#define CEIL_CONST_FLOAT              0.9999999999999999f   // XXX this should be derived from platform FLOAT_MIN


/*
 * Coord Rounding Functions
 */
inline nscoord NSToCoordFloor(float aValue)
{
  return ((0.0f <= aValue) ? nscoord(aValue) : nscoord(aValue - CEIL_CONST_FLOAT));
}

inline nscoord NSToCoordCeil(float aValue)
{
  return ((0.0f <= aValue) ? nscoord(aValue + CEIL_CONST_FLOAT) : nscoord(aValue));
}

inline nscoord NSToCoordRound(float aValue)
{
  return ((0.0f <= aValue) ? nscoord(aValue + ROUND_CONST_FLOAT) : nscoord(aValue - ROUND_CONST_FLOAT));
}

inline nscoord NSToCoordRoundExclusive(float aValue)
{
  return ((0.0f <= aValue) ? nscoord(aValue + ROUND_EXCLUSIVE_CONST_FLOAT) :
                             nscoord(aValue - ROUND_EXCLUSIVE_CONST_FLOAT));
}


/*
 * Int Rounding Functions
 */
inline PRInt32 NSToIntFloor(float aValue)
{
  return ((0.0f <= aValue) ? PRInt32(aValue) : PRInt32(aValue - CEIL_CONST_FLOAT));
}

inline PRInt32 NSToIntCeil(float aValue)
{
  return ((0.0f <= aValue) ? PRInt32(aValue + CEIL_CONST_FLOAT) : PRInt32(aValue));
}

inline PRInt32 NSToIntRound(float aValue)
{
  return ((0.0f <= aValue) ? PRInt32(aValue + ROUND_CONST_FLOAT) : PRInt32(aValue - ROUND_CONST_FLOAT));
}

inline PRInt32 NSToIntRoundExclusive(float aValue)
{
  return ((0.0f <= aValue) ? PRInt32(aValue + ROUND_EXCLUSIVE_CONST_FLOAT) :
                             PRInt32(aValue - ROUND_EXCLUSIVE_CONST_FLOAT));
}


/* 
 * Twips/Points conversions
 */
inline nscoord NSFloatPointsToTwips(float aPoints)
{
  return NSToCoordRound(aPoints * TWIPS_PER_POINT_FLOAT);
}

inline nscoord NSIntPointsToTwips(PRInt32 aPoints)
{
  return nscoord(aPoints * TWIPS_PER_POINT_INT);
}

inline PRInt32 NSTwipsToIntPoints(nscoord aTwips)
{
  return NSToIntRound(aTwips / TWIPS_PER_POINT_FLOAT);
}

inline PRInt32 NSTwipsToFloorIntPoints(nscoord aTwips)
{
  return NSToIntFloor(aTwips / TWIPS_PER_POINT_FLOAT);
}

inline PRInt32 NSTwipsToCeilIntPoints(nscoord aTwips)
{
  return NSToIntCeil(aTwips / TWIPS_PER_POINT_FLOAT);
}

inline float NSTwipsToFloatPoints(nscoord aTwips)
{
  return (float(aTwips) / TWIPS_PER_POINT_FLOAT);
}

/* 
 * Twips/Pixel conversions
 */
inline nscoord NSFloatPixelsToTwips(float aPixels, float aTwipsPerPixel)
{
  return NSToCoordRound(aPixels * aTwipsPerPixel);
}

inline nscoord NSIntPixelsToTwips(PRInt32 aPixels, float aTwipsPerPixel)
{
  return NSToCoordRound(float(aPixels) * aTwipsPerPixel);
}

inline float NSTwipsToFloatPixels(nscoord aTwips, float aPixelsPerTwip)
{
  return (float(aTwips) * aPixelsPerTwip);
}

inline PRInt32 NSTwipsToIntPixels(nscoord aTwips, float aPixelsPerTwip)
{
  return NSToIntRound(float(aTwips) * aPixelsPerTwip);
}

/* 
 * Twips/unit conversions
 */
inline nscoord NSUnitsToTwips(float aValue, float aPointsPerUnit)
{
  return NSToCoordRound(aValue * aPointsPerUnit * TWIPS_PER_POINT_FLOAT);
}

inline float NSTwipsToUnits(nscoord aTwips, float aUnitsPerPoint)
{
  return (aTwips * (aUnitsPerPoint / TWIPS_PER_POINT_FLOAT));
}


/// Unit conversion macros
//@{
#define NS_INCHES_TO_TWIPS(x)         NSUnitsToTwips((x), 72.0f)                      // 72 points per inch
#define NS_FEET_TO_TWIPS(x)           NSUnitsToTwips((x), (72.0f * 12.0f))            // 12 inches per foot
#define NS_MILES_TO_TWIPS(x)          NSUnitsToTwips((x), (72.0f * 12.0f * 5280.0f))  // 5280 feet per mile

#define NS_MILLIMETERS_TO_TWIPS(x)    NSUnitsToTwips((x), (72.0f * 0.03937f))
#define NS_CENTIMETERS_TO_TWIPS(x)    NSUnitsToTwips((x), (72.0f * 0.3937f))
#define NS_METERS_TO_TWIPS(x)         NSUnitsToTwips((x), (72.0f * 39.37f))
#define NS_KILOMETERS_TO_TWIPS(x)     NSUnitsToTwips((x), (72.0f * 39370.0f))

#define NS_PICAS_TO_TWIPS(x)          NSUnitsToTwips((x), 12.0f)                      // 12 points per pica
#define NS_DIDOTS_TO_TWIPS(x)         NSUnitsToTwips((x), (16.0f / 15.0f))            // 15 didots per 16 points
#define NS_CICEROS_TO_TWIPS(x)        NSUnitsToTwips((x), (12.0f * (16.0f / 15.0f)))  // 12 didots per cicero


#define NS_TWIPS_TO_INCHES(x)         NSTwipsToUnits((x), 1.0f / 72.0f)
#define NS_TWIPS_TO_FEET(x)           NSTwipsToUnits((x), 1.0f / (72.0f * 12.0f))
#define NS_TWIPS_TO_MILES(x)          NSTwipsToUnits((x), 1.0f / (72.0f * 12.0f * 5280.0f))

#define NS_TWIPS_TO_MILLIMETERS(x)    NSTwipsToUnits((x), 1.0f / (72.0f * 0.03937f))
#define NS_TWIPS_TO_CENTIMETERS(x)    NSTwipsToUnits((x), 1.0f / (72.0f * 0.3937f))
#define NS_TWIPS_TO_METERS(x)         NSTwipsToUnits((x), 1.0f / (72.0f * 39.37f))
#define NS_TWIPS_TO_KILOMETERS(x)     NSTwipsToUnits((x), 1.0f / (72.0f * 39370.0f))

#define NS_TWIPS_TO_PICAS(x)          NSTwipsToUnits((x), 1.0f / 12.0f)
#define NS_TWIPS_TO_DIDOTS(x)         NSTwipsToUnits((x), 1.0f / (16.0f / 15.0f))
#define NS_TWIPS_TO_CICEROS(x)        NSTwipsToUnits((x), 1.0f / (12.0f * (16.0f / 15.0f)))
//@}

#endif
