import pytest
from tests.support.asserts import assert_error, assert_success


@pytest.fixture
def check_doc(inline):
    return inline("""
    <input id=checked type=checkbox checked>
    <input id=notChecked type=checkbox>
    """)


@pytest.fixture
def option_doc(inline):
    return inline("""
    <select>
      <option id=notSelected>r-
      <option id=selected selected>r+
    </select>
    """)


def is_element_selected(session, element_id):
    return session.transport.send(
        "GET", "session/{session_id}/element/{element_id}/selected".format(
            session_id=session.session_id,
            element_id=element_id))


def test_no_top_browsing_context(session, closed_window):
    original_handle, element = closed_window

    response = is_element_selected(session, element.id)
    assert_error(response, "no such window")
    response = is_element_selected(session, "foo")
    assert_error(response, "no such window")

    session.window_handle = original_handle
    response = is_element_selected(session, element.id)
    assert_error(response, "no such element")


def test_no_browsing_context(session, closed_frame):
    response = is_element_selected(session, "foo")
    assert_error(response, "no such window")


def test_element_stale(session, check_doc):
    session.url = check_doc
    element = session.find.css("#checked", all=False)
    session.refresh()

    result = is_element_selected(session, element.id)
    assert_error(result, "stale element reference")


def test_element_checked(session, check_doc):
    session.url = check_doc
    element = session.find.css("#checked", all=False)

    result = is_element_selected(session, element.id)
    assert_success(result, True)


def test_checkbox_not_selected(session, check_doc):
    session.url = check_doc
    element = session.find.css("#notChecked", all=False)

    result = is_element_selected(session, element.id)
    assert_success(result, False)


def test_element_selected(session, option_doc):
    session.url = option_doc
    element = session.find.css("#selected", all=False)

    result = is_element_selected(session, element.id)
    assert_success(result, True)


def test_element_not_selected(session, option_doc):
    session.url = option_doc
    element = session.find.css("#notSelected", all=False)

    result = is_element_selected(session, element.id)
    assert_success(result, False)
