/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_dom_HTMLOptionElement_h__
#define mozilla_dom_HTMLOptionElement_h__

#include "mozilla/Attributes.h"
#include "nsGenericHTMLElement.h"
#include "mozilla/dom/HTMLFormElement.h"

namespace mozilla {
namespace dom {

class HTMLSelectElement;

class HTMLOptionElement final : public nsGenericHTMLElement
{
public:
  explicit HTMLOptionElement(already_AddRefed<mozilla::dom::NodeInfo>& aNodeInfo);

  static already_AddRefed<HTMLOptionElement>
    Option(const GlobalObject& aGlobal,
           const nsAString& aText,
           const Optional<nsAString>& aValue,
           bool aDefaultSelected,
           bool aSelected,
           ErrorResult& aError);

  NS_IMPL_FROMCONTENT_HTML_WITH_TAG(HTMLOptionElement, option)

  // nsISupports
  NS_DECL_ISUPPORTS_INHERITED

  using mozilla::dom::Element::SetText;
  using mozilla::dom::Element::GetText;

  bool Selected() const
  {
    return mIsSelected;
  }
  void SetSelected(bool aValue);


  void SetSelectedChanged(bool aValue)
  {
    mSelectedChanged = aValue;
  }

  virtual nsChangeHint GetAttributeChangeHint(const nsIAtom* aAttribute,
                                              int32_t aModType) const override;

  virtual nsresult BeforeSetAttr(int32_t aNamespaceID, nsIAtom* aName,
                                 const nsAttrValueOrString* aValue,
                                 bool aNotify) override;
  virtual nsresult AfterSetAttr(int32_t aNameSpaceID, nsIAtom* aName,
                                const nsAttrValue* aValue,
                                const nsAttrValue* aOldValue,
                                nsIPrincipal* aSubjectPrincipal,
                                bool aNotify) override;

  void SetSelectedInternal(bool aValue, bool aNotify);

  /**
   * This callback is called by an optgroup on all its option elements whenever
   * its disabled state is changed so that option elements can know their
   * disabled state might have changed.
   */
  void OptGroupDisabledChanged(bool aNotify);

  /**
   * Check our disabled content attribute and optgroup's (if it exists) disabled
   * state to decide whether our disabled flag should be toggled.
   */
  void UpdateDisabledState(bool aNotify);

  virtual nsresult BindToTree(nsIDocument* aDocument, nsIContent* aParent,
                              nsIContent* aBindingParent,
                              bool aCompileEventHandlers) override;
  virtual void UnbindFromTree(bool aDeep = true,
                              bool aNullParent = true) override;

  // nsIContent
  virtual EventStates IntrinsicState() const override;

  virtual nsresult Clone(mozilla::dom::NodeInfo* aNodeInfo, nsINode** aResult,
                         bool aPreallocateChildren) const override;

  nsresult CopyInnerTo(mozilla::dom::Element* aDest, bool aPreallocateChildren);

  virtual bool IsDisabled() const override {
    return State().HasState(NS_EVENT_STATE_DISABLED);
  }

  bool Disabled() const
  {
    return GetBoolAttr(nsGkAtoms::disabled);
  }

  void SetDisabled(bool aValue, ErrorResult& aRv)
  {
    SetHTMLBoolAttr(nsGkAtoms::disabled, aValue, aRv);
  }

  HTMLFormElement* GetForm();

  void GetLabel(DOMString& aLabel)
  {
    if (!GetAttr(kNameSpaceID_None, nsGkAtoms::label, aLabel)) {
      GetText(aLabel);
    }
  }
  void SetLabel(const nsAString& aLabel, ErrorResult& aError)
  {
    SetHTMLAttr(nsGkAtoms::label, aLabel, aError);
  }

  bool DefaultSelected() const
  {
    return HasAttr(kNameSpaceID_None, nsGkAtoms::selected);
  }
  void SetDefaultSelected(bool aValue, ErrorResult& aRv)
  {
    SetHTMLBoolAttr(nsGkAtoms::selected, aValue, aRv);
  }

  void GetValue(nsAString& aValue)
  {
    if (!GetAttr(kNameSpaceID_None, nsGkAtoms::value, aValue)) {
      GetText(aValue);
    }
  }
  void SetValue(const nsAString& aValue, ErrorResult& aRv)
  {
    SetHTMLAttr(nsGkAtoms::value, aValue, aRv);
  }

  void GetText(nsAString& aText);
  void SetText(const nsAString& aText, ErrorResult& aRv);

  int32_t Index();

protected:
  virtual ~HTMLOptionElement();

  virtual JSObject* WrapNode(JSContext* aCx, JS::Handle<JSObject*> aGivenProto) override;

  /**
   * Get the select content element that contains this option, this
   * intentionally does not return nsresult, all we care about is if
   * there's a select associated with this option or not.
   */
  HTMLSelectElement* GetSelect();

  bool mSelectedChanged;
  bool mIsSelected;

  // True only while we're under the SetOptionsSelectedByIndex call when our
  // "selected" attribute is changing and mSelectedChanged is false.
  bool mIsInSetDefaultSelected;
};

} // namespace dom
} // namespace mozilla

#endif // mozilla_dom_HTMLOptionElement_h__
