/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Netscape Security Services for Java.
 * 
 * The Initial Developer of the Original Code is Netscape
 * Communications Corporation.  Portions created by Netscape are 
 * Copyright (C) 1998-2000 Netscape Communications Corporation.  All
 * Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

package org.mozilla.jss.pkix.primitive;

import org.mozilla.jss.asn1.*;
import java.io.*;

/**
 * PKCS #5 <i>PBEParameter</i>, and PKCS #12 <i>pkcs-12PbeParams</i>. The only
 * difference between the two is that PKCS #5 dictates that the size of the
 * salt must be 8 bytes, while PKCS #12 leaves the salt length undefined.
 * To work with both standards, this class does not check the length of the
 * salt but rather leaves that to the application.
 */
public class PBEParameter implements ASN1Value {

    ///////////////////////////////////////////////////////////////////////
    // members and member access
    ///////////////////////////////////////////////////////////////////////
    private byte[] salt;
    private int iterations;
    private SEQUENCE sequence;

    public byte[] getSalt() {
        return salt;
    }

    public int getIterations() {
        return iterations;
    }

    ///////////////////////////////////////////////////////////////////////
    // constructors
    ///////////////////////////////////////////////////////////////////////
    private PBEParameter() { }

    /**
     * Creates a PBEParameter from a salt and iteration count. Neither
     * may be null.
     */
    public PBEParameter(byte[] salt, int iterations) {
        this.salt = salt;
        this.iterations = iterations;
        sequence = new SEQUENCE();
        sequence.addElement( new OCTET_STRING(salt) );
        sequence.addElement( new INTEGER(iterations) );
    }

    /**
     * Creates a PBEParameter from a salt and iteration count. Neither
     * may be null.
     */
    public PBEParameter(OCTET_STRING salt, INTEGER iterations) {
        this( salt.toByteArray(), iterations.intValue() );
    }

    ///////////////////////////////////////////////////////////////////////
    // DER encoding
    ///////////////////////////////////////////////////////////////////////

    private static final Tag TAG = SEQUENCE.TAG;
    public Tag getTag() {
        return TAG;
    }

    public void encode(OutputStream ostream) throws IOException {
        sequence.encode(ostream);
    }

    public void encode(Tag implicitTag, OutputStream ostream)
        throws IOException
    {
        sequence.encode(implicitTag, ostream);
    }


    private static final Template templateInstance = new Template();
    public static Template getTemplate() {
        return templateInstance;
    }

    /**
     * A template class for decoding a PBEParameter.
     */
    public static class Template implements ASN1Template {

        private SEQUENCE.Template seqt;

        public Template() {
            seqt = new SEQUENCE.Template();
            seqt.addElement( OCTET_STRING.getTemplate() );
            seqt.addElement( INTEGER.getTemplate() );
        }

        public boolean tagMatch(Tag tag) {
            return TAG.equals(tag);
        }

        public ASN1Value decode(InputStream istream)
            throws InvalidBERException, IOException
        {
            return decode(TAG, istream);
        }

        public ASN1Value decode(Tag implicitTag, InputStream istream)
            throws InvalidBERException, IOException
        {
            SEQUENCE seq = (SEQUENCE) seqt.decode(implicitTag, istream);

            return new PBEParameter( (OCTET_STRING) seq.elementAt(0),
                                     (INTEGER)      seq.elementAt(1) );
        }
    }
}
