// META: global=window,worker
// META: title=IndexedDB: Test IDBIndex.getAllKeys.
// META: script=resources/support.js

'use_strict';

const alphabet = 'abcdefghijklmnopqrstuvwxyz'.split('');

function getall_test(func, name) {
  indexeddb_test(
    function(t, connection, tx) {
      let store = connection.createObjectStore('generated',
            {autoIncrement: true, keyPath: 'id'});
      let index = store.createIndex('test_idx', 'upper');
      alphabet.forEach(function(letter) {
        store.put({ch: letter, upper: letter.toUpperCase()});
      });

      store = connection.createObjectStore('out-of-line', null);
      index = store.createIndex('test_idx', 'upper');
      alphabet.forEach(function(letter) {
        store.put({ch: letter, upper: letter.toUpperCase()}, letter);
      });

      store = connection.createObjectStore('out-of-line-multi', null);
      index = store.createIndex('test_idx', 'attribs', {multiEntry: true});
      alphabet.forEach(function(letter) {
        attrs = [];
        if (['a', 'e', 'i', 'o', 'u'].indexOf(letter) != -1)
          attrs.push('vowel');
        else
          attrs.push('consonant');
        if (letter == 'a')
          attrs.push('first');
        if (letter == 'z')
          attrs.push('last');
        store.put({ch: letter, attribs: attrs}, letter.toUpperCase());
      });

      store = connection.createObjectStore('empty', null);
      index = store.createIndex('test_idx', 'upper');
    },
    func,
    name
  );
}

function createGetAllKeysRequest(t, storeName, connection, range, maxCount) {
    const transaction = connection.transaction(storeName, 'readonly');
    const store = transaction.objectStore(storeName);
    const index = store.index('test_idx');
    const req = index.getAllKeys(range, maxCount);
    req.onerror = t.unreached_func('getAllKeys request should succeed');
    return req;
}

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection, 'C');
      req.onsuccess = t.step_func(function(evt) {
          const data = evt.target.result;
          assert_array_equals(evt.target.result, ['c']);
          t.done();
      });
    }, 'Single item get');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'empty', connection);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result, [],
              'getAllKeys() on empty object store should return empty array');
          t.done();
      });
    }, 'Empty object store');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result, alphabet,
              'getAllKeys() should return a..z');
          t.done();
      });
    }, 'Get all keys');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'generated', connection);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result,
              [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18,
               19, 20, 21, 22, 23, 24, 25, 26],
              'getAllKeys() should return 1..26');
          t.done();
      });
    }, 'Get all generated keys');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection, undefined,
                                    10);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result,
                             'abcdefghij'.split(''),
                             'getAllKeys() should return a..j');
          t.done();
      });
    }, 'maxCount=10');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection,
                                    IDBKeyRange.bound('G', 'M'));
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result,
                              'ghijklm'.split(''),
                              'getAllKeys() should return g..m');
          t.done();
      });
    }, 'Get bound range');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection,
                                    IDBKeyRange.bound('G', 'M'), 3);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result,
                             ['g', 'h', 'i'],
                             'getAllKeys() should return g..i');
          t.done();
      });
    }, 'Get bound range with maxCount');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection,
          IDBKeyRange.bound('G', 'K', false, true));
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result,
                             ['g', 'h', 'i', 'j'],
                             'getAllKeys() should return g..j');
          t.done();
      });
    }, 'Get upper excluded');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection,
          IDBKeyRange.bound('G', 'K', true, false));
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result,
                             ['h', 'i', 'j', 'k'],
                             'getAllKeys() should return h..k');
          t.done();
      });
    }, 'Get lower excluded');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'generated',
          connection, IDBKeyRange.bound(4, 15), 3);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result, [],
                              'getAllKeys() should return []');
          t.done();
      });
    }, 'Get bound range (generated) with maxCount');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line',
          connection, "Doesn't exist");
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result, [],
              'getAllKeys() using a nonexistent key should return empty array');
          t.done();
      req.onerror = t.unreached_func('getAllKeys request should succeed');
      });
    }, 'Non existent key');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line', connection,
          undefined, 0);
      req.onsuccess = t.step_func(function(evt) {
          assert_array_equals(evt.target.result, alphabet,
              'getAllKeys() should return a..z');
          t.done();
      });
    }, 'maxCount=0');

getall_test(function(t, connection) {
      const req = createGetAllKeysRequest(t, 'out-of-line-multi', connection,
                                        'vowel');
      req.onsuccess = t.step_func(function(evt) {
        assert_array_equals(evt.target.result, ['A','E','I','O','U'])
        t.done();
      });
      req.onerror = t.unreached_func('getAllKeys request should succeed');
    }, 'Retrieve multiEntry keys');
