/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */

"use strict";

// Globals
const { sinon } = ChromeUtils.importESModule(
  "resource://testing-common/Sinon.sys.mjs"
);

ChromeUtils.defineESModuleGetters(this, {
  ExperimentAPI: "resource://nimbus/ExperimentAPI.sys.mjs",
  ExperimentManager: "resource://nimbus/lib/ExperimentManager.sys.mjs",
  ExperimentTestUtils: "resource://testing-common/NimbusTestUtils.sys.mjs",
  ExperimentFakes: "resource://testing-common/NimbusTestUtils.sys.mjs",
});

add_setup(function () {
  let sandbox = sinon.createSandbox();

  /* We stub the functions that operate with enrollments and remote rollouts
   * so that any access to store something is implicitly validated against
   * the schema and no records have missing (or extra) properties while in tests
   */

  let origAddExperiment = ExperimentManager.store.addEnrollment.bind(
    ExperimentManager.store
  );
  sandbox
    .stub(ExperimentManager.store, "addEnrollment")
    .callsFake(enrollment => {
      ExperimentTestUtils.validateEnrollment(enrollment);
      return origAddExperiment(enrollment);
    });

  registerCleanupFunction(() => {
    sandbox.restore();
  });
});

async function setupTest() {
  await ExperimentAPI.ready();
  await ExperimentAPI._rsLoader.finishedUpdating();

  await ExperimentAPI._rsLoader.remoteSettingsClients.experiments.db.importChanges(
    {},
    Date.now(),
    [],
    { clear: true }
  );

  await ExperimentAPI._rsLoader.updateRecipes("test");

  return async function cleanup() {
    const store = ExperimentAPI._manager.store;

    store._store._saver.disarm();
    if (store._store._saver.isRunning) {
      await store._store._saver._runningPromise;
    }

    await IOUtils.remove(store._store.path);
  };
}

/**
 * Assert the store has no active experiments or rollouts.
 */
async function assertEmptyStore(store) {
  Assert.deepEqual(
    store
      .getAll()
      .filter(e => e.active)
      .map(e => e.slug),
    [],
    "Store should have no active enrollments"
  );

  store
    .getAll()
    .filter(e => !e.active)
    .forEach(e => store._deleteForTests(e.slug));

  Assert.deepEqual(
    store
      .getAll()
      .filter(e => !e.active)
      .map(e => e.slug),
    [],
    "Store should have no inactive enrollments"
  );

  store._store._saver.disarm();
  if (store._store._saver.isRunning) {
    await store._store._saver._runningPromise;
  }

  await IOUtils.remove(store._store.path);
}
