/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
"use strict";

/**
 * Build: <srcdir>/tools/@types/tspaths.json, and
 *        <srcdir>/tools/@types/lib.gecko.modules.d.ts
 *
 * from:  various import forms in all *.js and *.mjs sources.
 */

const fs = require("fs");
const { paths } = require("./config/fixed_paths.js");

const HEADER = `/**
 * NOTE: Do not modify this file by hand.
 * Content was generated by running "mach ts paths".
 */
`;

const IGNORE = [".git", ".hg", "node_modules", "test262"];
const IMPORT =
  /(\bimport |import\(|require\(|ChromeUtils\.(defineESModuleGetters?|importESModule)\()[^;]+/gm;
const URI = /("|')((resource|chrome|moz-src):\/\/[\w\d\/_.-]+\.m?js)\1/gm;

// Scan the root dir for *.js and *.mjs files, recursivelly.
function scan(root, dir, files) {
  for (let file of fs.readdirSync(`${root}/${dir}`, { withFileTypes: true })) {
    if (file.isDirectory() && !IGNORE.includes(file.name)) {
      scan(root, `${dir}${file.name}/`, files);
    } else if (file.name.match(/\.(x?html|m?js)$/)) {
      files.push(dir + file.name);
    }
  }
}

// Emit path mapping for all found module URIs.
function emitPaths(files, uris, modules) {
  for (let uri of [...uris].sort()) {
    if (uri in paths) {
      continue;
    }
    let parts = uri.split("/");
    let path = "";
    let matches;

    // Starting with just the file name, add each path part going backwards.
    do {
      path = "/" + parts.pop() + path;
      matches = files.filter(f => f.endsWith(path));
      // While multiple files match, add parts used for filtering.
    } while (matches.length > 1 && parts.length);

    // Unique match is almost certainy correct.
    if (matches.length === 1) {
      paths[uri] = [matches[0]];
    } else {
      // URI matched more than one, or failed to match any file.
      console.warn("[WARN]", uri);
      modules.delete(uri);
    }
  }

  let tspaths = { compilerOptions: { baseUrl: "../../", paths } };
  return JSON.stringify(tspaths, null, 2) + "\n";
}

// Emit type mapping for all modules imported via lazy getters.
function emitLazy(modules) {
  let lines = [HEADER];
  lines.push("export interface LazyModules {");
  for (let uri of [...modules].sort()) {
    lines.push(`  "${uri}": typeof import("${uri}"),`);
  }
  lines.push("}\n");
  return lines.join("\n");
}

function main(root_dir, paths_json, lib_lazy) {
  let files = [];
  scan(root_dir, "", files);

  let uris = new Set();
  let modules = new Set();

  for (let file of files) {
    let src = fs.readFileSync(`${root_dir}/${file}`, "utf-8");

    for (let [exp, , method] of src.matchAll(IMPORT)) {
      for (let [, , uri, proto] of exp.matchAll(URI)) {
        if (proto !== "moz-src") {
          uris.add(uri);
        }
        if (method?.startsWith("defineESModuleGetter")) {
          modules.add(uri);
        }
      }
    }
  }

  let json = emitPaths(files, uris, modules);
  console.log(`[INFO] ${paths_json} (${json.length.toLocaleString()} bytes)`);
  fs.writeFileSync(paths_json, json);

  let lib = emitLazy(modules);
  console.log(`[INFO] ${lib_lazy} (${lib.length.toLocaleString()} bytes)`);
  fs.writeFileSync(lib_lazy, lib);
}

main(...process.argv.slice(2));
