/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Update Service.
 *
 * The Initial Developer of the Original Code is Ben Goodger.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *  Ben Goodger <ben@mozilla.org> (Original Author)
 *  Darin Fisher <darin@meer.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

const PREF_APP_UPDATE_ENABLED             = "app.update.enabled";
const PREF_APP_UPDATE_AUTOINSTALL_ENABLED = "app.update.autoInstallEnabled";
// Defines the notification behavior when updates are encountered
//
// 0 = download and install all updates (including major versions, provided
//     no addons are made incompatible)
// 1 = download and install minor (security) updates
// 2 = ask before doing anything
//
const PREF_APP_UPDATE_AUTOINSTALL_MODE    = "app.update.autoInstallMode";
const PREF_APP_UPDATE_INTERVAL            = "app.update.interval";
const PREF_APP_UPDATE_TIMER               = "app.update.timer";
const PREF_APP_UPDATE_LOG_ENABLED         = "app.update.logEnabled";

const PREF_APP_UPDATE_URL                 = "app.update.url";
const PREF_APP_UPDATE_URL_OVERRIDE        = "app.update.url.override";

const PREF_UPDATE_LASTUPDATETIME_FMT      = "app.update.lastUpdateTime.%ID%";
const PREF_APP_EXTENSIONS_VERSION         = "app.extensions.version";

const URI_UPDATE_PROMPT_DIALOG  = "chrome://mozapps/content/update/updates.xul";
const URI_BRAND_PROPERTIES      = "chrome://branding/locale/brand.properties";
const URI_UPDATES_PROPERTIES    = "chrome://mozapps/locale/update/updates.properties";
const URI_UPDATE_NS             = "http://www.mozilla.org/2005/app-update";

const KEY_APPDIR          = "XCurProcD";

const DIR_UPDATES         = "updates";
const FILE_UPDATE_STATUS  = "update.status";
const FILE_UPDATE_ARCHIVE = "update.mar";
const FILE_UPDATE_INFO    = "update.info";
const FILE_UPDATES_DB     = "updates.xml";
const FILE_UPDATE_ACTIVE  = "active-update.xml";

const MODE_RDONLY   = 0x01;
const MODE_WRONLY   = 0x02;
const MODE_CREATE   = 0x08;
const MODE_APPEND   = 0x10;
const MODE_TRUNCATE = 0x20;

const PERMS_FILE      = 0644;
const PERMS_DIRECTORY = 0755;

const STATE_DOWNLOADING = "downloading";
const STATE_PENDING     = "pending";
const STATE_APPLYING    = "applying";
const STATE_SUCCEEDED   = "succeeded";
const STATE_FAILED      = "failed";

const DOWNLOAD_CHUNK_SIZE           = 65536;
const DOWNLOAD_BACKGROUND_INTERVAL  = 1;
const DOWNLOAD_FOREGROUND_INTERVAL  = 0;

const nsILocalFile            = Components.interfaces.nsILocalFile;
const nsIUpdateService        = Components.interfaces.nsIUpdateService;
const nsIUpdateItem           = Components.interfaces.nsIUpdateItem;
const nsIPrefLocalizedString  = Components.interfaces.nsIPrefLocalizedString;
const nsIIncrementalDownload  = Components.interfaces.nsIIncrementalDownload;
const nsIFileInputStream      = Components.interfaces.nsIFileInputStream;
const nsIFileOutputStream     = Components.interfaces.nsIFileOutputStream;
const nsICryptoHash           = Components.interfaces.nsICryptoHash;

const Node = Components.interfaces.nsIDOMNode;

var gApp        = null;
var gPref       = null;
var gOS         = null;
var gConsole    = null;
var gLogEnabled = false;

/**
 * Logs a string to the error console. 
 * @param   string
 *          The string to write to the error console..
 */  
function LOG(string) {
  if (gLogEnabled) {
    dump("*** " + string + "\n");
    gConsole.logStringMessage(string);
  }
}

/**
 * Convert a string containing binary values to hex.
 */
function binaryToHex(input) {
  var result = "";
  for (var i = 0; i < input.length; ++i) {
    var hex = input.charCodeAt(i).toString(16);
    if (hex.length == 1)
      hex = "0" + hex;
    result += hex;
  }
  return result;
}

/**
 * Gets a File URL spec for a nsIFile
 * @param   file
 *          The file to get a file URL spec to
 * @returns The file URL spec to the file
 */
function getURLSpecFromFile(file) {
  var ioServ = Components.classes["@mozilla.org/network/io-service;1"]
                         .getService(Components.interfaces.nsIIOService);
  var fph = ioServ.getProtocolHandler("file")
                  .QueryInterface(Components.interfaces.nsIFileProtocolHandler);
  return fph.getURLSpecFromFile(file);
}

/**
 * Gets the specified directory at the specified hierarchy under a 
 * Directory Service key. 
 * @param   key
 *          The Directory Service Key to start from
 * @param   pathArray
 *          An array of path components to locate beneath the directory 
 *          specified by |key|
 * @return  nsIFile object for the location specified. If the directory
 *          requested does not exist, it is created, along with any
 *          parent directories that need to be created.
 */
function getDir(key, pathArray) {
  return getDirInternal(key, pathArray, true);
}

/**
 * Gets the specified directory at the speciifed hierarchy under a 
 * Directory Service key. 
 * @param   key
 *          The Directory Service Key to start from
 * @param   pathArray
 *          An array of path components to locate beneath the directory 
 *          specified by |key|
 * @return  nsIFile object for the location specified. If the directory
 *          requested does not exist, it is NOT created.
 */
function getDirNoCreate(key, pathArray) {
  return getDirInternal(key, pathArray, false);
}

/**
 * Gets the specified directory at the speciifed hierarchy under a 
 * Directory Service key. 
 * @param   key
 *          The Directory Service Key to start from
 * @param   pathArray
 *          An array of path components to locate beneath the directory 
 *          specified by |key|
 * @param   shouldCreate
 *          true if the directory hierarchy specified in |pathArray|
 *          should be created if it does not exist,
 *          false otherwise.
 * @return  nsIFile object for the location specified. 
 */
function getDirInternal(key, pathArray, shouldCreate) {
  var fileLocator = Components.classes["@mozilla.org/file/directory_service;1"]
                              .getService(Components.interfaces.nsIProperties);
  var dir = fileLocator.get(key, Components.interfaces.nsIFile);
  for (var i = 0; i < pathArray.length; ++i) {
    dir.append(pathArray[i]);
    if (shouldCreate && !dir.exists())
      dir.create(nsILocalFile.DIRECTORY_TYPE, PERMS_DIRECTORY);
  }
  return dir;
}

/**
 * Gets the file at the speciifed hierarchy under a Directory Service key.
 * @param   key
 *          The Directory Service Key to start from
 * @param   pathArray
 *          An array of path components to locate beneath the directory 
 *          specified by |key|. The last item in this array must be the
 *          leaf name of a file.
 * @return  nsIFile object for the file specified. The file is NOT created
 *          if it does not exist, however all required directories along 
 *          the way are.
 */
function getFile(key, pathArray) {
  var file = getDir(key, pathArray.slice(0, -1));
  file.append(pathArray[pathArray.length - 1]);
  return file;
}

/**
 * An enumeration of items in a JS array.
 * @constructor
 */
function ArrayEnumerator(aItems) {
  this._index = 0;
  if (aItems) {
    for (var i = 0; i < aItems.length; ++i) {
      if (!aItems[i])
        aItems.splice(i, 1);      
    }
  }
  this._contents = aItems;
}

ArrayEnumerator.prototype = {
  _index: 0,
  _contents: [],
  
  hasMoreElements: function() {
    return this._index < this._contents.length;
  },
  
  getNext: function() {
    return this._contents[this._index++];      
  }
};

/**
 * Trims a prefix from a string.
 * @param   string
 *          The source string
 * @param   prefix
 *          The prefix to remove.
 * @returns The suffix (string - prefix)
 */
function stripPrefix(string, prefix) {
  return string.substr(prefix.length);
}

/**
 * Writes a string of text to a file.  A newline will be appended to the data
 * written to the file.  This function only works with ASCII text.
 */
function writeStringToFile(file, text) {
  var fos =
      Components.classes["@mozilla.org/network/safe-file-output-stream;1"].
      createInstance(nsIFileOutputStream);
  var modeFlags = MODE_WRONLY | MODE_CREATE | MODE_TRUNCATE;
  if (!file.exists()) 
    file.create(nsILocalFile.NORMAL_FILE_TYPE, PERMS_FILE);
  fos.init(file, modeFlags, PERMS_FILE, 0);
  text += "\n";
  fos.write(text, text.length);    
  if (fos instanceof Components.interfaces.nsISafeOutputStream)
    fos.finish();
  else
    fos.close();
}

/**
 * Reads a string of text from a file.  A trailing newline will be removed
 * before the result is returned.  This function only works with ASCII text.
 */
function readStringFromFile(file) {
  var fis =
      Components.classes["@mozilla.org/network/file-input-stream;1"].
      createInstance(nsIFileInputStream);
  var modeFlags = MODE_RDONLY;
  if (!file.exists())
    return null;
  fis.init(file, modeFlags, PERMS_FILE, 0);
  var sis =
      Components.classes["@mozilla.org/scriptableinputstream;1"].
      createInstance(Components.interfaces.nsIScriptableInputStream);
  sis.init(fis);
  var text = sis.read(sis.available());
  sis.close();
  if (text[text.length - 1] == "\n")
    text = text.slice(0, -1);
  return text;
}

/**
 *
 */
function UpdatePrompt() {
}
UpdatePrompt.prototype = {
  /**
   * See nsIUpdateService.idl
   */
  checkForUpdates: function() {
    this._showUI(0);
  },
    
  /**
   * See nsIUpdateService.idl
   */
  showUpdateAvailable: function(update) {
    var ary = Components.classes["@mozilla.org/supports-array;1"]
                        .createInstance(Components.interfaces.nsISupportsArray);
    ary.AppendElement(update);
      
    var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                       .getService(Components.interfaces.nsIWindowMediator);
    var win = wm.getMostRecentWindow("Update:Wizard");
    if (win) 
      win.focus();
    else {
      var ww = Components.classes["@mozilla.org/embedcomp/window-watcher;1"]
                         .getService(Components.interfaces.nsIWindowWatcher);
      ww.openWindow(null, URI_UPDATE_PROMPT_DIALOG, 
                    "", "chrome,centerscreen,dialog,titlebar", ary);
    }
  },
  
  /**
   * See nsIUpdateService.idl
   */
  showInstalledUpdates: function() {
    this._showUI(1);
  },
  
  /**
   * Shows the update wizard in the specified mode.
   * @param   mode
   *          The mode to show the update wizard in:
   *          0 - check for updates
   *          1 - show all installed updates
   */
  _showUI: function(mode) {
    var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                       .getService(Components.interfaces.nsIWindowMediator);
    var win = wm.getMostRecentWindow("Update:Wizard");
    if (win)
      win.focus();
    else {
      var ary = Components.classes["@mozilla.org/supports-array;1"]
                          .createInstance(Components.interfaces.nsISupportsArray);
      var wizardMode = Components.classes["@mozilla.org/supports-PRUint8;1"]
                                 .createInstance(Components.interfaces.nsISupportsPRUint8);
      wizardMode.data = mode;
      ary.AppendElement(wizardMode);
      var ww = Components.classes["@mozilla.org/embedcomp/window-watcher;1"]
                         .getService(Components.interfaces.nsIWindowWatcher);
      ww.openWindow(null, URI_UPDATE_PROMPT_DIALOG, 
                    "", "chrome,centerscreen,dialog,titlebar", ary);
    }
  },
  
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIUpdatePrompt) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

/**
 * UpdateService
 * A Service for managing the discovery and installation of software updates.
 */
function UpdateService() {
  gApp  = Components.classes["@mozilla.org/xre/app-info;1"]
                    .getService(Components.interfaces.nsIXULAppInfo)
                    .QueryInterface(Components.interfaces.nsIXULRuntime);
  gPref = Components.classes["@mozilla.org/preferences-service;1"]
                    .getService(Components.interfaces.nsIPrefBranch2);
  gOS   = Components.classes["@mozilla.org/observer-service;1"]
                    .getService(Components.interfaces.nsIObserverService);
  gConsole = Components.classes["@mozilla.org/consoleservice;1"]
                       .getService(Components.interfaces.nsIConsoleService);  

  // Start the update timer only after a profile has been selected so that the
  // appropriate values for the update check are read from the user's profile.  
  gOS.addObserver(this, "profile-after-change", false);

  // Observe xpcom-shutdown to unhook pref branch observers above to avoid 
  // shutdown leaks.
  gOS.addObserver(this, "xpcom-shutdown", false);

  gLogEnabled = getPref("getBoolPref", PREF_APP_UPDATE_LOG_ENABLED, false);
}

UpdateService.prototype = {
  _downloader: null,

  observe: function(subject, topic, data) {
    switch (topic) {
    case "app-startup":
      // Resume fetching...
      var um = Components.classes["@mozilla.org/updates/update-manager;1"]
                         .getService(Components.interfaces.nsIUpdateManager);
      var activeUpdate = um.activeUpdate;
      if (activeUpdate)
        this.downloadUpdate(activeUpdate, true);
      break;
    case "profile-after-change":
      gOS.removeObserver(this, "profile-after-change");

      // Register a background update check timer
      var tm = 
          Components.classes["@mozilla.org/updates/timer-manager;1"]
                    .getService(Components.interfaces.nsIUpdateTimerManager);
      var interval = getPref("getIntPref", PREF_APP_UPDATE_INTERVAL, 86400000);
      tm.registerTimer("background-update-timer", this, interval,
                       Components.interfaces.nsITimer.TYPE_REPEATING_SLACK);
      break;
    case "xpcom-shutdown":
      gOS.removeObserver(this, "xpcom-shutdown");
      break;
    }
  },
  
  _needsToPromptForUpdate: function(updates) {
    // First, check for Extension incompatibilities. These trump any preference
    // settings.
    var em = Components.classes["@mozilla.org/extensions/manager;1"]
                       .getService(Components.interfaces.nsIExtensionManager);
    var incompatibleList = { };
    for (var i = 0; i < updates.length; ++i) {
      var count = {};
      em.getIncompatibleItemList(gApp.ID, updates[i].extensionVersion,
                                 nsIUpdateItem.TYPE_ADDON, count);
      if (count.value > 0)
        return true;
    }

    // Now, inspect user preferences.
    
    // No prompt necessary, silently update...
    return false;
  },
  
  notify: function(timer) {
    // If a download is in progress, then do nothing.
    if (this._downloader && this._downloader.isBusy)
      return;

    var self = this;
    var listener = {
      onProgress: function() { },
      onCheckComplete: function(updates, updateCount) {
        self._selectAndInstallUpdate(updates);
      },
      onError: function() { },
    }

    this.checkForUpdates(listener);
  },
  
  /**
   * Determine whether or not an update requires user confirmation before it
   * can be installed.
   * @param   update
   *          The update to be installed
   * @returns true if a prompt UI should be shown asking the user if they want
   *          to install the update, false if the update should just be 
   *          silently downloaded and installed.
   */
  _shouldPrompt: function(update) {
    // There are two possible outcomes here:
    // 1. download and install the update automatically
    // 2. alert the user about the presence of an update before doing anything
    //
    // The outcome we follow is determined as follows:
    // 
    // Update Type      Mode      Incompatible    Outcome
    // Major            0         No              Auto Install
    // Major            0         Yes             Notify and Confirm
    // Major            1         Yes or No       Notify and Confirm
    // Major            2         Yes or No       Notify and Confirm
    // Minor            0         No              Auto Install
    // Minor            0         Yes             Notify and Confirm
    // Minor            1         No              Auto Install
    // Minor            1         Yes             Notify and Confirm
    // Minor            2         Yes or No       Notify and Confirm
    //
    // If app.update.enabled is set to false, an update check is not performed
    // at all, and so none of the decision making above is entered into.
    //
    var updateEnabled = getPref("getBoolPref", PREF_APP_UPDATE_ENABLED, true);
    if (!updateEnabled)
      return;
      
    var mode = getPref("getIntPref", PREF_APP_UPDATE_AUTOINSTALL_MODE, 0);
    var compatible = isCompatible(update);
    if ((update.type == "major" && (mode == 0)) ||
        (update.type == "minor" && (mode == 0 || mode == 1)))
      return !compatible;
    return true;
  },
  
  /**
   * Determine which of the specified updates should be installed.
   * @param   updates
   *          An array of available updates
   */
  selectUpdate: function(updates) {
    if (updates.length == 0)
      return null;
    
    // Choose the newest of the available minor and major updates. 
    var majorUpdate = null, minorUpdate = null;
    var newestMinor = updates[0], newestMajor = updates[0];

    var vc = new VersionChecker();
    for (var i = 0; i < updates.length; ++i) {
      if (updates[i].type == "major" && 
          vc.compare(newestMajor.version, updates[i].version) <= 0)
        majorUpdate = newestMajor = updates[i];
      if (updates[i].type == "minor" && 
          vc.compare(newestMinor.version, updates[i].version) <= 0)
        minorUpdate = newestMinor = updates[i];
    }

    // If there's a major update, always try and fetch that one first, 
    // otherwise fall back to the newest minor update.
    return majorUpdate || minorUpdate;
  },
  
  /**
   * Determine which of the specified updates should be installed and
   * begin the download/installation process, optionally prompting the
   * user for permission if required.
   * @param   updates
   *          An array of available updates
   */
  _selectAndInstallUpdate: function(updates) {
    var update = this.selectUpdate(updates, updates.length);
    if (!update)
      return;
    if (this._shouldPrompt(update)) {
      LOG("_selectAndInstallUpdate: need to prompt user before continuing...");
      var prompter = 
          Components.classes["@mozilla.org/updates/update-prompt;1"].
          createInstance(Components.interfaces.nsIUpdatePrompt);
      prompter.showUpdateAvailable(update);
    } else {
      this.downloadUpdate(update, true);
    }
  },

  /**
   * 
   */
  checkForUpdates: function(listener) {
    var checker = new Checker();
    checker.findUpdates(listener);
    return checker;
  },
  
  /**
   * See nsIUpdateService.idl
   */
  addDownloadListener: function(listener) {
    if (!this._downloader) {
      LOG("addDownloadListener: no downloader!\n");
      return;
    }
    this._downloader.addDownloadListener(listener);
  },
  
  /**
   * See nsIUpdateService.idl
   */
  removeDownloadListener: function(listener) {
    if (!this._downloader) {
      LOG("removeDownloadListener: no downloader!\n");
      return;
    }
    this._downloader.removeDownloadListener(listener);
  },
  
  /**
   * See nsIUpdateService.idl
   */
  downloadUpdate: function(update, background) {
    if (!update)
      throw Components.results.NS_ERROR_NULL_POINTER;
    if (this._downloader && this._downloader.isBusy) {
      if (update.isCompleteUpdate == this._downloader.isCompleteUpdate) {
        LOG("no support for downloading more than one update at a time");
        return;
      }
      this._downloader.cancel();
    }
    this._downloader = new Downloader(background);
    return this._downloader.downloadUpdate(update);
  },
  
  /**
   * See nsIUpdateService.idl
   */
  pauseDownload: function() {
    if (this._downloader && this._downloader.isBusy)
      this._downloader.cancel();
  },
  
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIApplicationUpdateService) &&
        !iid.equals(Components.interfaces.nsITimerCallback) && 
        !iid.equals(Components.interfaces.nsIObserver) && 
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

/**
 * A service to manage active and past updates.
 * @constructor
 */
function UpdateManager() {
}
UpdateManager.prototype = {
  /**
   * All previously downloaded and installed updates, as an array of nsIUpdate
   * objects.
   */
  _updates: null,
  
  /**
   * The current actively downloading/installing update, as a nsIUpdate object.
   */
  _activeUpdate: null,
  
  /**
   * Loads an updates.xml formatted file into an array of nsIUpdate items.
   * @param   file
   *          A nsIFile for the updates.xml file
   * @returns The array of nsIUpdate items held in the file.
   */
  _loadXMLFileIntoArray: function(file) {
    if (!file.exists())
      return [];
    
    var result = [];
    var fileStream = Components.classes["@mozilla.org/network/file-input-stream;1"]
                               .createInstance(Components.interfaces.nsIFileInputStream);
    fileStream.init(file, MODE_RDONLY, PERMS_FILE, 0);
    if (fileStream instanceof Components.interfaces.nsIScriptableInputStream) {
      var data = fileStream.read(fileStream.available());    
      var parser = Components.classes["@mozilla.org/xmlextras/domparser;1"]
                             .createInstance(Components.interfaces.nsIDOMParser);
      var doc = parser.parseFromString(data, "text/xml");
      
      var updateCount = doc.documentElement.childNodes.length;
      for (var i = 0; i < updateCount; ++i) {
        var updateElement = doc.documentElement.childNodes[i];
        if (updateElement.nodeType != Node.ELEMENT_NODE ||
            updateElement.localName != "update")
          continue;

        result.push(new Update(updateNode));
      }
    }
    return result;
  },
  
  /**
   * Load the update manager, initializing state from state files.
   */
  _ensureUpdates: function() {
    if (!this._updates)
      this._updates = this._loadXMLFileIntoArray(getFile(KEY_APPDIR, 
                        [FILE_UPDATES_DB]));
    this._ensureActiveUpdate();
  },
  
  /**
   * Ensure that the Active Update file is loaded.
   */
  _ensureActiveUpdate: function() {
    if (!this._activeUpdate) {
      var updates = this._loadXMLFileIntoArray(getFile(KEY_APPDIR, 
                             [FILE_UPDATE_ACTIVE]));
      if (updates.length > 0)
        this._activeUpdate = updates[0];
    }
  },
  
  /**
   * See nsIUpdateService.idl
   */
  getUpdateAt: function(index) {
    this._ensureUpdates();
    if (this._activeUpdate && index == 0) {
      // The "current" or "active" update is always first in the list.
      return this._activeUpdate;
    }
    return this._updates[index - (this._activeUpdate ? 1 : 0)];
  },
  
  /**
   * See nsIUpdateService.idl
   */
  get updateCount() {
    this._ensureUpdates();
    this._updates.length + (this._activeUpdate ? 1 : 0);
  },
  
  /**
   * See nsIUpdateService.idl
   */
  get activeUpdate() {
    this._ensureActiveUpdate();
    return this._activeUpdate;
  },
  set activeUpdate(activeUpdate) {
    this._ensureActiveUpdate();
    if (!activeUpdate)
      this._updates = [this._activeUpdate].concat(this._updates);
    this._activeUpdate = activeUpdate;
    this._writeUpdatesToXMLFile([this._activeUpdate], 
                                getFile(KEY_APPDIR, [FILE_UPDATE_ACTIVE]));
    return activeUpdate;
  },
  
  /**
   * See nsIUpdateService.idl
   */
  removeUpdateAtIndex: function(index) {
    this._ensureUpdates();
    if (this._activeUpdate && index == 0) 
      this._activeUpdate = null;
    if (update == this._activeUpdate)
      this._activeUpdate = null;
    else {
      for (var i = this._updates.length; i > 0; --i) {
        if (this._updates[i] == update)
          this._updates.splice(i, 1);
      }
    }
  },
  
  /**
   *
   */
  _writeUpdatesToXMLFile: function(updates, file) {
    var fos = Components.classes["@mozilla.org/network/safe-file-output-stream;1"]
                        .createInstance(Components.interfaces.nsIFileOutputStream);
    var modeFlags = MODE_WRONLY | MODE_CREATE | MODE_TRUNCATE;
    if (!file.exists()) 
      file.create(nsILocalFile.NORMAL_FILE_TYPE, PERMS_FILE);
    fos.init(file, modeFlags, PERMS_FILE, 0);
    
    var parser = Components.classes["@mozilla.org/xmlextras/domparser;1"]
                           .createInstance(Components.interfaces.nsIDOMParser);
    const EMPTY_UPDATES_DOCUMENT = "<?xml version=\"1.0\"?><updates xmlns=\"http://www.mozilla.org/2005/app-update\"></updates>";
    var doc = parser.parseFromString(EMPTY_UPDATES_DOCUMENT, "text/xml");

    for (var i = 0; i < updates.length; ++i) {
      if (updates[i])
        doc.documentElement.appendChild(updates[i].serialize(doc));
    }

    var serializer = Components.classes["@mozilla.org/xmlextras/xmlserializer;1"]
                               .createInstance(Components.interfaces.nsIDOMSerializer);
    serializer.serializeToStream(doc.documentElement, fos, null);

    if (fos instanceof Components.interfaces.nsISafeOutputStream) {
      try {
        fos.finish();
      }
      catch (e) {
        fos.close();
      }
    }
    else
      fos.close();
  },

  /**
   * See nsIUpdateService.idl
   */
  saveUpdates: function() {
    if (this._activeUpdate) {
      this._writeUpdatesToXMLFile([this._activeUpdate], 
                                  getFile(KEY_APPDIR, [FILE_UPDATE_ACTIVE]));
    }
    if (this._updates) {
      this._writeUpdatesToXMLFile(this._updates, 
                                  getFile(KEY_APPDIR, [FILE_UPDATES_DB]));
    }
  },
  
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIUpdateManager) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};


/**
 * Gets a preference value, handling the case where there is no default.
 * @param   func
 *          The name of the preference function to call, on nsIPrefBranch
 * @param   preference
 *          The name of the preference
 * @param   defaultValue
 *          The default value to return in the event the preference has 
 *          no setting
 * @returns The value of the preference, or undefined if there was no
 *          user or default value.
 */
function getPref(func, preference, defaultValue) {
  try {
    return gPref[func](preference);
  }
  catch (e) {
  }
  return defaultValue;
}

/**
 * Update Patch
 * @param   patch
 *          A <patch> element to initialize this object with
 * @constructor
 */
function UpdatePatch(patch) {
  this.type = patch.getAttribute("type");
  this.URL = patch.getAttribute("URL");
  this.hashFunction = patch.getAttribute("hashFunction");
  this.hashValue = patch.getAttribute("hashValue");
  this.size = parseInt(patch.getAttribute("size"));
  this.percentage = parseFloat(patch.getAttribute("percentage"));
  this.state = patch.getAttribute("state");
  this.status = patch.getAttribute("status");
  this.selected = patch.getAttribute("selected") == "true";
}
UpdatePatch.prototype = {
  /**
   * See nsIUpdateService.idl
   */
  serialize: function(updates) {
    var patch = updates.createElementNS(URI_UPDATE_NS, "patch");
    patch.setAttribute("type", this.type);
    patch.setAttribute("URL", this.URL);
    patch.setAttribute("hashFunction", this.hashFunction);
    patch.setAttribute("hashValue", this.hashValue);
    patch.setAttribute("size", this.size);
    patch.setAttribute("percentage", this.percentage);
    patch.setAttribute("state", this.state);
    patch.setAttribute("status", this.status);
    patch.setAttribute("selected", this.selected);
    return patch; 
  },
  
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIUpdatePatch) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

/**
 * Update
 * Implements nsIUpdate
 * @param   update
 *          An <update> element to initialize this object with
 * @constructor
 */
function Update(update) {
  this._patches = [];

  for (var i = 0; i < update.childNodes.length; ++i) {
    var patchElement = update.childNodes[i];
    if (patchElement.nodeType != Node.ELEMENT_NODE ||
        patchElement.localName != "patch")
      continue;

    this._patches.push(new UpdatePatch(patchElement));
  }
  
  this.type = update.getAttribute("type");
  this.version = update.getAttribute("version");
  this.extensionVersion = update.getAttribute("extensionVersion");
  this.detailsURL = update.getAttribute("detailsURL");
  this.licenseURL = update.getAttribute("licenseURL");
  this.installDate = update.hasAttribute("installDate") ? parseInt(update.getAttribute("installDate")) : 0;
  this.isCompleteUpdate = false;
  
  // The Update Name is either the string provided by the <update> element, or
  // the string: "<App Name> <Update App Version>"
  var name = "";
  if (update.hasAttribute("name"))
    name = update.getAttribute("name");
  else {
    var sbs = Components.classes["@mozilla.org/intl/stringbundle;1"]
                        .getService(Components.interfaces.nsIStringBundleService);
    var brandBundle = sbs.createBundle(URI_BRAND_PROPERTIES);
    var updateBundle = sbs.createBundle(URI_UPDATES_PROPERTIES);
    var appName = brandBundle.GetStringFromName("brandShortName");
    name = updateBundle.formatStringFromName("updateName", 
                                             [appName, this.version], 2);
  }
  
  this.name = name;
}
Update.prototype = {
  /**
   * See nsIUpdateService.idl
   */
  get patchCount() {
    return this._patches.length;
  },
  
  /**
   * See nsIUpdateService.idl
   */
  getPatchAt: function(index) {
    return this._patches[index];
  },
  
  get selectedPatch() {
    for (var i = 0; i < this.patchCount; ++i) {
      if (this._patches[i].selected)
        return this._patches[i];
    }
    return null;
  },
  
  /**
   * See nsIUpdateService.idl
   */
  serialize: function(updates) {
    var update = updates.createElementNS(URI_UPDATE_NS, "update");
    update.setAttribute("type", this.type);
    update.setAttribute("name", this.name);
    update.setAttribute("version", this.version);
    update.setAttribute("extensionVersion", this.extensionVersion);
    update.setAttribute("detailsURL", this.detailsURL);
    update.setAttribute("licenseURL", this.licenseURL);
    update.setAttribute("installDate", this.installDate);
    updates.documentElement.appendChild(update);
    
    for (var i = 0; i < this.patchCount; ++i)
      update.appendChild(this.getPatchAt(i).serialize(updates));
    
    return update;
  },
   
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIUpdate) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
}; 
 
/**
 * Checker
 */
function Checker() {
}
Checker.prototype = {
  /**
   * The XMLHttpRequest object that performs the connection.
   */
  _request  : null,
  
  /**
   *
   */
  _callback : null,
  
  /**
   *
   */
  observer  : null,
  
  /**
   * The URL of the update service XML file to connect to that contains details
   * about available updates.
   */
  get _updateURL() {
    var url;
    try {
      // Use the override URL if specified.
      url = gPref.getComplexValue(PREF_APP_UPDATE_URL_OVERRIDE,
                                  nsIPrefLocalizedString).data;
      if (url && url != "")
        return url;
    } catch (e) {}

    // Otherwise, construct the update URL from component parts.
    try {
      url = gPref.getComplexValue(PREF_APP_UPDATE_URL,
                                  nsIPrefLocalizedString).data;
    } catch (e) {
      LOG(PREF_APP_UPDATE_URL + " not defined");
      return null;
    }

    var locale = gPref.getCharPref("general.useragent.locale");

    url = url.replace(/%PRODUCT%/g, gApp.name);
    url = url.replace(/%VERSION%/g, gApp.version);
    url = url.replace(/%BUILD_ID%/g, gApp.appBuildID);
    url = url.replace(/%BUILD_TARGET%/g, gApp.OS + "_" + gApp.XPCOMABI);
    url = url.replace(/%LOCALE%/g, locale);
    url = url.replace(/\+/g, "%2B");

    LOG("update url: " + url);
    return url;
  },
  
  /**
   *
   */
  findUpdates: function(callback) {
    if (!callback)
      throw Components.results.NS_ERROR_NULL_POINTER;
      
    if (!this._updateURL)
      return;
      
    this._request = Components.classes["@mozilla.org/xmlextras/xmlhttprequest;1"]
                              .createInstance(Components.interfaces.nsIXMLHttpRequest);
    this._request.open("GET", this._updateURL, true);
    this._request.overrideMimeType("text/xml");
    this._request.setRequestHeader("Cache-Control", "no-cache");
    
    var self = this;
    this._request.onerror     = function(event) { self.onError(event);    };
    this._request.onload      = function(event) { self.onLoad(event);     };
    this._request.onprogress  = function(event) { self.onProgress(event); };

    LOG("Checker.findUpdates: sending request to " + this._updateURL);
    this._request.send(null);
    
    this._callback = callback;
  },
  
  /**
   *
   */
  onProgress: function(event) {
    LOG("Checker.onProgress: " + event.position + "/" + event.totalSize);
    this._callback.onProgress(event.target, event.position, event.totalSize);
  },
  
  /**
   *
   */
  get _updates() {
    var updatesElement = this._request.responseXML.documentElement;
    if (!updatesElement) {
      LOG("Checker.get_updates: empty updates document?!");
      return [];
    }

    if (updatesElement.nodeName != "updates") {
      LOG("Checker.get_updates: unexpected node name!");
      return [];
    }
    
    var updates = [];
    for (var i = 0; i < updatesElement.childNodes.length; ++i) {
      var updateElement = updatesElement.childNodes[i];
      if (updateElement.nodeType != Node.ELEMENT_NODE ||
          updateElement.localName != "update")
        continue;

      updates.push(new Update(updateElement));
    }

    return updates;
  },
  
  /**
   *
   */
  onLoad: function(event) {
    LOG("Checker.onLoad: request completed downloading document");
    
    // Notify the front end that we're complete
    if (this.observer)
      this.observer.onLoad(event.target);

    // Analyze the resulting DOM and determine the set of updates to install
    var updates = this._updates;

    LOG("Updates available: " + updates.length);
    
    // ... and tell the Update Service about what we discovered.
    this._callback.onCheckComplete(updates, updates.length);
  },
  
  /**
   *
   */
  onError: function(event) {
    LOG("Checker.onError: error during load");
    this._callback.onError(event.target);
  },
  
  /**
   * See nsIUpdateService.idl
   */
  stopChecking: function() {
    this._request.abort();
  },
  
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIUpdateChecker) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

/**
 * Formats status messages for a download operation based on the progress
 * of the download.
 * @constructor
 */
function DownloadStatusFormatter() {
  this._downloadStartTime = (new Date()).getTime();
  this._elapsed = 0;
  
  var sbs = Components.classes["@mozilla.org/intl/stringbundle;1"]
                      .getService(Components.interfaces.nsIStringBundleService);
  var bundle = sbs.createBundle(URI_UPDATES_PROPERTIES);
  
  this._statusFormat = bundle.GetStringFromName("statusFormat");
  this._statusFormatKBMB = bundle.GetStringFromName("statusFormatKBMB");
  this._statusFormatKBKB = bundle.GetStringFromName("statusFormatKBKB");
  this._statusFormatMBMB = bundle.GetStringFromName("statusFormatMBMB");
  this._statusFormatUnknownMB = bundle.GetStringFromName("statusFormatUnknownMB");
  this._statusFormatUnknownKB = bundle.GetStringFromName("statusFormatUnknownKB");
  this._remain = bundle.GetStringFromName("remain");
  this._unknownFilesize = bundle.GetStringFromName("unknownFilesize");
  this._longTimeFormat = bundle.GetStringFromName("longTimeFormat");
  this._shortTimeFormat = bundle.GetStringFromName("shortTimeFormat");
}
DownloadStatusFormatter.prototype = {
  /**
   * Time when the download started (in seconds since epoch)
   */
  _downloadStartTime: 0,

  /**
   * Time elapsed since the start of the download operation (in seconds)
   */
  _elapsed: 0,

  /**
   * Format a human-readable status message based on the current download
   * progress.
   * @param   currSize
   *          The current number of bytes transferred
   * @param   finalSize
   *          The total number of bytes to be transferred
   * @returns A human readable status message, e.g.
   *          "3.4 of 4.7MB; 1.15 remain"
   */
  formatStatus: function(currSize, finalSize) {
    var now = (new Date()).getTime();
    
    // 1) Determine the Download Progress in Kilobytes
    var KBTotal = parseInt(finalSize/1024 + 0.5);
    var KBProgress = this._formatKBytes(parseInt(currSize/1024 + 0.5), KBTotal);
    LOG("KBP = " + KBProgress);

    // 2) Determine the Transfer Rate
    this._elapsed = now - (this._startTime / 1000);
    var rate = this._elapsed ? (currSize * 1024) / this.elapsed : 0
    var KBRate = "??.?";
    if (rate) {    
      var KBRate = parseInt((rate / 1024) * 10 + 0.5);
      var fraction = KBRate % 10;
      KBRate = parseInt((KBRate - remainder) / 10);
      if (KBRate < 100)
        KBRate += "." + fraction;
    }

    // 3) Determine the Time Remaining
    var remainingTime = this._unknownFileSize;
    if (rate && (finalSize > 0)) {
      remainingTime = (finalSize - currSize) / rate;
      remainingTime = parseInt(remainingTime + 0.5);
      remainingTime = this._formatSeconds(remainingTime); 
    }
      
    var status = this._statusFormat;
    status = this._replaceInsert(status, 1, KBProgress);
    status = this._replaceInsert(status, 2, KBRate);
    status = this._replaceInsert(status, 3, remainingTime);
    return status;
  },

  /**
   * Inserts a string into another string at the specified index, e.g. for
   * the format string var foo ="#1 #2 #3", |_replaceInsert(foo, 2, "test")|
   * returns "#1 test #3";
   * @param   format
   *          The format string
   * @param   index
   *          The Index to insert into
   * @param   value
   *          The value to insert
   * @returns The string with the value inserted. 
   */  
  _replaceInsert: function(format, index, value) {
    return format.replace(new RegExp("#" + index), value);
  },

  /**
   * Formats progress in the form of kilobytes transfered vs. total to 
   * transfer.
   * @param   currentKB
   *          The current amount of data transfered, in kilobytes.
   * @param   totalKB
   *          The total amount of data that must be transfered, in kilobytes.
   * @returns A string representation of the progress, formatted according to:
   * 
   *            KB           totalKB           returns
   *            x, < 1MB     y < 1MB           x of y KB
   *            x, < 1MB     y >= 1MB          x KB of y MB
   *            x, >= 1MB    y >= 1MB          x of y MB
   */
   _formatKBytes: function(currentKB, totalKB) {
    var progressHasMB = parseInt(currentKB / 1024) > 0;
    var totalHasMB = parseInt(totalKB / 1024) > 0;
    
    var format = "";
    if (!progressHasMB && !totalHasMB) {
      if (!totalKB) {
        format = this._statusFormatUnknownKB;
        format = this._replaceInsert(format, 1, currentKB);
      } else {
        format = this._statusFormatKBKB;
        format = this._replaceInsert(format, 1, currentKB);
        format = this._replaceInsert(format, 2, totalKB);
      }
    }
    else if (progressHasMB && totalHasMB) {
      format = this._statusFormatMBMB;
      format = this._replaceInsert(format, 1, (currentKB / 1024).toFixed(1));
      format = this._replaceInsert(format, 2, (totalKB / 1024).toFixed(1));
    }
    else if (totalHasMB && !progressHasMB) {
      format = this._statusFormatKBMB;
      format = this._replaceInsert(format, 1, currentKB);
      format = this._replaceInsert(format, 2, (totalKB / 1024).toFixed(1));
    }
    else if (progressHasMB && !totalHasMB) {
      format = this._statusFormatUnknownMB;
      format = this._replaceInsert(format, 1, (currentKB / 1024).toFixed(1));
    }
    return format;  
  },

  /**
   * Formats a time in seconds into something human readable.
   * @param   seconds
   *          The time to format
   * @returns A human readable string representing the date.
   */
  _formatSeconds: function(seconds) {
    // Round the number of seconds to remove fractions.
    seconds = parseInt(seconds + .5);
    var hours = parseInt(seconds/3600);
    seconds -= hours * 3600;
    
    var minutes = parseInt(seconds/60);
    seconds -= minutes * 60;
    var result = hours ? this._longTimeFormat : this._shortTimeFormat;

    if (hours < 10)
      hours = "0" + hours;
    if (minutes < 10)
      minutes = "0" + minutes;
    if (seconds < 10)
      seconds = "0" + seconds;

    // Insert hours, minutes, and seconds into result string.
    result = this._replaceInsert(result, 1, hours);
    result = this._replaceInsert(result, 2, minutes);
    result = this._replaceInsert(result, 3, seconds);

    return result;
  }
};


/**
 * Manages the download of updates
 * @param   background
 *          Whether or not this downloader is operating in background
 *          update mode. 
 * @constructor
 */
function Downloader(background) {
  this.background = background;
}
Downloader.prototype = {
  /**
   * The nsIUpdatePatch that we are downloading
   */
  _patch: null,
  
  /**
   * The nsIUpdate that we are downloading
   */
  _update: null,
  
  /**
   * The nsIIncrementalDownload object handling the download
   */
  _request: null,

  /**
   * Whether or not the update being downloaded is a complete replacement of
   * the user's existing installation or a patch representing the difference
   * between the new version and the previous version.
   */
  isCompleteUpdate: null,

  /**
   *
   */  
  cancel: function() {
    if (this._request && 
        this._request instanceof Components.interfaces.nsIRequest) {
      const NS_BINDING_ABORTED = 0x804b0002;
      this._request.cancel(NS_BINDING_ABORTED);
    }
  },

  /**
   * 
   */
  _getUpdatesDir: function() {
    // Right now, we only support downloading one patch at a time, so we always
    // use the same target directory.
    var fileLocator =
        Components.classes["@mozilla.org/file/directory_service;1"].
        getService(Components.interfaces.nsIProperties);
    var appDir = fileLocator.get(KEY_APPDIR, Components.interfaces.nsIFile);
    appDir.append(DIR_UPDATES);
    appDir.append("0");
    if (!appDir.exists())
      appDir.create(nsILocalFile.DIRECTORY_TYPE, PERMS_DIRECTORY);
    return appDir;
  },
  
  /**
   * Writes the current update operation/state to a file in the patch 
   * directory, indicating to the patching system that operations need
   * to be performed.
   * @param   dir
   *          The patch directory where the update.status file should be 
   *          written.
   * @param   state
   *          The state value to write.
   */
  _writeStatusFile: function(dir, state) {
    var statusFile = dir.clone();
    statusFile.append(FILE_UPDATE_STATUS);
    writeStringToFile(statusFile, state);
  },

  /**
   * Reads the current update state...
   */
  _readStatusFile: function(dir) {
    var statusFile = dir.clone();
    statusFile.append(FILE_UPDATE_STATUS);
    LOG(statusFile.path);
    return readStringFromFile(statusFile);
  },

  /**
   * Verify the downloaded file.  We assume that the download is complete at
   * this point.
   */
  _verifyDownload: function() {
    var fileStream = Components.classes["@mozilla.org/network/file-input-stream;1"].
        createInstance(nsIFileInputStream);
    fileStream.init(this._request.destination, MODE_RDONLY, PERMS_FILE, 0);

    try {
      var hash = Components.classes["@mozilla.org/security/hash;1"].
          createInstance(nsICryptoHash);
      var hashFunction = nsICryptoHash[this._patch.hashFunction.toUpperCase()];
      if (hashFunction == undefined)
        return false;
      hash.init(hashFunction);
      hash.updateFromStream(fileStream, -1);
      // NOTE: For now, we assume that the format of _patch.hashValue is hex
      // encoded binary (such as what is typically output by programs like
      // sha1sum).  In the future, this may change to base64 depending on how
      // we choose to compute these hashes.
      digest = binaryToHex(hash.finish(false));
    } catch (e) {
      LOG("failed to compute hash of downloaded update archive");
      digest = "";
    }

    fileStream.close();

    return digest == this._patch.hashValue.toLowerCase();
  },

  /**
   * Select the patch to use given the current state of updateDir and the given
   * set of update patches.
   * @param   update
   *          A nsIUpdate object to select a patch from
   * @param   updateDir
   *          A nsIFile representing the update directory
   * @returns A nsIUpdatePatch object to download
   */
  _selectPatch: function(update, updateDir) {
    // Given an update to download, we will always try to download the patch
    // for a partial update over the patch for a full update.

    /**
     * Return the first UpdatePatch with the given type.
     * @param   type
     *          The type of the patch ("complete" or "partial")
     * @returns A nsIUpdatePatch object matching the type specified
     */
    function getPatchOfType(type) {
      for (var i = 0; i < update.patchCount; ++i) {
        var patch = update.getPatchAt(i);
        if (patch && patch.type == type)
          return patch;
      }
      return null;
    }

    // Look to see if any of the patches in the Update object has been
    // pre-selected for download, otherwise we must figure out which one
    // to select ourselves. 
    var selectedPatch = update.selectedPatch;
    
    // If this is a patch that we know about, then select it.  If it is a patch
    // that we do not know about, then remove it and use our default logic.
    var useComplete = false;
    if (selectedPatch) {
      var state = this._readStatusFile(updateDir)
      LOG("found existing patch [state="+state+"]");
      switch (state) {
      case STATE_DOWNLOADING: 
        LOG("resuming download");
        return selectedPatch;
      case STATE_PENDING:
        LOG("already downloaded and staged");
        return null;
      case STATE_FAILED:
      case STATE_APPLYING:
        // Something went wrong when we tried to apply the previous patch.
        // Try the complete patch next time.
        if (update && selectedPatch.type == "partial") {
          useComplete = true;
        } else {
          // This is a pretty fatal error.  Just bail.
          LOG("failed to apply complete patch!");
          return null;
        }
      }

      // The patch is no longer known to us, so we should assume that it is
      // no longer valid.  Remove all from the updateDir.
      // XXX We may want to preserve an update.log file from a failed update.
      // We expect this operation may fail since some files (e.g., updater.exe)
      // may still be in use. That's ok.
      try {
        updateDir.remove(true);
      } catch (e) {}
      // Restore the updateDir since we may have deleted it.
      updateDir = this._getUpdatesDir();

      selectedPatch = null;
    }

    // If we were not able to discover an update from a previous download, we 
    // select the best patch from the given set.
    var partialPatch = getPatchOfType("partial");
    if (!useComplete)
      selectedPatch = partialPatch;
    if (!selectedPatch) {
      partialPatch.selected = false;
      selectedPatch = getPatchOfType("complete");
    }
    selectedPatch.selected = true;
    update.isCompleteUpdate = useComplete;
    
    // Reset the Active Update object on the Update Manager and flush the
    // Active Update DB. 
    var um = Components.classes["@mozilla.org/updates/update-manager;1"]
                       .getService(Components.interfaces.nsIUpdateManager);
    um.activeUpdate = update;

    return selectedPatch;
  },

  /**
   * Whether or not we are currently downloading something.
   */
  get isBusy() {
    return this._request != null;
  },
  
  /**
   * Download and stage the given update.
   * @param   update
   *          A nsIUpdate object to download a patch for. Cannot be null.
   */
  downloadUpdate: function(update) {
    if (!update)
      throw Components.results.NS_ERROR_NULL_POINTER;
    
    var updateDir = this._getUpdatesDir();

    this._update = update;

    // This function may return null, which indicates that there are no patches
    // to download.
    this._patch = this._selectPatch(update, updateDir);
    if (!this._patch) {
      LOG("no patch to download");
      return this._readStatusFile(updateDir);
    }
    this.isCompleteUpdate = this._patch.type == "complete";

    var patchFile = updateDir.clone();
    patchFile.append(FILE_UPDATE_ARCHIVE);

    var ios = Components.classes["@mozilla.org/network/io-service;1"].
        getService(Components.interfaces.nsIIOService);
    var uri = ios.newURI(this._patch.URL, null, null);

    this._request =
        Components.classes["@mozilla.org/network/incremental-download;1"].
        createInstance(nsIIncrementalDownload);

    LOG("Downloader.downloadUpdate: Downloading from " + uri.spec + " to " + 
        patchFile.path);

    var interval = this.background ? DOWNLOAD_BACKGROUND_INTERVAL
                                   : DOWNLOAD_FOREGROUND_INTERVAL;
    this._request.init(uri, patchFile, DOWNLOAD_CHUNK_SIZE, interval);
    this._request.start(this, null);

    this._writeStatusFile(updateDir, STATE_DOWNLOADING);
    this._patch.state = STATE_DOWNLOADING;
    var um = Components.classes["@mozilla.org/updates/update-manager;1"]
                       .getService(Components.interfaces.nsIUpdateManager);
    um.saveUpdates();
    return STATE_DOWNLOADING;
  },
  
  /**
   *
   */
  _listeners: [],

  /** 
   * 
   */
  addDownloadListener: function(listener) {
    for (var i = 0; i < this._listeners.length; ++i) {
      if (this._listeners[i] == listener)
        return;
    }
    this._listeners.push(listener);
  },
  
  /** 
   *
   */
  removeDownloadListener: function(listener) {
    LOG("LISTENERS0 = " + this._listeners.length);
    for (var i = 0; i < this._listeners.length; ++i) {
      if (this._listeners[i] == listener) {
        this._listeners.splice(i, 1);
        return;
      }
    }
    LOG("LISTENERS1 = " + this._listeners.length);
  },
  
  _statusFormatter: null,
  
  /**
   *
   */
  onStartRequest: function(request, context) {
    request.QueryInterface(nsIIncrementalDownload);
    LOG("Downloader.onStartRequest: " + request.URI.spec);
    
    this._statusFormatter = new DownloadStatusFormatter();
    
    var listenerCount = this._listeners.length;
    for (var i = 0; i < listenerCount; ++i)
      this._listeners[i].onStartRequest(request, context);
  },
  
  /**
   *
   */
  onProgress: function(request, context, progress, maxProgress) {
    request.QueryInterface(nsIIncrementalDownload);
    // LOG("Downloader.onProgress: " + request.URI.spec + ", " + progress + "/" + maxProgress);
    
    this._patch.progress = Math.round(100 * (progress/maxProgress));
    this._patch.status = this._statusFormatter.formatStatus(progress, maxProgress);

    var listenerCount = this._listeners.length;
    for (var i = 0; i < listenerCount; ++i) {
      var listener = this._listeners[i];
      if (listener instanceof Components.interfaces.nsIProgressEventSink)
        listener.onProgress(request, context, progress, maxProgress);
    }
  },
  
  /**
   *
   */
  onStatus: function(request, context, status, statusText) {
    request.QueryInterface(nsIIncrementalDownload);
    LOG("Downloader.onStatus: " + request.URI.spec + " status = " + status + ", text = " + statusText);
    var listenerCount = this._listeners.length;
    for (var i = 0; i < listenerCount; ++i) {
      var listener = this._listeners[i];
      if (listener instanceof Components.interfaces.nsIProgressEventSink)
        listener.onStatus(request, context, status, statusText);
    }
  },
  
  /**
   *
   */
  onStopRequest: function(request, context, status) {
    request.QueryInterface(nsIIncrementalDownload);
    LOG("Downloader.onStopRequest: " + request.URI.spec + ", status = " + status);

    if (Components.isSuccessCode(status)) {
      var state;
      if (this._verifyDownload()) {
        state = STATE_PENDING;
      } else {
        LOG("download verification failed");
        state = STATE_FAILED;
        // TODO: use more informative error code here
        status = Components.results.NS_ERROR_UNEXPECTED;
        
        if (!this._update.isCompleteUpdate) {
          // If we were downloading a patch and the patch verification phase 
          // failed, log this and then commence downloading the complete update.
          LOG("Downloader.onStopRequest: Verification of patch failed, downloading complete update");
          this._update.isCompleteUpdate = true;
          var rv = this.downloadUpdate(this._update);
          LOG("US:STATE = " + rv);
          // This will reset the |.state| property on this._update if a new 
          // download initiates.
        }
      }
      this._writeStatusFile(this._getUpdatesDir(), state);
      this._patch.state = state;
      var um = Components.classes["@mozilla.org/updates/update-manager;1"]
                         .getService(Components.interfaces.nsIUpdateManager);
      um.activeUpdate = null;
      um.saveUpdates();
    }

    var listenerCount = this._listeners.length;
    for (var i = 0; i < listenerCount; ++i)
      this._listeners[i].onStopRequest(request, context, status);

    this._request = null;
  },
   
  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIRequestObserver) &&
        !iid.equals(Components.interfaces.nsIProgressEventSink) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

/**
 * A manager for update check timers. Manages timers that fire over long 
 * periods of time (e.g. days, weeks).
 * @constructor
 */
function TimerManager() {
}
TimerManager.prototype = {
  /**
   * The set of registered timers.
   */
  _timers: { },
  
  /**
   *
   */
  registerTimer: function(id, callback, interval, type) {
    const nsITimer = Components.interfaces.nsITimer;
    var timer = Components.classes["@mozilla.org/timer;1"]
                          .createInstance(nsITimer);
    var timerInterval = getPref("getIntPref", PREF_APP_UPDATE_TIMER, 5000);
    
    var self = this;
    
    /**
     * A callback object implementing nsITimerCallback that determines if the
     * user-registered callback should be invoked yet. 
     * @param   id
     *          The id of the timer that fired
     * @param   callback
     *          The nsITimerCallback object supplied by the user that should be
     *          notified if the user's interval has expired.
     * @param   interval
     *          The user's interval
     * @constructor
     */
    function TimerCallback(id, callback, interval) {
      this.id = id;
      this.callback = callback;
      this.interval = interval;
    }
    TimerCallback.prototype = {
      /**
       *
       */
      notify: function(timer) {
        // LOG("self._timers = " + self._timers.toSource());
        var lastUpdateTime = self._timers[this.id].lastUpdateTime;
        var now = Math.round(Date.now() / 1000);
        // LOG("notify = " + (now - lastUpdateTime) + " > " + this.interval);
        
        // Fudge the lastUpdateTime by some random increment of the update 
        // check interval (e.g. some random slice of 10 minutes) so that when
        // the time comes to check, we offset each client request by a random
        // amount so they don't all hit at once.
        lastUpdateTime += Math.round(Math.random() * this.interval);
        
        if ((now - lastUpdateTime) > this.interval &&
            this.callback instanceof Components.interfaces.nsITimerCallback) {
          this.callback.notify(timer);
          self._timers[this.id].lastUpdateTime = now;
          var preference = PREF_UPDATE_LASTUPDATETIME_FMT.replace(/%ID%/, this.id);
          gPref.setIntPref(preference, now);
        }
      },

      /**
       * See nsISupports.idl
       */
      QueryInterface: function(iid) {
        if (!iid.equals(Components.interfaces.nsITimerCallback) &&
            !iid.equals(Components.interfaces.nsISupports))
          throw Components.results.NS_ERROR_NO_INTERFACE;
        return this;
      }
    };
    var tc = new TimerCallback(id, callback, interval);
    LOG("timerInterval = " + timerInterval);
    timer.initWithCallback(tc, timerInterval, type);
    var preference = PREF_UPDATE_LASTUPDATETIME_FMT.replace(/%ID%/, id);
    var lastUpdateTime = getPref("getIntPref", preference, 
                                 Math.round(Date.now() / 1000));
    this._timers[id] = { timer: timer, lastUpdateTime: lastUpdateTime };
  },

  /**
   * See nsISupports.idl
   */
  QueryInterface: function(iid) {
    if (!iid.equals(Components.interfaces.nsIUpdateTimerManager) &&
        !iid.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

/**
 * Represents the version of an entity, in Firefox Version Format.
 * @constructor
 */
function Version(aMajor, aMinor, aRelease, aBuild, aPlus) { 
  this.major    = aMajor    || 0;
  this.minor    = aMinor    || 0;
  this.release  = aRelease  || 0;
  this.build    = aBuild    || 0;
  this.plus     = aPlus     || 0;
}
Version.prototype = {
  toString: function Version_toString() {
    return this.major + "." + this.minor + "." + this.subminor + "." + this.release + (this.plus ? "+" : "");
  },
  
  compare: function (aVersion) {
    var fields = ["major", "minor", "release", "build", "plus"];
    
    for (var i = 0; i < fields.length; ++i) {
      var field = fields[i];
      if (aVersion[field] > this[field])
        return -1;
      else if (aVersion[field] < this[field])
        return 1;
    }
    return 0;
  }
}

/**
 * A service that provides a means to compare two FVF strings.
 * @constructor
 */
function VersionChecker() {
}

VersionChecker.prototype = {
  /////////////////////////////////////////////////////////////////////////////
  // nsIVersionChecker
  
  // -ve      if B is newer
  // equal    if A == B
  // +ve      if A is newer
  compare: function(aVersionA, aVersionB) {
    var a = this._decomposeVersion(aVersionA);
    var b = this._decomposeVersion(aVersionB);
    
    return a.compare(b);
  },
  
  _decomposeVersion: function(aVersion) {
    var plus = 0;
    if (aVersion.charAt(aVersion.length-1) == "+") {
      aVersion = aVersion.substr(0, aVersion.length-1);
      plus = 1;
    }

    var parts = aVersion.split(".");
    
    return new Version(this._getValidInt(parts[0]),
                       this._getValidInt(parts[1]),
                       this._getValidInt(parts[2]),
                       this._getValidInt(parts[3]),
                       plus);
  },
  
  _getValidInt: function(aPartString) {
    var integer = parseInt(aPartString);
    if (isNaN(integer))
      return 0;
    return integer;
  },
  
  isValidVersion: function(aVersion) {
    return /^([0-9]+\.){0,3}[0-9]+\+?$/.test(aVersion);
  },

  /////////////////////////////////////////////////////////////////////////////
  // nsISupports
  QueryInterface: function(aIID) {
    if (!aIID.equals(Components.interfaces.nsIVersionChecker) &&
        !aIID.equals(Components.interfaces.nsISupports))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

var gModule = {
  registerSelf: function(componentManager, fileSpec, location, type) {
    componentManager = componentManager.QueryInterface(Components.interfaces.nsIComponentRegistrar);
    
    for (var key in this._objects) {
      var obj = this._objects[key];
      componentManager.registerFactoryLocation(obj.CID, obj.className, obj.contractID,
                                               fileSpec, location, type);
    }

    // Make the Update Service a startup observer
    var categoryManager = Components.classes["@mozilla.org/categorymanager;1"]
                                    .getService(Components.interfaces.nsICategoryManager);
    categoryManager.addCategoryEntry("app-startup", this._objects.service.className,
                                     "service," + this._objects.service.contractID, 
                                     true, true, null);
  },
  
  getClassObject: function(componentManager, cid, iid) {
    if (!iid.equals(Components.interfaces.nsIFactory))
      throw Components.results.NS_ERROR_NOT_IMPLEMENTED;

    for (var key in this._objects) {
      if (cid.equals(this._objects[key].CID))
        return this._objects[key].factory;
    }
    
    throw Components.results.NS_ERROR_NO_INTERFACE;
  },
  
  _makeFactory: #1= function(ctor) {
    function ci(outer, iid) {
      if (outer != null)
        throw Components.results.NS_ERROR_NO_AGGREGATION;
      return (new ctor()).QueryInterface(iid);
    } 
    return { createInstance: ci };
  },
  
  _objects: {
    service: { CID        : Components.ID("{B3C290A6-3943-4B89-8BBE-C01EB7B3B311}"),
               contractID : "@mozilla.org/updates/update-service;1",
               className  : "Update Service",
               factory    : #1#(UpdateService)
             },
    manager: { CID        : Components.ID("{093C2356-4843-4C65-8709-D7DBCBBE7DFB}"),
               contractID : "@mozilla.org/updates/update-manager;1",
               className  : "Update Manager",
               factory    : #1#(UpdateManager)
             },
    prompt:  { CID        : Components.ID("{27ABA825-35B5-4018-9FDD-F99250A0E722}"),
               contractID : "@mozilla.org/updates/update-prompt;1",
               className  : "Update Prompt",
               factory    : #1#(UpdatePrompt)
             },
    version: { CID        : Components.ID("{9408E0A5-509E-45E7-80C1-0F35B99FF7A9}"),
               contractID : "@mozilla.org/updates/version-checker;1",
               className  : "Version Checker",
               factory    : #1#(VersionChecker)
             },
    timers:  { CID        : Components.ID("{B322A5C0-A419-484E-96BA-D7182163899F}"),
               contractID : "@mozilla.org/updates/timer-manager;1",
               className  : "Timer Manager",
               factory    : #1#(TimerManager)
             }
  },
  
  canUnload: function(componentManager) {
    return true;
  }
};

function NSGetModule(compMgr, fileSpec) {
  return gModule;
}

/**
 * Determines whether or there are installed addons which are incompatible 
 * with this update.
 * @param   update
 *          The update to check compatibility against
 * @returns true if there are no addons installed that are incompatible with
 *          the specified update, false otherwise.
 */
function isCompatible(update) {
//@line 2013 "/builds/tinderbox/XR-Trunk/Linux_2.4.18-14_Depend/mozilla/toolkit/mozapps/update/src/nsUpdateService.js.in"
}

